# ##### BEGIN GPL LICENSE BLOCK #####
#
#  This program is free software; you can redistribute it and/or
#  modify it under the terms of the GNU General Public License
#  as published by the Free Software Foundation; either version 2
#  of the License, or (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software Foundation,
#  Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#
# ##### END GPL LICENSE BLOCK #####

import bpy

from enum import IntEnum
import math

from .factories import get_sets_mgr, get_sets_ids
from ..preferences import ZsOperatorOptions, get_prefs, display_element_modes, display_compact_element_modes, display_object_element_modes
from ..ico import zs_tool_icon
from ..labels import ZsLabels
from ..blender_zen_utils import ZenLocks, ZenPolls, ZsUiConstants


class ZsSelectType(IntEnum):
    Box = 0,
    Circle = 1,
    Lasso = 2,
    UV = 3


class ZsWorkSpaceToolHelper:
    @classmethod
    def do_draw_settings(cls, context, layout, tool):
        p_scene = context.scene

        p_cls_mgr = get_sets_mgr(p_scene)
        if not p_cls_mgr:
            return

        addon_prefs = get_prefs()

        if addon_prefs.common.compact_mode:
            row = layout.row(align=True)

            display_compact_element_modes(row, context, get_sets_ids())
        else:
            not_header = context.region.type != 'TOOL_HEADER'

            row_element = layout.row(align=True)
            if not_header:
                row_element = row_element.column(align=False)
            else:
                r_sync = row_element.row(align=True)
                r_sync.prop(addon_prefs.common, 'sync_with_mesh_selection', icon_only=True, icon=ZsUiConstants.ZSTS_SYNC_MODE_ICON)
                r_sync.active = not p_cls_mgr.is_uv_area_and_not_sync()
                row_element.separator()

            row_blender = layout.row(align=True)
            if not_header:
                row_blender = row_element.column(align=False)

            row_mode = layout.row(align=True)
            if not_header:
                row_mode = row_element.column(align=False)

            display_element_modes(row_element, row_blender, row_mode, context, display_mode_labels=not_header)

            p_cls_mgr._do_draw_highlight(context, layout, type='TOOL')

        if context.region.type == 'TOOL_HEADER':
            row = layout.row(align=True)
            row.alignment = 'RIGHT'
            row.label(text="Group")
            row.prop(p_scene, "zen_sets_groups", text="")
        else:
            layout.prop(p_scene, "zen_sets_groups")

    @classmethod
    def get_keymap(cls, type, mode='MESH'):
        select_km = []
        if type == ZsSelectType.Box:
            if bpy.app.version < (3, 2, 0):
                select_km = [
                    ("view3d.select_box", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": False},
                        {"properties": [("wait_for_input", False)]}),
                    ("view3d.select_box", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB'), ("wait_for_input", False)]}),
                    ("view3d.select_box", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
            else:
                select_km = [
                    ("view3d.select_box", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": False},
                        {"properties": [("wait_for_input", False)]}),
                    ("view3d.select_box", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB'), ("wait_for_input", False)]}),
                    ("view3d.select_box", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
        elif type == ZsSelectType.UV:
            if bpy.app.version < (3, 2, 0):
                select_km = [
                    ("uv.select_box", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": False},
                        {"properties": [("wait_for_input", False)]}),
                    ("uv.select_box", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB'), ("wait_for_input", False)]}),
                    ("uv.select_box", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
            else:
                select_km = [
                    ("uv.select_box", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": False},
                        {"properties": [("wait_for_input", False)]}),
                    ("uv.select_box", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB'), ("wait_for_input", False)]}),
                    ("uv.select_box", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
        elif type == ZsSelectType.Circle:
            if bpy.app.version < (3, 2, 0):
                select_km = [
                    ("view3d.select_circle", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": False},
                        {"properties": [("wait_for_input", False)]}),
                    ("view3d.select_circle", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB'), ("wait_for_input", False)]}),
                    ("view3d.select_circle", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
            else:
                select_km = [
                    ("view3d.select_circle", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": False},
                        {"properties": [("wait_for_input", False)]}),
                    ("view3d.select_circle", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB'), ("wait_for_input", False)]}),
                    ("view3d.select_circle", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
        elif type == ZsSelectType.Lasso:
            if bpy.app.version < (3, 2, 0):
                select_km = [
                    ("view3d.select_lasso", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": False},
                        {}),
                    ("view3d.select_lasso", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB')]}),
                    ("view3d.select_lasso", {"type": 'EVT_TWEAK_L', "value": 'ANY', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]
            else:
                select_km = [
                    ("view3d.select_lasso", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": False},
                        {}),
                    ("view3d.select_lasso", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": True, "shift": False},
                        {"properties": [("mode", 'SUB')]}),
                    ("view3d.select_lasso", {"type": 'LEFTMOUSE', "value": 'CLICK_DRAG', "ctrl": False, "shift": True},
                        {"properties": [("mode", 'ADD'), ("wait_for_input", False)]}),
                ]

        mesh_only_keymaps = [
            ("zsts.assign_tool_pinned", {"type": 'RIGHTMOUSE', "value": 'PRESS', "ctrl": True, "shift": True},
                {}),
        ]

        zen_keymaps = [
            ("zsts.show_only_one_group", {"type": 'LEFTMOUSE', "value": 'PRESS', "ctrl": True, "shift": True},
                {}),
            ("zsts.select_only_one_group", {"type": 'RIGHTMOUSE', "value": 'PRESS', "ctrl": True, "shift": False},
                {"properties": [("keep_active_group", False)]}),
            ("zsts.select_only_one_group", {"type": 'RIGHTMOUSE', "value": 'PRESS', "ctrl": False, "shift": True},
                {"properties": [("keep_active_group", True)]}),

            ("zsts.group_selector", {"type": 'WHEELDOWNMOUSE', "value": 'ANY', "ctrl": True},
                {}),
            ("zsts.group_selector", {"type": 'WHEELUPMOUSE', "value": 'ANY', "ctrl": True},
                {}),
            ("zsts.mode_selector", {"type": 'WHEELDOWNMOUSE', "value": 'ANY', "ctrl": True, "shift": True},
                {}),
            ("zsts.mode_selector", {"type": 'WHEELUPMOUSE', "value": 'ANY', "ctrl": True, "shift": True},
                {}),

            ("zsts.draw_show_tool_help", {"type": 'F1', "value": 'PRESS', "ctrl": False, "shift": False},
                {}),
        ]

        return tuple(select_km + zen_keymaps) if mode == 'OBJECT' else tuple(select_km + mesh_only_keymaps + zen_keymaps)


class ZsBoxWorkSpaceTool(bpy.types.WorkSpaceTool):
    bl_space_type = 'VIEW_3D'
    bl_context_mode = 'EDIT_MESH'

    # The prefix of the idname should be your add-on name.
    bl_idname = "zsts.select_box_tool"
    bl_label = ZsLabels.TOOL_LABEL
    bl_description = ZsLabels.TOOL_DESCRIPTION
    bl_icon = zs_tool_icon()
    bl_keymap = ZsWorkSpaceToolHelper.get_keymap(ZsSelectType.Box)

    def draw_settings(context, layout, tool):
        ZsWorkSpaceToolHelper.do_draw_settings(context, layout, tool)


class ZsUvWorkSpaceTool(bpy.types.WorkSpaceTool):
    bl_space_type = 'IMAGE_EDITOR'
    bl_context_mode = 'UV'

    # The prefix of the idname should be your add-on name.
    bl_idname = "zsts.select_uv_tool"
    bl_label = ZsLabels.TOOL_LABEL
    bl_description = ZsLabels.TOOL_DESCRIPTION
    bl_icon = zs_tool_icon()
    bl_keymap = ZsWorkSpaceToolHelper.get_keymap(ZsSelectType.UV)

    def draw_settings(context, layout, tool):
        if context.mode == 'EDIT_MESH':
            ZsWorkSpaceToolHelper.do_draw_settings(context, layout, tool)


class ZsObjectWorkSpaceTool(bpy.types.WorkSpaceTool):
    bl_space_type = 'VIEW_3D'
    bl_context_mode = 'OBJECT'

    # The prefix of the idname should be your add-on name.
    bl_idname = "zsts.select_object_tool"
    bl_label = ZsLabels.TOOL_LABEL
    bl_description = ZsLabels.TOOL_DESCRIPTION
    bl_icon = zs_tool_icon()
    bl_keymap = ZsWorkSpaceToolHelper.get_keymap(ZsSelectType.Box, mode='OBJECT')

    def draw_settings(context, layout, tool):
        p_scene = context.scene

        is_tool_header = context.region.type == 'TOOL_HEADER'

        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr:

            row_mode = layout.row(align=True)
            if not is_tool_header:
                row_mode = layout.column(align=False)

            display_object_element_modes(row_mode, context, display_mode_labels=(not is_tool_header))

            p_cls_mgr._do_draw_highlight(context, layout, type='TOOL')

            col_toolbar_row = layout
            if not is_tool_header:
                row = layout.row(align=True)
                row_s = row.split(factor=0.4)
                row_s.separator()
                col_toolbar_row = row_s

            p_cls_mgr._do_draw_collection_toolbar(
                context, col_toolbar_row, is_tool_header)

        if is_tool_header:
            row = layout.row(align=False)
            row.alignment = 'RIGHT'
            row.label(text="Group")
            row.prop(p_scene, "zen_sets_groups", text='')
        else:
            layout.prop(p_scene, "zen_sets_groups")


class ZsCircleWorkSpaceTool(bpy.types.WorkSpaceTool):
    bl_space_type = 'VIEW_3D'
    bl_context_mode = 'EDIT_MESH'

    # The prefix of the idname should be your add-on name.
    bl_idname = "zsts.select_circle_tool"
    bl_label = "Zen Sets Circle Select"
    bl_description = (
        "Zen Sets Circle Select"
    )
    bl_icon = "ops.generic.select_circle"
    bl_keymap = ZsWorkSpaceToolHelper.get_keymap(ZsSelectType.Circle)

    def draw_settings(context, layout, tool):
        ZsWorkSpaceToolHelper.do_draw_settings(context, layout, tool)


class ZsLassoWorkSpaceTool(bpy.types.WorkSpaceTool):
    bl_space_type = 'VIEW_3D'
    bl_context_mode = 'EDIT_MESH'

    # The prefix of the idname should be your add-on name.
    bl_idname = "zsts.select_lasso_tool"
    bl_label = "Zen Sets Lasso Select"
    bl_description = (
        "Zen Sets Lasso Select"
    )
    bl_icon = "ops.generic.select_lasso"
    bl_keymap = ZsWorkSpaceToolHelper.get_keymap(ZsSelectType.Lasso)

    def draw_settings(context, layout, tool):
        ZsWorkSpaceToolHelper.do_draw_settings(context, layout, tool)


_flag_not_skip_empty = False


class ZSTS_OT_GroupSelector(bpy.types.Operator):
    """ Select group by wheel """
    bl_idname = 'zsts.group_selector'
    bl_label = ZsLabels.OT_TOOL_GROUP_SELECTOR
    bl_description = ZsLabels.OT_TOOL_GROUP_DESCRIPTION
    bl_options = {'REGISTER', 'UNDO'}

    def _get_skip_empty(self):
        addon_prefs = get_prefs()
        return addon_prefs.group_selector_options.skip_empty

    def _set_skip_empty(self, value):
        addon_prefs = get_prefs()
        addon_prefs.group_selector_options.skip_empty = value

    def _get_select(self):
        addon_prefs = get_prefs()
        return addon_prefs.group_selector_options.select

    def _set_select(self, value):
        addon_prefs = get_prefs()
        addon_prefs.group_selector_options.select = value

    def _get_frame_selected(self):
        addon_prefs = get_prefs()
        return addon_prefs.group_selector_options.frame_selected

    def _set_frame_selected(self, value):
        addon_prefs = get_prefs()
        addon_prefs.group_selector_options.frame_selected = value

    def _get_mode(self):
        addon_prefs = get_prefs()
        enums = addon_prefs.group_selector_options.bl_rna.properties['mode'].enum_items
        return enums.find(addon_prefs.group_selector_options.mode)

    def _set_mode(self, value):
        addon_prefs = get_prefs()
        enums = addon_prefs.group_selector_options.bl_rna.properties['mode'].enum_items
        was_mode = addon_prefs.group_selector_options.mode
        new_mode = enums[value].identifier
        if was_mode != new_mode:
            addon_prefs.group_selector_options.mode = new_mode
            global _flag_not_skip_empty
            if new_mode == 'ISOLATE':
                if not self.skip_empty:
                    self.skip_empty = True
                    _flag_not_skip_empty = True
            else:
                # restore previous settings
                if _flag_not_skip_empty:
                    _flag_not_skip_empty = False
                    if self.skip_empty:
                        self.skip_empty = False

    mode: bpy.props.EnumProperty(
        name='Mode',
        items=[
            ('ALL', 'All', 'All Groups'),
            ('VISIBLE', 'Visible', 'Visible Groups Only'),
            ('ISOLATE', 'Isolate', 'Isolate Selected Group')
        ],
        get=_get_mode,
        set=_set_mode
    )

    skip_empty: bpy.props.BoolProperty(
        name='Skip Empty Groups',
        description='Skip Groups that do not contain elements',
        get=_get_skip_empty,
        set=_set_skip_empty)

    select: bpy.props.BoolProperty(
        name='Select',
        get=_get_select,
        set=_set_select)

    frame_selected: bpy.props.BoolProperty(
        name='Frame Selected',
        description='Move the view to the selection center',
        get=_get_frame_selected,
        set=_set_frame_selected)

    new_index: bpy.props.IntProperty(
        name='New Index',
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    def draw(self, context):
        layout = self.layout

        row = layout.row(align=True)
        row.prop(self, 'mode', expand=True)

        box = layout.box()
        col = box.column(align=True)
        col.prop(self, 'skip_empty')
        col.prop(self, 'select')
        subrow = col.row(align=True)
        subrow.active = self.select
        subrow.prop(self, 'frame_selected')

    @classmethod
    def poll(cls, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if len(p_cls_mgr.get_current_group_pairs(context)):
                return True
        return False

    def is_interrupted(self, p_cls_mgr, context: bpy.types.Context, p_group_pair):
        interrupt_condition = True
        idx, p_group = p_group_pair
        is_collection_mode = ZenPolls.is_object_and_collection_mode(context)

        n_mesh_group_count = None
        n_mesh_group_hide_count = None

        if self.mode == 'VISIBLE':
            if is_collection_mode:
                lay_col = p_cls_mgr.get_internal_layer_collection(context, idx)
                if not lay_col or not lay_col.is_visible:
                    interrupt_condition = False
            else:
                t_info = p_cls_mgr.get_lookup_extra_info_by_index(context, idx)
                n_mesh_group_count = t_info.get('count', 0)
                n_mesh_group_hide_count = t_info.get('hide_count', 0)
                interrupt_condition = n_mesh_group_hide_count == 0

        if interrupt_condition and self.skip_empty:
            if is_collection_mode:
                interrupt_condition = p_group.group_count != 0
            else:
                if n_mesh_group_count is None:
                    t_info = p_cls_mgr.get_lookup_extra_info_by_index(context, idx)
                    n_mesh_group_count = t_info.get('count', 0)
                interrupt_condition = n_mesh_group_count != 0

        return interrupt_condition

    def invoke(self, context, event):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            p_group_pairs = p_cls_mgr.get_current_group_pairs(context)
            i_groups_count = len(p_group_pairs)
            if i_groups_count > 0:
                i_old_index = p_cls_mgr.get_current_group_pair_index(context)
                i_new_index = i_old_index
                if event.type in {'WHEELDOWNMOUSE'}:
                    for i in range(len(p_group_pairs)):
                        i_new_index += 1
                        if i_new_index > i_groups_count - 1:
                            i_new_index = 0

                        if self.is_interrupted(p_cls_mgr, context, p_group_pairs[i_new_index]):
                            break
                elif event.type in {'WHEELUPMOUSE'}:
                    for i in range(len(p_group_pairs)):
                        i_new_index -= 1
                        if i_new_index < 0:
                            i_new_index = i_groups_count - 1

                        if self.is_interrupted(p_cls_mgr, context, p_group_pairs[i_new_index]):
                            break

                self.new_index = i_new_index

                return self.execute(context)

        return {'CANCELLED'}

    def execute(self, context):
        p_scene = context.scene
        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr:
            p_group_pairs = p_cls_mgr.get_current_group_pairs(context)
            i_groups_count = len(p_group_pairs)
            if i_groups_count > 0:
                i_old_index = p_cls_mgr.get_current_group_pair_index(context)
                if self.new_index != i_old_index:
                    if self.mode == 'ISOLATE':
                        p_group_pair = p_group_pairs[self.new_index]
                        self.nested = False
                        p_cls_mgr.set_group_pair_invert_hide(context, p_group_pair, self)
                        p_cls_mgr.set_current_group_pair_index(context, self.new_index)
                    else:
                        p_cls_mgr.set_current_group_pair_index(context, self.new_index)
                        if self.select:
                            self.nested = False
                            p_cls_mgr.execute_SelectOnlyGroup(self, context)

                    if self.select and self.frame_selected:
                        if bpy.ops.view3d.view_selected.poll():
                            bpy.ops.view3d.view_selected('INVOKE_DEFAULT')
                return {'FINISHED'}
        return {'CANCELLED'}


class ZSTS_OT_ModeSelector(bpy.types.Operator):
    """ Select mode by wheel """
    bl_idname = 'zsts.mode_selector'
    bl_description = 'Mode Selector'
    bl_label = 'Mode Selector'
    bl_options = {'REGISTER', 'UNDO'}

    event_type: bpy.props.StringProperty(
        name='Mouse Wheel Event Type',
        description='WHEELDOWNMOUSE | WHEELUPMOUSE',
        options={'HIDDEN', 'SKIP_SAVE'},
        default='')

    def execute(self, context):
        if self.event_type != '':
            p_scene = context.scene

            if context.mode == 'EDIT_MESH':
                addon_prefs = get_prefs()

                modes = tuple(id for id in get_sets_ids() if getattr(addon_prefs.modes, f'enable_{id}'))

                if len(modes):
                    i_old_index = 0
                    for i, id in enumerate(modes):
                        if id == p_scene.zen_sets_active_mode:
                            i_old_index = i
                            break

                    i_new_index = i_old_index
                    if self.event_type in {'WHEELDOWNMOUSE'}:
                        i_new_index = i_old_index + 1
                        if i_new_index > len(modes) - 1:
                            i_new_index = 0
                    elif self.event_type in {'WHEELUPMOUSE'}:
                        i_new_index = i_old_index - 1
                        if i_new_index < 0:
                            i_new_index = len(modes) - 1

                    if i_new_index != -1 and i_new_index != i_old_index:
                        p_scene.zen_sets_active_mode = modes[i_new_index]
                        return {'FINISHED'}

            elif context.mode == 'OBJECT':
                if not ZenPolls.is_collection_mode(context):
                    return {'CANCELLED'}

                p_cls_mgr = get_sets_mgr(context.scene)
                if p_cls_mgr:
                    p_list = p_cls_mgr.get_list(p_scene)
                    i_groups_count = len(p_list)
                    if i_groups_count > 1 and len(context.selected_objects) > 0:
                        i_old_index = p_cls_mgr.get_list_index(p_scene)
                        i_new_index = i_old_index
                        if self.event_type in {'WHEELDOWNMOUSE'}:
                            for i in range(len(p_list)):
                                i_new_index += 1
                                if i_new_index > i_groups_count - 1:
                                    i_new_index = 0
                                if set(p_list[i_new_index].collection.all_objects).intersection(context.selected_objects):
                                    lay_col = p_cls_mgr.get_internal_layer_collection(context, i_new_index)
                                    if lay_col and not lay_col.exclude:
                                        context.view_layer.active_layer_collection = lay_col
                                        return {'FINISHED'}

                        elif self.event_type in {'WHEELUPMOUSE'}:
                            for i in range(len(p_list)):
                                i_new_index -= 1
                                if i_new_index < 0:
                                    i_new_index = i_groups_count - 1
                                if set(p_list[i_new_index].collection.all_objects).intersection(context.selected_objects):
                                    lay_col = p_cls_mgr.get_internal_layer_collection(context, i_new_index)
                                    if lay_col and not lay_col.exclude:
                                        context.view_layer.active_layer_collection = lay_col
                                        return {'FINISHED'}
        return {'CANCELLED'}

    def invoke(self, context, event):
        self.event_type = event.type

        return self.execute(context)


def _get_show_hide_select(self):
    return get_prefs().op_options.show_hide_group_select


def _set_show_hide_select(self, value):
    get_prefs().op_options.show_hide_group_select = value


class ZSTS_OT_ShowHideGroup(bpy.types.Operator):
    """ Show one group """
    bl_idname = 'zsts.show_only_one_group'
    bl_description = ZsLabels.OT_TOOL_SHOW_HIDE_GROUP_DESC
    bl_label = ZsLabels.OT_TOOL_SHOW_HIDE_GROUP_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    x: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )
    y: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    region_x: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )
    region_y: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    mode: bpy.props.EnumProperty(
        items=[
            ('CLICK', 'Click', ''),
            ('DRAG', 'Drag', ''),
        ],
        default='CLICK',
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    submode: bpy.props.EnumProperty(
        name='Mode:',
        items=[
            ('DEFAULT', 'Default', ''),
            ('UNHIDE_ALL', 'Unhide All', ''),
            ('SMART_SELECT', 'Smart Select', ''),
            ('SMART_EXTEND', 'Smart Extend', ''),
            ('INVERT', 'Hide Invert', ''),
            ('NOTHING_INVERT', 'Nothing to Invert', ''),
        ],
        default='DEFAULT',
        options={'SKIP_SAVE'}
    )

    select: bpy.props.BoolProperty(
        name='Select',
        get=_get_show_hide_select,
        set=_set_show_hide_select
    )

    @classmethod
    def poll(cls, context):
        return ZenPolls.poll_object_edit_or_uv_with_objs_in_mode(context)

    def draw(self, context):
        layout = self.layout
        layout.label(text=layout.enum_item_name(self, 'submode', self.submode))
        if self.submode in {'UNHIDE_ALL', 'SMART_EXTEND', 'INVERT'}:
            layout.prop(self, 'select', expand=True)

    def do_click(self, context: bpy.context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:

            is_mesh_mode = context.mode == 'EDIT_MESH'
            is_object_mode = context.mode == 'OBJECT'

            b_is_uv_area = p_cls_mgr.is_uv_area()
            b_is_uv_not_sync = p_cls_mgr.check_uv_not_sync()

            if b_is_uv_area:
                p_cls_mgr.check_uv_select_mode()

                x_uv, y_uv = context.region.view2d.region_to_view(self.region_x, self.region_y)

                bpy.ops.uv.select(extend=False, deselect_all=True, location=(x_uv, y_uv))
            else:
                bpy.ops.view3d.select(
                    toggle=False, deselect_all=True, location=(self.region_x, self.region_y))

            addon_prefs = get_prefs()
            b_sync_collection_hidden_state = addon_prefs.object_options.hide_collection_with_objects

            sel_count = p_cls_mgr.get_context_selected_count(context)
            if sel_count:
                if p_cls_mgr.is_anyone_hidden(context):
                    self.submode = 'SMART_EXTEND'
                    if is_mesh_mode:
                        selection_stats = p_cls_mgr.smart_select(context, True, True)
                        i_uv_sel = selection_stats.new_uv_sel_count
                        if b_is_uv_not_sync:
                            if i_uv_sel != 0:
                                bpy.ops.uv.hide(unselected=False)
                                if self.select:
                                    bpy.ops.uv.select_all(action='SELECT')
                            else:
                                bpy.ops.uv.reveal(select=self.select)
                                self.submode = 'UNHIDE_ALL'
                        else:
                            bpy.ops.mesh.hide(unselected=False)
                            if b_is_uv_area:
                                if self.select:
                                    bpy.ops.uv.select_all(action='SELECT')
                            ZenLocks.unlock_depsgraph_update_one()
                    elif is_object_mode:
                        p_cls_mgr.smart_select(context, True, True)
                        bpy.ops.object.hide_view_set(unselected=False)

                        if b_sync_collection_hidden_state:
                            p_cls_mgr.finalize_hide_state(context)
                else:
                    self.submode = 'SMART_SELECT'
                    if is_mesh_mode:
                        selection_stats = p_cls_mgr.smart_select(context, True, True)
                        i_uv_sel = selection_stats.new_uv_sel_count
                        if b_is_uv_not_sync:
                            if i_uv_sel != 0:
                                bpy.ops.uv.hide(unselected=False)
                                i_count = p_cls_mgr.get_context_selected_count(context)
                                if i_count:
                                    p_cls_mgr.invert_context_uv_selection(context)
                                    bpy.ops.uv.select_all(action='SELECT')
                                else:
                                    bpy.ops.uv.reveal(select=self.select)
                                    self.submode = 'UNHIDE_ALL'
                            else:
                                bpy.ops.uv.reveal(select=self.select)
                                self.submode = 'UNHIDE_ALL'
                        else:
                            bpy.ops.mesh.hide(unselected=True)
                            if b_is_uv_area:
                                bpy.ops.uv.select_all(action='SELECT')
                    elif is_object_mode:
                        p_cls_mgr.smart_select(context, True, False)
                        bpy.ops.object.hide_view_set(unselected=True)

                        if b_sync_collection_hidden_state:
                            p_cls_mgr.finalize_hide_state(context)
            else:
                self.submode = 'UNHIDE_ALL'
                if is_mesh_mode:
                    if b_is_uv_not_sync:
                        bpy.ops.uv.reveal(select=self.select)
                    else:
                        bpy.ops.mesh.reveal(select=self.select)
                        if b_is_uv_area and self.select:
                            bpy.ops.uv.select_all(action='SELECT')
                        ZenLocks.unlock_depsgraph_update_one()
                elif is_object_mode:
                    p_cls_mgr.unhide_all(context, self.select, self)

    def do_drag(self, context):
        self.submode = 'NOTHING_INVERT'
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:
            if p_cls_mgr.is_anyone_hidden(context):
                self.submode = 'INVERT'
                if context.mode == 'EDIT_MESH':
                    b_is_uv_not_sync = p_cls_mgr.check_uv_not_sync()

                    if b_is_uv_not_sync:
                        bpy.ops.mesh.select_all(action='INVERT')
                    else:
                        bpy.ops.mesh.select_all(action='SELECT')
                        bpy.ops.mesh.reveal(select=False)
                        bpy.ops.mesh.hide(unselected=False)

                    if b_is_uv_not_sync:
                        bpy.ops.uv.select_all(action=('SELECT' if self.select else 'DESELECT'))
                    else:
                        if self.select:
                            bpy.ops.mesh.select_all(action='SELECT')
                elif context.mode == 'OBJECT':
                    was_visible = set(context.visible_objects)

                    if ZenPolls.is_collection_mode(context):
                        for idx, _ in p_cls_mgr.get_current_group_pairs(context):
                            lay_col = p_cls_mgr.get_internal_layer_collection(context, idx)
                            if lay_col and not lay_col.exclude:
                                if lay_col.hide_viewport:
                                    lay_col.hide_viewport = False

                    for p_obj in context.view_layer.objects:
                        try:
                            p_obj.hide_set(p_obj in was_visible)
                        except Exception:
                            pass

                    addon_prefs = get_prefs()
                    b_sync_collection_hidden_state = addon_prefs.object_options.hide_collection_with_objects
                    if b_sync_collection_hidden_state:
                        p_cls_mgr.finalize_hide_state(context)

                    if self.select:
                        bpy.ops.object.select_all(action='SELECT')

    def modal(self, context, event):

        diff_x = event.mouse_x - self.x
        diff_y = event.mouse_y - self.y

        dist = math.hypot(diff_x, diff_y)

        MAX_DRAG_DISTANCE = 8

        try:
            if event.type == 'LEFTMOUSE':
                if event.value == 'PRESS':
                    pass
                elif event.value == 'RELEASE':
                    self.region_x = event.mouse_region_x
                    self.region_y = event.mouse_region_y
                    self.mode = 'CLICK'
                    return self.execute(context)
            elif event.type in {'ESC', 'RIGHTMOUSE'}:
                return {'CANCELLED'}
            elif event.type == 'MOUSEMOVE':
                if dist > MAX_DRAG_DISTANCE:
                    self.mode = 'DRAG'
                    return self.execute(context)
        except Exception as e:
            self.report({'WARNING'}, str(e))
            return {'CANCELLED'}

        return {'PASS_THROUGH'}

    def invoke(self, context, event):
        self.x = event.mouse_x
        self.y = event.mouse_y

        context.window_manager.modal_handler_add(self)

        return {'RUNNING_MODAL'}

    def execute(self, context):
        try:
            self.submode = 'DEFAULT'
            if self.mode == 'CLICK':
                self.do_click(context)
            elif self.mode == 'DRAG':
                self.do_drag(context)
            else:
                return {'CANCELLED'}
        except Exception as e:
            self.report({'WARNING'}, str(e))
            return {'CANCELLED'}
        return {'FINISHED'}


class ZSTS_OT_SelectOnlyOneGroup(bpy.types.Operator):
    """ Select one group """
    bl_idname = 'zsts.select_only_one_group'
    bl_description = ZsLabels.OT_TOOL_SMART_SELECT_DESC
    bl_label = ZsLabels.OT_TOOL_SMART_SELECT_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    region_x: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )
    region_y: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    # Do not remember in Prefs !!!
    keep_active_group: bpy.props.BoolProperty(
        name=ZsLabels.PROP_KEEP_ACTIVE_GROUP_NAME,
        description=ZsLabels.PROP_KEEP_ACTIVE_GROUP_DESC,
        default=True
    )

    @classmethod
    def poll(cls, context):
        return ZenPolls.poll_object_edit_or_uv_with_objs_in_mode(context)

    def invoke(self, context, event):
        self.region_x = event.mouse_region_x
        self.region_y = event.mouse_region_y
        return self.execute(context)

    def execute(self, context):
        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr:

            is_object_mode = context.mode == 'OBJECT'

            try:
                if p_cls_mgr.is_uv_area():

                    b_is_not_sync = p_cls_mgr.check_uv_not_sync()

                    p_cls_mgr.check_uv_select_mode()

                    x_uv, y_uv = context.region.view2d.region_to_view(self.region_x, self.region_y)

                    if ZenPolls.version_greater_3_2_0:
                        bpy.ops.uv.select(
                            extend=self.keep_active_group,
                            toggle=False, deselect_all=True, location=(x_uv, y_uv))
                    else:
                        was_selection = {}
                        if self.keep_active_group:
                            for p_obj in context.objects_in_mode_unique_data:
                                if p_obj.type != 'MESH':
                                    continue

                                bm = p_cls_mgr._get_bm(p_obj)
                                bm.faces.ensure_lookup_table()
                                uv_layer = bm.loops.layers.uv.active
                                if uv_layer:
                                    selected_items = []
                                    if b_is_not_sync:
                                        selected_items = [
                                            loop
                                            for f in bm.faces for loop in f.loops
                                            if not f.hide and loop[uv_layer].select
                                        ]
                                    else:
                                        selected_items = [
                                            item
                                            for item in p_cls_mgr.get_bm_items(bm)
                                            if not item.hide and item.select
                                        ]
                                    was_selection[p_obj] = (bm, uv_layer, selected_items)

                        bpy.ops.uv.select(extend=False, deselect_all=True, location=(x_uv, y_uv))

                        if self.keep_active_group:

                            is_any_selected = False
                            for p_obj, v in was_selection.items():
                                bm = v[0]
                                uv_layer = v[1]
                                if b_is_not_sync:
                                    is_any_selected = any(loop[uv_layer].select for f in bm.faces for loop in f.loops if not f.hide)
                                else:
                                    is_any_selected = p_cls_mgr.get_selected_count(p_obj)
                                if is_any_selected:
                                    break

                            if is_any_selected:
                                if b_is_not_sync:
                                    for v in was_selection.values():
                                        for loop in v[2]:
                                            loop[v[1]].select = True
                                else:
                                    for v in was_selection.values():
                                        for item in v[2]:
                                            item.select = True
                                        bm = v[0]
                                        bm.select_flush_mode()

                else:
                    if not is_object_mode:
                        p_cls_mgr.set_mesh_select_mode(context)
                    bpy.ops.view3d.select(
                        extend=self.keep_active_group,
                        toggle=False, deselect_all=True, location=(self.region_x, self.region_y))

                sel_count = p_cls_mgr.get_context_selected_count(context)
                if sel_count:
                    if context.active_object:
                        p_cls_mgr.smart_select(context, False, self.keep_active_group)
                else:
                    if is_object_mode:
                        bpy.ops.object.select_all(action='DESELECT')
                    else:
                        bpy.ops.mesh.select_all(action='DESELECT')
            except Exception as e:
                self.report({'WARNING'}, str(e))
                return {'CANCELLED'}
        return {'FINISHED'}


def _get_tool_assign_pinned_clear_select(self):
    return get_prefs().op_options.tool_assign_pinned_clear_select


def _set_tool_assign_pinned_clear_select(self, value):
    get_prefs().op_options.tool_assign_pinned_clear_select = value


class ZSTS_OT_AssignToolPinnedGroup(bpy.types.Operator):
    """ Assign to pinned group in Tool Mode """
    bl_idname = 'zsts.assign_tool_pinned'
    bl_description = ZsLabels.OT_ASSIGN_ITEM_PINNED_DESC
    bl_label = ZsLabels.OT_ASSIGN_ITEM_PINNED_LABEL
    bl_options = {'REGISTER', 'UNDO'}

    region_x: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )
    region_y: bpy.props.IntProperty(
        options={'HIDDEN', 'SKIP_SAVE'}
    )

    clear_selection: bpy.props.BoolProperty(
        name="Clear Selection",
        get=_get_tool_assign_pinned_clear_select,
        set=_set_tool_assign_pinned_clear_select
    )

    group_name: ZsOperatorOptions.get_tool_assign_pinned_group_name()
    group_color: ZsOperatorOptions.get_tool_assign_pinned_group_color(options={'HIDDEN', 'SKIP_SAVE'})

    @classmethod
    def poll(cls, context):
        if not ZenPolls.poll_edit_or_uv_with_objs_in_mode(context):
            return False

        p_cls_mgr = get_sets_mgr(context.scene)
        if p_cls_mgr and not p_cls_mgr.is_unique:
            return True

        return False

    def invoke(self, context, event):
        self.region_x = event.mouse_region_x
        self.region_y = event.mouse_region_y
        return self.execute(context)

    def execute(self, context):
        p_scene = context.scene
        p_cls_mgr = get_sets_mgr(p_scene)
        if p_cls_mgr:
            try:
                if self.clear_selection:
                    if p_cls_mgr.is_uv_area():
                        p_cls_mgr.check_uv_not_sync()

                        p_cls_mgr.check_uv_select_mode()

                        x_uv, y_uv = context.region.view2d.region_to_view(self.region_x, self.region_y)

                        bpy.ops.uv.select(extend=False, deselect_all=True, location=(x_uv, y_uv))
                    else:
                        bpy.ops.view3d.select(
                            extend=False,
                            toggle=False, deselect_all=True, location=(self.region_x, self.region_y))

                res = p_cls_mgr.execute_AssignToGroup(self, context)
                p_list = p_cls_mgr.get_list(p_scene)
                idx = p_cls_mgr._index_of_group_name(p_list, self.group_name)
                if idx != -1:
                    t_info = p_cls_mgr.get_lookup_extra_info_by_index(context, idx)
                    n_group_count = t_info.get('count', 0)
                    if n_group_count == 0:
                        self.report({'WARNING'}, 'Nothing Selected to Assign!')

                return res
            except Exception as e:
                self.report({'WARNING'}, str(e))
        return {'CANCELLED'}


class ZsToolHelpGizmoButton:

    def draw_prepare(self, context):
        ui_scale = context.preferences.view.ui_scale
        widget_size = 32 * ui_scale

        n_panel_width = [region.width for region in context.area.regions if region.type == "UI"][0]
        base_position = context.region.width - n_panel_width - widget_size
        self.foo_gizmo.matrix_basis[0][3] = base_position

        self.foo_gizmo.matrix_basis[1][3] = context.region.height * 5 * 0.01

    def setup(self, context):
        mpr = self.gizmos.new("GIZMO_GT_button_2d")
        mpr.show_drag = False
        mpr.icon = 'HELP'
        mpr.draw_options = {'BACKDROP', 'OUTLINE'}

        mpr.color = 0.0, 0.0, 0.0
        mpr.alpha = 0.5
        mpr.color_highlight = 0.8, 0.8, 0.8
        mpr.alpha_highlight = 0.2

        mpr.scale_basis = (80 * 0.35) / 2  # Same as buttons defined in C
        _ = mpr.target_set_operator("zsts.draw_show_tool_help")
        self.foo_gizmo = mpr


class ZSTS_UI_ToolHelpGizmoButton(bpy.types.GizmoGroup, ZsToolHelpGizmoButton):
    bl_idname = "ZSTS_UI_ToolHelpGizmoButton"
    bl_label = "Zen Sets Tool Help Button"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'WINDOW'
    bl_options = {'PERSISTENT', 'SCALE'}

    @classmethod
    def poll(cls, context: bpy.types.Context):
        s_mode = context.mode
        if s_mode in {'EDIT_MESH', 'OBJECT'}:
            _id = getattr(context.workspace.tools.from_space_view3d_mode(s_mode, create=False), 'idname', None)
            if isinstance(_id, str) and _id.startswith('zsts.select_'):
                return True
        return False


class ZSTS_UI_UvToolHelpGizmoButton(bpy.types.GizmoGroup, ZsToolHelpGizmoButton):
    bl_idname = "ZSTS_UI_UvToolHelpGizmoButton"
    bl_label = "Zen Sets Tool Help Button"
    bl_space_type = 'IMAGE_EDITOR'
    bl_region_type = 'WINDOW'
    bl_options = {'PERSISTENT', 'SCALE'}

    @classmethod
    def poll(cls, context: bpy.types.Context):
        s_mode = context.mode
        if s_mode in {'EDIT_MESH'}:
            _id = getattr(context.workspace.tools.from_space_image_mode('UV', create=False), 'idname', None)
            if isinstance(_id, str) and _id.startswith('zsts.select_'):
                return True
        return False


class ZsToolFactory:
    classes = (
        ZSTS_OT_GroupSelector,
        ZSTS_OT_ModeSelector,
        ZSTS_OT_ShowHideGroup,
        ZSTS_OT_SelectOnlyOneGroup,
        ZSTS_OT_AssignToolPinnedGroup,
        ZSTS_UI_ToolHelpGizmoButton,
        ZSTS_UI_UvToolHelpGizmoButton
    )
