import bpy
import bmesh
from math import radians
from mathutils import Vector
from ..utils import vc_processor as vc
from ..ui.main_pie_v_01 import ZUV_MT_Main_Pie
from ..utils import get_uv_islands as island_util
from ..utils import checker_setup
from ..ico import icon_get
from ..utils.messages import zen_message
from bpy.props import EnumProperty


def get_mesh_data():
    "Return me as obj.data and bm as bmesh.from_edit_mesh(me)"
    obj = bpy.context.edit_object
    # me = obj.data
    return obj.data, bmesh.from_edit_mesh(obj.data)


def create_uv_material_old(obj):
    mat_name = "Zen_UV_Checker"
    zen_texture_node_name = "ZenUV_Texture_node"
    zen_image_name = "ZenUV_Texture"

    def get_img_reference(suffix):
        images = bpy.data.images
        for img in images:
            if suffix in img.name:
                return img

    def get_node_reference(nodes, suffix):
        for node in nodes:
            if suffix in node.name:
                return node

    def create_image_node(nodes_collecton, node_name):
        node = nodes_collecton.new(type='ShaderNodeTexImage')
        node.name = node_name
        return node

    # Create material
    mat = (bpy.data.materials.get(mat_name) or
           bpy.data.materials.new(mat_name))

    # Enable 'Use nodes'
    mat.use_nodes = True
    nodes = mat.node_tree.nodes

    # UV SamplerImage Creation
    zen_uv_image = bpy.data.images.get(zen_image_name)

    if not zen_uv_image:
        # print('CreateNewTexture:', zen_uv_image)
        zen_uv_image = bpy.ops.image.new(name=zen_image_name,
                                        width=1024,
                                        height=1024,
                                        alpha=False,
                                        generated_type='UV_GRID',
                                        float=False,
                                        use_stereo_3d=False,
                                        tiled=False)

    # Texture Node creation
    uv_texture_node = (nodes.get(zen_texture_node_name) or
                       create_image_node(nodes, zen_texture_node_name))

    # Set image reference to image node
    zen_uv_image = get_img_reference(zen_image_name)
    if zen_uv_image:
        uv_texture_node.image = zen_uv_image

    # Set all to current object
    obj.active_material = mat


def set_vcolor_to_display():
    for area in bpy.context.screen.areas:
        if area.type == 'VIEW_3D':
            for space in area.spaces:
                if space.type == 'VIEW_3D':
                    space.shading.color_type = 'VERTEX'
                    # print('Shading is set to VERTEX')
                    return True
    return False


def set_texture_to_display():
    for area in bpy.context.screen.areas:
        if area.type == 'VIEW_3D':
            for space in area.spaces:
                if space.type == 'VIEW_3D':
                    space.shading.color_type = 'TEXTURE'
                    # print('Shading is set to VERTEX')
                    return True
    return False


def set_fit_UV_to_view():
    for area in bpy.context.screen.areas:
        # print(area.type)
        if area.type == 'IMAGE_EDITOR':
            for space in area.spaces:
                if space.type == 'IMAGE_EDITOR':
                    pass
                    # print("\n\n\n")
                    # print(space.type)
                    # space.shading.color_type = 'VERTEX'
                    # print('Fit UV Here')
                    # bpy.context.scene.tool_settings.use_uv_select_sync
                    # bpy.ops.image.view_all(fit_view=True)
                    return True
    return False


def select_seam_edges():
    obj = bpy.context.edit_object
    me = obj.data

    bm = bmesh.from_edit_mesh(me)
    for e in bm.edges:
        if e.seam:
            e.select = True

    bmesh.update_edit_mesh(me, False)


def switch_mesh_to_smooth_mode():
    obj = bpy.context.edit_object
    me = obj.data

    bm = bmesh.from_edit_mesh(me)
    for f in bm.faces:
        if not f.smooth:
            #e.select = True
            f.smooth = True

    bmesh.update_edit_mesh(me, False)


def set_seams_as_sharp(bm):
    for edge in bm.edges:
        if edge.seam:
            edge.smooth = False
        elif not edge.seam:
            edge.smooth = True


def check_selection():
    me, bm = get_mesh_data()
    return len([f for f in bm.faces if f.select]) > 0 or len([e for e in bm.edges if e.select]) > 0

def call_from_zen_check():
    return ZUV_OT_Mark_Seams.call_from_zen

def set_auto_seams(sharp_edges, not_sharp_edges): # context, bm, me, angle):
    # bpy.ops.mesh.edges_select_sharp(sharpness=(radians(angle)))
    # print("Radians", radians(angle), "Degrees", angle)
    for edge in sharp_edges:
        edge.seam = True
        edge.smooth = False
    for edge in not_sharp_edges:
        edge.seam = False
        edge.smooth = True


    

def zen_pack_islands():
        uv_select_sync = bpy.context.scene.tool_settings.use_uv_select_sync
        # Pack UV Islands after any isolate action for better visual perception
        bpy.context.scene.tool_settings.use_uv_select_sync = True
        bpy.ops.mesh.select_all(action='SELECT')
        bpy.ops.uv.average_islands_scale()
        bpy.ops.uv.pack_islands()
        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.context.scene.tool_settings.use_uv_select_sync = uv_select_sync

def fit_uv_view_to_selected(context, mode="selected"):
    addon_prefs = context.preferences.addons["ZenUV"].preferences
    if addon_prefs.autoFitUV:
        for window in bpy.context.window_manager.windows:
            screen = window.screen
            for area in screen.areas:
                if area.type == 'IMAGE_EDITOR':
                    # print(area.type)
                    override = {'window': window, 'screen': screen, 'area': area}
                    if mode == "selected":  # mode - True meant fit to selected
                        bpy.ops.image.view_selected(override)
                    elif mode == "all": # mode - False meant fit to All
                        bpy.ops.image.view_all(override, fit_view=True)
                    elif mode == "checker":
                        bpy.ops.image.view_zoom_ratio(override, ratio=0.5)
                    break


class ZUV_OT_ZenUnwrap(bpy.types.Operator):
    bl_idname = "uv.zenuv_unwrap"
    bl_label = "Zen Unwrap"
    bl_description = """Unwrap by Marked edges and Pack after. Also, you can select edges/faces and do Zen Unwrap.
All selected edges/faces will be Marked as Seams and/or Sharp edges and unwrapped after"""

    action: EnumProperty(
        items=[
            ("NONE", "Zen Unwrap", "Default Zen Unwrap Mode."),
            ("AUTO", "Auto Seams Mode", "Perform Auto Seams Before Zen Unwrap.")]
    )

    def invoke(self, context, event):
        me, bm = get_mesh_data()
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        if True in [x.select for x in bm.edges] or True in [x.select for x in bm.faces]:
            return self.execute(context)
        else:
            if addon_prefs.workOnSelected:
                return context.window_manager.invoke_props_dialog(self)
            return self.execute(context)

    def draw(self, context):
        layout = self.layout
        layout.label(text="Nothing is selected. The entire object will be unwrapped.")
        layout.separator()

    def execute(self, context):
        me, bm = get_mesh_data()
        addon_prefs = context.preferences.addons["ZenUV"].preferences

        if self.action == "AUTO":
            bpy.ops.uv.zenuv_auto_seams()

        if addon_prefs.autoSeamsWithUnwrap:
            # Operator Zen Mark Seams is poll with selection. Howewer I need to call it from Zen Unwrap in no selection mode.
            ZUV_OT_Mark_Seams.call_from_zen = True
            bpy.ops.uv.zenuv_mark_seams()
            ZUV_OT_Mark_Seams.call_from_zen = False

        selfaces = []
        selected_edges = []
        work_mode = None
        # seams_edges = []

        # If user want to work on selected only:

        if addon_prefs.workOnSelected:
            # Face selection mode
            if bpy.context.tool_settings.mesh_select_mode[2]:
                work_mode = 2
                # print("Face MODE")
                # Store selected faces for future restore
                selfaces = [f for f in bm.faces if f.select]

            # Edge selection mode
            elif bpy.context.tool_settings.mesh_select_mode[1]:
                work_mode = 1
                # print("Edges MODE")
                selected_edges = [e for e in bm.edges if e.select]
                if selected_edges:
                    bpy.ops.uv.zenuv_select_island()
                    selfaces = [f for f in bm.faces if f.select]

        # "Unwrap selected" mechanism. If no selected - do unwrap for all. Else only for selection
        if not selfaces:
            # if nothing selected and mesh have no assigned seams
            if not True in [x.seam for x in bm.edges]:
                bpy.ops.wm.call_menu(name="ZUV_MT_ZenUnwrap_Popup")
            else:
                bpy.ops.mesh.select_all(action='SELECT')
                fit_uv_view_to_selected(context, mode='all')
        
        if addon_prefs.sortIslands:
            bm.faces.ensure_lookup_table()
            uv_layer = bm.loops.layers.uv.verify()
            for face in bm.faces:
                for loop in face.loops:
                    uv = loop[uv_layer].uv
                    if uv[0] > 0:
                        uv[0] -= 1.2
        
        bpy.ops.uv.unwrap(method='CONFORMAL', margin=addon_prefs.margin)

        fit_uv_view_to_selected(context, mode="selected")

        bpy.ops.mesh.select_all(action='DESELECT')

        # Restore initial selection
        if work_mode == 2:
            bpy.context.tool_settings.mesh_select_mode = (False, False, True)
            for face in selfaces:
                face.select = True
        elif work_mode == 1:
            bpy.context.tool_settings.mesh_select_mode = (False, True, False)
            for edge in selected_edges:
                edge.select = True

        bmesh.update_edit_mesh(me, True)
        
        return {'FINISHED'}


class ZUV_OT_Clear_UV_Seams(bpy.types.Operator):
    bl_idname = "uv.zenuv_full_clear_seams"
    bl_label = "Unmark All"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Remove all the Seams and/or Sharp edges from the mesh"

    def execute(self, context):
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        me, bm = get_mesh_data()
        uv_layer = bm.loops.layers.uv.verify()

        for face in bm.faces:
            for loop in face.loops:
                loop[uv_layer].pin_uv = False

        bpy.ops.mesh.select_all(action='SELECT')

        if addon_prefs.markSeamEdges:
            # print("Seam is true")
            bpy.ops.mesh.mark_seam(clear=True)
        if addon_prefs.markSharpEdges:
            bpy.ops.mesh.mark_sharp(clear=True)

        # if bpy.context.scene.zen_uv_props.markSharpEdges:
        #     # print("Seam is true")
        #     bpy.ops.mesh.mark_sharp(clear=True)

        bpy.ops.mesh.select_all(action='DESELECT')

        return {'FINISHED'}


class ZUV_OT_Select_UV_Overlap(bpy.types.Operator):
    bl_idname = "uv.zenuv_select_uv_overlap"
    bl_label = "Select Overlapped Islands"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Select Overlapped Islands (works in 2.81 and above)"

    @classmethod
    def poll(cls, context):
        # if not (2, 81, 0) < bpy.app.version:
        #     zen_message(message="Select overlap works in 2.81 and above")
        #     return False
        return (2, 81, 0) < bpy.app.version

    def execute(self, context):

        bpy.ops.mesh.select_all(action='DESELECT')
        bpy.context.scene.tool_settings.use_uv_select_sync = True
        bpy.ops.uv.select_overlap()

        if not check_selection():
            zen_message(message="Overlappings are not found.",)

        return {'FINISHED'}


class ZUV_OT_Seams_By_UV_Borders(bpy.types.Operator):
    bl_idname = "uv.zenuv_seams_by_uv_islands"
    bl_label = "Mark Seams by UV Borders"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Mark Seams by existing UV Borders"

    def execute(self, context):
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        bpy.ops.uv.seams_from_islands()
        if addon_prefs.markSharpEdges:
            bpy.ops.uv.zenuv_sharp_by_seams()


        return {'FINISHED'}


class ZUV_OT_Seams_By_Sharp(bpy.types.Operator):
    bl_idname = "uv.zenuv_seams_by_sharp"
    bl_label = "Mark Seams by Sharp Edges"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Mark Seams by existing Sharp edges"

    def execute(self, context):
        me, bm = get_mesh_data()

        for edge in bm.edges:
            edge.seam = not edge.smooth
        bmesh.update_edit_mesh(me)
        return {'FINISHED'}


class ZUV_OT_Sharp_By_Seam(bpy.types.Operator):
    bl_idname = "uv.zenuv_sharp_by_seams"
    bl_label = "Mark Sharp Edges by Seams"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Mark Sharp edges by existing Seams"

    def execute(self, context):
        me, bm = get_mesh_data()
        for edge in bm.edges:
            edge.smooth = not edge.seam
        bmesh.update_edit_mesh(me)
        return {'FINISHED'}


class ZUV_OT_Autosmooth(bpy.types.Operator):
    bl_idname = "view3d.zenuv_set_autosmooth"
    bl_label = "Smooth Mode (Toggle)"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Toggle between Auto Smooth 180° (with sharp edges) and regular smooth modes"

    def execute(self, context):
        me, bm = get_mesh_data()

        switch_mesh_to_smooth_mode()

        # set_seams_as_sharp(bm)

        bpy.context.object.data.use_auto_smooth = not bpy.context.object.data.use_auto_smooth
        bpy.context.object.data.auto_smooth_angle = 3.14159

        return {'FINISHED'}


class ZUV_OT_Auto_Seams(bpy.types.Operator):
    bl_idname = "uv.zenuv_auto_seams"
    bl_label = "Auto Mark"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Automatically Mark edges as Seams and/or Sharp edges"

    angle: bpy.props.FloatProperty(
        name="Angle",
        description="Set angle",
        min=0.0,
        max=180.0,
        default=30.03
    )
    
    def invoke(self, context, event):
        return self.execute(context)
        
    @classmethod
    def poll(cls, context):
        return context.active_object is not None

    def execute(self, context):
        me, bm = get_mesh_data()

        sharp_edges = [edge for edge in bm.edges if len(edge.link_faces) == 2 and edge.calc_face_angle() > radians(self.angle)]
        holes_edges = [edge for edge in bm.edges if len(edge.link_faces) == 1]
        sharp_edges.extend(holes_edges)

        # sharp_edges = [edge for edge in bm.edges if edge.calc_face_angle() > radians(self.angle)]
        not_sharp_edges = [edge for edge in bm.edges if edge not in sharp_edges]

        bpy.context.tool_settings.mesh_select_mode = (False, True, False)

        for edge in sharp_edges:
            edge.seam = True
            edge.smooth = False
        for edge in not_sharp_edges:
            edge.seam = False
            edge.smooth = True

        bmesh.update_edit_mesh(me)

        return {'FINISHED'}
    
    def draw(self, context):
        layout = self.layout
        layout.prop(self, "angle")
    


class ZUV_OT_Auto_Seams_legacy(bpy.types.Operator):
    bl_idname = "uv.zenuv_auto_seams"
    bl_label = "Auto Seams"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Setup Auto Seams"

    sharp_angle: bpy.props.FloatProperty(
        name="Angle",
        description="Set angle",
        min=0.0,
        max=180.0,
        default=30.0
    )
    
    @classmethod
    def poll(cls, context):
        return context.active_object is not None

    def execute(self, context):
        me, bm = get_mesh_data()
        bpy.ops.mesh.select_mode(type="EDGE")
        bpy.ops.mesh.select_all(action='DESELECT')
        set_autoseams(me, bm, self.sharp_angle)

        return {'FINISHED'}
    
    def draw(self, context):
        layout = self.layout
        layout.prop(self.sharp_angle)

        # bpy.ops.mesh.mark_seam(clear=False)

def set_autoseams_legacy(me, bm, sharp_angle):
    for edge in bm.edges:
        bpy.ops.mesh.edges_select_sharp(sharpness=sharp_angle)
        if edge.select:
            edge.seam = True
    

class ZUV_OT_Mark_Seams(bpy.types.Operator):
    bl_idname = "uv.zenuv_mark_seams"
    bl_label = "Mark"
    bl_description = "Mark selected edges or face borders as Seams and/or Sharp edges"
    bl_options = {'REGISTER', 'UNDO'}

    call_from_zen = False

    @classmethod
    def poll(cls, context):
        return check_selection() or call_from_zen_check()

    def execute(self, context):
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        me, bm = get_mesh_data()
        self.selfaces = []
        self.seledges = []

        # Check if face selection mode
        # Check if have currently selected faces
        # Check Mark Seams is True
        if bm.select_mode == {'FACE'} and True in [x.select for x in bm.faces]:

            self.selfaces = [f for f in bm.faces if f.select]
            bpy.ops.mesh.region_to_loop()
            region_loop_edges = [e for e in bm.edges if e.select]

            # Test if selected edges exist - restore face selection and assign seams to borders
            if len(region_loop_edges):
                self.restore_selected_faces()
                bpy.ops.mesh.mark_seam(clear=True)
                bpy.ops.mesh.mark_sharp(clear=True)
                if addon_prefs.markSeamEdges:
                    # print("Seam is true")
                    self.assign_seam_to_edges(region_loop_edges)
                if addon_prefs.markSharpEdges:
                    # print("Sharp is true")
                    self.assign_sharp_to_edges(region_loop_edges)
            else:
                zen_message(message="Nothing is produced. Selected polygons do not have a borders.")


            # Toggle to face selection mode and restore previous faces selection
            bpy.ops.mesh.select_mode(type="FACE")
            self.restore_selected_faces()

        # Check if Edge selection mode
        if bm.select_mode == {'EDGE'} and True in [x.select for x in bm.edges]:
            self.seledges = [e for e in bm.edges if e.select]
            if addon_prefs.markSeamEdges:
                # print("Seam is true")
                self.assign_seam_to_edges(self.seledges)
            if addon_prefs.markSharpEdges:
                # print("Sharp is true")
                self.assign_sharp_to_edges(self.seledges)
        
        # Show changes in object
        bmesh.update_edit_mesh(me)

        return {'FINISHED'}


    def restore_selected_faces(self):
        for face in self.selfaces:
            face.select = True
    
    def assign_seam_to_edges(self, edges):
        for edge in edges:
            edge.seam = True

    def assign_sharp_to_edges(self, edges):
        for edge in edges:
            edge.smooth = False

    def assign_seam_to_selected_edges(self, bm):
        for edge in bm.edges:
            if edge.select:
                edge.seam = True


class ZUV_OT_Unmark_Seams(bpy.types.Operator):
    bl_idname = "uv.zenuv_unmark_seams"
    bl_label = "Unmark"
    bl_description = "Unmark selected edges or face borders as Seams and/or Sharp edges"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return check_selection()

    def execute(self, context):
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        if addon_prefs.markSeamEdges:
            # print("Seam is true")
            bpy.ops.mesh.mark_seam(clear=True)
        if addon_prefs.markSharpEdges:
            # print("Seam is true")
            bpy.ops.mesh.mark_sharp(clear=True)
       
        return {'FINISHED'}


class ZUV_OT_PinUV_Island(bpy.types.Operator):
    bl_idname = "uv.zenuv_pin_island"
    bl_label = "Pin UV Island"
    bl_description = "Select at least one edge/face of the Island(s)"
    bl_options = {'REGISTER', 'UNDO'}

    # pin_action = None

    pin_action: bpy.props.BoolProperty(
        name="Pin Current Island",
        description="Set Current Island Pinned or Not.",
        default=True
    )

    color: bpy.props.FloatVectorProperty(
        name="UV Island Color",
        description="Color of pinned Island.",
        subtype='COLOR',
        default=[0.25, 0.4, 0.4, 1],
        size=4,
        min=0,
        max=1
    )
        # update=vc.set_v_color
    # )

    randomize_color: bpy.props.BoolProperty(
        name="Randomize Color",
        description="Adds some variations to the current color.",
        default=True
    )
    unpin_color = [1, 1, 1]

    @classmethod
    def poll(cls, context):
        return check_selection()

    def execute(self, context):
        me, bm = get_mesh_data()
        uv_layer = bm.loops.layers.uv.verify()
        islands_for_pin = island_util.get_island(bm)
        bmesh.update_edit_mesh(me, True)

        # Pinning islands, create vcolor
        # print('islands_for_pin', len(islands_for_pin))
        for island in islands_for_pin:
            self.shift_island(island, uv_layer)
            self.pin_island(island, uv_layer)

            # if need_to_recolor and self.color == [255, 255, 255]:
            color_layer = vc.set_color_layer(me)

            # if pin action is clear color (False). I just set color to white
            if not self.pin_action:
                self.randomize_color = False
                vc.set_v_color(me, color_layer, island, self.unpin_color, self.randomize_color)
            else:
                self.color = vc.set_v_color(me, color_layer, island, self.color, self.randomize_color)

        vc.update_vcolor(me)
        # checker_setup.switch_shading_style('VERTEX', switch=False)

        return {'FINISHED'}

    def shift_island(self, island, uv_layer):
        for face in island:
            for loop in face.loops:
                uv = loop[uv_layer].uv
                if uv[1] >= 2:
                    return False
                else:
                    uv[1] += 2
        return True

    def pin_island(self, island, uv_layer):
        for face in island:
            for loop in face.loops:
                loop[uv_layer].pin_uv = self.pin_action


class ZUV_OT_Isolate_Island(bpy.types.Operator):
    bl_idname = "uv.zenuv_isolate_island"
    bl_label = "Isolate Islands (Toggle)"
    bl_description = "Isolate UV Island (Toggle). Select at least one edge/face of the Island(s)"
    bl_options = {'REGISTER', 'UNDO'}

    # pin_action: bpy.props.BoolProperty(
    #     name="Isolate Current Island",
    #     description="Set Current Island Isolation Mode.",
    #     default=True
    # )

    isolate_mode = False

    @classmethod
    def poll(cls, context):
        return check_selection()

    def execute(self, context):
        me, bm = get_mesh_data()
        uv_layer = bm.loops.layers.uv.verify()
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        preselected_faces = [f for f in bm.faces if f.select]
        islands_for_isolate = island_util.get_island(bm)

        blender_sync_mode = bpy.context.scene.tool_settings.use_uv_select_sync

        # Turn off blender sync UV mode
        bpy.context.scene.tool_settings.use_uv_select_sync = False

        # Isolate island
        faces_for_isolate = []
        # print(len(islands_for_isolate))
        for island in islands_for_isolate:
            faces_for_isolate.extend(island)
        self.isolate_island(bm, faces_for_isolate, uv_layer)

        bpy.ops.uv.zenuv_pack()

        bpy.context.scene.tool_settings.use_uv_select_sync = True

        for f in faces_for_isolate:
            f.select = True
        
        if not self.isolate_mode:
            # print("Mode - ", self.isolate_mode)
            if addon_prefs.packEngine == "PKM":
                fit_uv_view_to_selected(context, mode="all")
            else:
                fit_uv_view_to_selected(context, mode="selected")
        else:
            if addon_prefs.packEngine == "PKM":
                fit_uv_view_to_selected(context, mode="all")
            else:
                fit_uv_view_to_selected(context, mode="selected")

        bpy.ops.mesh.select_all(action='DESELECT')


        # restore initial selection
        for f in preselected_faces:
            f.select = True

        bpy.context.scene.tool_settings.use_uv_select_sync = blender_sync_mode
        bmesh.update_edit_mesh(me, True)

       
        return {'FINISHED'}

    def isolate_island(self, bm, island, uv_layer):
        """ if mesh in island isolation mode, need to back to regular mode """
        check_isolation = self.check_isolation(bm)
        self.isolate_mode = check_isolation
        if check_isolation:
            # print("Return To Regular")
            self.return_to_regular_mode(bm)
        elif not check_isolation:
            # print("Set Isolation")
            self.go_to_isolate_mode(bm, island, uv_layer)

    def return_to_regular_mode(self, bm):
        for face in bm.faces:
            face.hide_set(False)

    def go_to_isolate_mode(self, bm, island, uv_layer):
        # start = time.time()
        # faces_for_shift = []
        """
        for face in bm.faces:
            if face not in island:
                face.hide_set(True)
        """
        # bpy.ops.mesh.select_all(action='DESELECT')
        faces_to_hide = [f for f in bm.faces if f not in island]
        for face in faces_to_hide:
            face.hide_set(True)
        # for face in island:
        #     face.select = True
        # bpy.ops.mesh.hide(unselected=True)
        # temporary disabled Shift Island
        #         faces_for_shift.append(face)
        # self.shift_island(faces_for_shift, uv_layer)
        # end = time.time()
        # print("def go_to_isolate_mode", end - start)

    def check_isolation(self, bm):
        """ Return True if mesh in isolation island mode. Else return False """
        """
        # start = time.time()
        faces = [f for f in bm.faces if f.hide]
        True in [f.hide in bm.faces]
        if faces:
            return True

        for face in bm.faces:
            if face.hide:
                return True
        # end = time.time()
        # print("def check_isolation", end - start)
        return False
        """
        return True in [f.hide for f in bm.faces]

    def shift_island(self, island, uv_layer):
        for face in island:
            for loop in face.loops:
                uv = loop[uv_layer].uv
                if uv[1] >= 2:
                    return False
                else:
                    uv[1] += 2
        return True


class ZUV_OT_Select_UV_Island(bpy.types.Operator):
    bl_idname = "uv.zenuv_select_island"
    bl_label = "Select UV Islands"
    bl_description = "Select UV Island. Select at least one edge/face of the Island(s)"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return check_selection()

    def execute(self, context):
        me, bm = get_mesh_data()
        # uv_layer = bm.loops.layers.uv.verify()
        islands_for_select = island_util.get_island(bm)
        # islands_for_select, selected_faces = get_island_by_face(bm, uv_layer)

        for island in islands_for_select:
            for face in island:
                face.select = True

        bmesh.update_edit_mesh(me, True)

        return {'FINISHED'}


class ZUV_OT_Toggle_UV_Check_Texture(bpy.types.Operator):
    bl_idname = "uv.zenuv_toggle_checker_texture"
    bl_label = "Checker Texture (Toggle)"
    bl_description = "Add Checker Texture to the mesh (Toggle)"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        obj = bpy.context.edit_object
        if checker_setup.switch_checker(obj):
            fit_uv_view_to_selected(context, mode="checker")
        else:
            fit_uv_view_to_selected(context, mode="all")
        # bpy.ops.mesh.select_all(action='DESELECT')

        return {'FINISHED'}


class ZUV_OT_Toggle_Object_Color(bpy.types.Operator):
    bl_idname = "uv.zenuv_toggle_color_to_pinned_islands"
    bl_label = "Show Pinned Islands (Toggle)"
    bl_description = "Show Pinned Islands (Toggle)"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        checker_setup.switch_shading_style("VERTEX", switch=True)
        return {'FINISHED'}


class ZUV_OT_Quadrify(bpy.types.Operator):
    bl_idname = "uv.zenuv_quadrify"
    bl_label = "Quadrify Islands"
    bl_description = "Straighten rectangular shaped UV Islands"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return check_selection()

    def execute(self, context):
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        obj = bpy.context.edit_object
        obj.select_set(True)

        me, bm = get_mesh_data()
        preselected_faces = [f for f in bm.faces if f.select]

        blender_sync_mode = bpy.context.scene.tool_settings.use_uv_select_sync
        
        # Turn off blender sync UV mode
        bpy.context.scene.tool_settings.use_uv_select_sync = False
        bpy.ops.uv.zenuv_select_island()

        selected_island = [f for f in bm.faces if f.select]

        bpy.ops.uv.select_all(action='SELECT')

        bpy.ops.uv.uv_squares_by_shape()

        bpy.ops.mesh.select_all(action='DESELECT')

        
        if addon_prefs.autoPinQuadrified:
            bpy.ops.uv.zenuv_pin_island()
        
        # zen_pack_islands()
        # bpy.ops.uv.zenuv_pack()

        # restore initial selection
        for f in selected_island:
            f.select = True
        
        if addon_prefs.packEngine == "PKM":
            fit_uv_view_to_selected(context, mode="all")
        else:
            fit_uv_view_to_selected(context, mode="selected")

        bpy.ops.mesh.select_all(action='DESELECT')

        # restore initial selection
        for f in preselected_faces:
            f.select = True
        
        bmesh.update_edit_mesh(me, True)

        # Zen Pack if needed
        if addon_prefs.packAfQuadrify:
            bpy.ops.uv.zenuv_pack()
            
        bpy.context.scene.tool_settings.use_uv_select_sync = blender_sync_mode

        return {'FINISHED'}


class ZUV_OT_Select_Loop(bpy.types.Operator):
    bl_idname = "ops.zenuv_select_loop"
    bl_label = "Select Edge Loop"
    bl_description = "Select Edge Loop with Seams respect"
    bl_options = {'REGISTER', 'UNDO'}


    @classmethod
    def poll(cls, context):
        me, bm = get_mesh_data()
        if context.active_object.mode == 'EDIT' and True in [x.select for x in bm.edges]:
            return True

    def execute(self, context):
        me, bm = get_mesh_data()
        bm.edges.ensure_lookup_table()
        is_edge = isinstance(bm.select_history.active, bmesh.types.BMEdge)
        selected_edge = [edge for edge in bm.edges if edge.select][0]
        start_loop = selected_edge.link_loops[0]

        if selected_edge:
            # CW direction
            for i in range(len(bm.edges)):
                next_edge_in_loop = start_loop.link_loop_next.link_loop_radial_next.edge
                next_loop = start_loop.link_loop_next.link_loop_radial_next.link_loop_next
                
                if next_edge_in_loop.seam == True \
                    or next_edge_in_loop == selected_edge \
                    or next_loop.edge == start_loop.link_loop_next.link_loop_next.edge \
                    or next_loop.edge == start_loop.link_loop_radial_next.link_loop_prev.edge \
                    or len(start_loop.link_loop_next.vert.link_edges) > 4:
                    break
                else:
                    
                    next_loop.edge.select = True
                    start_loop = next_loop
                    
            # CCW direction
            for i in range(len(bm.edges)):
                next_edge_in_loop = start_loop.link_loop_prev.link_loop_radial_next.edge
                next_loop = start_loop.link_loop_prev.link_loop_radial_next.link_loop_prev
                
                if next_edge_in_loop.seam == True \
                    or next_edge_in_loop == selected_edge \
                    or next_loop.edge == start_loop.link_loop_next.link_loop_next.edge \
                    or next_loop.edge == start_loop.link_loop_radial_next.link_loop_next.edge \
                    or len(start_loop.vert.link_edges) > 4:
                    break
                else:
                    
                    next_loop.edge.select = True
                    start_loop = next_loop
                    
            bmesh.update_edit_mesh(me)
            
            return {'FINISHED'}


class ZUV_OT_Pack(bpy.types.Operator):
    bl_idname = "uv.zenuv_pack"
    bl_label = "Pack Islands"
    bl_options = {'REGISTER', 'UNDO'}
    bl_description = "Pack all Islands"

    def execute(self, context):
        me, bm = get_mesh_data()
        addon_prefs = context.preferences.addons["ZenUV"].preferences
        uv_layer = bm.loops.layers.uv.verify()

        blender_sync_mode = bpy.context.scene.tool_settings.use_uv_select_sync

        bpy.context.scene.tool_settings.use_uv_select_sync = True

        bpy.ops.mesh.select_all(action='SELECT')

        if addon_prefs.packEngine == "PKM":
            # print("Seam is true")
            if addon_prefs.averageBeforePack:
                bpy.ops.uv.average_islands_scale()
            if hasattr(bpy.types, bpy.ops.uvpackmaster2.uv_pack.idname()):
                print("Zen UV: Packing UV by UVPackmaster 2...")
                if hasattr(bpy.context.scene, "uvp2_props") and hasattr(bpy.context.scene.uvp2_props, "margin"):
                    member = bpy.context.scene.uvp2_props.margin
                    # print("UVP Margin before:", bpy.context.scene.uvp2_props.margin)
                    bpy.context.scene.uvp2_props.margin = addon_prefs.margin
                    bpy.ops.uvpackmaster2.uv_pack()
                    fit_uv_view_to_selected(context, mode="all")
                    bpy.context.scene.uvp2_props.margin = member
                    # print("UVP Margin:", member)
            else:
                # zen_message(message="Nothing is produced. Seems like Pack Master 2 is not installed on your system.")
                bpy.ops.wm.call_menu(name="ZUV_MT_ZenPack_Popup")
        elif addon_prefs.packEngine == "BLDR":
            print("Zen UV: Packing UV by Blender Pack...")
            if addon_prefs.averageBeforePack:
                bpy.ops.uv.average_islands_scale()
            bpy.ops.uv.pack_islands(margin=addon_prefs.margin)
            fit_uv_view_to_selected(context, mode="all")
        

        bpy.ops.mesh.select_all(action='DESELECT')

        bpy.context.scene.tool_settings.use_uv_select_sync = blender_sync_mode

        return {'FINISHED'}