/*
 * Decompiled with CFR 0.152.
 */
package com.android.traceview;

import com.android.traceview.Call;
import com.android.traceview.MethodData;
import com.android.traceview.ProfileProvider;
import com.android.traceview.ThreadData;
import com.android.traceview.TimeLineView;
import com.android.traceview.TraceReader;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.nio.BufferUnderflowException;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public class DmTraceReader
extends TraceReader {
    private int mVersionNumber = 0;
    private boolean mDebug = false;
    private static final int TRACE_MAGIC = 1464814675;
    private boolean mRegression;
    private ProfileProvider mProfileProvider;
    private String mTraceFileName;
    private MethodData mTopLevel;
    private ArrayList<Call> mCallList;
    private ArrayList<Call> mSwitchList;
    private HashMap<Integer, MethodData> mMethodMap;
    private HashMap<Integer, ThreadData> mThreadMap;
    private ThreadData[] mSortedThreads;
    private MethodData[] mSortedMethods;
    private long mGlobalEndTime;
    private MethodData mContextSwitch;
    private int mOffsetToData;
    private byte[] mBytes = new byte[8];
    private static final Pattern mIdNamePattern = Pattern.compile("(\\d+)\t(.*)");
    static final int PARSE_VERSION = 0;
    static final int PARSE_THREADS = 1;
    static final int PARSE_METHODS = 2;
    static final int PARSE_OPTIONS = 4;

    DmTraceReader(String traceFileName, boolean regression) {
        this.mTraceFileName = traceFileName;
        this.mRegression = regression;
        this.mMethodMap = new HashMap();
        this.mThreadMap = new HashMap();
        this.mTopLevel = new MethodData(0, "(toplevel)");
        this.mContextSwitch = new MethodData(-1, "(context switch)");
        this.mMethodMap.put(0, this.mTopLevel);
        this.generateTrees();
    }

    void generateTrees() {
        try {
            long offset = this.parseKeys();
            this.parseData(offset);
            this.analyzeData();
        }
        catch (IOException e) {
            System.err.println(e.getMessage());
            System.exit(1);
        }
    }

    @Override
    public ProfileProvider getProfileProvider() {
        if (this.mProfileProvider == null) {
            this.mProfileProvider = new ProfileProvider(this);
        }
        return this.mProfileProvider;
    }

    Call readCall(MappedByteBuffer buffer, Call call) {
        long time;
        int methodId;
        short threadId;
        try {
            threadId = this.mVersionNumber == 1 ? buffer.get() : buffer.getShort();
            methodId = buffer.getInt();
            time = buffer.getInt();
        }
        catch (BufferUnderflowException ex) {
            return null;
        }
        int methodAction = methodId & 3;
        MethodData methodData = this.mMethodMap.get(methodId &= 0xFFFFFFFC);
        if (methodData == null) {
            String name = String.format("(0x%1$x)", methodId);
            methodData = new MethodData(methodId, name);
        }
        if (call != null) {
            call.set(threadId, methodData, time, methodAction);
        } else {
            call = new Call(threadId, methodData, time, methodAction);
        }
        return call;
    }

    private MappedByteBuffer mapFile(String filename, long offset) {
        MappedByteBuffer buffer = null;
        try {
            FileInputStream dataFile = new FileInputStream(filename);
            File file = new File(filename);
            FileChannel fc = dataFile.getChannel();
            buffer = fc.map(FileChannel.MapMode.READ_ONLY, offset, file.length() - offset);
            buffer.order(ByteOrder.LITTLE_ENDIAN);
        }
        catch (FileNotFoundException ex) {
            System.err.println(ex.getMessage());
            System.exit(1);
        }
        catch (IOException ex) {
            System.err.println(ex.getMessage());
            System.exit(1);
        }
        return buffer;
    }

    private void readDataFileHeader(MappedByteBuffer buffer) {
        int magic = buffer.getInt();
        if (magic != 1464814675) {
            System.err.printf("Error: magic number mismatch; got 0x%x, expected 0x%x\n", magic, 1464814675);
            throw new RuntimeException();
        }
        short version = buffer.getShort();
        this.mOffsetToData = buffer.getShort() - 16;
        buffer.getLong();
        for (int ii = 0; ii < this.mOffsetToData; ++ii) {
            buffer.get();
        }
        buffer.mark();
    }

    private void parseData(long offset) {
        MappedByteBuffer buffer = this.mapFile(this.mTraceFileName, offset);
        this.readDataFileHeader(buffer);
        this.parseDataPass1(buffer);
        buffer.reset();
        this.parseDataPass2(buffer);
    }

    private void parseDataPass1(MappedByteBuffer buffer) {
        this.mSwitchList = new ArrayList();
        Call call = new Call();
        if ((call = this.readCall(buffer, call)) == null) {
            return;
        }
        long callTime = call.mThreadStartTime;
        long prevCallTime = 0L;
        ThreadData threadData = this.mThreadMap.get(call.getThreadId());
        if (threadData == null) {
            String name = String.format("[%1$d]", call.getThreadId());
            threadData = new ThreadData(call.getThreadId(), name, this.mTopLevel);
            this.mThreadMap.put(call.getThreadId(), threadData);
        }
        ThreadData prevThreadData = threadData;
        while (true) {
            if (prevThreadData != threadData) {
                Call switchEnter = new Call(prevThreadData.getId(), this.mContextSwitch, prevCallTime, 0);
                prevThreadData.setLastContextSwitch(switchEnter);
                this.mSwitchList.add(switchEnter);
                Call contextSwitch = threadData.getLastContextSwitch();
                if (contextSwitch != null) {
                    long prevStartTime = contextSwitch.mThreadStartTime;
                    long elapsed = callTime - prevStartTime;
                    long beforeSwitch = elapsed / 2L;
                    long afterSwitch = elapsed - beforeSwitch;
                    long exitTime = callTime - afterSwitch;
                    contextSwitch.mThreadStartTime = prevStartTime + beforeSwitch;
                    Call switchExit = new Call(threadData.getId(), this.mContextSwitch, exitTime, 1);
                    this.mSwitchList.add(switchExit);
                }
                prevThreadData = threadData;
            }
            if ((call = this.readCall(buffer, call)) == null) break;
            prevCallTime = callTime;
            callTime = call.mThreadStartTime;
            threadData = this.mThreadMap.get(call.getThreadId());
            if (threadData != null) continue;
            String name = String.format("[%d]", call.getThreadId());
            threadData = new ThreadData(call.getThreadId(), name, this.mTopLevel);
            this.mThreadMap.put(call.getThreadId(), threadData);
        }
    }

    void parseDataPass2(MappedByteBuffer buffer) {
        ThreadData threadData;
        long callTime;
        this.mCallList = new ArrayList();
        Call call = this.readCall(buffer, null);
        long prevCallTime = callTime = call.mThreadStartTime;
        ThreadData prevThreadData = threadData = this.mThreadMap.get(call.getThreadId());
        threadData.setGlobalStartTime(0L);
        int nthContextSwitch = 0;
        long globalTime = 0L;
        while (true) {
            long elapsed = callTime - prevCallTime;
            if (threadData != prevThreadData) {
                Call contextSwitch = this.mSwitchList.get(nthContextSwitch++);
                this.mCallList.add(contextSwitch);
                elapsed = contextSwitch.mThreadStartTime - prevCallTime;
                globalTime += elapsed;
                elapsed = 0L;
                contextSwitch.mGlobalStartTime = globalTime;
                prevThreadData.handleCall(contextSwitch, globalTime);
                if (!threadData.isEmpty()) {
                    contextSwitch = this.mSwitchList.get(nthContextSwitch++);
                    this.mCallList.add(contextSwitch);
                    contextSwitch.mGlobalStartTime = globalTime;
                    elapsed = callTime - contextSwitch.mThreadStartTime;
                    threadData.handleCall(contextSwitch, globalTime);
                }
                if (threadData.getGlobalStartTime() == -1L) {
                    threadData.setGlobalStartTime(globalTime);
                }
                prevThreadData = threadData;
            }
            call.mGlobalStartTime = globalTime += elapsed;
            threadData.handleCall(call, globalTime);
            this.mCallList.add(call);
            call = this.readCall(buffer, null);
            if (call == null) break;
            prevCallTime = callTime;
            callTime = call.mThreadStartTime;
            threadData = this.mThreadMap.get(call.getThreadId());
        }
        for (int id : this.mThreadMap.keySet()) {
            threadData = this.mThreadMap.get(id);
            long endTime = threadData.endTrace();
            if (endTime <= 0L) continue;
            this.mTopLevel.addElapsedInclusive(endTime, false, null);
        }
        this.mGlobalEndTime = globalTime;
        if (this.mRegression) {
            this.dumpCallTimes();
        }
    }

    long parseKeys() throws IOException {
        BufferedReader in = null;
        try {
            in = new BufferedReader(new FileReader(this.mTraceFileName));
        }
        catch (FileNotFoundException ex) {
            System.err.println(ex.getMessage());
        }
        long offset = 0L;
        int mode = 0;
        String line = null;
        while (true) {
            if ((line = in.readLine()) == null) {
                throw new IOException("Key section does not have an *end marker");
            }
            offset += (long)(line.length() + 1);
            if (line.startsWith("*")) {
                if (line.equals("*version")) {
                    mode = 0;
                    continue;
                }
                if (line.equals("*threads")) {
                    mode = 1;
                    continue;
                }
                if (line.equals("*methods")) {
                    mode = 2;
                    continue;
                }
                if (line.equals("*end")) {
                    return offset;
                }
            }
            switch (mode) {
                case 0: {
                    this.mVersionNumber = Integer.decode(line);
                    mode = 4;
                    break;
                }
                case 1: {
                    this.parseThread(line);
                    break;
                }
                case 2: {
                    this.parseMethod(line);
                    break;
                }
            }
        }
    }

    void parseThread(String line) {
        String idStr = null;
        String name = null;
        Matcher matcher = mIdNamePattern.matcher(line);
        if (matcher.find()) {
            idStr = matcher.group(1);
            name = matcher.group(2);
        }
        if (idStr == null) {
            return;
        }
        if (name == null) {
            name = "(unknown)";
        }
        int id = Integer.decode(idStr);
        this.mThreadMap.put(id, new ThreadData(id, name, this.mTopLevel));
    }

    void parseMethod(String line) {
        String[] tokens = line.split("\t");
        int id = Long.decode(tokens[0]).intValue();
        String className = tokens[1];
        String methodName = null;
        String signature = null;
        String pathname = null;
        int lineNumber = -1;
        if (tokens.length == 6) {
            methodName = tokens[2];
            signature = tokens[3];
            pathname = tokens[4];
            lineNumber = Integer.decode(tokens[5]);
            pathname = this.constructPathname(className, pathname);
        } else if (tokens.length > 2) {
            if (tokens[3].startsWith("(")) {
                methodName = tokens[2];
                signature = tokens[3];
            } else {
                pathname = tokens[2];
                lineNumber = Integer.decode(tokens[3]);
            }
        }
        this.mMethodMap.put(id, new MethodData(id, className, methodName, signature, pathname, lineNumber));
    }

    private String constructPathname(String className, String pathname) {
        int index = className.lastIndexOf(47);
        if (index > 0 && index < className.length() - 1 && pathname.endsWith(".java")) {
            pathname = className.substring(0, index + 1) + pathname;
        }
        return pathname;
    }

    private void analyzeData() {
        Collection<ThreadData> tv = this.mThreadMap.values();
        this.mSortedThreads = tv.toArray(new ThreadData[tv.size()]);
        Arrays.sort(this.mSortedThreads, new Comparator<ThreadData>(){

            @Override
            public int compare(ThreadData td1, ThreadData td2) {
                if (td2.getCpuTime() > td1.getCpuTime()) {
                    return 1;
                }
                if (td2.getCpuTime() < td1.getCpuTime()) {
                    return -1;
                }
                return td2.getName().compareTo(td1.getName());
            }
        });
        long sum = 0L;
        for (ThreadData t : this.mSortedThreads) {
            if (t.isEmpty()) continue;
            Call root = t.getCalltreeRoot();
            root.mGlobalStartTime = t.getGlobalStartTime();
        }
        Collection<MethodData> mv = this.mMethodMap.values();
        MethodData[] methods = mv.toArray(new MethodData[mv.size()]);
        Arrays.sort(methods, new Comparator<MethodData>(){

            @Override
            public int compare(MethodData md1, MethodData md2) {
                if (md2.getElapsedInclusive() > md1.getElapsedInclusive()) {
                    return 1;
                }
                if (md2.getElapsedInclusive() < md1.getElapsedInclusive()) {
                    return -1;
                }
                return md1.getName().compareTo(md2.getName());
            }
        });
        int nonZero = 0;
        for (MethodData md : methods) {
            if (md.getElapsedInclusive() == 0L) break;
            ++nonZero;
        }
        this.mSortedMethods = new MethodData[nonZero];
        int ii = 0;
        for (MethodData md : methods) {
            if (md.getElapsedInclusive() == 0L) break;
            md.setRank(ii);
            this.mSortedMethods[ii++] = md;
        }
        for (MethodData md : this.mSortedMethods) {
            md.analyzeData();
        }
        for (Call call : this.mCallList) {
            call.updateName();
        }
        if (this.mRegression) {
            this.dumpMethodStats();
        }
    }

    @Override
    public ArrayList<TimeLineView.Record> getThreadTimeRecords() {
        TimeLineView.Record record;
        ArrayList<TimeLineView.Record> timeRecs = new ArrayList<TimeLineView.Record>();
        for (ThreadData threadData : this.mSortedThreads) {
            if (threadData.isEmpty() || threadData.getId() == 0) continue;
            Call call = new Call(threadData.getId(), this.mTopLevel, threadData.getGlobalStartTime(), 0);
            call.mGlobalStartTime = threadData.getGlobalStartTime();
            call.mGlobalEndTime = threadData.getGlobalEndTime();
            record = new TimeLineView.Record(threadData, call);
            timeRecs.add(record);
        }
        for (Call call : this.mCallList) {
            if (call.getMethodAction() != 0 || call.getThreadId() == 0) continue;
            ThreadData threadData = this.mThreadMap.get(call.getThreadId());
            record = new TimeLineView.Record(threadData, call);
            timeRecs.add(record);
        }
        if (this.mRegression) {
            this.dumpTimeRecs(timeRecs);
            System.exit(0);
        }
        return timeRecs;
    }

    private void dumpCallTimes() {
        System.out.format("id thread  global start,end   method\n", new Object[0]);
        for (Call call : this.mCallList) {
            String action = call.getMethodAction() == 0 ? "+" : " ";
            long callTime = call.mThreadStartTime;
            System.out.format("%2d %6d %8d %8d %s %s\n", call.getThreadId(), callTime, call.mGlobalStartTime, call.mGlobalEndTime, action, call.getMethodData().getName());
        }
    }

    private void dumpMethodStats() {
        System.out.format("\nExclusive Inclusive     Calls  Method\n", new Object[0]);
        for (MethodData md : this.mSortedMethods) {
            System.out.format("%9d %9d %9s  %s\n", md.getElapsedExclusive(), md.getElapsedInclusive(), md.getCalls(), md.getProfileName());
        }
    }

    private void dumpTimeRecs(ArrayList<TimeLineView.Record> timeRecs) {
        System.out.format("\nid thread  global start,end  method\n", new Object[0]);
        for (TimeLineView.Record record : timeRecs) {
            Call call = (Call)record.block;
            long callTime = call.mThreadStartTime;
            System.out.format("%2d %6d %8d %8d  %s\n", call.getThreadId(), callTime, call.mGlobalStartTime, call.mGlobalEndTime, call.getMethodData().getName());
        }
    }

    @Override
    public HashMap<Integer, String> getThreadLabels() {
        HashMap<Integer, String> labels = new HashMap<Integer, String>();
        for (ThreadData t : this.mThreadMap.values()) {
            labels.put(t.getId(), t.getName());
        }
        return labels;
    }

    @Override
    public MethodData[] getMethods() {
        return this.mSortedMethods;
    }

    @Override
    public ThreadData[] getThreads() {
        return this.mSortedThreads;
    }

    @Override
    public long getEndTime() {
        return this.mGlobalEndTime;
    }
}

