//---------------------------------------------------------------------------------------
//  FILE:    X2Ability_InfantryAbilitySet.uc
//  AUTHOR:  Lucubration
//  PURPOSE: Defines abilities used by the Infantry class.
//---------------------------------------------------------------------------------------

class X2Ability_InfantryAbilitySet extends X2Ability
	config(LucubrationsInfantryClass);

// Local variables are defined and initialized entirely in code
var name LightEmUpAbilityName;
var name CrippledEffectName;
var name StickAndMoveMobilityAbilityName;
var name StickAndMoveMobilityEffectName;
var name StickAndMoveDodgeAbilityName;
var name StickAndMoveDodgeEffectName;
var name FlareStatusEffectName;
var name FlareRemoveTriggerName;
var name ZoneOfControlActionPointName;
var name ZoneOfControlReactionFireAbilityName;
var name EscapeAndEvadeActionPointName;
var name EscapeAndEvadeStealthAbilityName;

// Config variables are defined here but initialized using values in 'XComLucubrationsInfantryClass.ini'.
// This config file name has the safe suffix as the package name
// Localized strings are defined here but initialized using values in 'LucubrationsInfantryClass.int'.
// I'm truly unsure of how localization works, but I figure it doesn't hurt to get things set up in case
// I should localize the class, ability and effect descriptions later
var config int HarrierAimBonus;
var config int HarrierCritBonus;
var config int CripplingShotAimBonus;
var config int CripplingShotAmmoCost;
var config int CripplingShotActionPointCost;
var config int CripplingShotCooldown;
var config int CripplingShotDuration;
var config int CripplingShotDodgeReduction;
var config int CripplingShotMobilityReduction;
var localized string CrippledFriendlyName;
var localized string CrippledFriendlyDesc;
var localized string CrippledEffectAcquiredString;
var localized string CrippledEffectTickedString;
var localized string CrippledEffectLostString;
var config int ShakeItOffCharges;
var config int StickAndMoveMobilityBonus;
var config int StickAndMoveMobilityDuration;
var config int StickAndMoveDodgeBonus;
var config int StickAndMoveDodgeDuration;
var localized string StickAndMoveMobilityFriendlyName;
var localized string StickAndMoveMobilityFriendlyDesc;
var localized string StickAndMoveDodgeFriendlyName;
var localized string StickAndMoveDodgeFriendlyDesc;
var config int ExplosiveActionCooldown;
var config int ExplosiveActionBonusActionPoints;
var config int ExplosiveActionRecoveryDelay;
var config int ExplosiveActionRecoveryActionPoints;
var localized string ExplosiveActionRecoveryFriendlyName;
var localized string ExplosiveActionRecoveryFriendlyDesc;
var config int ZoneOfControlReactionFireRadius;
var config int EscapeAndEvadeDodgeBonus;
var config int FlareCharges;
var config int FlareRadius;
var config int FlareHeight;
var config int FlareRange;
var config int FlareDuration;
var localized string FlareFriendlyName;
var localized string FlareFriendlyDesc;
var localized string FlareEffectAcquiredString;
var localized string FlareEffectTickedString;
var localized string FlareEffectLostString;
var config float DeepReservesDamagePercentToHeal;
var config int DeepReservesMaxHealPerTurn;
var config int FireForEffectCooldown;
var config int FireForEffectTileWidth;
var config int FireForEffectAmmoCost;
var config int FireForEffectAbilityPointCost;

// This method is natively called for subclasses of X2DataSet. It'll create and return ability templates for our new class
static function array<X2DataTemplate> CreateTemplates()
{
	local array<X2DataTemplate> Templates;
	
	Templates.Length = 0;
	// Abilities that don't need much can be created using this handy function from Firaxis
	// We have to access local variables using default values because these are static methods. Defaults are defined at the end
	// of the file, apparently by convention. Look for 'DefaultProperties'
	Templates.AddItem(PurePassive(default.LightEmUpAbilityName, "img:///UILibrary_InfantryClass.UIPerk_lightemup"));
	Templates.AddItem(EstablishedDefenses()); // Many abilities will only require one template
	Templates.AddItem(Harrier());
	Templates.AddItem(CripplingShot());
	Templates.AddItem(CripplingShotDamage());
	Templates.AddItem(ShakeItOff());
	Templates.AddItem(StickAndMove()); // Some abilities use a series of templates for performing more complex ability interactions
	Templates.AddItem(StickAndMoveDodge());
	Templates.AddItem(StickAndMoveMobility());
	Templates.AddItem(ExplosiveAction());
	Templates.Additem(ZoneOfControl());
	Templates.Additem(ZoneOfControlShot());
	Templates.AddItem(EscapeAndEvade());
	Templates.AddItem(EscapeAndEvadeStealth());
	Templates.AddItem(DeepReserves());
	Templates.AddItem(FireForEffect());
	Templates.AddItem(Flare());

	return Templates;
}


//---------------------------------------------------------------------------------------------------
// Established Defenses
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate EstablishedDefenses()
{
	local X2AbilityTemplate						Template;
	local X2Effect_EstablishedDefenses          Effect;

	// This is some sort of macro by Firaxis that sets up an ability template with localized text from XComGame.int (and maybe some other stuff?)
	`CREATE_X2ABILITY_TEMPLATE(Template, 'EstablishedDefenses');

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow; // This ability doesn't show up on the action HUD (can't click to activate it)
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_fortress";

	Template.AbilityToHitCalc = default.DeadEye; // Always hits
	Template.AbilityTargetStyle = default.SelfTarget; // Applies to the unit with the ability
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger); // Basically begins immediately when the unit is spawned with the ability

	// This passive effect contains the logic for updating armor values based on changing conditions
	Effect = new class'X2Effect_EstablishedDefenses';
	Effect.EffectName = 'EstablishedDefenses';
	Effect.DuplicateResponse = eDupe_Ignore; // Shouldn't be a case where multiple copies of the passive effect are applied, but if they are ignore the new one
	Effect.BuildPersistentEffect(1, true, false); // Lasts forever
	Effect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage,,,Template.AbilitySourceName);
	Template.AddTargetEffect(Effect); // Effects added to the primary target of an ability. In this case, that's our unit

	// Function delegate for setting up a standard XComGameState_Ability object for this ability
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!
	// No function delegates for creating function visualizations because this passive ability isn't showy

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Harrier
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate Harrier()
{
	local X2AbilityTemplate						Template;
	local X2Effect_ToHitModifier                Effect;
	local X2Condition_FlankedTarget             FlankedCondition;

	// Log statements write to the debug log window
	`LOG("Lucubration Infantry Class: Harrier aim bonus=" @ string(default.HarrierAimBonus));
	`LOG("Lucubration Infantry Class: Harrier crit bonus=" @ string(default.HarrierCritBonus));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'Harrier');

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_shadowstrike";

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger);

	Effect = new class'X2Effect_ToHitModifier';
	Effect.EffectName = 'Harrier';
	Effect.DuplicateResponse = eDupe_Ignore;
	Effect.BuildPersistentEffect(1, true, false);
	Effect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage,,,Template.AbilitySourceName);
	Effect.AddEffectHitModifier(eHit_Success, default.HarrierAimBonus, Template.LocFriendlyName);//, /*StandardAim*/, true, true, true, false);
	Effect.AddEffectHitModifier(eHit_Crit, default.HarrierCritBonus, Template.LocFriendlyName);//, /*StandardAim*/, true, true, true, false);
	FlankedCondition = new class'X2Condition_FlankedTarget';
	Effect.ToHitConditions.AddItem(FlankedCondition);
	Template.AddTargetEffect(Effect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!
	
	Template.bCrossClassEligible = true;

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Crippling Shot
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate CripplingShot()
{
	local X2AbilityTemplate                 Template;
	local X2AbilityCost_Ammo                AmmoCost;
	local X2AbilityCost_ActionPoints        ActionPointCost;
	local X2Effect_PersistentStatChange     CripplingEffect;
	local array<name>                       SkipExclusions;
	local X2AbilityToHitCalc_StandardAim    StandardAim;
	local X2AbilityCooldown                 Cooldown;
	
	`LOG("Lucubration Infantry Class: Crippling Shot aim bonus=" @ string(default.CripplingShotAimBonus));
	`LOG("Lucubration Infantry Class: Crippling Shot ammo cost=" @ string(default.CripplingShotAmmoCost));
	`LOG("Lucubration Infantry Class: Crippling Shot action point cost=" @ string(default.CripplingShotActionPointCost));
	`LOG("Lucubration Infantry Class: Crippling Shot aim bonus=" @ string(default.CripplingShotCooldown));
	`LOG("Lucubration Infantry Class: Crippling Shot duration=" @ string(default.CripplingShotDuration));
	`LOG("Lucubration Infantry Class: Crippling Shot mobility reduction=" @ string(default.CripplingShotMobilityReduction));
	`LOG("Lucubration Infantry Class: Crippling Shot dodge reduction=" @ string(default.CripplingShotDodgeReduction));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'CripplingShot');
	
	Template.AdditionalAbilities.AddItem('CripplingShotDamage');

	// Icon Properties
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_bulletshred";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_LIEUTENANT_PRIORITY;
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_AlwaysShow;
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.DisplayTargetHitChance = true;
	Template.AbilityConfirmSound = "TacticalUI_ActivateAbility";

	// Activated by a button press; additionally, tells the AI this is an activatable
	Template.AbilityTriggers.AddItem(default.PlayerInputTrigger);

	// *** VALIDITY CHECKS *** //
	//  Normal effect restrictions (except disoriented)
	SkipExclusions.AddItem(class'X2AbilityTemplateManager'.default.DisorientedName);
	Template.AddShooterEffectExclusions(SkipExclusions);

	// Targeting Details
	// Can only shoot visible enemies
	Template.AbilityTargetConditions.AddItem(default.GameplayVisibilityCondition);
	// Can't target dead; Can't target friendlies
	Template.AbilityTargetConditions.AddItem(default.LivingHostileTargetProperty);
	// Can't shoot while dead
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	// Only at single targets that are in range.
	Template.AbilityTargetStyle = default.SimpleSingleTarget;

	// Action Point
	ActionPointCost = new class'X2AbilityCost_ActionPoints';
	ActionPointCost.iNumPoints = default.CripplingShotActionPointCost;
	ActionPointCost.bConsumeAllPoints = true;
	Template.AbilityCosts.AddItem(ActionPointCost);

	// Ammo
	AmmoCost = new class'X2AbilityCost_Ammo';	
	AmmoCost.iAmmo = default.CripplingShotAmmoCost;
	Template.AbilityCosts.AddItem(AmmoCost);
	Template.bAllowAmmoEffects = true;

	// Cooldown
	Cooldown = new class'X2AbilityCooldown';
	Cooldown.iNumTurns = default.CripplingShotCooldown;
	Template.AbilityCooldown = Cooldown;

	// Weapon Upgrade Compatibility
	Template.bAllowFreeFireWeaponUpgrade = true; // Flag that permits action to become 'free action' via 'Hair Trigger' or similar upgrade / effects

	// Allows this attack to work with the Holo-Targeting and Shredder perks, in case of AWC perkage
	Template.AddTargetEffect(class'X2Ability_GrenadierAbilitySet'.static.HoloTargetEffect());
	Template.AddTargetEffect(class'X2Ability_GrenadierAbilitySet'.static.ShredderDamageEffect());

	// There's some nice stuff built into the standard aim calculations, including a place to apply the aim bonus
	StandardAim = new class'X2AbilityToHitCalc_StandardAim';
	StandardAim.BuiltInHitMod = default.CripplingShotAimBonus;
	Template.AbilityToHitCalc = StandardAim;
	Template.AbilityToHitOwnerOnMissCalc = StandardAim;
		
	// Targeting Method. There's other ones that let you do grenade spheres, cones, etc. This is the standard, single-target selection
	Template.TargetingMethod = class'X2TargetingMethod_OverTheShoulder';
	Template.bUsesFiringCamera = true;
	Template.CinescriptCameraType = "StandardGunFiring";
	
	CripplingEffect = new class'X2Effect_PersistentStatChange';
	CripplingEffect.EffectName = default.CrippledEffectName;
	CripplingEffect.DuplicateResponse = eDupe_Refresh; // Refresh the duration of this debuff if it's already active
	CripplingEffect.BuildPersistentEffect(default.CripplingShotDuration,, false,,eGameRule_PlayerTurnBegin);
	CripplingEffect.SetDisplayInfo(ePerkBuff_Penalty, default.CrippledFriendlyName, default.CrippledFriendlyDesc, "img:///UILibrary_PerkIcons.UIPerk_disoriented");
	CripplingEffect.AddPersistentStatChange(eStat_Mobility, default.CripplingShotMobilityReduction);
	CripplingEffect.AddPersistentStatChange(eStat_Dodge, default.CripplingShotDodgeReduction);
	CripplingEffect.VisualizationFn = static.CrippledVisualization;
	CripplingEffect.EffectTickedVisualizationFn = static.CrippledVisualizationTicked;
	CripplingEffect.EffectRemovedVisualizationFn = static.CrippledVisualizationRemoved;
	CripplingEffect.EffectHierarchyValue = class'X2StatusEffects'.default.DISORIENTED_HIERARCHY_VALUE;
	CripplingEffect.bRemoveWhenTargetDies = true;
	CripplingEffect.bIsImpairingMomentarily = true;
	Template.AddTargetEffect(CripplingEffect);

	// MAKE IT LIVE!
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	Template.BuildVisualizationFn = TypicalAbility_BuildVisualization;	

	Template.bCrossClassEligible = true;

	return Template;	
}

// I could either create a new X2Effect_Crippled to display these visualizations (the flyover text), or I could
// use these static functions as delegates for creating the visualizations using the X2Effect_PersistentStateChange.
// I'm not going to make a new class just for this if it's not required
static function CrippledVisualization(XComGameState VisualizeGameState, out VisualizationTrack BuildTrack, const name EffectApplyResult)
{
	if( EffectApplyResult != 'AA_Success' )
	{
		return;
	}
	if (XComGameState_Unit(BuildTrack.StateObject_NewState) == none)
		return;

	class'X2StatusEffects'.static.AddEffectSoundAndFlyOverToTrack(BuildTrack, VisualizeGameState.GetContext(), default.CrippledFriendlyName, '', eColor_Bad, class'UIUtilities_Image'.const.UnitStatus_Marked);
	class'X2StatusEffects'.static.AddEffectMessageToTrack(BuildTrack, default.CrippledEffectAcquiredString, VisualizeGameState.GetContext());
	class'X2StatusEffects'.static.UpdateUnitFlag(BuildTrack, VisualizeGameState.GetContext());
}

static function CrippledVisualizationTicked(XComGameState VisualizeGameState, out VisualizationTrack BuildTrack, const name EffectApplyResult)
{
	local XComGameState_Unit UnitState;

	UnitState = XComGameState_Unit(BuildTrack.StateObject_NewState);
	if (UnitState == none)
		return;

	// dead units should not be reported
	if( !UnitState.IsAlive() )
	{
		return;
	}

	class'X2StatusEffects'.static.AddEffectSoundAndFlyOverToTrack(BuildTrack, VisualizeGameState.GetContext(), default.CrippledFriendlyName, '', eColor_Bad, class'UIUtilities_Image'.const.UnitStatus_Marked);
	class'X2StatusEffects'.static.AddEffectMessageToTrack(BuildTrack, default.CrippledEffectTickedString, VisualizeGameState.GetContext());
	class'X2StatusEffects'.static.UpdateUnitFlag(BuildTrack, VisualizeGameState.GetContext());
}

static function CrippledVisualizationRemoved(XComGameState VisualizeGameState, out VisualizationTrack BuildTrack, const name EffectApplyResult)
{
	local XComGameState_Unit UnitState;

	UnitState = XComGameState_Unit(BuildTrack.StateObject_NewState);
	if (UnitState == none)
		return;

	// dead units should not be reported
	if( !UnitState.IsAlive() )
	{
		return;
	}

	class'X2StatusEffects'.static.AddEffectMessageToTrack(BuildTrack, default.CrippledEffectLostString, VisualizeGameState.GetContext());
	class'X2StatusEffects'.static.UpdateUnitFlag(BuildTrack, VisualizeGameState.GetContext());
}

// The damage reduction is a seperate effect because it affects the shooter, not the shooter's target. It's a persistent
// effect that hangs around on the shooter and just modifies the damage for the Crippling Shot ability when it sees it
static function X2AbilityTemplate CripplingShotDamage()
{
	local X2AbilityTemplate						Template;
	local X2Effect_CripplingShotDamage          DamageEffect;

	// Icon Properties
	`CREATE_X2ABILITY_TEMPLATE(Template, 'CripplingShotDamage');
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_momentum";

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = EAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger);

	DamageEffect = new class'X2Effect_CripplingShotDamage';
	DamageEffect.BuildPersistentEffect(1, true, false);
	DamageEffect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage, false,,Template.AbilitySourceName);
	Template.AddTargetEffect(DamageEffect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Shake it Off
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate ShakeItOff()
{
	local X2AbilityTemplate                 Template;
	local X2AbilityCharges					Charges;
	local X2AbilityCost_ActionPoints        ActionPointCost;
	local X2AbilityCost_Charges				ChargeCost;
	local X2Condition_UnitEffects			ExcludeEffects;
	local X2Condition_ShakeItOff            Condition;
	local X2Effect_RemoveEffects            RemoveEffects;
	
	`LOG("Lucubration Infantry Class: Shake It Off charges=" @ string(default.ShakeItOffCharges));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'ShakeItOff');
	
	ActionPointCost = new class'X2AbilityCost_ActionPoints';
	ActionPointCost.iNumPoints = 1;
	ActionPointCost.bFreeCost = true;
	Template.AbilityCosts.AddItem(ActionPointCost);
	
	Charges = new class 'X2AbilityCharges';
	Charges.InitialCharges = default.ShakeItOffCharges;
	Template.AbilityCharges = Charges;
	
	ChargeCost = new class'X2AbilityCost_Charges';
	ChargeCost.NumCharges = 1;
	Template.AbilityCosts.AddItem(ChargeCost);

	Template.AbilityToHitCalc = default.DeadEye;

	Template.AbilityTargetStyle = default.SelfTarget;
	
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);

	// Rather than the normal shooter exclusion conditions, we're just going to exclude Carry Unit and Bound, as well as adding Mind Control
	ExcludeEffects = new class'X2Condition_UnitEffects';
	ExcludeEffects.AddExcludeEffect(class'X2Effect_MindControl'.default.EffectName, 'AA_UnitIsMindControlled');
	ExcludeEffects.AddExcludeEffect(class'X2Ability_CarryUnit'.default.CarryUnitEffectName, 'AA_CarryingUnit');
	ExcludeEffects.AddExcludeEffect(class'X2AbilityTemplateManager'.default.BoundName, 'AA_UnitIsBound');
	Template.AbilityShooterConditions.AddItem(ExcludeEffects);

	Condition = new class'X2Condition_ShakeItOff';
	Template.AbilityTargetConditions.AddItem(Condition);

	// Remove all the things
	RemoveEffects = new class'X2Effect_RemoveEffects';
	RemoveEffects.EffectNamesToRemove.AddItem(class'X2AbilityTemplateManager'.default.ConfusedName);
	RemoveEffects.EffectNamesToRemove.AddItem(class'X2AbilityTemplateManager'.default.DisorientedName);
	RemoveEffects.EffectNamesToRemove.AddItem(class'X2AbilityTemplateManager'.default.StunnedName);
	RemoveEffects.EffectNamesToRemove.AddItem(class'X2StatusEffects'.default.UnconsciousName);
	Template.AddTargetEffect(RemoveEffects);
	// Put the unit back to full actions
	Template.AddTargetEffect(new class'X2Effect_RestoreActionPoints');
	
	Template.AbilityTriggers.AddItem(default.PlayerInputTrigger);

	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_revivalprotocol";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_CORPORAL_PRIORITY;
	Template.Hostility = eHostility_Defensive;
	Template.bDisplayInUITooltip = false;
	Template.bLimitTargetIcons = true;
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.ActivationSpeech = 'CombatStim';
	
	Template.bShowActivation = true;
	Template.bSkipFireAction = true;
	Template.CustomSelfFireAnim = 'FF_FireMedkitSelf';
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	Template.BuildVisualizationFn = TypicalAbility_BuildVisualization;
	
	Template.bCrossClassEligible = true;

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Stick and Move
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate StickAndMove()
{
	local X2AbilityTemplate         Template;
	local X2Effect_StickAndMove     Effect;
	
	`CREATE_X2ABILITY_TEMPLATE(Template, 'StickAndMove');

	Template.AdditionalAbilities.AddItem(default.StickAndMoveMobilityAbilityName);
	Template.AdditionalAbilities.AddItem(default.StickAndMoveDodgeAbilityName);
	
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_implacable";
	Template.bIsPassive = true;
	
	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger);
		
	Effect = new class'X2Effect_StickAndMove';
	Effect.EffectName='StickAndMove';
	Effect.BuildPersistentEffect(1, true, false);
	Effect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage,,,Template.AbilitySourceName);
	Effect.DuplicateResponse = eDupe_Ignore;
	Template.AddTargetEffect(Effect);
	
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}

static function X2AbilityTemplate StickAndMoveMobility()
{
	local X2AbilityTemplate                 Template;
	local X2Effect_PersistentStatChange		MoveEffect;
	local array<name>                       SkipExclusions;
	
	`LOG("Lucubration Infantry Class: Stick And Move mobility bonus=" @ string(default.StickAndMoveMobilityBonus));
	`LOG("Lucubration Infantry Class: Stick And Move mobility duration=" @ string(default.StickAndMoveMobilityDuration));

	`CREATE_X2ABILITY_TEMPLATE(Template, default.StickAndMoveMobilityAbilityName);

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;

	// Triggered by the passive effect gamestate instead of using ability triggers. Not sure this
	// is the best way to do this, but I wanted to be pretty explicit about the conditions under
	// which it gets activated
	Template.AbilityTriggers.AddItem(new class'X2AbilityTrigger_Placeholder');
	
	// Specifically exclude activation while dead or already under the effect
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	SkipExclusions.AddItem(default.StickAndMoveMobilityEffectName);
	Template.AddShooterEffectExclusions(SkipExclusions);

	// The active effect
	MoveEffect = new class'X2Effect_PersistentStatChange';
	MoveEffect.EffectName = default.StickAndMoveMobilityEffectName;
	MoveEffect.SetDisplayInfo(ePerkBuff_Bonus, default.StickAndMoveMobilityFriendlyName, default.StickAndMoveMobilityFriendlyDesc, "img:///UILibrary_PerkIcons.UIPerk_momentum");
	MoveEffect.AddPersistentStatChange(eStat_Mobility, default.StickAndMoveMobilityBonus);
	MoveEffect.BuildPersistentEffect(default.StickAndMoveMobilityDuration,,,, eGameRule_PlayerTurnEnd);
	Template.AddTargetEffect(MoveEffect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}

static function X2AbilityTemplate StickAndMoveDodge()
{
	local X2AbilityTemplate                 Template;
	local X2Effect_PersistentStatChange		MoveEffect;
	local array<name>                       SkipExclusions;
	
	`LOG("Lucubration Infantry Class: Stick And Move dodge bonus=" @ string(default.StickAndMoveDodgeBonus));
	`LOG("Lucubration Infantry Class: Stick And Move dodge duration=" @ string(default.StickAndMoveDodgeDuration));

	`CREATE_X2ABILITY_TEMPLATE(Template, default.StickAndMoveDodgeAbilityName);

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;

	// Specifically exclude activation while dead or already under the effect
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	SkipExclusions.AddItem(default.StickAndMoveDodgeEffectName);
	Template.AddShooterEffectExclusions(SkipExclusions);

	// Triggered by the passive effect gamestate instead of using ability triggers. Not sure this
	// is the best way to do this, but I wanted to be pretty explicit about the conditions under
	// which it gets activated
	Template.AbilityTriggers.AddItem(new class'X2AbilityTrigger_Placeholder');
	
	// The active effect
	MoveEffect = new class'X2Effect_PersistentStatChange';
	MoveEffect.EffectName = 'StickAndMoveDodge';
	MoveEffect.SetDisplayInfo(ePerkBuff_Bonus, default.StickAndMoveDodgeFriendlyName, default.StickAndMoveDodgeFriendlyDesc, "img:///UILibrary_PerkIcons.UIPerk_momentum");
	MoveEffect.AddPersistentStatChange(eStat_Dodge, default.StickAndMoveDodgeBonus);
	MoveEffect.BuildPersistentEffect(default.StickAndMoveDodgeDuration,,,, eGameRule_PlayerTurnBegin);
	Template.AddTargetEffect(MoveEffect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Explosive Action
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate ExplosiveAction()
{
	local X2AbilityTemplate                 Template;
	local X2AbilityCost_ActionPoints        ActionPointCost;
	local X2AbilityCooldown					Cooldown;
	local X2Effect_ExplosiveAction          ActionEffect;
	local X2Effect_ExplosiveActionRecovery	RecoveryEffect;
	
	`LOG("Lucubration Infantry Class: Explosive Action cooldown=" @ string(default.ExplosiveActionCooldown));
	`LOG("Lucubration Infantry Class: Explosive Action bonus action points=" @ string(default.ExplosiveActionBonusActionPoints));
	`LOG("Lucubration Infantry Class: Explosive Action recovery delay=" @ string(default.ExplosiveActionRecoveryDelay));
	`LOG("Lucubration Infantry Class: Explosive Action recovery action points=" @ string(default.ExplosiveActionRecoveryActionPoints));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'ExplosiveAction');
	
	// Icon Properties
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_shaken";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_COLONEL_PRIORITY;
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_AlwaysShow;
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.AbilityConfirmSound = "TacticalUI_ActivateAbility";
	
	// Activated by a button press; additionally, tells the AI this is an activatable
	Template.AbilityTriggers.AddItem(default.PlayerInputTrigger);

	// Can't shoot while dead
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	
	ActionPointCost = new class'X2AbilityCost_ActionPoints';
	ActionPointCost.iNumPoints = 1;
	ActionPointCost.bFreeCost = true;
	Template.AbilityCosts.AddItem(ActionPointCost);
	
	Cooldown = new class'X2AbilityCooldown';
	Cooldown.iNumTurns = default.ExplosiveActionCooldown;
	Template.AbilityCooldown = Cooldown;

	Template.AbilityToHitCalc = default.DeadEye;

	Template.AbilityTargetStyle = default.SelfTarget;
	
	// Effect to immediately add action point(s)
	ActionEffect = new class'X2Effect_ExplosiveAction';
	ActionEffect.BonusActionPoints = default.ExplosiveActionBonusActionPoints;
	Template.AddTargetEffect(ActionEffect);

	// Delayed effect to eventually remove action point(s)
	RecoveryEffect = new class'X2Effect_ExplosiveActionRecovery';
	RecoveryEffect.EffectName = 'ExplosivelyRecovering';
	RecoveryEffect.DuplicateResponse = eDupe_Ignore;
	RecoveryEffect.EffectHierarchyValue = class'X2StatusEffects'.default.FRENZY_HIERARCHY_VALUE;
	RecoveryEffect.bRemoveWhenTargetDies = true;
	RecoveryEffect.RecoveryActionPoints = default.ExplosiveActionRecoveryActionPoints;
	RecoveryEffect.SetDisplayInfo(ePerkBuff_Penalty, default.ExplosiveActionRecoveryFriendlyName, default.ExplosiveActionRecoveryFriendlyDesc, "img:///UILibrary_PerkIcons.UIPerk_shaken", true,, Template.AbilitySourceName);
	RecoveryEffect.BuildPersistentEffect(default.ExplosiveActionRecoveryDelay,,,, eGameRule_PlayerTurnBegin);
	Template.AddTargetEffect(RecoveryEffect);
	
	Template.AbilityTriggers.AddItem(default.PlayerInputTrigger);

	Template.Hostility = eHostility_Defensive;
	Template.bDisplayInUITooltip = false;
	Template.bLimitTargetIcons = true;
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.ActivationSpeech = 'CombatStim';
	
	Template.bShowActivation = true;
	Template.bSkipFireAction = true;
	Template.CustomSelfFireAnim = 'FF_FireMedkitSelf';
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	Template.BuildVisualizationFn = TypicalAbility_BuildVisualization;
	
	Template.bCrossClassEligible = true;

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Zone of Control
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate ZoneOfControl()
{
	local X2AbilityTemplate						Template;
	local X2Effect_ZoneOfControl				Effect;
	
	`LOG("Lucubration Infantry Class: Zone of Control reaction fire radius=" @ string(default.ZoneOfControlReactionFireRadius));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'ZoneOfControl');

	Template.AdditionalAbilities.AddItem(default.ZoneOfControlReactionFireAbilityName);

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_sentinel";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_LIEUTENANT_PRIORITY;

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger);

	Effect = new class'X2Effect_ZoneOfControl';
	Effect.EffectName = 'ZoneOfControl';
	Effect.AbilityToActivate = default.ZoneOfControlReactionFireAbilityName;
	Effect.ActionPointName = default.ZoneOfControlActionPointName;
	Effect.ReactionFireRadius = default.ZoneOfControlReactionFireRadius;
	Effect.DuplicateResponse = eDupe_Ignore;
	Effect.BuildPersistentEffect(1, true, false);
	Effect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage,,,Template.AbilitySourceName);
	Template.AddTargetEffect(Effect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}

static function X2AbilityTemplate ZoneOfControlShot()
{
	local X2AbilityTemplate                 Template;
	local X2AbilityCost_ReserveActionPoints ReserveActionPointCost;

	`CREATE_X2ABILITY_TEMPLATE(Template, default.ZoneOfControlReactionFireAbilityName);	
	class'X2Ability_DefaultAbilitySet'.static.PistolOverwatchShotHelper(Template);
	Template.bShowActivation = true;

	ReserveActionPointCost = X2AbilityCost_ReserveActionPoints(Template.AbilityCosts[0]);
	ReserveActionPointCost.AllowedTypes.AddItem(default.ZoneOfControlActionPointName);
	
	return Template;
}


//---------------------------------------------------------------------------------------------------
// Escape and Evade
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate EscapeAndEvade()
{
	local X2AbilityTemplate						Template;
	local X2Effect_EscapeAndEvade				Effect;
	
	`LOG("Lucubration Infantry Class: Escape and Evade dodge bonus=" @ string(default.EscapeAndEvadeDodgeBonus));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'EscapeAndEvade');

	Template.AdditionalAbilities.AddItem(default.EscapeAndEvadeStealthAbilityName);

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_stealth";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_MAJOR_PRIORITY;

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger);

	// Combine stat change and event registration on the same effect
	Effect = new class'X2Effect_EscapeAndEvade';
	Effect.EffectName = 'EscapeAndEvade';
	Effect.AbilityToActivate = default.EscapeAndEvadeStealthAbilityName;
	Effect.ActionPointName = default.EscapeAndEvadeActionPointName;
	Effect.DuplicateResponse = eDupe_Ignore;
	Effect.AddPersistentStatChange(eStat_Dodge, default.EscapeAndEvadeDodgeBonus);
	Effect.BuildPersistentEffect(1, true, false);
	Effect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage,,,Template.AbilitySourceName);
	Template.AddTargetEffect(Effect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}


static function X2AbilityTemplate EscapeAndEvadeStealth()
{
	local X2AbilityTemplate						Template;
	local X2AbilityCost_ReserveActionPoints		ReserveActionPointCost;
	local X2AbilityTrigger_Event				Trigger;
	local X2Effect_RangerStealth                StealthEffect;
	//local X2AbilityCharges                      Charges;

	`CREATE_X2ABILITY_TEMPLATE(Template, default.EscapeAndEvadeStealthAbilityName);

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_stealth";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_MAJOR_PRIORITY;
	Template.bShowActivation = true;
	Template.bDontDisplayInAbilitySummary = true;

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	//Template.AbilityTriggers.AddItem(default.PlayerInputTrigger);
	//Template.AbilityCosts.AddItem(new class'X2AbilityCost_Charges');
	ReserveActionPointCost = new class'X2AbilityCost_ReserveActionPoints';
	ReserveActionPointCost.iNumPoints = 1;
	ReserveActionPointCost.AllowedTypes.AddItem(default.EscapeAndEvadeActionPointName);
	Template.AbilityCosts.AddItem(ReserveActionPointCost);

	//Charges = new class'X2AbilityCharges';
	//Charges.InitialCharges = default.STEALTH_CHARGES;
	//Template.AbilityCharges = Charges;

	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	Template.AbilityShooterConditions.AddItem(new class'X2Condition_Stealth');
	
	//Trigger on movement - interrupt the move
	Trigger = new class'X2AbilityTrigger_Event';
	Trigger.EventObserverClass = class'X2TacticalGameRuleset_AttackObserver';
	Trigger.MethodName = 'InterruptGameState';
	Template.AbilityTriggers.AddItem(Trigger);

	StealthEffect = new class'X2Effect_RangerStealth';
	StealthEffect.BuildPersistentEffect(1, true, true, false, eGameRule_PlayerTurnEnd);
	StealthEffect.SetDisplayInfo(ePerkBuff_Bonus, Template.LocFriendlyName, Template.GetMyHelpText(), Template.IconImage, true);
	StealthEffect.bRemoveWhenTargetConcealmentBroken = true;
	Template.AddTargetEffect(StealthEffect);

	Template.AddTargetEffect(class'X2Effect_Spotted'.static.CreateUnspottedEffect());

	Template.ActivationSpeech = 'ActivateConcealment';
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	Template.BuildVisualizationFn = TypicalAbility_BuildVisualization;
	Template.bSkipFireAction = true;

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Deep Reserves
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate DeepReserves()
{
	local X2AbilityTemplate						Template;
	local X2Effect_DeepReserves					Effect;
	
	`LOG("Lucubration Infantry Class: Deep Reserves damage percent to heal=" @ string(default.EscapeAndEvadeDodgeBonus));
	`LOG("Lucubration Infantry Class: Deep Reserves max heal per turn=" @ string(default.EscapeAndEvadeDodgeBonus));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'DeepReserves');

	Template.AdditionalAbilities.AddItem(default.EscapeAndEvadeStealthAbilityName);

	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_NeverShow;
	Template.Hostility = eHostility_Neutral;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_rapidregeneration";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_MAJOR_PRIORITY;

	Template.AbilityToHitCalc = default.DeadEye;
	Template.AbilityTargetStyle = default.SelfTarget;
	Template.AbilityTriggers.AddItem(default.UnitPostBeginPlayTrigger);

	// Combine stat change and event registration on the same effect
	Effect = new class'X2Effect_DeepReserves';
	Effect.EffectName = 'DeepReserves';
	Effect.DuplicateResponse = eDupe_Ignore;
	Effect.BuildPersistentEffect(1, true, false, , eGameRule_PlayerTurnBegin);
	Effect.SetDisplayInfo(ePerkBuff_Passive, Template.LocFriendlyName, Template.GetMyLongDescription(), Template.IconImage,,,Template.AbilitySourceName);
	Effect.HealAmountPerTurn = default.DeepReservesMaxHealPerTurn;
	Effect.HealDamagePercent = default.DeepReservesDamagePercentToHeal;
	Effect.HealthRegeneratedName = 'DeepReservesHealthRegenerated';
	Effect.DamageTakenName = 'DeepReservesDamageTaken';
	Template.AddTargetEffect(Effect);

	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	//  NOTE: No visualization on purpose!

	return Template;
}


//---------------------------------------------------------------------------------------------------
// Fire for Effect
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate FireForEffect()
{
	local X2AbilityTemplate					Template;
	local X2AbilityCost_ActionPoints		ActionPointCost;
	local X2AbilityCost_Ammo				AmmoCost;
	local X2AbilityMultiTarget_Cone         ConeMultiTarget;
	local X2AbilityToHitCalc_StandardAim    ToHitCalc;
	local X2AbilityCooldown                 Cooldown;
	local X2Effect_ApplyWeaponDamage		WeaponEffect;
	
	`LOG("Lucubration Infantry Class: Fire for Effect damage ability point cost=" @ string(default.FireForEffectAbilityPointCost));
	`LOG("Lucubration Infantry Class: Fire for Effect damage ammo cost=" @ string(default.FireForEffectAmmoCost));
	`LOG("Lucubration Infantry Class: Fire for Effect damage cooldown=" @ string(default.FireForEffectCooldown));
	`LOG("Lucubration Infantry Class: Fire for Effect damage cone width=" @ string(default.FireForEffectTileWidth));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'FireForEffect');

	ActionPointCost = new class'X2AbilityCost_ActionPoints';
	ActionPointCost.iNumPoints = default.FireForEffectAbilityPointCost;
	ActionPointCost.bConsumeAllPoints = true;
	Template.AbilityCosts.AddItem(ActionPointCost);

	AmmoCost = new class'X2AbilityCost_Ammo';
	AmmoCost.iAmmo = default.FireForEffectAmmoCost;
	Template.AbilityCosts.AddItem(AmmoCost);

	Cooldown = new class'X2AbilityCooldown';
	Cooldown.iNumTurns = default.FireForEffectCooldown;
	Template.AbilityCooldown = Cooldown;

	ToHitCalc = new class'X2AbilityToHitCalc_StandardAim';
	ToHitCalc.bGuaranteedHit = true;
	ToHitCalc.bAllowCrit = true;
	Template.AbilityToHitCalc = ToHitCalc;
	Template.AbilityToHitOwnerOnMissCalc = ToHitCalc;
	
	WeaponEffect = new class'X2Effect_ApplyWeaponDamage';
	Template.AddMultiTargetEffect(WeaponEffect);
	
	Template.AbilityTargetStyle = default.SimpleSingleTarget;
	
	ConeMultiTarget = new class'X2AbilityMultiTarget_Cone';
	ConeMultiTarget.bExcludeSelfAsTargetIfWithinRadius = true;
	ConeMultiTarget.ConeEndDiameter = default.FireForEffectTileWidth * class'XComWorldData'.const.WORLD_StepSize;
	ConeMultiTarget.bUseWeaponRangeForLength = true;
	ConeMultiTarget.fTargetRadius = 99;     //  large number to handle weapon range - targets will get filtered according to cone constraints
	ConeMultiTarget.bIgnoreBlockingCover = true;
	Template.AbilityMultiTargetStyle = ConeMultiTarget;

	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	Template.AddShooterEffectExclusions();

	Template.AbilityTargetConditions.AddItem(default.LivingHostileTargetProperty);
	Template.AbilityTargetConditions.AddItem(default.GameplayVisibilityCondition);
	
	// Allows this attack to work with the Holo-Targeting and Shredder perks, in case of AWC perkage
	Template.AddTargetEffect(class'X2Ability_GrenadierAbilitySet'.static.HoloTargetEffect());
	Template.AddTargetEffect(class'X2Ability_GrenadierAbilitySet'.static.ShredderDamageEffect());
	Template.bAllowAmmoEffects = true;

	Template.AbilityTriggers.AddItem(default.PlayerInputTrigger);
	
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_COLONEL_PRIORITY;
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.eAbilityIconBehaviorHUD = eAbilityIconBehavior_AlwaysShow;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_hailofbullets";
	Template.AbilityConfirmSound = "TacticalUI_ActivateAbility";
	
	Template.ActionFireClass = class'X2Action_Fire_Faceoff';

	Template.ActivationSpeech = 'SaturationFire';
	Template.CinescriptCameraType = "Grenadier_SaturationFire";
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	Template.BuildVisualizationFn = FireForEffect_BuildVisualization;
	Template.BuildInterruptGameStateFn = TypicalAbility_BuildInterruptGameState;

	Template.bCrossClassEligible = true;

	return Template;
}

// A custom attack visualization that's basically the same as the normal attack visualization except that we step out of cover
// and play the shoot animation twice, to reinforce the impression that we are sending more rounds downrange than normal
function FireForEffect_BuildVisualization(XComGameState VisualizeGameState, out array<VisualizationTrack> OutVisualizationTracks)
{		
	local X2AbilityTemplate             AbilityTemplate;
	local XComGameStateContext_Ability  Context;
	local AbilityInputContext           AbilityContext;
	local StateObjectReference          ShootingUnitRef;	
	local X2Action                      AddedAction;
	local array<X2Action>				AddedActions;
	local XComGameState_BaseObject      TargetStateObject;//Container for state objects within VisualizeGameState	
	local XComGameState_Item            SourceWeapon;
	local X2GrenadeTemplate             GrenadeTemplate;
	local X2AmmoTemplate                AmmoTemplate;
	local X2WeaponTemplate              WeaponTemplate;
	local array<X2Effect>               MultiTargetEffects;
	local bool							bSourceIsAlsoTarget;
	local bool							bMultiSourceIsAlsoTarget;
	
	local Actor                     TargetVisualizer, ShooterVisualizer;
	local X2VisualizerInterface     TargetVisualizerInterface, ShooterVisualizerInterface;
	local int                       EffectIndex, TargetIndex;
	local XComGameState_EnvironmentDamage EnvironmentDamageEvent;
	local XComGameState_WorldEffectTileData WorldDataUpdate;

	local VisualizationTrack        EmptyTrack;
	local VisualizationTrack        BuildTrack;
	local VisualizationTrack        SourceTrack, InterruptTrack;
	local int						TrackIndex;
	local bool						bAlreadyAdded;
	local XComGameStateHistory      History;
	local X2Action_MoveTurn         MoveTurnAction;

	local XComGameStateContext_Ability CounterAttackContext;
	local X2AbilityTemplate			CounterattackTemplate;
	local array<VisualizationTrack> OutCounterattackVisualizationTracks;
	local int						ActionIndex;

	local X2Action_PlaySoundAndFlyOver SoundAndFlyover;
	local name         ApplyResult;

	local XComGameState_InteractiveObject InteractiveObject;
	local XComGameState_Ability     AbilityState;
		
	History = `XCOMHISTORY;
	Context = XComGameStateContext_Ability(VisualizeGameState.GetContext());
	AbilityContext = Context.InputContext;
	AbilityState = XComGameState_Ability(History.GetGameStateForObjectID(AbilityContext.AbilityRef.ObjectID));
	AbilityTemplate = class'XComGameState_Ability'.static.GetMyTemplateManager().FindAbilityTemplate(AbilityContext.AbilityTemplateName);
	ShootingUnitRef = Context.InputContext.SourceObject;

	//Configure the visualization track for the shooter, part I. We split this into two parts since
	//in some situations the shooter can also be a target
	//****************************************************************************************
	ShooterVisualizer = History.GetVisualizer(ShootingUnitRef.ObjectID);
	ShooterVisualizerInterface = X2VisualizerInterface(ShooterVisualizer);

	SourceTrack = EmptyTrack;
	SourceTrack.StateObject_OldState = History.GetGameStateForObjectID(ShootingUnitRef.ObjectID, eReturnType_Reference, VisualizeGameState.HistoryIndex - 1);
	SourceTrack.StateObject_NewState = VisualizeGameState.GetGameStateForObjectID(ShootingUnitRef.ObjectID);
	if (SourceTrack.StateObject_NewState == none)
		SourceTrack.StateObject_NewState = SourceTrack.StateObject_OldState;
	SourceTrack.TrackActor = ShooterVisualizer;

	SourceTrack.AbilityName = AbilityTemplate.DataName;

	SourceWeapon = XComGameState_Item(History.GetGameStateForObjectID(AbilityContext.ItemObject.ObjectID));
	if (SourceWeapon != None)
	{
		WeaponTemplate = X2WeaponTemplate(SourceWeapon.GetMyTemplate());
		AmmoTemplate = X2AmmoTemplate(SourceWeapon.GetLoadedAmmoTemplate(AbilityState));
	}
	if(AbilityTemplate.bShowPostActivation)
	{
		//Show the text flyover at the end of the visualization after the camera pans back
		Context.PostBuildVisualizationFn.AddItem(ActivationFlyOver_PostBuildVisualization);
	}
	if (AbilityTemplate.bShowActivation || AbilityTemplate.ActivationSpeech != '')
	{
		SoundAndFlyOver = X2Action_PlaySoundAndFlyOver(class'X2Action_PlaySoundAndFlyover'.static.AddToVisualizationTrack(SourceTrack, Context));

		if (SourceWeapon != None)
		{
			GrenadeTemplate = X2GrenadeTemplate(SourceWeapon.GetMyTemplate());
		}

		if (GrenadeTemplate != none)
		{
			SoundAndFlyOver.SetSoundAndFlyOverParameters(None, "", GrenadeTemplate.OnThrowBarkSoundCue, eColor_Good);
		}
		else
		{
			SoundAndFlyOver.SetSoundAndFlyOverParameters(None, AbilityTemplate.bShowActivation ? AbilityTemplate.LocFriendlyName : "", AbilityTemplate.ActivationSpeech, eColor_Good, AbilityTemplate.bShowActivation ? AbilityTemplate.IconImage : "");
		}
	}

	if( Context.IsResultContextMiss() && AbilityTemplate.SourceMissSpeech != '' )
	{
		SoundAndFlyOver = X2Action_PlaySoundAndFlyOver(class'X2Action_PlaySoundAndFlyover'.static.AddToVisualizationTrack(BuildTrack, Context));
		SoundAndFlyOver.SetSoundAndFlyOverParameters(None, "", AbilityTemplate.SourceMissSpeech, eColor_Bad);
	}
	else if( Context.IsResultContextHit() && AbilityTemplate.SourceHitSpeech != '' )
	{
		SoundAndFlyOver = X2Action_PlaySoundAndFlyOver(class'X2Action_PlaySoundAndFlyover'.static.AddToVisualizationTrack(BuildTrack, Context));
		SoundAndFlyOver.SetSoundAndFlyOverParameters(None, "", AbilityTemplate.SourceHitSpeech, eColor_Good);
	}

	if (!AbilityTemplate.bSkipFireAction)
	{
		if (!AbilityTemplate.bSkipExitCoverWhenFiring)
		{
			class'X2Action_ExitCover'.static.AddToVisualizationTrack(SourceTrack, Context);
		}

		// no move, just add the fire action

		// Lucubration: The only special thing I'm going to do here is to try to chain together several attack animations in sequence
		// for this single attack ability
		AddedAction = class'X2Action_Fire_OpenUnfinishedAnim'.static.AddToVisualizationTrack(SourceTrack, Context);

		AddedAction = AbilityTemplate.ActionFireClass.static.AddToVisualizationTrack(SourceTrack, Context);
		AddedActions.AddItem(AddedAction);
		AddedAction = AbilityTemplate.ActionFireClass.static.AddToVisualizationTrack(SourceTrack, Context);
		AddedActions.AddItem(AddedAction);

		if( AbilityTemplate.AbilityToHitCalc != None )
		{
			foreach AddedActions(AddedAction)
			{
				X2Action_Fire(AddedAction).SetFireParameters( Context.IsResultContextHit() );
			}
		}

		//Process a potential counter attack from the target here
		if (Context.ResultContext.HitResult == eHit_CounterAttack)
		{
			CounterAttackContext = class'X2Ability'.static.FindCounterAttackGameState(Context, XComGameState_Unit(SourceTrack.StateObject_OldState));
			if (CounterAttackContext != none)
			{
				//Entering this code block means that we were the target of a counter attack to our original attack. Here, we look forward in the history
				//and append the necessary visualization tracks so that the counter attack can happen visually as part of our original attack.

				//Get the ability template for the counter attack against us
				CounterattackTemplate = class'X2AbilityTemplateManager'.static.GetAbilityTemplateManager().FindAbilityTemplate(CounterAttackContext.InputContext.AbilityTemplateName);
				CounterattackTemplate.BuildVisualizationFn(CounterAttackContext.AssociatedState, OutCounterattackVisualizationTracks);

				//Take the visualization actions from the counter attack game state ( where we are the target )
				for(TrackIndex = 0; TrackIndex < OutCounterattackVisualizationTracks.Length; ++TrackIndex)
				{
					if(OutCounterattackVisualizationTracks[TrackIndex].StateObject_OldState.ObjectID == SourceTrack.StateObject_OldState.ObjectID)
					{
						for(ActionIndex = 0; ActionIndex < OutCounterattackVisualizationTracks[TrackIndex].TrackActions.Length; ++ActionIndex)
						{
							//Don't include waits
							if(!OutCounterattackVisualizationTracks[TrackIndex].TrackActions[ActionIndex].IsA('X2Action_WaitForAbilityEffect'))
							{
								SourceTrack.TrackActions.AddItem(OutCounterattackVisualizationTracks[TrackIndex].TrackActions[ActionIndex]);
							}
						}
						break;
					}
				}
				
				//Notify the visualization mgr that the counter attack visualization is taken care of, so it can be skipped
				`XCOMVISUALIZATIONMGR.SkipVisualization(CounterAttackContext.AssociatedState.HistoryIndex);
			}
		}
	}

	//If there are effects added to the shooter, add the visualizer actions for them
	for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityShooterEffects.Length; ++EffectIndex)
	{
		AbilityTemplate.AbilityShooterEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, SourceTrack, Context.FindShooterEffectApplyResult(AbilityTemplate.AbilityShooterEffects[EffectIndex]));		
	}
	//****************************************************************************************

	//Configure the visualization track for the target(s). This functionality uses the context primarily
	//since the game state may not include state objects for misses.
	//****************************************************************************************	
	bSourceIsAlsoTarget = AbilityContext.PrimaryTarget.ObjectID == AbilityContext.SourceObject.ObjectID; //The shooter is the primary target
	if (AbilityTemplate.AbilityTargetEffects.Length > 0 &&			//There are effects to apply
		AbilityContext.PrimaryTarget.ObjectID > 0)				//There is a primary target
	{
		TargetVisualizer = History.GetVisualizer(AbilityContext.PrimaryTarget.ObjectID);
		TargetVisualizerInterface = X2VisualizerInterface(TargetVisualizer);

		if( bSourceIsAlsoTarget )
		{
			BuildTrack = SourceTrack;
		}
		else
		{
			BuildTrack = InterruptTrack;        //  interrupt track will either be empty or filled out correctly
		}

		BuildTrack.TrackActor = TargetVisualizer;

		TargetStateObject = VisualizeGameState.GetGameStateForObjectID(AbilityContext.PrimaryTarget.ObjectID);
		if( TargetStateObject != none )
		{
			History.GetCurrentAndPreviousGameStatesForObjectID(AbilityContext.PrimaryTarget.ObjectID, 
															   BuildTrack.StateObject_OldState, BuildTrack.StateObject_NewState,
															   eReturnType_Reference,
															   VisualizeGameState.HistoryIndex);
			`assert(BuildTrack.StateObject_NewState == TargetStateObject);
		}
		else
		{
			//If TargetStateObject is none, it means that the visualize game state does not contain an entry for the primary target. Use the history version
			//and show no change.
			BuildTrack.StateObject_OldState = History.GetGameStateForObjectID(AbilityContext.PrimaryTarget.ObjectID);
			BuildTrack.StateObject_NewState = BuildTrack.StateObject_OldState;
		}

		// if this is a melee attack, make sure the target is facing the location he will be melee'd from
		if(!AbilityTemplate.bSkipFireAction 
			&& !bSourceIsAlsoTarget 
			&& AbilityContext.MovementPaths.Length > 0
			&& AbilityContext.MovementPaths[0].MovementData.Length > 0
			&& XGUnit(TargetVisualizer) != none)
		{
			MoveTurnAction = X2Action_MoveTurn(class'X2Action_MoveTurn'.static.AddToVisualizationTrack(BuildTrack, Context));
			MoveTurnAction.m_vFacePoint = AbilityContext.MovementPaths[0].MovementData[AbilityContext.MovementPaths[0].MovementData.Length - 1].Position;
			MoveTurnAction.m_vFacePoint.Z = TargetVisualizerInterface.GetTargetingFocusLocation().Z;
			MoveTurnAction.UpdateAimTarget = true;
		}

		//Make the target wait until signaled by the shooter that the projectiles are hitting
		if (!AbilityTemplate.bSkipFireAction && !bSourceIsAlsoTarget)
		{
			class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);
		}
		
		//Add any X2Actions that are specific to this effect being applied. These actions would typically be instantaneous, showing UI world messages
		//playing any effect specific audio, starting effect specific effects, etc. However, they can also potentially perform animations on the 
		//track actor, so the design of effect actions must consider how they will look/play in sequence with other effects.
		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityTargetEffects.Length; ++EffectIndex)
		{
			ApplyResult = Context.FindTargetEffectApplyResult(AbilityTemplate.AbilityTargetEffects[EffectIndex]);

			// Target effect visualization
			AbilityTemplate.AbilityTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, ApplyResult);

			// Source effect visualization
			AbilityTemplate.AbilityTargetEffects[EffectIndex].AddX2ActionsForVisualizationSource(VisualizeGameState, SourceTrack, ApplyResult);
		}

		//the following is used to handle Rupture flyover text
		if (XComGameState_Unit(BuildTrack.StateObject_OldState).GetRupturedValue() == 0 &&
			XComGameState_Unit(BuildTrack.StateObject_NewState).GetRupturedValue() > 0)
		{
			//this is the frame that we realized we've been ruptured!
			class 'X2StatusEffects'.static.RuptureVisualization(VisualizeGameState, BuildTrack);
		}

		if (AbilityTemplate.bAllowAmmoEffects && AmmoTemplate != None)
		{
			for (EffectIndex = 0; EffectIndex < AmmoTemplate.TargetEffects.Length; ++EffectIndex)
			{
				ApplyResult = Context.FindTargetEffectApplyResult(AmmoTemplate.TargetEffects[EffectIndex]);
				AmmoTemplate.TargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, ApplyResult);
				AmmoTemplate.TargetEffects[EffectIndex].AddX2ActionsForVisualizationSource(VisualizeGameState, SourceTrack, ApplyResult);
			}
		}
		if (AbilityTemplate.bAllowBonusWeaponEffects && WeaponTemplate != none)
		{
			for (EffectIndex = 0; EffectIndex < WeaponTemplate.BonusWeaponEffects.Length; ++EffectIndex)
			{
				ApplyResult = Context.FindTargetEffectApplyResult(WeaponTemplate.BonusWeaponEffects[EffectIndex]);
				WeaponTemplate.BonusWeaponEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, ApplyResult);
				WeaponTemplate.BonusWeaponEffects[EffectIndex].AddX2ActionsForVisualizationSource(VisualizeGameState, SourceTrack, ApplyResult);
			}
		}

		if (Context.IsResultContextMiss() && (AbilityTemplate.LocMissMessage != "" || AbilityTemplate.TargetMissSpeech != ''))
		{
			SoundAndFlyOver = X2Action_PlaySoundAndFlyOver(class'X2Action_PlaySoundAndFlyover'.static.AddToVisualizationTrack(BuildTrack, Context));
			SoundAndFlyOver.SetSoundAndFlyOverParameters(None, AbilityTemplate.LocMissMessage, AbilityTemplate.TargetMissSpeech, eColor_Bad);
		}
		else if( Context.IsResultContextHit() && (AbilityTemplate.LocHitMessage != "" || AbilityTemplate.TargetHitSpeech != '') )
		{
			SoundAndFlyOver = X2Action_PlaySoundAndFlyOver(class'X2Action_PlaySoundAndFlyover'.static.AddToVisualizationTrack(BuildTrack, Context));
			SoundAndFlyOver.SetSoundAndFlyOverParameters(None, AbilityTemplate.LocHitMessage, AbilityTemplate.TargetHitSpeech, eColor_Good);
		}

		if( TargetVisualizerInterface != none )
		{
			//Allow the visualizer to do any custom processing based on the new game state. For example, units will create a death action when they reach 0 HP.
			TargetVisualizerInterface.BuildAbilityEffectsVisualization(VisualizeGameState, BuildTrack);
		}

		if (!bSourceIsAlsoTarget && BuildTrack.TrackActions.Length > 0)
		{
			OutVisualizationTracks.AddItem(BuildTrack);
		}

		if( bSourceIsAlsoTarget )
		{
			SourceTrack = BuildTrack;
		}
	}

	if (AbilityTemplate.bUseLaunchedGrenadeEffects)
	{
		MultiTargetEffects = X2GrenadeTemplate(SourceWeapon.GetLoadedAmmoTemplate(AbilityState)).LaunchedGrenadeEffects;
	}
	else if (AbilityTemplate.bUseThrownGrenadeEffects)
	{
		MultiTargetEffects = X2GrenadeTemplate(SourceWeapon.GetMyTemplate()).ThrownGrenadeEffects;
	}
	else
	{
		MultiTargetEffects = AbilityTemplate.AbilityMultiTargetEffects;
	}

	//  Apply effects to multi targets
	if( MultiTargetEffects.Length > 0 && AbilityContext.MultiTargets.Length > 0)
	{
		for( TargetIndex = 0; TargetIndex < AbilityContext.MultiTargets.Length; ++TargetIndex )
		{	
			bMultiSourceIsAlsoTarget = false;
			if( AbilityContext.MultiTargets[TargetIndex].ObjectID == AbilityContext.SourceObject.ObjectID )
			{
				bMultiSourceIsAlsoTarget = true;
				bSourceIsAlsoTarget = bMultiSourceIsAlsoTarget;				
			}

			//Some abilities add the same target multiple times into the targets list - see if this is the case and avoid adding redundant tracks
			bAlreadyAdded = false;
			for( TrackIndex = 0; TrackIndex < OutVisualizationTracks.Length; ++TrackIndex )
			{
				if( OutVisualizationTracks[TrackIndex].StateObject_NewState.ObjectID == AbilityContext.MultiTargets[TargetIndex].ObjectID )
				{
					bAlreadyAdded = true;
				}
			}

			if( !bAlreadyAdded )
			{
				TargetVisualizer = History.GetVisualizer(AbilityContext.MultiTargets[TargetIndex].ObjectID);
				TargetVisualizerInterface = X2VisualizerInterface(TargetVisualizer);

				if( bMultiSourceIsAlsoTarget )
				{
					BuildTrack = SourceTrack;
				}
				else
				{
					BuildTrack = EmptyTrack;
				}
				BuildTrack.TrackActor = TargetVisualizer;

				TargetStateObject = VisualizeGameState.GetGameStateForObjectID(AbilityContext.MultiTargets[TargetIndex].ObjectID);
				if( TargetStateObject != none )
				{
					History.GetCurrentAndPreviousGameStatesForObjectID(AbilityContext.MultiTargets[TargetIndex].ObjectID, 
																	   BuildTrack.StateObject_OldState, BuildTrack.StateObject_NewState,
																	   eReturnType_Reference,
																	   VisualizeGameState.HistoryIndex);
					`assert(BuildTrack.StateObject_NewState == TargetStateObject);
				}			
				else
				{
					//If TargetStateObject is none, it means that the visualize game state does not contain an entry for the primary target. Use the history version
					//and show no change.
					BuildTrack.StateObject_OldState = History.GetGameStateForObjectID(AbilityContext.PrimaryTarget.ObjectID);
					BuildTrack.StateObject_NewState = BuildTrack.StateObject_OldState;
				}

				//Make the target wait until signaled by the shooter that the projectiles are hitting
				if (!AbilityTemplate.bSkipFireAction && !bMultiSourceIsAlsoTarget)
				{
					class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);
				}
		
				//Add any X2Actions that are specific to this effect being applied. These actions would typically be instantaneous, showing UI world messages
				//playing any effect specific audio, starting effect specific effects, etc. However, they can also potentially perform animations on the 
				//track actor, so the design of effect actions must consider how they will look/play in sequence with other effects.
				for (EffectIndex = 0; EffectIndex < MultiTargetEffects.Length; ++EffectIndex)
				{
					ApplyResult = Context.FindMultiTargetEffectApplyResult(MultiTargetEffects[EffectIndex], TargetIndex);

					// Target effect visualization
					MultiTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, ApplyResult);

					// Source effect visualization
					MultiTargetEffects[EffectIndex].AddX2ActionsForVisualizationSource(VisualizeGameState, SourceTrack, ApplyResult);
				}			

				//the following is used to handle Rupture flyover text
				if (XComGameState_Unit(BuildTrack.StateObject_OldState).GetRupturedValue() == 0 &&
					XComGameState_Unit(BuildTrack.StateObject_NewState).GetRupturedValue() > 0)
				{
					//this is the frame that we realized we've been ruptured!
					class 'X2StatusEffects'.static.RuptureVisualization(VisualizeGameState, BuildTrack);
				}

				if( TargetVisualizerInterface != none )
				{
					//Allow the visualizer to do any custom processing based on the new game state. For example, units will create a death action when they reach 0 HP.
					TargetVisualizerInterface.BuildAbilityEffectsVisualization(VisualizeGameState, BuildTrack);
				}

				if( !bMultiSourceIsAlsoTarget && BuildTrack.TrackActions.Length > 0 )
				{
					OutVisualizationTracks.AddItem(BuildTrack);
				}

				if( bMultiSourceIsAlsoTarget )
				{
					SourceTrack = BuildTrack;
				}
			}
		}
	}
	//****************************************************************************************

	//Finish adding the shooter's track
	//****************************************************************************************
	if( !bSourceIsAlsoTarget && ShooterVisualizerInterface != none)
	{
		ShooterVisualizerInterface.BuildAbilityEffectsVisualization(VisualizeGameState, SourceTrack);				
	}	

	if (!AbilityTemplate.bSkipFireAction)
	{
		if (!AbilityTemplate.bSkipExitCoverWhenFiring)
		{
			class'X2Action_EnterCover'.static.AddToVisualizationTrack(SourceTrack, Context);
		}
	}
	
	OutVisualizationTracks.AddItem(SourceTrack);
	//****************************************************************************************

	//Configure the visualization tracks for the environment
	//****************************************************************************************
	foreach VisualizeGameState.IterateByClassType(class'XComGameState_EnvironmentDamage', EnvironmentDamageEvent)
	{
		BuildTrack = EmptyTrack;
		BuildTrack.TrackActor = none;
		BuildTrack.StateObject_NewState = EnvironmentDamageEvent;
		BuildTrack.StateObject_OldState = EnvironmentDamageEvent;

		//Wait until signaled by the shooter that the projectiles are hitting
		if (!AbilityTemplate.bSkipFireAction)
			class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityShooterEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityShooterEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');		
		}

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityTargetEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');
		}

		for (EffectIndex = 0; EffectIndex < MultiTargetEffects.Length; ++EffectIndex)
		{
			MultiTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');	
		}

		OutVisualizationTracks.AddItem(BuildTrack);
	}

	foreach VisualizeGameState.IterateByClassType(class'XComGameState_WorldEffectTileData', WorldDataUpdate)
	{
		BuildTrack = EmptyTrack;
		BuildTrack.TrackActor = none;
		BuildTrack.StateObject_NewState = WorldDataUpdate;
		BuildTrack.StateObject_OldState = WorldDataUpdate;

		//Wait until signaled by the shooter that the projectiles are hitting
		if (!AbilityTemplate.bSkipFireAction)
			class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityShooterEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityShooterEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');		
		}

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityTargetEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');
		}

		for (EffectIndex = 0; EffectIndex < MultiTargetEffects.Length; ++EffectIndex)
		{
			MultiTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');	
		}

		OutVisualizationTracks.AddItem(BuildTrack);
	}
	//****************************************************************************************

	//Process any interactions with interactive objects
	foreach VisualizeGameState.IterateByClassType(class'XComGameState_InteractiveObject', InteractiveObject)
	{
		// Add any doors that need to listen for notification
		if (InteractiveObject.IsDoor() && InteractiveObject.HasDestroyAnim()) //Is this a closed door?
		{
			BuildTrack = EmptyTrack;
			//Don't necessarily have a previous state, so just use the one we know about
			BuildTrack.StateObject_OldState = InteractiveObject;
			BuildTrack.StateObject_NewState = InteractiveObject;
			BuildTrack.TrackActor = History.GetVisualizer(InteractiveObject.ObjectID);

			if (!AbilityTemplate.bSkipFireAction)
				class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);

			class'X2Action_BreakInteractActor'.static.AddToVisualizationTrack(BuildTrack, Context);

			OutVisualizationTracks.AddItem(BuildTrack);
		}
	}
}


//---------------------------------------------------------------------------------------------------
// Flare
//---------------------------------------------------------------------------------------------------


static function X2AbilityTemplate Flare()
{
	local X2AbilityTemplate                 Template;	
	local X2AbilityCost_ActionPoints        ActionPointCost;
	local X2AbilityCharges					Charges;
	local X2AbilityCost_Charges				ChargeCost;
	local X2AbilityTarget_Cursor            CursorTarget;
	local X2AbilityMultiTarget_Radius       RadiusMultiTarget;
	local X2Condition_UnitProperty          UnitPropertyCondition;
	local X2AbilityTrigger_PlayerInput      InputTrigger;
	local X2Effect_ApplyFlareTargetToWorld	FlareEffect;
	
	`LOG("Lucubration Infantry Class: Flare charges=" @ string(default.FlareCharges));
	`LOG("Lucubration Infantry Class: Flare radius=" @ string(default.FlareRadius));
	`LOG("Lucubration Infantry Class: Flare height=" @ string(default.FlareHeight));
	`LOG("Lucubration Infantry Class: Flare range=" @ string(default.FlareRange));

	`CREATE_X2ABILITY_TEMPLATE(Template, 'Flare');
	Template.bDontDisplayInAbilitySummary = false;
	Template.IconImage = "img:///UILibrary_PerkIcons.UIPerk_demolition";
	Template.ShotHUDPriority = class'UIUtilities_Tactical'.const.CLASS_CORPORAL_PRIORITY;
	Template.eAbilityIconBehaviorHUD = EAbilityIconBehavior_AlwaysShow;
	Template.AbilitySourceName = 'eAbilitySource_Perk';
	Template.bShowActivation = false;
	Template.bShowPostActivation = false;
	Template.bSkipFireAction = false;
	Template.CustomFireAnim = 'HL_SignalPoint';

	ActionPointCost = new class'X2AbilityCost_ActionPoints';
	ActionPointCost.iNumPoints = 1;
	Template.AbilityCosts.AddItem(ActionPointCost);
	
	Charges = new class 'X2AbilityCharges';
	Charges.InitialCharges = default.FlareCharges;
	Template.AbilityCharges = Charges;
	
	ChargeCost = new class'X2AbilityCost_Charges';
	ChargeCost.NumCharges = 1;
	Template.AbilityCosts.AddItem(ChargeCost);

	Template.AbilityToHitCalc = default.DeadEye;
	
	Template.AbilityShooterConditions.AddItem(default.LivingShooterProperty);
	Template.AddShooterEffectExclusions();

	// The actual flare FX goes in shooter array as there will be no single target hit and no side effects of the flare FX on other units
	FlareEffect = new class'X2Effect_ApplyFlareTargetToWorld';
	FlareEffect.DuplicateResponse = eDupe_Ignore;
	FlareEffect.BuildPersistentEffect(default.FlareDuration, false, false, false, eGameRule_PlayerTurnEnd);
	// TESTING: Let's see if this effect even shows up on our character
	FlareEffect.SetDisplayInfo(ePerkBuff_Bonus, "Flare", "TESING: Flare effect is applied.", "img:///UILibrary_PerkIcons.UIPerk_shaken", true,, 'Flare');
	FlareEffect.EffectName = 'Flare';
	Template.AddShooterEffect(FlareEffect);

	// Apply the negative status to targets
	Template.AddMultiTargetEffect(static.CreateFlareStatusEffect());

	CursorTarget = new class'X2AbilityTarget_Cursor';
	CursorTarget.FixedAbilityRange = default.FlareRange;
	Template.AbilityTargetStyle = CursorTarget;
	
	RadiusMultiTarget = new class'X2AbilityMultiTarget_Radius';
	RadiusMultiTarget.fTargetRadius = default.FlareRadius;
	Template.AbilityMultiTargetStyle = RadiusMultiTarget;

	// Prevent the effect from apply to dead targets
	UnitPropertyCondition = new class'X2Condition_UnitProperty';
	UnitPropertyCondition.ExcludeDead = true;
	Template.AbilityShooterConditions.AddItem(UnitPropertyCondition); 
	Template.AddShooterEffectExclusions();

	InputTrigger = new class'X2AbilityTrigger_PlayerInput';
	Template.AbilityTriggers.AddItem(InputTrigger);
	
	Template.BuildNewGameStateFn = TypicalAbility_BuildGameState;
	Template.BuildVisualizationFn = Flare_BuildVisualization;
	Template.CinescriptCameraType = "Viper_PoisonSpit";//"Mark_Target";

	Template.TargetingMethod = class'X2TargetingMethod_Flare';

	// This action is considered 'hostile' and can be interrupted!
	Template.Hostility = eHostility_Offensive;
	Template.BuildInterruptGameStateFn = TypicalAbility_BuildInterruptGameState;
	
	Template.bCrossClassEligible = true;

	return Template;
}

static function X2Effect_Illuminated CreateFlareStatusEffect()
{
	local X2Effect_Illuminated				FlareEffect;
	local X2Condition_UnitProperty			UnitPropCondition;

	// Flare effect always lasts for 1 turn. The ticking of the world spawn flare refreshes it on enemies
	// inside the flare AoE, and entering the flare AoE will apply it to enemies
	FlareEffect = new class'X2Effect_Illuminated';
	FlareEffect.EffectName = default.FlareStatusEffectName;
	FlareEffect.DuplicateResponse = eDupe_Refresh;
	FlareEffect.BuildPersistentEffect(1,,,,eGameRule_PlayerTurnEnd);
	FlareEffect.SetDisplayInfo(ePerkBuff_Penalty, default.FlareFriendlyName, default.FlareFriendlyDesc, "img:///UILibrary_PerkIcons.UIPerk_demolition", true,, 'eAbilitySource_Perk');
	FlareEffect.VisualizationFn = static.IlluminatedVisualization;
	FlareEffect.EffectTickedVisualizationFn = static.IlluminatedVisualizationTicked;
	FlareEffect.EffectRemovedVisualizationFn = static.IlluminatedVisualizationRemoved;
	FlareEffect.bRemoveWhenTargetDies = true;

	UnitPropCondition = new class'X2Condition_UnitProperty';
	UnitPropCondition.ExcludeFriendlyToSource = false;
	FlareEffect.TargetConditions.AddItem(UnitPropCondition);

	return FlareEffect;
}

static function IlluminatedVisualization(XComGameState VisualizeGameState, out VisualizationTrack BuildTrack, const name EffectApplyResult)
{
	if( EffectApplyResult != 'AA_Success' )
	{
		return;
	}
	if (XComGameState_Unit(BuildTrack.StateObject_NewState) == none)
		return;

	class'X2StatusEffects'.static.AddEffectSoundAndFlyOverToTrack(BuildTrack, VisualizeGameState.GetContext(), default.FlareFriendlyName, '', eColor_Bad, class'UIUtilities_Image'.const.UnitStatus_Marked);
	class'X2StatusEffects'.static.AddEffectMessageToTrack(BuildTrack, default.FlareEffectAcquiredString, VisualizeGameState.GetContext());
	class'X2StatusEffects'.static.UpdateUnitFlag(BuildTrack, VisualizeGameState.GetContext());
}

static function IlluminatedVisualizationTicked(XComGameState VisualizeGameState, out VisualizationTrack BuildTrack, const name EffectApplyResult)
{
	local XComGameState_Unit UnitState;

	UnitState = XComGameState_Unit(BuildTrack.StateObject_NewState);
	if (UnitState == none)
		return;

	// dead units should not be reported
	if( !UnitState.IsAlive() )
	{
		return;
	}

	class'X2StatusEffects'.static.AddEffectSoundAndFlyOverToTrack(BuildTrack, VisualizeGameState.GetContext(), default.FlareFriendlyName, '', eColor_Bad, class'UIUtilities_Image'.const.UnitStatus_Marked);
	class'X2StatusEffects'.static.AddEffectMessageToTrack(BuildTrack, default.FlareEffectTickedString, VisualizeGameState.GetContext());
	class'X2StatusEffects'.static.UpdateUnitFlag(BuildTrack, VisualizeGameState.GetContext());
}

static function IlluminatedVisualizationRemoved(XComGameState VisualizeGameState, out VisualizationTrack BuildTrack, const name EffectApplyResult)
{
	local XComGameState_Unit UnitState;

	UnitState = XComGameState_Unit(BuildTrack.StateObject_NewState);
	if (UnitState == none)
		return;

	// dead units should not be reported
	if( !UnitState.IsAlive() )
	{
		return;
	}

	class'X2StatusEffects'.static.AddEffectMessageToTrack(BuildTrack, default.FlareEffectLostString, VisualizeGameState.GetContext());
	class'X2StatusEffects'.static.UpdateUnitFlag(BuildTrack, VisualizeGameState.GetContext());
}

simulated function Flare_BuildVisualization(XComGameState VisualizeGameState, out array<VisualizationTrack> OutVisualizationTracks)
{
	local X2AbilityTemplate             AbilityTemplate;
	local XComGameStateContext_Ability  Context;
	local AbilityInputContext           AbilityContext;
	local StateObjectReference          ShootingUnitRef;
	local XComGameState_BaseObject      TargetStateObject;//Container for state objects within VisualizeGameState	
	local array<X2Effect>               MultiTargetEffects;
	local bool							bSourceIsAlsoTarget;
	local bool							bMultiSourceIsAlsoTarget;
	
	local Actor                     TargetVisualizer, ShooterVisualizer;
	local X2VisualizerInterface     TargetVisualizerInterface, ShooterVisualizerInterface;
	local int                       EffectIndex, TargetIndex;
	local XComGameState_EnvironmentDamage EnvironmentDamageEvent;
	local XComGameState_WorldEffectTileData WorldDataUpdate;

	local VisualizationTrack        EmptyTrack;
	local VisualizationTrack        BuildTrack;
	local VisualizationTrack        SourceTrack;
	local int						TrackIndex;
	local bool						bAlreadyAdded;
	local XComGameStateHistory      History;

	local X2Action_SendInterTrackMessage    SendMessageAction;
	local X2Action_WaitForAbilityEffect		WaitForAbilityEffect;
	local X2Action_PlayAnimation            PlayAnimation;

	local name         ApplyResult;

	local XComGameState_InteractiveObject InteractiveObject;
		
	History = `XCOMHISTORY;
	Context = XComGameStateContext_Ability(VisualizeGameState.GetContext());

	AbilityContext = Context.InputContext;
	AbilityTemplate = class'XComGameState_Ability'.static.GetMyTemplateManager().FindAbilityTemplate(AbilityContext.AbilityTemplateName);

	//Configure the visualization track for the shooter, part I. We split this into two parts since
	//in some situations the shooter can also be a target
	//****************************************************************************************
	ShootingUnitRef = Context.InputContext.SourceObject;
	ShooterVisualizer = History.GetVisualizer(ShootingUnitRef.ObjectID);
	ShooterVisualizerInterface = X2VisualizerInterface(ShooterVisualizer);

	SourceTrack = EmptyTrack;
	SourceTrack.StateObject_OldState = History.GetGameStateForObjectID(ShootingUnitRef.ObjectID, eReturnType_Reference, VisualizeGameState.HistoryIndex - 1);
	SourceTrack.StateObject_NewState = VisualizeGameState.GetGameStateForObjectID(ShootingUnitRef.ObjectID);
	SourceTrack.TrackActor = ShooterVisualizer;

	if (!AbilityTemplate.bSkipFireAction)
	{
		if (!AbilityTemplate.bSkipExitCoverWhenFiring)
		{
			class'X2Action_ExitCover'.static.AddToVisualizationTrack(SourceTrack, Context);
		}
	}
	
	PlayAnimation = X2Action_PlayAnimation(class'X2Action_PlayAnimation'.static.AddToVisualizationTrack(SourceTrack, Context));
	PlayAnimation.Params.AnimName = 'HL_SignalPoint';
	
	//If there are effects added to the shooter, add the visualizer actions for them
	for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityShooterEffects.Length; ++EffectIndex)
	{
		AbilityTemplate.AbilityShooterEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, SourceTrack, Context.FindShooterEffectApplyResult(AbilityTemplate.AbilityShooterEffects[EffectIndex]));
	}

	//****************************************************************************************

	//Configure the visualization track for the target(s). This functionality uses the context primarily
	//since the game state may not include state objects for misses.
	//****************************************************************************************
	MultiTargetEffects = AbilityTemplate.AbilityMultiTargetEffects;
	
	// Apply effects to multi targets
	if( MultiTargetEffects.Length > 0 && AbilityContext.MultiTargets.Length > 0)
	{
		for( TargetIndex = 0; TargetIndex < AbilityContext.MultiTargets.Length; ++TargetIndex )
		{	
			bMultiSourceIsAlsoTarget = false;
			if( AbilityContext.MultiTargets[TargetIndex].ObjectID == AbilityContext.SourceObject.ObjectID )
			{
				bMultiSourceIsAlsoTarget = true;
				bSourceIsAlsoTarget = bMultiSourceIsAlsoTarget;				
			}

			//Some abilities add the same target multiple times into the targets list - see if this is the case and avoid adding redundant tracks
			bAlreadyAdded = false;
			for( TrackIndex = 0; TrackIndex < OutVisualizationTracks.Length; ++TrackIndex )
			{
				if( OutVisualizationTracks[TrackIndex].StateObject_NewState.ObjectID == AbilityContext.MultiTargets[TargetIndex].ObjectID )
				{
					bAlreadyAdded = true;
				}
			}

			if( !bAlreadyAdded )
			{
				TargetVisualizer = History.GetVisualizer(AbilityContext.MultiTargets[TargetIndex].ObjectID);
				TargetVisualizerInterface = X2VisualizerInterface(TargetVisualizer);

				if( bMultiSourceIsAlsoTarget )
				{
					BuildTrack = SourceTrack;
				}
				else
				{
					BuildTrack = EmptyTrack;
				}
				BuildTrack.TrackActor = TargetVisualizer;

				TargetStateObject = VisualizeGameState.GetGameStateForObjectID(AbilityContext.MultiTargets[TargetIndex].ObjectID);
				if( TargetStateObject != none )
				{
					History.GetCurrentAndPreviousGameStatesForObjectID(AbilityContext.MultiTargets[TargetIndex].ObjectID, 
																	   BuildTrack.StateObject_OldState, BuildTrack.StateObject_NewState,
																	   eReturnType_Reference,
																	   VisualizeGameState.HistoryIndex);
					`assert(BuildTrack.StateObject_NewState == TargetStateObject);
				}			
				else
				{
					//If TargetStateObject is none, it means that the visualize game state does not contain an entry for the primary target. Use the history version
					//and show no change.
					BuildTrack.StateObject_OldState = History.GetGameStateForObjectID(AbilityContext.PrimaryTarget.ObjectID);
					BuildTrack.StateObject_NewState = BuildTrack.StateObject_OldState;
				}

				//Make the target wait until signaled by the shooter that the projectiles are hitting
				if (!AbilityTemplate.bSkipFireAction && !bMultiSourceIsAlsoTarget)
				{
					//WaitForAbilityEffect = X2Action_WaitForAbilityEffect(class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context));
					//WaitForAbilityEffect.ChangeTimeoutLength(5.0f);
				}
		
				//Add any X2Actions that are specific to this effect being applied. These actions would typically be instantaneous, showing UI world messages
				//playing any effect specific audio, starting effect specific effects, etc. However, they can also potentially perform animations on the 
				//track actor, so the design of effect actions must consider how they will look/play in sequence with other effects.
				for (EffectIndex = 0; EffectIndex < MultiTargetEffects.Length; ++EffectIndex)
				{
					ApplyResult = Context.FindMultiTargetEffectApplyResult(MultiTargetEffects[EffectIndex], TargetIndex);

					// Target effect visualization
					MultiTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, ApplyResult);
				}			

				if( TargetVisualizerInterface != none )
				{
					//Allow the visualizer to do any custom processing based on the new game state. For example, units will create a death action when they reach 0 HP.
					TargetVisualizerInterface.BuildAbilityEffectsVisualization(VisualizeGameState, BuildTrack);
				}

				if( !bMultiSourceIsAlsoTarget && BuildTrack.TrackActions.Length > 0 )
				{
					OutVisualizationTracks.AddItem(BuildTrack);
				}

				if( bMultiSourceIsAlsoTarget )
				{
					SourceTrack = BuildTrack;
				}
			}
		}
	}
	//****************************************************************************************

	//Finish adding the shooter's track
	//****************************************************************************************
	if( !bSourceIsAlsoTarget && ShooterVisualizerInterface != none)
	{
		ShooterVisualizerInterface.BuildAbilityEffectsVisualization(VisualizeGameState, SourceTrack);				
	}	

	if (!AbilityTemplate.bSkipFireAction)
	{
		if (!AbilityTemplate.bSkipExitCoverWhenFiring)
		{
			class'X2Action_EnterCover'.static.AddToVisualizationTrack(SourceTrack, Context);
		}
	}

	OutVisualizationTracks.AddItem(SourceTrack);

	/*
	//Configure the visualization tracks for the environment
	foreach VisualizeGameState.IterateByClassType(class'XComGameState_EnvironmentDamage', EnvironmentDamageEvent)
	{
		BuildTrack = EmptyTrack;
		BuildTrack.TrackActor = none;
		BuildTrack.StateObject_NewState = EnvironmentDamageEvent;
		BuildTrack.StateObject_OldState = EnvironmentDamageEvent;

		//Wait until signaled by the shooter that the projectiles are hitting
		if (!AbilityTemplate.bSkipFireAction)
			class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityShooterEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityShooterEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');		
		}

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityTargetEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');
		}

		for (EffectIndex = 0; EffectIndex < MultiTargetEffects.Length; ++EffectIndex)
		{
			MultiTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');	
		}

		OutVisualizationTracks.AddItem(BuildTrack);
	}

	foreach VisualizeGameState.IterateByClassType(class'XComGameState_WorldEffectTileData', WorldDataUpdate)
	{
		BuildTrack = EmptyTrack;
		BuildTrack.TrackActor = none;
		BuildTrack.StateObject_NewState = WorldDataUpdate;
		BuildTrack.StateObject_OldState = WorldDataUpdate;

		//Wait until signaled by the shooter that the projectiles are hitting
		if (!AbilityTemplate.bSkipFireAction)
			class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityShooterEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityShooterEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');		
		}

		for (EffectIndex = 0; EffectIndex < AbilityTemplate.AbilityTargetEffects.Length; ++EffectIndex)
		{
			AbilityTemplate.AbilityTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');
		}

		for (EffectIndex = 0; EffectIndex < MultiTargetEffects.Length; ++EffectIndex)
		{
			MultiTargetEffects[EffectIndex].AddX2ActionsForVisualization(VisualizeGameState, BuildTrack, 'AA_Success');	
		}

		OutVisualizationTracks.AddItem(BuildTrack);
	}

	//Process any interactions with interactive objects
	foreach VisualizeGameState.IterateByClassType(class'XComGameState_InteractiveObject', InteractiveObject)
	{
		// Add any doors that need to listen for notification
		if (InteractiveObject.IsDoor() && InteractiveObject.HasDestroyAnim()) //Is this a closed door?
		{
			BuildTrack = EmptyTrack;
			//Don't necessarily have a previous state, so just use the one we know about
			BuildTrack.StateObject_OldState = InteractiveObject;
			BuildTrack.StateObject_NewState = InteractiveObject;
			BuildTrack.TrackActor = History.GetVisualizer(InteractiveObject.ObjectID);

			if (!AbilityTemplate.bSkipFireAction)
				class'X2Action_WaitForAbilityEffect'.static.AddToVisualizationTrack(BuildTrack, Context);

			class'X2Action_BreakInteractActor'.static.AddToVisualizationTrack(BuildTrack, Context);

			OutVisualizationTracks.AddItem(BuildTrack);
		}
	}
	*/
}

DefaultProperties
{
	// I don't know why names are initialized using strings
	LightEmUpAbilityName="LightEmUp"
	CrippledEffectName="Crippled"
	StickAndMoveMobilityAbilityName="StickAndMoveMobility"
	StickAndMoveMobilityEffectName="StickAndMoveMobility"
	StickAndMoveDodgeAbilityName="StickAndMoveDodge"
	StickAndMoveDodgeEffectName="StickAndMoveDodge"
	FlareStatusEffectName="Illuminated"
	ZoneOfControlActionPointName="zoneofcontrol"
	ZoneOfControlReactionFireAbilityName="ZoneOfControlShot"
	EscapeAndEvadeActionPointName="escapeandevade"
	EscapeAndEvadeStealthAbilityName="EscapeAndEvadeStealth"
}
