﻿/**
 *  PageItem
 */
define(function (require, exports) {

    require('browserContext');
    var path = require('../../common/path');
    var define = require('../define');
    var model = require('../model');
    var hotspotManger = require('./hotspot/hotspotmgr');
    var factory = require('./widgetfactory');
    var shellcmd = require('../../common/shellcmd');

    var ebkContext = require('../ebookcontext').Instance;
    var pageXmlParser = require('../../common/pagexmlparser').Instance;
    require('xml2json');

    /**
     *
     * @param pageProvider
     * @param pageIndex
     * @param audioPlayer
     * @constructor
     */
    function PageItem(pageProvider, pageIndex, audioPlayer) {

        var _hotspotMgr;

        /** @type {HTMLCanvasElement}  */
        var _paintCanvas;

        /** @type {HTMLCanvasElement} */
        var _hotspotCanvas;

        /** @type {boolean} 笔迹是否已经加载 */
        var _strokeLoaded = false;

        this.isStrokeChanged = false;
        this.provider = pageProvider;
        this.pageIndex = pageIndex;
        this.content = null;
        this.pageObject = null;
        this.xamlPath = '';
        this.player = audioPlayer;
        this.widgetFactory = new factory.WidgetFactory();

        this.hotspots = [];
        this.widgets = [];
        //位置数组
        this.widgetPositions = null;

        this._isModify = false;  //页面被修改
        this._isEditing = false;

        this.hotspotToolBar = null;

        this.getHotspotMgr = function () {
            return _hotspotMgr;
        };

        this.setHotspotMgr = function (hotspotMgr) {
            _hotspotMgr = hotspotMgr;
        };

        this.getStrokeLoaded = function () {
            return _strokeLoaded;
        };

        this._setStrokeLoaded = function (isLoaded) {
            _strokeLoaded = isLoaded;
        };

        /**
         * 设置笔迹Canvas
         * @param {HTMLCanvasElement} canvas
         * @private
         */
        this._setPaintCanvas = function (canvas) {
            _paintCanvas = canvas;
        };

        /**
         *  获取笔迹Canvas
         * @returns {HTMLCanvasElement}
         */
        this.getPaintCanvas = function () {
            return _paintCanvas;
        };

        this.saveStrokesToLocal = function () {
            this._savePainting();
        }

        this._setHotspotCanvas = function (canvas) {
            _hotspotCanvas = canvas;
        };

        this.getHotspotCanvas = function () {
            return _hotspotCanvas;
        };
    };

    PageItem.prototype.trigger = function (name, args) {
        this.content.triggerHandler(name, args);
    }

    PageItem.prototype.getContent = function () {
        return this.content;
    };

    /**
     * 获取页面JSON对象
     * @param {String} xaml xaml文件路径
     * @returns {JSON} 转换后的JSON对象
     * @private
     */
    PageItem.prototype._readPageObjectFromXaml = function (xaml) {
        console.log(xaml);
        var xml = shellcmd.readFile(xaml);
        if (isString(xml)) {
            if (xml.length > 0) {
                // console.time('xmltojson: Page');
                var obj2 = pageXmlParser.ToJSON(xml);
                // console.timeEnd('xmltojson: Page');
                return obj2;
            }
        }
        return xml;
    };

    PageItem.prototype._getUnDrmBackgroundImg = function (pageIndex) {
        var isPdf = /pdf_\w+/.test(this.provider.bookCode);
        if(!isPdf){
            var src = path.join(this.provider.getStdBookDir().replace(/\s/g, '%20'), this.pageIndex + '/' + this.pageIndex + '.jpg');
            return src;
        }
        else{
            var fmt = pageIndex.toString();
            var zeros = [];
            if(fmt.length < 6){
                for(var i=0;i<6-fmt.length;i++){
                    zeros.push(0);
                }
            }
            var file = "pdf-"+zeros.join("")+fmt+".jpg";
            var src = path.join(this.provider.getStdBookDir().replace(/\s/g, '%20'), file);
            return src;
        }
    };

    PageItem.prototype.getBackgroud = function () {
        if (!AppCfg.useDrm) {
            return this._getUnDrmBackgroundImg(this.pageIndex);
        } else {
            var winDst = this._getDrmTmpImgPath();
            var dst = winDst.replace(/\s/g, '%20');
            if (shellcmd.isFileExistSync(winDst)) {
                return dst;
            }
        }
        return null;
    };

    PageItem.prototype._getDrmTmpImgPath = function () {
        var isPdf = /pdf_\w+/.test(this.provider.bookCode);
        if(isPdf){
            var fmt = pageIndex.toString();
            var zeros = [];
            if(fmt.length < 6){
                for(var i=0;i<6-fmt.length;i++){
                    zeros.push(0);
                }
            }
            var file = "pdf-"+zeros.join("")+fmt+".png";
            var winDst = path.join(ebkContext.getSysTmpPath(), this.provider.bookCode, file);
            return winDst;
        }
        else{
            var file = '{0}\\{0}.jpg'.format(this.pageIndex, this.pageIndex);
            var winDst = path.join(ebkContext.getSysTmpPath(), this.provider.bookCode, '{0}.jpg'.format(this.pageIndex));
            return winDst;
        }
    };

    PageItem.prototype.loadBackground = function () {
        var _self = this;
        var def = $.Deferred();

        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;
        var isPdf = /pdf_\w+/.test(this.provider.bookCode);
        if (!AppCfg.useDrm) {
            //不使用DRM加密的资源
            var src = this._getUnDrmBackgroundImg(this.pageIndex).replace(/[\(\)]/g, function (str) {
                return ("\\" + str)
            });
            this.content = $('<div class="pageItem" style="background-image:url(' + src + ');height:' + h + 'px;width:' + w + 'px;"></div>');
            def.resolve();
        } else {   //使用DRM的资源
            //不缓存,基本上用不到了
            if (!AppCfg.saveDRMImage) {
                //使用canvas直接加载
                this._loadBgByCanvas(h, w).done(function () {
                    def.resolve();
                });
            }
            else { //直接解压到Tmp 文件夹

                var file = '{0}\\{0}.jpg'.format(this.pageIndex, this.pageIndex);
                var winDst = this._getDrmTmpImgPath();
                //网页显示图片必须把空格换成%20,而客户端里不必要这么做
                var dst = winDst.replace(/\s/g, '%20');
                if (shellcmd.isFileExistSync(winDst)) {
                    //存在缓存背景图
                    dst = dst.replace(/[\(\)]/g, function (str) {
                        return ("\\" + str)
                    });
                    _self.content = $('<div class="pageItem" style="background-image:url(' + 'file:///' + dst +
                        ');height:' + h + 'px;width:' + w + 'px;"></div>');
                    return def.resolve();
                } else {
                    //不存在缓存背景图
                    _self.provider.retriveDrmRes(file, winDst).done(function () {
                        dst = dst.replace(/[\(\)]/g, function (str) {
                            return ("\\" + str)
                        });
                        _self.content = $('<div class="pageItem" style="background-image:url(' + 'file:///' + dst +
                            ');height:' + h + 'px;width:' + w + 'px;"></div>');
                        def.resolve();
                    });
                }
            }
        }

        return def.promise();
    };

    PageItem.prototype._getBase64Str = function (pageIndex) {
        var file = '{0}\\{0}.jpg'.format(pageIndex, pageIndex);
        return this.provider.retriveDrmRes(file, '');
    };

    PageItem.prototype._loadBgByCanvas = function (h, w) {
        var _self = this;
        var def = $.Deferred();
        this.content = $('<div class="pageItem" style="height:' + h + 'px;width:' + w + 'px;"></div>');
        var canvas = $('<canvas class="background" />').attr({width: w, height: h}).css({'position': 'absolute', 'left': '0px', 'z-index': -3});
        this._getBase64Str(this.pageIndex).done(function (base64) {
            var ctx = canvas.get(0).getContext('2d');
            var backgroundImg = new Image();
            backgroundImg.onload = function () {
                ctx.drawImage(this, 0, 0);
            }
            backgroundImg.src = base64;
            _self.content.prepend(canvas);
            def.resolve();
        });

        return def.promise();
    }

    PageItem.prototype._bindCommands = function () {
        var _self = this;
        /**
         * 删除widget
         */
        this.content.on('widgetDeleted', function (event, id, isDeleted) {
            _self._isModify = true;
            if(json.TP.useDb){
                _self.removeWidgetByUuid(id, isDeleted,true);
            }else{
                _self.removeWidgetById(id, isDeleted,true);
            }
            event.stopPropagation();
        });

        /**
         * 对widget title和description的修改
         * @param lists [{
         *                 title:'',
         *                 description:'',
         *                 uuid:''
         *              },{}]  每个数组项都是JSON结构
         */
        this.content.on('widgetEdit',function(event, id, data){
            _self._isModify = true;
            if(json.TP.useDb){
                var lists = [];
                if(data && data.hasOwnProperty('title') && data.hasOwnProperty('description')){
                    var title = data.title;
                    var description = data.description;
                    var _description = "暂无资源描述";
                    if (description != ""){
                        _description = description;
                    }

                    lists.push({
                        title: title,
                        description: _description,
                        uuid: id});
                    _self.provider.updateTPE(lists).done(function(ret){
                        // 编辑成功后改变样式
                        if(app_global_curWidget && app_global_curWidget.Content){
                            var curDom = $(app_global_curWidget.Content);
                            var saveinfo = curDom.find("button[name='saveinfo']");
                            var editinfo = curDom.find("button[name='editinfo']");
                            var atitle = curDom.find("p[name='restile']");
                            var inputtitle = curDom.find("input[name='restile']");
                            var adescription = curDom.find("span[name='adescription']");
                            var inputdescription = curDom.find("p[name='inputdescription']");

                            app_global_curWidget.Title = title;
                            app_global_curWidget.description = _description;
                            app_global_curWidget.reSetTitle(title);

                            saveinfo.parent().hide();
                            editinfo.parent().show();
                            var _showTitle = title.length > 10 ? title.substr(0, 10) + "···" : title;
                            atitle.text(_showTitle);
                            atitle.attr("title", title);
                            atitle.show();
                            inputtitle.val(title).hide;
                            inputtitle.parent().hide();

                            adescription.text(_description);
                            adescription.attr("title", _description);
                            adescription.parent().show();
                            inputdescription.find("input").val(_description).hide();
                            inputdescription.hide();
                        }
                        $(document).trigger("widgetLoaded");
                    }).fail(function(e){
                        console.log(e);
                        title = app_global_curWidget.Title || '';
                        _description = (app_global_curWidget.description == "") ? '暂无资源描述' : app_global_curWidget.description;
                        var _showTitle = title.length > 10 ? title.substr(0, 10) + "···" : title;
                        // 保持编辑状态
                        // if(app_global_curWidget && app_global_curWidget.Content){
                        //     var curDom = $(app_global_curWidget.Content);
                        //     curDom.find("p[name='restile']").attr(title);
                        //     curDom.find("p[name='restile']").html(_showTitle);
                        //     curDom.find("input[name='restile']").val(title);
                        //     curDom.find("span[name='adescription']").attr(description);
                        //     curDom.find("span[name='adescription']").html(description);
                        //     curDom.find("p[name='inputdescription']").find("input").val(description);
                        // }

                        // 恢复原样
                        if(app_global_curWidget && app_global_curWidget.Content) {
                            var curDom = $(app_global_curWidget.Content);
                            var saveinfo = curDom.find("button[name='saveinfo']");
                            var editinfo = curDom.find("button[name='editinfo']");
                            var atitle = curDom.find("p[name='restile']");
                            var inputtitle = curDom.find("input[name='restile']");
                            var adescription = curDom.find("span[name='adescription']");
                            var inputdescription = curDom.find("p[name='inputdescription']");

                            saveinfo.parent().hide();
                            editinfo.parent().show();
                            var _showTitle = title.length > 10 ? title.substr(0, 10) + "···" : title;
                            atitle.text(_showTitle);
                            atitle.attr("title", title);
                            atitle.show();
                            inputtitle.val(title).hide;
                            inputtitle.parent().hide();

                            adescription.text(_description);
                            adescription.attr("title", _description);
                            adescription.parent().show();
                            inputdescription.find("input").val(_description).hide();
                            inputdescription.hide();
                        }
                        tipsDialog({
                            "id": "warn" + new Date().getTime(),
                            "content": "编辑资源失败..."
                        });
                        return 0;
                    });
                }else{
                    //不做处理
                    return 0;
                }
            }
        });

        this.content.on('hideAllNote', function () {
            $.each(_self.provider.bookWidgets, function (i, widget) {
                widget.WidgetType === define.WidgetType.Note && widget.hide();
            });
        });

        this.content.bind('insertWidget', function (event, pageOffset, widgetId, srcPos, curPos) {
            if(json.TP.useDb){
                var w = _self.getWidgetByUuid(widgetId);
            }else{
                var w = _self.getWidgetById(widgetId);
            }
            if (w) {
                var curPageItem = pageOffset ? _self.provider.getPageItem(parseInt(_self.pageIndex) + parseInt(pageOffset)) : _self;
                _self._onDrop(_self, curPageItem, w, srcPos, curPos);
            }else{
                var f = _self.provider.fadeWidgets.filter(function(el){
                    return el.uuid == widgetId;
                });
                if(f.length > 0){
                    var curPageItem = pageOffset ? _self.provider.getPageItem(parseInt(_self.pageIndex) + parseInt(pageOffset)) : _self;
                    _self._onDrop(_self, curPageItem, f[0], srcPos, curPos);
                }
            }
        });
    }

    PageItem.prototype._loadContent = function () {
        if (this.pageObject && this.content) {
            this.widgetFactory.init(this.content, path.join(ebkContext.getUserBookPath(), this.pageIndex + ""), this.xamlPath, this.player);

            this.loadBlackOverlay();
            // 创建热区
            this._loadHotspots(this.pageObject.hotspot);
            if(json.TP.useDb){
                var currpageWidgets = this.pageObject.currpageWidgets;
                if(currpageWidgets.length > 0) {
                    this._loadWidget_ext(currpageWidgets);
                }
            }else{
                // 创建绝对定位对象
                this._loadWidget(this.pageObject.absolutelayout);
            }
            // 创建笔迹画布
            this._loadPaintCanvas();
            //创建位置数组
            this._initWidgetPosition();
        } else {
            console.warn('pageObject or content is null');
        }
    };

    PageItem.prototype._loadWidget = function (absolutelayout) {
        if (!absolutelayout)  return;
        var value, _self = this;
        for (var key in absolutelayout) {
            if (absolutelayout.hasOwnProperty(key)) {
                value = $.makeArray(absolutelayout[key]);
                $.each(value, function (i, data) {
                    //注意约定把xaml中的类型字段的第一个字母大写后作为卡片的类型，如absolutelayout.video,则创建define.WidgetType.Video类型的卡片,这样减少许多if else
                    _self._insertWidget(define.WidgetType[key[0].toUpperCase() + key.substr(1)], data);
                });
            }
        }
    }

    PageItem.prototype._loadWidget_ext = function (dbData) {
        if (!dbData)  return;
        var value, _self = this;
        for(var i = 0,count = dbData.length;i < count;i++){
            var item = dbData[i];
            //解析json字符串
            var _widgetType;
            var multipleType = _self.parseWidgetTypeField(item.widgetType);
            _widgetType = multipleType.widgetType;
            var key = _widgetType;
            if(key){
                var data = _self.formatWidget(item);
                //注意约定把xaml中的类型字段的第一个字母大写后作为卡片的类型，如absolutelayout.video,则创建define.WidgetType.Video类型的卡片,这样减少许多if else
                var cachedPageItem = _self.provider.getPageItem(_self.pageIndex);
                var pageItem = cachedPageItem || _self;
                _self._insertWidget(define.WidgetType[key[0].toUpperCase() + key.substr(1)], data);
            }
        }
    }

    PageItem.prototype.formatWidget = function(widget){
        var _self = this;
        var standard = {
            //uuid是数据库数据唯一标识
            uuid: '',
            id:'',
            bound: '0;0;0;0',
            editable: 'true',
            style: 'tiny:false',
            event: [{action:'opened',style:'zoomin_anim'},{action:'closed',style:'zoomin_anim'}],
            layout: {src: 'stack',style: 'pop'}
        };
        //资源属性（扩展独立字段的key）
        var resourceProprety;

        //属性填充(自定义属性)
        standard.uuid = widget.uuid;
        standard.sort = widget.sort || '';
        standard.pageIndex = widget.pageIndex || '';
        standard.unit1 = widget.unit1;
        standard.unit2 = widget.unit2;
        standard.unit3 = widget.unit3;
        standard.unit4 = widget.unit4;
        standard.md5 = widget.md5;

        standard.id = widget.widgetId;
        standard.bound = widget.position;
        standard.editable = 'true';

        //解析json字符串
        var _widgetType;
        var _secondWidgetType;
        var multipleType = _self.parseWidgetTypeField(widget.widgetType);
        _widgetType = multipleType.widgetType;
        _secondWidgetType = multipleType.secondWidgetType;

        if(_widgetType){
            switch (_widgetType) {
                case define.WidgetType.Img:
                    resourceProprety = 'img';
                    break;
                case define.WidgetType.Video:
                    resourceProprety = 'video';
                    break;
                case define.WidgetType.Flash:
                    resourceProprety = 'flash';
                    break;
                case define.WidgetType.Gallery:
                    resourceProprety = 'file';
                    break;
                case define.WidgetType.Evaluate:
                    resourceProprety = 'file';
                    break;
                case define.WidgetType.Choice:
                    resourceProprety = 'file';
                    break;
                case define.WidgetType.OnlineRes:
                    resourceProprety = 'file';
                    break;
                case define.WidgetType.Proc3rd:
                    resourceProprety = 'file';
                    break;
                case define.WidgetType.Audio:
                    resourceProprety = 'audio';
                    break;
                case define.WidgetType.Note:
                    resourceProprety = 'note';
                    break;
                case define.WidgetType.FadeCurtain:
                    resourceProprety = 'fadeCurtain';
                    break;
                case define.WidgetType.Icw :
                    resourceProprety = 'icw';
                    break;
                case define.WidgetType.ImgPaint :
                    resourceProprety = 'imgPaint';
                    break;
                case define.WidgetType.Vote:
                    resourceProprety = 'vote';
                    break;
                case define.WidgetType.InteractiveCard:
                    resourceProprety = 'file';
                    break;
                default:
                    resourceProprety = _widgetType;
                    break;
                }
             standard[resourceProprety] = {
                 description: widget.description,
                 // realPath: path.join(path.trimPosix(Config.nwexePath), 'user/' + ebkContext.getUser().getName() + '/books/' + widget.bookCode + '/' + widget.uuid + '/' + widget.fileName + '.' + widget.extension),
                 realPath: path.join(ebkContext.getUser().getUserBookResourceBasePath(widget.bookCode),widget.uuid + '/' + widget.fileName + '.' + widget.extension),
                 // fileFolderPath: path.join(path.trimPosix(Config.nwexePath), 'user/' + ebkContext.getUser().getName() + '/books/' + widget.bookCode + '/' + widget.uuid + '/'),
                 fileFolderPath: path.join(ebkContext.getUser().getUserBookResourceBasePath(widget.bookCode), widget.uuid + '/'),
                 resourceId: widget.resId,
                 score: "0",
                 showIconFlag: "false",
                 showTitleFlag: "false",
                 src: widget.fileName + '.' + widget.extension,
                 timestamp: widget.modifyTime,
                 title: widget.title,
                 uploadTime: "",
                 userAddressInfo: "",
                 userName: widget.creator || ebkContext.getUser().getUserName()
             }
            //互动卡片对象数据需要secondWidgetType值，属性名是secondWidgetType，node的入参属性名是type
            if(define.WidgetType.InteractiveCard == _widgetType){
                standard.type = _secondWidgetType;
            }
        }
        return standard;
    };

    /**
     * 解析数据库widgetType字段
     * @param str 普通字符串或者json字符串
     * @returns {{widgetType: string, secondWidgetType: string}}
     */
    PageItem.prototype.parseWidgetTypeField = function (str) {
        var _self = this;
        var result = {
            widgetType: '',
            secondWidgetType: ''
        };
        // 识别为字符串
        if((typeof str === 'string') && str.length > 0){
            var _widgetType;
            var _secondWidgetType;
            try{
                var multipleType = JSON.parse(str);
                _widgetType = multipleType.widgetType || '';
                _secondWidgetType = multipleType.secondWidgetType || '';
            }catch(e){
                _widgetType = str;
            }
            result.widgetType = _widgetType;
            result.secondWidgetType = _secondWidgetType;
        }
        // 识别为对象
        if((typeof str === 'object') && c.hasOwnProperty('widgetType')){
            result.widgetType = str.widgetType;
            result.secondWidgetType = str.secondWidgetType;
        }
        return result;
    };

    /**
     * 加载笔迹画布
     * @private
     */
    PageItem.prototype._loadPaintCanvas = function () {
        var _self = this;
        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;
        var canvas = $('<canvas class="paint" />').attr({width: w, height: h}).css({'position': 'absolute', 'z-index': -2});

        this.content.prepend(canvas);
        this._setPaintCanvas(canvas[0]);
        _self._loadPainting();
    };

    /**
     * 加载热区画布
     * @param hotspotObj
     * @private
     */
    PageItem.prototype._loadHotspots = function (hotspotObj) {
        if (!hotspotObj) return;
        var rects = hotspotObj.Rectangle;
        if (rects) {
            if ($.isArray(rects)) {
                for (var i = 0, len = rects.length; i < len; i++) {
                    var rect = rects[i];
                    this.hotspots.push(new model.HotspotMeta(define.HotspotType.Rectangle, rect));
                }
            } else {
                this.hotspots.push(new model.HotspotMeta(define.HotspotType.Rectangle, rects));
            }
        }

        var ellipses = hotspotObj.Ellipse;
        if (ellipses) {
            if ($.isArray(ellipses)) {
                for (var i = 0, len = ellipses.length; i < len; i++) {
                    var ellipse = ellipses[i];
                    this.hotspots.push(new model.HotspotMeta(define.HotspotType.Ellipse, ellipse));
                }
            } else {
                this.hotspots.push(new model.HotspotMeta(define.HotspotType.Ellipse, ellipses));
            }
        }

        var paths = hotspotObj.Path;
        if (paths) {
            if ($.isArray(paths)) {
                for (var i = 0, len = paths.length; i < len; i++) {
                    var path = paths[i];
                    this.hotspots.push(new model.HotspotMeta(define.HotspotType.Path, path));
                }
            } else {
                this.hotspots.push(new model.HotspotMeta(define.HotspotType.Path, paths));
            }
        }

        $.each(this.hotspots, function () {
            // console.log(this.toString());
        });

        this._createHotspotCanvas(this.hotspots);
    };

    /**
     * 创建热区
     * @param {Array.<HotspotMeta>}  hotspotMetaArr
     * @private
     */
    PageItem.prototype._createHotspotCanvas = function (hotspotMetaArr) {

        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;
        var canvas = $('<canvas class="hotspot" />').attr({
            width: w,
            height: h
        }).css({'z-index': -1});

        var hotspotMgr = new hotspotManger.HotspotMgr(canvas[0], hotspotMetaArr);
        hotspotMgr.setIsDoubleLanguage(this.provider.isDoubleLanguage);
        if (this.pageObject.ReadLan === "Multi") {
            hotspotMgr.setIsDoubleLanguage(true);
        }
        this.setHotspotMgr(hotspotMgr);

        this.content.prepend(canvas);
        this._setHotspotCanvas(canvas[0]);

        //创建热区的同时，每个页面创建一个热区工具栏
        var hotspotToolBar = $('<div class="hotspotPopToolBar" style="position: absolute;display:none;-webkit-transform:matrix(1.3,0,0,1.4,0,0);">\
            <dl class="toolbar" style="width:80px;">\
                <dd class="toolbar_black" onMouseDown="this.className=\'toolbar_black_current\'" onmouseup="this.className=\'toolbar_black\'">\
                    <img src="assert/images/han.png" >\
                </dd>  \
                <dd class="toolbar_black" onMouseDown="this.className=\'toolbar_black_current\'" onmouseup="this.className=\'toolbar_black\'">\
                    <img src="assert/images/wei.png" >\
                </dd>\
            </dl>\
            </div>');
        this.content.prepend(hotspotToolBar);
        this.hotspotToolBar = hotspotToolBar;
        hotspotMgr.setHotspotToolBar(hotspotToolBar);

        // Author: xinfang3@iflytek.com
        // Date：2016/05/10
        var hotspotWaitplayIcon = $('<div class="btn-waitplay" style="position: absolute;display:none;">\
            <i class="icon iconfont">&#xe641;</i>\
            </div>');
        this.content.prepend(hotspotWaitplayIcon);
        this.hotspotWaitplayIcon = hotspotWaitplayIcon;
        hotspotMgr.setHotspotWaitplayIcon(hotspotWaitplayIcon);
    };

    /**
     * 根据 id 获取widget
     * @param id
     */
    PageItem.prototype.getWidgetById = function (id) {
        for (var i = 0; i < this.widgets.length; i++) {
            var w = this.widgets[i];
            if (w.Id === id) {
                return w;
            }
        }
        return null;
    };

    /**
     * 根据 uuid 获取widget
     * @param uuid
     */
    PageItem.prototype.getWidgetByUuid = function (uuid) {
        var _self = this;
        var widgets = _self.provider.bookWidgets;
        for (var i = 0; i < widgets.length; i++) {
            var w = widgets[i];
            if (w.uuid === uuid && 1 == w._packedFlag) {
                return w;
            }
        }
        return null;
    };

    /**
     * 删除某个Widget
     * @param id
     * @param {boolean=} deleteSrc   是否删除物理文件
     * @param {boolean=} deleteTA    是否删除TA文件及数据库
     */
    PageItem.prototype.removeWidgetById = function (id, deleteSrc,deleteTA) {

        var deleteWidget = null;
        this.widgets = this.widgets.filter(function (w) {
            if (w.Id != id)
                return true;
            else {
                deleteWidget = w;
                return false;
            }
        });
        if (deleteWidget) {
            deleteSrc = (deleteSrc == undefined) ? true : deleteSrc;
            if (deleteSrc) {
                //避免删除有多个引用的文件
                $.each(this.widgets, function (i, e) {
                    if (e.src === deleteWidget.src) {
                        deleteSrc = false;
                        return false;
                    }
                });
            }
            this.widgetPositions[this._getWidgetPosition([deleteWidget.Content[0].offsetLeft, deleteWidget.Content[0].offsetTop])] -= 1;
            //删除ifb物理文件
            deleteWidget.destroy(deleteSrc);
            var resourceId=deleteWidget.resourceId;
            var timestamp=deleteWidget.timestamp;
            //删除TA文件
            //只有存在resourceIdc和timestamp才会删除TA文件及数据库
            if(resourceId&&timestamp&&deleteTA){
                var taFilePath=ebkContext.getUser().getUserPpticwsBasePath()+"/"+timestamp;
                var taDbPath =ebkContext.getUser().getUserPpticwsBasePath()+"/file.db";
                var sqlite3 = nodereq('sqlite3').verbose();
                var taDb = new sqlite3.Database(taDbPath);
                var deleteSql="delete from ppticws where fileid='"+ resourceId +"' and timestamp='"+timestamp +"'and fromapp='iflybook'";
                shellcmd._rmdirASync(taFilePath,function(){
                    // TA删除文件后删除数据库
                    taDb.exec(deleteSql, function (err, rows) {
                        if(err){
                            console.log("删除TA数据库失败");
                        }else{
                            console.log("删除TA数据库成功");
                        };
                        taDb.close();
                    });
                });
            }else{
                console.log("widget没有timestamp，删除TA文件夹及数据库失败！")
            };
            this._isModify = true;
        }
    };

    PageItem.prototype.removeWidgetByUuid = function (uuid, deleteSrc) {
        var _this = this;
        var deleteWidget = null;
        _this.provider.deleteTPUuids([uuid]).done(function(ret){
            deleteWidget = _this.provider.deleteBookWidget(uuid);
            if(deleteWidget) {
                // 这里已清空该widget位置，后续不能再执行removeWidgetById（里面仍会执行）
                _this.widgetPositions[_this._getWidgetPosition([deleteWidget.Content[0].offsetLeft, deleteWidget.Content[0].offsetTop])] -= 1;

                deleteSrc = (deleteSrc == undefined) ? false : deleteSrc;

                //start 这里要同步删除书本pageItem里widget缓存数据，否则移除资源后widget.Content = null会引发异常
                var pageDeleteWidget;
                var w_id = deleteWidget.Id;
                _this.widgets = _this.widgets.filter(function (w) {
                    if (w.Id != w_id)
                        return true;
                    else {
                        pageDeleteWidget = w;
                        return false;
                    }
                });
                if (pageDeleteWidget) {
                    if (deleteSrc) {
                        //避免删除有多个引用的文件
                        $.each(_this.widgets, function (i, e) {
                            if (e.src === pageDeleteWidget.src) {
                                deleteSrc = false;
                                return false;
                            }
                        });
                    }
                    //删除ifb物理文件
                    pageDeleteWidget.destroy(deleteSrc);
                }
                //end 这里要同步删除书本pageItem里widget缓存数据

                if(!pageDeleteWidget){
                    deleteWidget.destroy(false);
                }
                _this._isModify = true;
                //通知同步精选资源刷新数据
                $(document).trigger("bubble_add");
                $(document).trigger("deleteWidgetSuccess",{"id":uuid,"file":"","origin":{},"pageIndex":_this.pageIndex}); //删除页面widgets成功
            }
        }).fail(function(e){
            console.log(e);
        });
    };

    PageItem.prototype._getPageObj = function (pageIndex) {
        var _self = this;
        var isPdf = /pdf_\w+/.test(_self.provider.bookCode);
        var def = $.Deferred();
        if(!isPdf){
            var tmpPath = _self.pageIndex + '/' + _self.pageIndex + '.xaml';
            var userPageXaml = path.trimPosix(path.join(ebkContext.getUserBookPath(), tmpPath));
            _self.xamlPath = userPageXaml;
            _self._getStdPageObj(pageIndex).done(function (stdPgObj) {
                console.log(stdPgObj);
                if(!json.TP.useDb){
                    if (shellcmd.isFileExistSync(userPageXaml)) {
                        var usrPgObj = _self._readPageObjectFromXaml(userPageXaml);
                        if (usrPgObj != null) {
                            //使用授课资源里的卡片信息,使用原资源里的点读信息
                            stdPgObj.absolutelayout = usrPgObj.absolutelayout;
                        }
                    } else {
                        _self._initUserPageXaml(userPageXaml);
                        _self._isModify = true;
                    }
                }
                def.resolve(stdPgObj);
            });
        }
        else{
            var tmpPath = _self.pageIndex + '/' + _self.pageIndex + '.xaml';
            var userPageXaml = path.trimPosix(path.join(ebkContext.getUserBookPath(), tmpPath));
            var stdPgObj = {};
            _self.xamlPath = userPageXaml;
            if(!json.TP.useDb) {
                if (shellcmd.isFileExistSync(userPageXaml)) {
                    var usrPgObj = _self._readPageObjectFromXaml(userPageXaml);
                    if (usrPgObj != null) {
                        //使用授课资源里的卡片信息,使用原资源里的点读信息
                        stdPgObj.absolutelayout = usrPgObj.absolutelayout;
                    }
                } else {
                    _self._initUserPageXaml(userPageXaml);
                    _self._isModify = true;
                }
            }
            var fmt = pageIndex.toString();
            var zeros = [];
            if(fmt.length < 6){
                for(var i=0;i<6-fmt.length;i++){
                    zeros.push(0);
                }
            }
            fmt = zeros.join("")+fmt;
            stdPgObj.background = "pdf-"+fmt+".png";
            def.resolve(stdPgObj);
        }
        return def.promise();
    };

    PageItem.prototype._getStdPageObj = function (pageIndex) {

        var def = $.Deferred();
        if (AppCfg.useDrm) {
            var file = '{0}\\{0}.xaml'.format(pageIndex, pageIndex);
//            AppCfg.saveDRMXAML = true;
            if (!AppCfg.saveDRMXAML) {
                this.provider.retriveDrmRes(file, '').done(function (xml) {
                    var obj = {};
                    if (xml) {
                        try {
                            var p = $.xml2json(xml);
                            if (p && typeof p === "object") {
                                obj = p;
                            }
                        } catch (e) {
                        }
                    }
                    def.resolve(obj);
                });
            }
            else {
                var dst = path.join(ebkContext.getSysTmpPath(), this.provider.bookCode, '{0}.xaml'.format(pageIndex));
                if (shellcmd.isFileExistSync(dst)) {
                    var xml = shellcmd.readFile(dst);
                    def.resolve($.xml2json(xml));
                } else {
                    this.provider.retriveDrmRes(file, dst).done(function () {
                        var xml = shellcmd.readFile(dst);
                        var obj = {};
                        if (xml) {
                            try {
                                var p = $.xml2json(xml);
                                if (p && typeof p === "object") {
                                    obj = p;
                                }
                            } catch (e) {
                            }
                        }
                        def.resolve(obj);
                    });
                }
            }
        } else {
            var tmpPath = this.pageIndex + '/' + this.pageIndex + '.xaml';
            var xaml = path.trimPosix(path.join(ebkContext.getStdBookPath(), tmpPath));
            var obj = this._readPageObjectFromXaml(xaml);
            def.resolve(obj);
        }

        return def.promise();
    };

    PageItem.prototype._initUserPageXaml = function (userPageXaml) {
        var ub = path.trimPosix(ebkContext.getUserBookPath());
        var temp = path.trimPosix(ebkContext.getTmpPath());
        var pagDir = path.trimPosix(path.dirname(userPageXaml));

        shellcmd.createDirIfNotExist(ub);
        shellcmd.createDirIfNotExist(temp);
        shellcmd.createDirIfNotExist(pagDir, 0);
    };

    /**
     * 页面加载
     * @param {function=} callback
     */
    PageItem.prototype.load = function () {
        var def = $.Deferred();
        var _self = this;
        var pageIndex = _self.pageIndex;
        var provider = _self.provider;
        var units = provider.getPageMetadata(pageIndex);
        // _self._getPageObj(_self.pageIndex).done(function (ret) {
        //     _self.pageObject = ret;
        //     _self.loadBackground().done(function () {
        //         _self._loadContent();
        //         _self._bindCommands();
        //         _self.content.dragable = true;
        //         //把当前数据保存到data里,否则有些回调里取不到当前的引用对象
        //         _self.content.data('pagedata', _self);
        //         // 等页面完成加载之后才能确定笔迹Canvas或是一些Html元素的动态大小
        //         setTimeout(function () {
        //             var hotspotMgr = _self.getHotspotMgr();
        //             hotspotMgr && hotspotMgr.reRender();
        //             def.resolve();
        //         }, 0);
        //     });
        // });

        _self._getPageObj(_self.pageIndex).done(function (ret) {
            _self.pageObject = ret || {};
            try {
                provider.getTPUnit(units).done(function (unitDate) {
                    //缓存这份数据
                    provider.setBookWidgets(units, unitDate);
                    var currpageWidgets = unitDate.filter(function (el) {
                        return el.pageIndex == pageIndex;
                    });
                    _self.pageObject.currpageWidgets = currpageWidgets;
                    _self.loadBackground().done(function () {
                        _self._loadContent();
                        _self._bindCommands();
                        _self.content.dragable = true;
                        //把当前数据保存到data里,否则有些回调里取不到当前的引用对象
                        _self.content.data('pagedata', _self);
                        // 等页面完成加载之后才能确定笔迹Canvas或是一些Html元素的动态大小
                        setTimeout(function () {
                            var hotspotMgr = _self.getHotspotMgr();
                            hotspotMgr && hotspotMgr.reRender();
                            def.resolve();
                        }, 0);
                    });
                }).fail(function (e) {
                    console.log(e);
                    _self.pageObject.currpageWidgets = [];
                    _self.loadBackground().done(function () {
                        _self._loadContent();
                        _self._bindCommands();
                        _self.content.dragable = true;
                        //把当前数据保存到data里,否则有些回调里取不到当前的引用对象
                        _self.content.data('pagedata', _self);
                        // 等页面完成加载之后才能确定笔迹Canvas或是一些Html元素的动态大小
                        setTimeout(function () {
                            var hotspotMgr = _self.getHotspotMgr();
                            hotspotMgr && hotspotMgr.reRender();
                            def.resolve();
                        }, 0);
                    });
                });
            } catch (e) {
                console.error(e);
                _self.pageObject.currpageWidgets = [];
                _self.loadBackground().done(function () {
                    _self._loadContent();
                    _self._bindCommands();
                    _self.content.dragable = true;
                    //把当前数据保存到data里,否则有些回调里取不到当前的引用对象
                    _self.content.data('pagedata', _self);
                    // 等页面完成加载之后才能确定笔迹Canvas或是一些Html元素的动态大小
                    setTimeout(function () {
                        var hotspotMgr = _self.getHotspotMgr();
                        hotspotMgr && hotspotMgr.reRender();
                        def.resolve();
                    }, 0);
                });
            }
        });
        return def.promise();
    };

    PageItem.prototype._onDrop = function (srcPageItem, curPageItem, w, srcPos, curPos) {
        if(json.TP.useDb){
            var condition = [];
            // 是否跨单元
            var crossUnitFlag = false;
            var srcPageUnit = srcPageItem.provider.getPageMetadata(srcPageItem.pageIndex);
            var curPageUnit = curPageItem.provider.getPageMetadata(curPageItem.pageIndex);
            if(app_global_bookContainer.getPageViewType() == define.PageViewType.SimpleDoublePage){
                if(!(curPageUnit.unit1 == srcPageUnit.unit1 && curPageUnit.unit2 == srcPageUnit.unit2 && curPageUnit.unit3 == srcPageUnit.unit3 && curPageUnit.unit4 == srcPageUnit.unit4)){
                    crossUnitFlag = true;
                }
            }

            var flag = curPageItem._correctPos(curPos, w);//更正位置信息
            if (flag) {
                if (srcPageItem.pageIndex === curPageItem.pageIndex) {
                    srcPageItem.widgetPositions[srcPageItem._getWidgetPosition(srcPos)] -= 1;
                    w.Content.css({
                        left: curPos[0],
                        top: curPos[1]
                    });

                    //针对互动卡片特殊处理widgetType(取w.WidgetType，注意W大写)
                    var _widgetType = w.WidgetType;
                    if(define.WidgetType.InteractiveCard == w.WidgetType){
                        if(w.secondWidgetType){
                            _widgetType = JSON.stringify({
                                widgetType: w.WidgetType,
                                secondWidgetType: w.secondWidgetType
                            });
                        }
                    }
                    condition.push({
                        unit1: curPageUnit.unit1 || '',
                        unit2: curPageUnit.unit2 || '',
                        unit3: curPageUnit.unit3 || '',
                        unit4: curPageUnit.unit4 || '',
                        uuid: w.uuid,
                        pageCode: w.pageCode,
                        pageIndex: curPageItem.pageIndex,
                        position: curPos[0] + ';' + curPos[1] + ';' + w.PlayAreaWidth + ";" + w.PlayAreaHeight,
                        widgetId: w.Id,
                        widgetType: _widgetType,
                        resId: w.resourceId || '',
                        md5: w.md5 || ""
                    });
                    srcPageItem._isModify = true;
                    curPageItem.provider.updateTPP(condition).done(function(ret){
                        $(document).trigger('widgetLoaded');
                    }).fail(function(e){
                        console.log(e);
                    });
                } else {
                    //从别的页面拖进来
                    if (w) {
                        //
                        if(w.WidgetType == "fadeCurtain"){
                            srcPageItem.removeWidgetByUuid(w.uuid);
                            srcPageItem.widgetPositions[srcPageItem._getWidgetPosition([w.Content[0].offsetLeft, w.Content[0].offsetTop])] -= 1;

                            w.Content.remove();
                        }
                        //
                        var arr = [];
                        arr.push(w.src);
                        if (w.WidgetType != "interactiveCard") {
                            arr.push(w.src.substring(0, w.src.lastIndexOf('/')) + '/' + w.resourceId + '.jpg');
                        }
                        var timestamp = w.timestamp;
                        var _objhere = {
                            file: w.WidgetType != "fadeCurtain" ? w.src : ("assert/images/curtains/" + w.src),
                            multiFiles: arr,
                            realPath: w.realPath,
                            title: w.Title,
                            showTitleFlag: w.showTitleFlag,
                            showIconFlag: w.showIconFlag,
                            resourceId: w.resourceId,
                            timestamp: timestamp,
                            left: curPos[0],
                            top: curPos[1],
                            width: w.PlayAreaWidth,
                            height: w.PlayAreaHeight,
                            type: w.type,
                            style: w.style.toString(),
                            userName: w.userName,
                            uploadTime: w.uploadTime,
                            score: w.score,
                            userAddressInfo: w.userAddressInfo,
                            description: w.description,
                            notCopySrc: true,
                            md5:w.md5
                        }

                        //上传的地方同样要做如此处理，很挫，但还有4小时上线。
                        if (w.WidgetType == "icw") {
                            _objhere = {
                                file: w.src + ".icw",
                                insertFromOuter: false,
                                multiFiles: [w.src + ".icw"],
                                userName: w.userName,
                                left: curPos[0],
                                top: curPos[1],
                                notCopySrc: true
                            }
                        }

                        var options = _objhere;

                        // 特殊参数，用于传递widget src文件路径信息
                        options.realPath = w.src;
                        options.uuid = w.uuid;

                        var widgetType = w.WidgetType;
                        options.file = path.toPosix(options.file || "");
                        if (!widgetType) {
                            var ext = path.extname(options.file);
                            widgetType = define.getWidgetTypeByExt(ext);
                        }
                        if(w.WidgetType == "fadeCurtain"){
                            curPageItem.insertWidget(w.WidgetType, _objhere);
                            if(crossUnitFlag){
                                // 跨单元时，删除原数据存储的排序信息，再移动资源回原页面时序号重新计算
                                $(document).trigger('deleteWidgetSuccess',{id:w.uuid,pageIndex:w.pageIndex});
                            }else{
                                $(document).trigger('widgetLoaded');
                            }
                        }
                        else{
                            var widget = curPageItem.widgetFactory.createWidget(widgetType, null, options);
                            widget.Content.hide();
                            var pos = [widget.Left, widget.Top];
                            var flag = curPageItem._correctPos(pos, widget, options);
                            if(flag){
                                widget.Left = pos[0];
                                widget.Top = pos[1];
                            }

                            //针对互动卡片特殊处理widgetType(取w.WidgetType，注意W大写)
                            var _widgetType = widget.WidgetType;
                            if(define.WidgetType.InteractiveCard == widget.WidgetType){
                                if(widget.secondWidgetType){
                                    _widgetType = JSON.stringify({
                                        widgetType: widget.WidgetType,
                                        secondWidgetType: widget.secondWidgetType
                                    });
                                }
                            }

                            // 资源移动位置
                            if(crossUnitFlag){
                                // 跨单元
                                condition.push({
                                    unit1: curPageUnit.unit1 || '',
                                    unit2: curPageUnit.unit2 || '',
                                    unit3: curPageUnit.unit3 || '',
                                    unit4: curPageUnit.unit4 || '',
                                    uuid: w.uuid,
                                    sort: '', //跨单元设置为空
                                    pageCode: '',
                                    pageIndex: curPageItem.pageIndex,
                                    position: widget.Left + ';' + widget.Top + ';' + w.PlayAreaWidth + ";" + w.PlayAreaHeight,
                                    widgetId: widget.Id,
                                    widgetType: _widgetType,
                                    resId: widget.resourceId || '',
                                    md5: widget.md5 || ""
                                });
                            }else{
                                // 不跨单元
                                condition.push({
                                    unit1: curPageUnit.unit1 || '',
                                    unit2: curPageUnit.unit2 || '',
                                    unit3: curPageUnit.unit3 || '',
                                    unit4: curPageUnit.unit4 || '',
                                    uuid: w.uuid,
                                    sort: w.sort,
                                    pageCode: '',
                                    pageIndex: curPageItem.pageIndex,
                                    position: widget.Left + ';' + widget.Top  + ';' + w.PlayAreaWidth + ";" + w.PlayAreaHeight,
                                    widgetId: widget.Id,
                                    widgetType: _widgetType,
                                    resId: widget.resourceId || '',
                                    md5: widget.md5 || ""
                                });
                            }
                            curPageItem.provider.updateTPP(condition).done(function(ret){
                                srcPageItem.widgetPositions[srcPageItem._getWidgetPosition([w.Content[0].offsetLeft, w.Content[0].offsetTop])] -= 1;
                                w.Content.remove();
                                //新的widget数据
                                widget.Content.data('pagedata', curPageItem);
                                widget.uuid = w.uuid;
                                widget.unit1 = condition[0] && condition[0].unit1;
                                widget.unit2 = condition[0] && condition[0].unit2;
                                widget.unit3 = condition[0] && condition[0].unit3;
                                widget.unit4 = condition[0] && condition[0].unit4;
                                widget.pageIndex = condition[0] && condition[0].pageIndex;
                                widget.position = condition[0] && condition[0].position;
                                widget.md5 = condition[0] && condition[0].md5;
                                if(!crossUnitFlag){
                                    widget.sort = condition[0] && condition[0].sort;
                                }
                                widget.Content.css({'left':widget.Left , 'top': widget.Top});
                                widget.Content.show();
                                curPageItem.provider.updateBookWidget(widget);
                                if(crossUnitFlag){
                                    // 跨单元时，删除原数据存储的排序信息，再移动资源回原页面时序号重新计算
                                    $(document).trigger('deleteWidgetSuccess',{id:w.uuid,pageIndex:w.pageIndex});
                                }else{
                                    $(document).trigger('widgetLoaded');
                                }
                            }).fail(function(e){
                                console.log(e);
                                $(document).trigger('widgetLoaded');
                            });
                        }    
                    }
                }
                //发送位置变化消息
                w.trigger('reposition', curPos);
            }
        }else{
            var flag = curPageItem._correctPos(curPos, w);//更正位置信息
            if (flag) {
                if (srcPageItem.pageIndex === curPageItem.pageIndex) {
                    srcPageItem.widgetPositions[srcPageItem._getWidgetPosition(srcPos)] -= 1;
                    w.Content.css({
                        left: curPos[0],
                        top: curPos[1]
                    });
                    srcPageItem._isModify = true;
                } else {
                    //从别的页面拖进来
                    if (w) {
                        srcPageItem.removeWidgetById(w.Id, false);
                        var arr = [];
                        arr.push(w.src);
                        if (w.WidgetType != "interactiveCard") {
                            arr.push(w.src.substring(0, w.src.lastIndexOf('/')) + '/' + w.resourceId + '.jpg');
                        }
                        var timestamp = w.timestamp;
                        var _objhere = {
                            file: w.src,
                            multiFiles: arr,
                            realPath: w.src,
                            title: w.Title,
                            showTitleFlag: w.showTitleFlag,
                            showIconFlag: w.showIconFlag,
                            resourceId: w.resourceId,
                            timestamp: timestamp,
                            left: curPos[0],
                            top: curPos[1],
                            width: w.PlayAreaWidth,
                            height: w.PlayAreaHeight,
                            type: w.type,
                            style: w.style.toString(),
                            userName: w.userName,
                            uploadTime: w.uploadTime,
                            score: w.score,
                            userAddressInfo: w.userAddressInfo,
                            description: w.description
                        }

                        //上传的地方同样要做如此处理，很挫，但还有4小时上线。
                        if (w.WidgetType == "icw") {
                            _objhere = {
                                file: w.src + ".icw",
                                insertFromOuter: false,
                                multiFiles: [w.src + ".icw"],
                                userName: w.userName,
                                left: curPos[0],
                                top: curPos[1],
                                notCopySrc: true
                            }
                        }

                        curPageItem.insertWidget(w.WidgetType, _objhere);
                    }
                }
                //发送位置变化消息
                w.trigger('reposition', curPos);
            }
        }
    }

    PageItem.prototype.onDragOver = function (event) {
        /*event.preventDefault();
         return false;*/
        event.stopPropagation();
        event.preventDefault();
    };

    /**
     * 页面卸载
     * @param {boolean=} index 默认为false，只是卸载当前页面，如果为true，则卸载整个书本页面。如从书本跳转到书架
     */
    PageItem.prototype.unload = function (index) {
        //TODO: need impl
        //console.log('PageItem::unload, pageIndex:%d', this.pageIndex);
        $(document).trigger('pageItemUnload');
        this._savePainting();
        if(!json.TP.useDb){
            this.saveXaml();
        }

        this.content.data('pagedata', null);

        $(this.getPaintCanvas()).remove();
        this._setPaintCanvas(null);

        var hotspotMgr = this.getHotspotMgr();
        if (hotspotMgr) {
            hotspotMgr.dispose();
            this.setHotspotMgr(null);
        }
        this.content.empty();
        //释放资源
        for (var i = 0, len = this.widgets.length; i < len; ++i) {
            var widget = this.widgets[i];
            widget.releaseRes && widget.releaseRes();
        }
        !index && this.content.remove();
    };

    /**
     * 插入卡片
     * @param widgetType 卡片类型
     * @param node  卡片节点信息
     * @param {object=} options 卡片额外参数
     *  options.posType : 1 - 拖动到指定位置；2 - 位置随机，不重复
     */
    PageItem.prototype._insertWidget = function (widgetType, node, options) {
        var _this = this,
            _fileFullPath = "",
            _fileId = "",
            _fromText = options && options.fileInfo && options.fileInfo.from || "",
            _from = 1, //1 -云端资源 2-备课本资源 3-本地资源
            _fileName = "";
        var exsit = 0;
        var args = {"files": []};
        //来源判断
        switch (_fromText){
            case "resource":
                _from = 1;
                break;
            case "beike":
                _from = 2;
                break;
            case "local":
                _from = 3;
                break;
            default:
                _from = 1;
                break;
        }
        if (options && options.file) {
            if(widgetType == "fadeCurtain"){
                // var pathPrefix = "assert/images/curtains/";
                _fileFullPath = path.trimPosix(Config.binPath + "webapp/"  +options.file);
            }else{
                _fileFullPath = options.file;
            }
            _fileName = _fileFullPath.substr(_fileFullPath.lastIndexOf("/") + 1, _fileFullPath.length - 1);
        }

        if (options && options.noNewIcon) {
            _this.widgetFactory.createWidget(widgetType, node, options);
            return;
        }

        if (options && options.fileInfo) {
            //分类推荐
            if(_from == 1 && options.fileInfo.id){
                _fileId = options.fileInfo.id;
            }

            //备课本（云端）
            if(_from == 2 && options.fileInfo.fileId){
                _fileId = options.fileInfo.fileId;
            }

            //备课本（本地）
            if(json.TP.useDb){
                if(_from == 3 && options.fileInfo.uuid){
                    _fileId = options.fileInfo.uuid;
                }
            }else{
                if(_from == 3 && options.fileInfo.timestamp){
                    _fileId = options.fileInfo.timestamp;
                }
            }
        }
        if (options){
            options.realPath = ( options.toTPInfo && options.toTPInfo.path) || _fileFullPath;
            options.pageIndex =  options.toTPInfo && options.toTPInfo.pageIndex || _this.pageIndex;
            options.uuid =  options.toTPInfo && options.toTPInfo.uuid;
            options.title =  options.toTPInfo && options.toTPInfo.title;
            options.description = options.toTPInfo && options.toTPInfo.description || "";
            options.userName = options.toTPInfo && options.toTPInfo.creator || options.userName;
            options.uploader = options.toTPInfo && options.toTPInfo.uploader || options.userName;
        }
        if(options){
            $.when(shellcmd.getMd5(options.realPath)).done(function(ret){
                if(ret){
                    //获取当前页的缓存widgets
                    var curPageIndexWidgets = _this.provider.getWidgetsByPageIndex(options.pageIndex);
                    if(curPageIndexWidgets.length > 0){
                        var flag = curPageIndexWidgets.filter(function(el){
                            return el.md5 == ret && el.WidgetType != "note" && el.WidgetType !="fadeCurtain";
                        });
                        if(flag.length>0){
                            $.Zebra_Dialog(AppImportAndExportTips.ExistDownLoadFile, {
                                'type': 'question',
                                'title': AppTips.WarmTips,
                                'buttons': [AppTips.CancelTips, AppTips.OkTips],
                                'onClose': function (capition) {
                                    /**
                                     * widegts和页面已有文件md5相同后，点击确定才覆盖文件
                                     */
                                    if (capition == AppTips.OkTips) {
                                        var p = {
                                            title:_fileName,
                                            description:flag[0].description,
                                            uuid:flag[0].uuid,
                                            resId: "",
                                            resFrom:""
                                        };
                                        if((_from == 1 || _from == 2) && _fileId){
                                            p.resId = _fileId;
                                            p.resFrom = options && options.fileInfo && options.fileInfo.from || "";
                                            flag[0].resourceId = _fileId;
                                        }
                                        _this.provider.updateTPER([p]).done(function(ret){
                                            if(ret && ret.code == 1){
                                                tipsDialog({
                                                    "id": "updateTPE" + new Date().getTime(),
                                                    "content": "资源添加成功"
                                                });
                                                $(document).trigger('insertWidgetSuccess', {"id": _fileId,"insertType":"overlay"});
                                                $(document).trigger('widgetLoaded');
            
                                                flag[0].Title = _fileName;
                                                $(document).trigger("widgetLoaded");
                                                _this.provider.updateBookWidget(flag[0]);
                                            }else{
                                                tipsDialog({
                                                    "id": "updateTPE" + new Date().getTime(),
                                                    "content": "资源添加失败"
                                                });
                                            }
                                        });
                                    }
                                }
                            });
                        }else{
                            insertTP();
                        }
                    }else{
                        insertTP();
                    }
                }else{
                    tipsDialog({
                        "id": "updateTPE" + new Date().getTime(),
                        "content": "资源添加失败"
                    });
                } 
            }).fail(function(e){
                insertTP();
            });
        }else{
            insertTP();
        }
        // var index = 0 ;
        // var exsit = 0;
        // var allwidgets = app_global_bookContainer.getCurrentPageBookWidgets();
        // for (; index < allwidgets.length; index++) {
        //     var _exsitFileName = allwidgets[index].File;
        //     if (_fileName == _exsitFileName && _fileName != "") {
        //         exsit = 1;
        //         var originpath =allwidgets[index].src;
        //         $(document).trigger('insertWidgetSuccess', {"id": _fileId,"insertType":"overlay"});
        //         $.Zebra_Dialog(AppImportAndExportTips.ExistDownLoadFile, {
        //             'type': 'question',
        //             'title': AppTips.WarmTips,
        //             'buttons': [AppTips.CancelTips, AppTips.OkTips],
        //             'onClose': function (capition) {
        //                 /**
        //                  * widegts和页面已有文件重名后，点击确定才覆盖文件
        //                  */
        //                 if (capition == AppTips.OkTips) {
        //                     // 只覆盖文件，不添加图标
        //                     options.realPath = (options && options.toTPInfo && options.toTPInfo.path) || _fileFullPath;
        //                     // var dstpath = path.trimPosix( path.join(ebkContext.getUserBookPath(), _this.pageIndex + "/" + _fileName));
        //                     args.files[0] = {
        //                         "oriname": options.realPath,
        //                         "newname": originpath
        //                     }
        //                     shellcmd.multiCopyFile(JSON.stringify(args));
        //                     // 文件重名覆盖后，widget序号不变,这里不重新计算序号
        //                     // $(document).trigger('insertWidgetSuccess',{"id":_fileId});
        //                     tipsDialog({
        //                         "id": "uploading" + new Date().getTime(),
        //                         "content": "资源添加成功"
        //                     });
        //                     $(document).trigger('insertWidgetSuccess', {"id": _fileId,"insertType":"overlay"});
        //                 }
        //             }
        //         });

        //     }
        // }
        function insertTP(){
            if (exsit == 0){
                if(json.TP.useDb && options){   
                    // $(document).queue("saveToTp",function(){
                        //写入tp数据库
                        var bookobj = _this.provider.getPageMetadata(options.pageIndex);
                        var param = {
                            userName:ebkContext.getUser().getName()|| "",
                            bookCode:_this.provider.bookCode||"",
                            path:options.realPath || "",
                            fileName:_fileName || "未知文件",
                            unit1:bookobj.unit1||"",
                            unit2:bookobj.unit2||"",
                            unit3:bookobj.unit3||"",
                            unit4:bookobj.unit4||"",
                            uuid:options && options.toTPInfo && options.toTPInfo.uuid || "",
                            title:_fileName || "",
                            pageCode:"",
                            pageIndex:options.pageIndex || "",
                            position:"0;0;0;0",
                            creator:options.userName || "",
                            uploader:options.uploader || "",
                            description:"",
                            fid:"",
                            source:"",
                            resFrom:options && options.fileInfo && options.fileInfo.from || ""
                        };
                        _this.saveToTp(param).done(function(ret){
                            if(ret && ret.code == 1 && ret.data){
                                // console.log("%c保存成功", "color:green;font-weight:bold");
                                // console.log(param);
                                var data = ret.data || {};
                                // 临时处理，覆盖传入的file为文件真实地址
                                if (options.file && !shellcmd.isFileExistSync(options.file)) {
                                    options.file = path.join(ebkContext.getUser().getUserBookResourceBasePath(data.bookCode),data.uuid + '/' + data.fileName + '.' + data.extension);
                                }
                                options.uuid = data.uuid;
                                options.unit1 = data.unit1 || "";
                                options.unit2 = data.unit2 || "";
                                options.unit3 = data.unit3 || "";
                                options.unit4 = data.unit4 || "";
                                options.sort = data.sort || "";
                                options.pageIndex = data.pageIndex || "";
                                options.realPath = path.join(ebkContext.getUser().getUserBookResourceBasePath(data.bookCode),data.uuid + '/' + data.fileName + '.' + data.extension);
                                options.md5 = data.md5 || "";

                                if (_this.pageIndex != options.pageIndex) { //不是当前页
                                    options.noNewIcon = true;   //创建widget不生成水滴图标
                                    var widget = _this.widgetFactory.createWidget(widgetType, node, options);
                                    widget.Content.hide();
                                    //位置计算
                                    var pos = [widget.Left, widget.Top];
                                    //获取当前页的缓存widgets
                                    var pageIndexWidgets = _this.provider.getWidgetsByPageIndex(options.pageIndex);
                                    //资源下载的页面不在缓存的pageItem中时，计算位置，只插入到数据库，不插入到页面
                                    if (_this._correctPosNew(pos, widget, options, pageIndexWidgets)) {
                                        widget.Content.css({'left': pos[0], 'top': pos[1]});
                                        widget.Left = pos[0];
                                        widget.Top = pos[1];
                                    }
                                } else {
                                    var widget = _this.widgetFactory.createWidget(widgetType, node, options);
                                    widget.Content.hide();
                                    //位置计算
                                    var pos = [widget.Left, widget.Top];

                                    if (_this._correctPos(pos, widget, options)) {
                                        widget.Content.css({'left': pos[0], 'top': pos[1]});
                                        widget.Left = pos[0];
                                        widget.Top = pos[1];
                                    }
                                }

                                var _widgetType = widget.WidgetType;
                                if(define.WidgetType.InteractiveCard == widgetType){
                                    if(widget.secondWidgetType){
                                        _widgetType = JSON.stringify({
                                            widgetType: widgetType,
                                            secondWidgetType: widget.secondWidgetType
                                        });
                                    }
                                }

                                data.widgetId = widget.Id;
                                data.widgetType = _widgetType;
                                data.resId = widget.resourceId;

                                data.position = pos[0] + ";" + pos[1] + ";" + widget.PlayAreaWidth + ";" + widget.PlayAreaHeight;
                                // 更新缓存
                                if (json.TP.useDb) {
                                    _this.provider.updateBookWidget(widget);
                                } else {
                                    _this.widgets.push(widget);
                                }
                                _this.provider.updateTPP([data]).done(function(ret){
                                    if(ret && ret.code == 1){
                                        tipsDialog({
                                            "id": "saveToTp" + new Date().getTime(),
                                            "content": "资源添加成功"
                                        });
                                        //同步精选资源下载成功后，去掉原有红色气泡
                                        $(document).find("#bubble_"+data.resId) && $(document).find("#bubble_"+data.resId).remove();
                                        // console.log("当前页"+_this.pageIndex);
                                        // console.log("下载页"+options.pageIndex);
                                        if (_this.pageIndex == options.pageIndex) {
                                            insertWidgetAndover(widget);
                                        } else {    //不是下载的当前页，删除widget
                                            if(_fileId != "") {
                                                $(document).trigger('insertWidgetSuccess', {"id": _fileId});
                                            }
                                            widget.destroy();
                                            // console.log("*********删除widget");
                                        }
                                        // $(document).dequeue("saveToTp");
                                    } else {
                                        if (_this.pageIndex == options.pageIndex) {
                                            _this.widgetPositions[_this._getWidgetPosition([widget.Content[0].offsetLeft, widget.Content[0].offsetTop])] -= 1;
                                        }
                                        // console.log("%c插入失败","color:red;font-weight:bold");
                                        // console.log(ret);
                                        tipsDialog({
                                            "id": "saveToTp" + new Date().getTime(),
                                            "content": "资源添加失败"
                                        });
                                        //插入失败，删除数据库中的资源
                                        _this.removeWidgetByUuid(data.uuid, false);
                                        // $(document).dequeue("saveToTp");
                                    }
                                }).fail(function(e){
                                    if (_this.pageIndex == options.pageIndex) {
                                        _this.widgetPositions[_this._getWidgetPosition([widget.Content[0].offsetLeft, widget.Content[0].offsetTop])] -= 1;
                                    }
                                    // console.log("%c插入失败-更新","color:red;font-weight:bold");
                                    // console.log(e);
                                    // console.log(data);
                                    tipsDialog({
                                        "id": "saveToTp" + new Date().getTime(),
                                        "content": "资源添加失败"
                                    });
                                    console.log("更新数据库失败");
                                    //插入失败，删除数据库中的资源
                                    _this.removeWidgetByUuid(data.uuid, false);
                                    // $(document).dequeue("saveToTp");
                                });
                            }else{
                                // console.log("%c插入失败","color:red;font-weight:bold");
                                // console.log(ret);
                                // console.log(param);
                                tipsDialog({
                                    "id": "saveToTp" + new Date().getTime(),
                                    "content": "资源添加失败"
                                });
                                console.log("插入tp失败");
                                // $(document).dequeue("saveToTp");
                            }
                        }).fail(function(e){
                            // console.log("%c插入失败","color:red;font-weight:bold");
                            // console.log(e);
                            // console.log(param);
                            tipsDialog({
                                "id": "saveToTp" + new Date().getTime(),
                                "content": "资源添加失败"
                            });
                            console.log("插入tp失败");
                        });
                    // });
                    // $(document).dequeue("saveToTp");
                }else{
                    var widget = _this.widgetFactory.createWidget(widgetType, node, options);
                    widget.Content.hide();
                    //位置计算
                    var pos = [widget.Left, widget.Top];
                    if (_this._correctPos(pos, widget, options)){
                        widget.Content.css({'left': pos[0], 'top': pos[1]});
                    }
                    insertWidgetAndover(widget);
                }
            }
        }

        /**
         * 文件不存在直接插入、若存在覆盖文件
         */
        function insertWidgetAndover(widget) {
            if (widget) {
                var _self = _this;
                widget.Content.data('pagedata', _this);
                if(json.TP.useDb){
                    _this.provider.updateBookWidget(widget);
                }else{
                    _this.widgets.push(widget);
                }
                _this._isModify = true;
                //这里是想让这段代码在当前环境下最后执行 ---- by yczhu
                setTimeout(function () {
                    // var pos = [widget.Left, widget.Top];
                    // if (_self._correctPos(pos, widget, options)){
                    //     widget.Content.css({'left': pos[0], 'top': pos[1]});
                    // }
                    widget.Content.show();
                    //correctToBaseResolution(widget.Content, {scaleFromCenter: false, scaleOrder: false, keepScale: true, width: page_base_resolution.width, height: page_base_resolution.height});
                    //通知页面加载完成
                    widget.trigger('pageLoaded');
                    widget.setIsEditing(_self._isEditing);
                    //新增资源时触发
                    if(_fileId != "") {
                        $(document).trigger('insertWidgetSuccess', {"id": _fileId});
                    }else{
                        /**
                         * 某一个widget被加载到页面
                         * 来源于其他页面或者外部导入未带资源信息的文件（兼容资源导入和mtbk导入）
                         */
                        if(options){
                            var pages = _this.provider.pages.filter(function(el){
                                return options.pageIndex == el.pageIndex;
                            });
                            if(pages.length > 0){
                                $(document).trigger('widgetLoaded');
                            }
                            // if(options.pageIndex == _this.pageIndex || options.pageIndex == (_this.pageIndex+1)){
                            //     $(document).trigger('widgetLoaded');
                            // }
                        }else{
                            $(document).trigger('widgetLoaded');
                        }
                    }
                }, 0);


                // widget.Content.hide();
                // //处理options,如果有些options属性为空，则将widget的属性赋值给它
                // if(options){
                //     if(!options.file){
                //         options.file=widget.src;
                //     };
                //     if(options.fileInfo && !options.fileInfo.fileName){
                //         options.fileInfo.fileName=widget.Title;
                //     }
                // }
            }
        };

        //options转换为数据库字段
        // function optionsToTPdb(options){
        //     var dbparam = {};
        //     if(options){
        //         var path = options.toTPInfo && options.toTPInfo.path || options.file;
        //         var fileName = path.substr(path.lastIndexOf("/") + 1,path.length-1) || "";
        //         var pageIndex = options.toTPInfo && options.toTPInfo.pageIndex || _this.pageIndex;
        //         var bookobj = _this.provider.getPageMetadata(pageIndex);
        //         dbparam.userName = ebkContext.getUser().getName() || "";
        //         dbparam.bookCode = _this.provider.bookCode||"",
        //         dbparam.path = path || "";
        //         dbparam.realPath = path || "";
        //         dbparam.fileName = fileName || "";
        //         dbparam.unit1 = bookobj.unit1||"";
        //         dbparam.unit2 = bookobj.unit2||"";
        //         dbparam.unit3 = bookobj.unit3||"";
        //         dbparam.unit4 = bookobj.unit4||"";
        //         dbparam.title = fileName || "";
        //         dbparam.pageCode = "";
        //         dbparam.pageIndex = pageIndex || "";
        //         dbparam.position = "0;0;0;0";
        //         dbparam.creator = options.userName || "";
        //         dbparam.uploader = options.userName || "";
        //         dbparam.description = options.toTPInfo && options.toTPInfo.description || "";
        //         dbparam.fid = "";
        //         dbparam.source = "";
        //         dbparam.resFrom = (options.toTPInfo && options.toTPInfo.resFrom) || (options.fileInfo && options.fileInfo.from);
        //     }
        //     return $.extend(options,dbparam);
        // }

        /*
        *  写数据库的方式已替换，下面几个方法的代码已废弃
        */ 
        // 查询TA数据库中有多少条记录
        function selectTADb(resourceId){
            var sql="select count(1) from ppticws where" +
                " fileid='"+resourceId +"' and timestamp='"+options.fileInfo.timestamp +"'"
            tadb.all(sql, function (err, rows) {
                if(err||rows[0]["count(1)"]<0){
                    tadb.close();
                    return ;
                }else if(rows[0]["count(1)"]==0){
                    insertTADb(options);
                    copyFiletoTa();
                }else{
                    updateTADb(options);
                }
            });
        };

        // insert TA数据库
        function insertTADb(){
            var uuid=createUuid();
            var uuid1=createUuid();
            var lasttime=new Date().Format("yyyy-MM-dd hh:mm:ss");
            var filename=options.fileInfo.fileName||options.fileInfo.name||options.fileInfo.title||"未知文件";
            var type=options.fileInfo.fileExt||options.fileInfo.extension||"";
            var paramToTADb={
                filename:filename,
                unit1:app_global_bookContainer.getCurrPageMetadata().unit1||"",
                unit2:app_global_bookContainer.getCurrPageMetadata().unit2||"",
                unit3:app_global_bookContainer.getCurrPageMetadata().unit3||"",
                unit4:app_global_bookContainer.getCurrPageMetadata().unit4||"",
                fileid:options.resourceId||uuid,
                timestamp:options.timestamp||uuid1,
                type:type,
                bookcode:app_global_bookContainer.bookMetaInfo.bookCode||"",
                secondType:options.type||"",
                creator:options.fileInfo.fromapp||options.userName||"",
                size:options.fileInfo.fileSize||options.fileInfo.length||"",
                lasttime:options.fileInfo.lasttime||"",
                filepath:filename+"."+type,
            };
            var insertSql = "insert into ppticws  values('"+
                paramToTADb.filename.replace(/'/g,'\'\'') + "','" + paramToTADb.unit1 + "','" + paramToTADb.unit2 + "','" +
                paramToTADb.unit3 + "','"+ paramToTADb.unit4 + "','"+ paramToTADb.timestamp + "','"+
                paramToTADb.fileid + "','"+ paramToTADb.type +"','"+  paramToTADb.bookcode + "','"+paramToTADb.secondType+"','iflybook','0','0','"+
                paramToTADb.filepath.replace(/'/g,'\'\'') +"','0','"+paramToTADb.lasttime+ "','"+
                paramToTADb.creator + "','"+ paramToTADb.size +
                "')";
            tadb.all(insertSql, function (err, rows) {
                tadb.close();
                if(err){
                    console.log("insert TA数据库失败！")
                    return ;
                }else{
                    console.log("insert TA数据库成功！")
                }
            });
        };
        // update TA数据库
        function updateTADb(){
            var updataSql="update ppticws set fromapp= 'iflybook'" +
                " where fileid='"+ options.resourceId +"' and timestamp='"+options.fileInfo.timestamp +"'";
            tadb.all(updataSql, function (err, rows) {
                tadb.close();
                if(err){
                    console.log("update TA数据库失败！");
                    return ;
                }else{
                    console.log("update TA数据库成功！");
                }
            });

        };
        //把文件拷贝到Ta目录
        function copyFiletoTa(){
            var timestamp = options.timestamp;
            //源文件路径
            var fileFullPath=options.file;
            //如果没有fileid，退出
            if(timestamp) {
                //文件名（带后缀）
                var filepath = options.fileInfo.filepath;
                //云资源不存在filepath，手动拼装
                if(!filepath){
                    var filename=options.fileInfo.fileName||options.fileInfo.name||options.fileInfo.title||"未知文件";
                    var type=options.fileInfo.fileExt||options.fileInfo.extension||"";
                    filepath=filename+"."+type;
                };
                //创建文件夹
                var destFolderPath = ebkContext.getUser().getUserPpticwsBasePath() + '/' + timestamp;
                shellcmd.createDirIfNotExist(destFolderPath).done(function (ret) {
                    if (!ret) {
                        console.log("创建文件夹失败");
                        def.reject(false);
                    }
                    var destFileFullPath = destFolderPath + '/' + filepath;
                    //拷贝文件
                    shellcmd.copyDir_ext(fileFullPath, destFileFullPath).done(function () {
                        consol.log("复制文件到TA数据库成功")
                    });
                });
            };
        };
        //为本地文件设置uuid;
        function createUuid() {
            var s = [];
            var hexDigits = "0123456789abcdef";
            for (var i = 0; i < 36; i++) {
                s[i] = hexDigits.substr(Math.floor(Math.random() * 0x10), 1);
            }
            s[14] = "4";  // bits 12-15 of the time_hi_and_version field to 0010
            s[19] = hexDigits.substr((s[19] & 0x3) | 0x8, 1);  // bits 6-7 of the clock_seq_hi_and_reserved to 01
            s[8] = s[13] = s[18] = s[23] = "";

            var uuid = s.join("");
            return uuid;
        };

    }

    /**
     * 插入卡片，外部模块调用
     * @param {!object} widgetType 卡片类型
     * @param {!object} options  其他参数
     */
    PageItem.prototype.insertWidget = function (widgetType, options) {
        if (!options) return;
        options.file = path.toPosix(options.file || "");
        if (!widgetType) {
            var ext = path.extname(options.file);
            widgetType = define.getWidgetTypeByExt(ext);
        }
        var cachedPageItem = this.provider.getPageItem(this.pageIndex);
        var pageItem = cachedPageItem ? cachedPageItem : this;
        pageItem._insertWidget(widgetType, null, options);
    }

    PageItem.prototype.toXml = function () {
        var hotSpotXaml = "";
        if (hotSpotXaml !== "") hotSpotXaml = "\n<eb:hotspot>" + hotSpotXaml + "\n</eb:hotspot>";
        var widgetsXaml = "";
        $.each(this.widgets, function (i, e) {
            widgetsXaml += e.toXml();
        });
        if (widgetsXaml !== "") widgetsXaml = "\n<eb:absolutelayout>" + widgetsXaml + "\n</eb:absolutelayout>"

        var xml = "<eb:page background=\"" + this.pageIndex + ".jpg\" xmlns:eb=\"EBookLib\">" + hotSpotXaml + widgetsXaml + "\n</eb:page>";
        return xml;
    };

    PageItem.prototype.setIsEditing = function (isediting) {
        var _this = this;
        this._isEditing = isediting;
        $.each(_this.provider.fadeWidgets, function () {
            this.setIsEditing(isediting);
        });
        var hotspotMgr = this.getHotspotMgr();
        if (hotspotMgr) hotspotMgr.setIsEditing(isediting);
    };

    /**
     * 保存笔迹
     * @private
     */
    PageItem.prototype._savePainting = function () {
        if (!this.isStrokeChanged) {
            return;
        }

        var paintCanvas = this.getPaintCanvas();
        var strokeBase64 = paintCanvas.toDataURL();
        var strokeFilePath = path.join(this.provider.getUserBookDir(), this.pageIndex + '/' + this.pageIndex + '.stroke.base64');
        strokeFilePath = path.trimPosix(strokeFilePath);
        shellcmd.saveFile(strokeBase64, strokeFilePath);
    };


    PageItem.prototype.saveXaml = function () {
        if (!this._isModifyChanged()) return;
        var xml = this.toXml();
        shellcmd.saveFile(xml, this.xamlPath);
        this._isModify = false;
    }


    PageItem.prototype._isModifyChanged = function () {
        if (this._isModify) return true;
        for (var i = 0, len = this.widgets.length; i < len; ++i) {
            if (this.widgets[i].isModify) return true;
        }
        return false;
    };

    /**
     * 更正位置信息，防止widget移动时超出边界和重叠时的bug
     * @param {!Array.<number>} pos
     * @param {object} widget
     * @param {object=} options
     * @returns {boolean}是否更正了位置信息
     */
    PageItem.prototype._correctPos = function (pos, widget, options) {

        if (!pos || pos.length < 2) return false;
        var flag = true;
        options = options || {};
        pos[0] = parseFloat(pos[0]);
        pos[1] = parseFloat(pos[1]);

        if (!this.widgetPositions) this._initWidgetPosition();

        //是否从外部插入，防止刚插入时重叠
        if (options.insertFromOuter && options.posType != 1) {
            var emptyPos = -1;//有任何可用的空位
            for (var i = 0; i < this.widgetPositions.length; ++i) {

                //如果当前位置可用
                if (this.widgetPositions[i] <= 0) {
                    emptyPos = i;
                    break;
                    /*if (Math.floor(i / this.widgetPositions.leftCount) * this.widgetPositions.topOffset >= 200) {
                        emptyPos = i;
                        break;
                    }*/
                }
            }
            if (emptyPos < 0) {
                emptyPos = 0;
                var minPosCount = this.widgetPositions[0];
                for (var i = 0; i < this.widgetPositions.length; ++i) {
                    if (this.widgetPositions[i] < minPosCount) {
                        minPosCount = this.widgetPositions[i];
                        emptyPos = i;
                    }
                }
            }
            pos[0] = (emptyPos % this.widgetPositions.leftCount) * this.widgetPositions.leftOffset + (this.widgetPositions.leftOffset - widget.Width);
            pos[1] = Math.floor(emptyPos / this.widgetPositions.leftCount) * this.widgetPositions.topOffset + (this.widgetPositions.topOffset ) / 2;
        }

        if (isNaN(pos[0]) || isNaN(pos[1])) return false;

        // 防止超出边界
        var width = widget.Content.width(), height = widget.Content.height();
        // var title = widget.Content.find('.widgetTitle:first');
        var title = widget.Content.find('.widgetTitle:visible');
        if (title.length) {
            try{
                //--zqwei 居中：字长/2-便宜量  非居中：字长-偏移量
                if( "center" == title.css("text-align") )
                    minLeft = parseInt(/\d+/.exec(title.css("font-size"))[0])*title.text().length/2 + title.parents(".user-popover").position().left;
                else
                    minLeft = parseInt(/\d+/.exec(title.css("font-size"))[0])*title.text().length + title.parents(".user-popover").position().left;
            }
            catch(e){
                minLeft = 0;
            }
            // minLeft = Math.max(title.outerWidth() - width, 0) / 2;
            width += minLeft;
        }

        var matrixTransform = this.getTransformMatrix(widget.Content);
        var offsetX = (matrixTransform[0] - 1) * width / 2, offsetY = (matrixTransform[3] - 1) * height / 2;
        var maxLeft = this.content.width() - offsetX, maxTop = this.content.height() - offsetY;
        var minLeft = offsetX, minTop = offsetY;
        //   minTop = 300//offsetY;

        pos[0] = pos[0] <= minLeft ? minLeft : pos[0], pos[1] = pos[1] <= minTop ? minTop : pos[1];

        if (pos[0] + width > maxLeft) pos[0] = maxLeft - 10 - width;
        if (widget.Content.find(".volet_clos").length > 0){
            if (pos[0] + 188 > maxLeft){
                $(widget.Content.find(".volet_clos")[0]).addClass('rotleft');

                $(widget.Content.find(".rotleft")[0]) .css({"-webkit-transform":"translateX(-162px) rotateY(180deg)"});
                $(widget.Content.find(".volet_clos")[0]) .css({"width":"200px"});
                $(widget.Content.find(".volet_clos .wrap")[0]) .css({"width":"200px"});
            }
            else{
                $(widget.Content.find(".rotleft")[0]) .css({"-webkit-transform":"translateX(0px) rotateY(0deg)"});
                $(widget.Content.find(".volet_clos")[0]) .css({"width":"40px"});
                $(widget.Content.find(".volet_clos .wrap")[0]) .css({"width":"40px"});
                $(widget.Content.find(".volet_clos")[0]).removeClass('rotleft');

            }
        }
        if (pos[1] + height > maxTop) pos[1] = maxTop - 10 - height;

        pos[0] = pos[0] <= minLeft ? minLeft : pos[0], pos[1] = pos[1] <= minTop ? minTop : pos[1];

        //更正位置数组
        this.widgetPositions[this._getWidgetPosition(pos)] += 1;

        console.log("**********原位置*************");
        console.log("x="+pos[0]);
        console.log("y="+pos[1]);
        return flag;
    }

    /**
     * 生成资源下载位置（下载时翻页）
     * @param {!Array.<number>} pos
     * @param {object} widget
     * @param {object=} options
     * @returns {boolean}是否更正了位置信息
     */
    PageItem.prototype._correctPosNew = function (pos, widget, options, cacheWidgets) {

        /*var timestamp1  = new Date().getTime();
        console.log("开始时间："+timestamp1);*/
        if (!pos || pos.length < 2) return false;
        var flag = true;
        options = options || {};
        pos[0] = parseFloat(pos[0]);
        pos[1] = parseFloat(pos[1]);

        if (!this.widgetPositions) this._initWidgetPosition();

        var widgetPositionsForPageIndex = {
            positions : []
        };
        widgetPositionsForPageIndex.leftCount = this.widgetPositions.leftCount;
        widgetPositionsForPageIndex.topCount = this.widgetPositions.topCount;
        widgetPositionsForPageIndex.leftOffset = this.widgetPositions.leftOffset;
        widgetPositionsForPageIndex.topOffset = this.widgetPositions.topOffset;
        var counts = widgetPositionsForPageIndex.leftCount * widgetPositionsForPageIndex.topCount;
        for (var j = 0; j < counts; j++){
            widgetPositionsForPageIndex.positions[j] = 0;
        }

        for (var i = 0, len = cacheWidgets.length; i < len; i++){
            var pointPos = [];
            pointPos[0] = cacheWidgets[i].Left;
            pointPos[1] = cacheWidgets[i].Top;
            widgetPositionsForPageIndex.positions[this._getWidgetPosition(pointPos)] += 1;
        }

        //是否从外部插入，防止刚插入时重叠
        if (options.insertFromOuter && options.posType != 1) {
            var emptyPos = -1;//有任何可用的空位
            for (var i = 0; i < widgetPositionsForPageIndex.positions.length; ++i) {
                //如果当前位置可用
                if (widgetPositionsForPageIndex.positions[i] <= 0) {
                    emptyPos = i;
                    break;
                }
            }
            if (emptyPos < 0) {
                emptyPos = 0;
                var minPosCount = widgetPositionsForPageIndex.positions[0];
                for (var i = 0; i < widgetPositionsForPageIndex.positions.length; ++i) {
                    if (widgetPositionsForPageIndex.positions[i] < minPosCount) {
                        minPosCount = widgetPositionsForPageIndex.positions[i];
                        emptyPos = i;
                    }
                }
            }
            pos[0] = (emptyPos % widgetPositionsForPageIndex.leftCount) * widgetPositionsForPageIndex.leftOffset + (widgetPositionsForPageIndex.leftOffset - widget.Width);
            pos[1] = Math.floor(emptyPos / widgetPositionsForPageIndex.leftCount) * widgetPositionsForPageIndex.topOffset + (widgetPositionsForPageIndex.topOffset ) / 2;
        }

        if (isNaN(pos[0]) || isNaN(pos[1])) return false;

        console.log("**********位置*************");
        console.log("x="+pos[0]);
        console.log("y="+pos[1]);
       /* var timestamp2  = new Date().getTime();
        console.log("结束时间："+timestamp2);
        console.log("时间差："+(timestamp2 - timestamp1));*/
        return flag;
    };

    /**
     * 初始化widget的位置数组
     * @private
     */

    PageItem.prototype._initWidgetPosition = function () {
        if (this.widgetPositions) return;
        this.widgetPositions = [];
        var matrixTransform = this.getTransformMatrix(this.content), leftOffset, topOffset;
        leftOffset = topOffset = parseInt(80 / matrixTransform[0] * 2) + 80; //每个widget的距离
        if (leftOffset < 130) leftOffset = 130;
        if (topOffset < 130) topOffset = 130;

        var maxLeft = this.content.width(), maxTop = this.content.height();

        var leftCount = Math.floor(maxLeft / leftOffset), topCount = Math.floor(maxTop / topOffset);

        for (var i = 0, len = leftCount * topCount; i < len; ++i) this.widgetPositions[i] = 0;

        $.extend(this.widgetPositions, {
            leftCount: leftCount,
            topCount: topCount,
            leftOffset: leftOffset,
            topOffset: topOffset
        });
    }

    /**
     * 根据当前位置获取在widget位置数组中的位置
     * @param {Array} pos
     * @returns {number}
     * @private
     */
    PageItem.prototype._getWidgetPosition = function (pos) {
        if (!pos || pos.length != 2 || !pos[0] || !pos[1]) return -1;
        var index = Math.floor(pos[0] / this.widgetPositions.leftOffset) + Math.floor(pos[1] / this.widgetPositions.topOffset) * this.widgetPositions.leftCount;
        return index;
    }

    /**
     * 获得变换矩阵
     * @param {jQuery} ele
     * @return {Array}
     */
    PageItem.prototype.getTransformMatrix = function (ele) {
        var matrix = [1, 0, 0, 1, 0, 0];
        if (ele) {
            var matrixTransform = ele.css('-webkit-transform');
            if (matrixTransform && matrixTransform != 'none')  matrix = matrixTransform.substr(7, matrixTransform.length - 8).split(",");
        }
        return matrix;
    };

    /**
     * 加载笔迹
     * @private
     */
    PageItem.prototype._loadPainting = function () {
        var paintCanvas = this.getPaintCanvas();
        var self = this;

        var strokeFilePath = path.join(this.provider.getUserBookDir(), this.pageIndex + '/' + this.pageIndex + '.stroke.base64');
        strokeFilePath = path.trimPosix(strokeFilePath);

        var strokeFileExist = shellcmd.isFileExistSync(strokeFilePath);
        if (strokeFileExist) {
            var strokeBase64 = shellcmd.readFile(strokeFilePath);
            var ctx = paintCanvas.getContext('2d');
            var image = new Image();
            image.src = strokeBase64;
            image.onload = function () {
                ctx.clearRect(0, 0, self.provider.bookImageSize.width, self.provider.bookImageSize.height);
                ctx.drawImage(image, 0, 0);
                self._setStrokeLoaded(true);
                $(self).trigger('strokeLoaded');
            }
        } else {
            self._setStrokeLoaded(true);
        }
    };

    /**
     * 绘制笔迹
     * @param {HTMLCanvasElement} canvas
     * @param {Bound} boundd 原始画布中的边界信息
     */
    PageItem.prototype.paintStroke = function (canvas, bound, isStrokeChanged) {
        var paintCanvas = this.getPaintCanvas();
        if (isStrokeChanged) {
            this.isStrokeChanged = isStrokeChanged;
            var w = this.provider.bookImageSize.width;
            var h = this.provider.bookImageSize.height;
            var ctx = paintCanvas.getContext('2d');
            ctx.clearRect(0, 0, w, h);
            ctx.drawImage(canvas, bound.left, bound.top, bound.width, bound.height, 0, 0, w, h);
        }
        $(paintCanvas).show();
    };

    /**
     * 恢复笔迹画布
     * @param {HTMLCanvasElement} canvas
     * @param {Bound} boundd 原始画布中的边界信息
     */
    PageItem.prototype.restoreStroke = function (canvas, bound) {
        var w = this.provider.bookImageSize.width;
        var h = this.provider.bookImageSize.height;

        var self = this;
        var paintCanvas = self.getPaintCanvas();
        var ctx = canvas.getContext('2d');

        var strokeLoaded = self.getStrokeLoaded();
        if (strokeLoaded) {
            ctx.drawImage(paintCanvas, 0, 0, w, h, bound.left, bound.top, bound.width, bound.height);
        } else {
            $(self).bind('strokeLoaded', function () {
                ctx.drawImage(paintCanvas, 0, 0, w, h, bound.left, bound.top, bound.width, bound.height);
                $(self).unbind('strokeLoaded');
            })
        }
        $(paintCanvas).hide();
        if (AppCfg.useTurnJSDoublePage) {
            $(canvas).css("zIndex", 1000);
        }
    };

    /**
     * 显示所有热区
     */
    PageItem.prototype.showAllHotspots = function () {
        var hotspotMgr = this.getHotspotMgr();
        hotspotMgr && hotspotMgr.showAllHotspots();
    }

    /**
     * 隐藏所有热区
     */
    PageItem.prototype.hideAllHotspots = function () {
        var hotspotMgr = this.getHotspotMgr();
        hotspotMgr && hotspotMgr.hideAllHotspots();
    };

    /**
     * 清除选中的热区显示
     */
    PageItem.prototype.renderSelectedHotspot = function () {
        var hotspotMgr = this.getHotspotMgr();
        hotspotMgr && hotspotMgr.renderSelectedHotspot();
    };

    /**
     * 清空本页资源
     */
    PageItem.prototype.clearCurrentPageRes = function () {

        this.player && this.player.releaseRes();

        $.each(this.widgets, function (i, e) {
            e.destroy(false);
        });

        var userPageXaml = path.trimPosix(path.join(ebkContext.getUserBookPath(), this.pageIndex + ""));
        var args = {
            "filter": ".base64",
            "dirs": userPageXaml
        };
        shellcmd.clearDir(JSON.stringify(args));
        if (this.widgets.length) {
            this.widgets = [];
            for (var i = 0, len = this.widgetPositions.length; i < len; ++i) this.widgetPositions[i] = 0;
            this._isModify = true;
        }
    }

    /**
     * 获取本页资源 去除幕布
     */
    PageItem.prototype.getCurrentPageRes = function () {
        var tmpWidgets = [];
        var widgets = this.widgets;
        for (var i = 0; i < widgets.length; i++) {
            if (widgets[i].WidgetType != "fadeCurtain") {
                tmpWidgets.push(widgets[i]);
            }
        }
        return tmpWidgets;
    }

    /**
     * 导入时重新加载widgets
     */
    PageItem.prototype.importBookRes = function () {
        var _self = this;
        this._getPageObj(this.pageIndex).done(function (ret) {
            _self.pageObject = ret;
            if (_self.pageObject.absolutelayout != null) {
                _self._loadWidget(_self.pageObject.absolutelayout);
                _self._isModify = true;
            }
        });
    }

    PageItem.prototype.loadBlackOverlay = function () {
        var _DOM = $('<div class="itemOverlay" style="display:none;height:100%;width:100%;position: absolute;top: 0px;z-index:0;background:#000;opacity: 0.4;"></div>');
        this.content.append(_DOM);
    };

    //add by njfu on 2017-1-20
    //写入tp.db文件中
    PageItem.prototype.saveToTp = function(param){
        var def = jQuery.Deferred();
        var p = {
            method:'addTPFile',
            data:{
                userName:param.userName,
                bookCode:param.bookCode,
                path:param.path,
                fileName:param.fileName || "未知文件",
                unit1:param.unit1 || "",
                unit2:param.unit2 || "",
                unit3:param.unit3 || "",
                unit4:param.unit4 || "",
                uuid:param.uuid || "",
                title:param.title,
                pageCode:param.pageCode || "",
                pageIndex:param.pageIndex || "",
                position:param.position,
                creator:param.creator || "",
                uploader:param.uploader || "",
                description:param.description || "",
                widgetId:param.widgetId || "",
                widgetType:param.widgetType || "",
                resId:param.resId || "",
                fid:param.fid || "",
                source:param.source || "",
                resFrom:param.resFrom || ""
            }
        }
        hproseClient.hproseCall(p,function(ret){
            def.resolve(ret);
        },function(e){
            console.error(e);
            def.reject(e);
        });
        return def.promise();
    }

    exports.PageItem = PageItem;
});