/*
 * Decompiled with CFR 0.152.
 */
package se.llbit.chunky.launcher;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import se.llbit.chunky.launcher.JreUtil;
import se.llbit.chunky.launcher.LaunchMode;
import se.llbit.chunky.launcher.LauncherSettings;
import se.llbit.chunky.launcher.Logger;
import se.llbit.chunky.launcher.VersionInfo;
import se.llbit.chunky.launcher.ui.ChunkyLauncherController;
import se.llbit.chunky.resources.SettingsDirectory;
import se.llbit.json.JsonArray;
import se.llbit.json.JsonObject;
import se.llbit.json.JsonParser;
import se.llbit.json.JsonValue;
import se.llbit.log.Level;
import se.llbit.log.Log;

public final class ChunkyDeployer {
    private ChunkyDeployer() {
    }

    public static boolean checkVersionIntegrity(String version) {
        File chunkyDir = SettingsDirectory.getSettingsDirectory();
        if (chunkyDir == null) {
            return false;
        }
        File versionsDir = new File(chunkyDir, "versions");
        File libDir = new File(chunkyDir, "lib");
        if (!versionsDir.isDirectory() || !libDir.isDirectory()) {
            return false;
        }
        File versionFile = new File(versionsDir, version + ".json");
        if (!versionFile.isFile()) {
            return false;
        }
        try {
            FileInputStream in = new FileInputStream(versionFile);
            JsonParser parser = new JsonParser(in);
            JsonObject obj = parser.parse().object();
            in.close();
            String versionName = obj.get("name").stringValue("");
            if (!versionName.equals(version)) {
                System.err.println("Stored version name does not match file name");
                return false;
            }
            JsonArray array = obj.get("libraries").array();
            for (JsonValue value : array) {
                VersionInfo.Library lib = new VersionInfo.Library(value.object());
                switch (lib.testIntegrity(libDir)) {
                    case INCOMPLETE_INFO: {
                        System.err.println("Missing library name or checksum");
                        return false;
                    }
                    case MD5_MISMATCH: {
                        System.err.println("Library MD5 checksum mismatch");
                        return false;
                    }
                    case MISSING: {
                        System.err.println("Missing library " + lib.name);
                        return false;
                    }
                }
            }
            return true;
        }
        catch (IOException e) {
            System.err.println("Could not read version info file: " + e.getMessage());
        }
        catch (JsonParser.SyntaxError e) {
            System.err.println("Corrupted version info file: " + e.getMessage());
        }
        return false;
    }

    public static void deploy(LauncherSettings settings) {
        List<VersionInfo> versions = ChunkyDeployer.availableVersions();
        VersionInfo embedded = ChunkyDeployer.embeddedVersion();
        if (!(embedded == null || versions.contains(embedded) && ChunkyDeployer.checkVersionIntegrity(embedded.name))) {
            Log.infof("Deploying embedded version: %s", embedded.name);
            ChunkyDeployer.deployEmbeddedVersion(embedded);
            if (!settings.version.equals(VersionInfo.LATEST.name)) {
                settings.version = VersionInfo.LATEST.name;
                settings.save();
            }
        }
    }

    public static List<VersionInfo> availableVersions() {
        File chunkyDir = SettingsDirectory.getSettingsDirectory();
        if (chunkyDir == null) {
            return Collections.emptyList();
        }
        File versionsDir = new File(chunkyDir, "versions");
        if (!versionsDir.isDirectory()) {
            return Collections.emptyList();
        }
        File[] versionFiles = versionsDir.listFiles();
        if (versionFiles == null) {
            return Collections.emptyList();
        }
        ArrayList<VersionInfo> versions = new ArrayList<VersionInfo>();
        for (File versionFile : versionFiles) {
            if (!versionFile.getName().endsWith(".json")) continue;
            try {
                FileInputStream in = new FileInputStream(versionFile);
                JsonParser parser = new JsonParser(in);
                versions.add(new VersionInfo(parser.parse().object()));
                in.close();
            }
            catch (IOException e) {
                System.err.println("Could not read version info file: " + e.getMessage());
            }
            catch (JsonParser.SyntaxError e) {
                System.err.println("Corrupted version info file: " + e.getMessage());
            }
        }
        Collections.sort(versions);
        return versions;
    }

    private static void deployEmbeddedVersion(VersionInfo version) {
        File libDir;
        File chunkyDir = SettingsDirectory.getSettingsDirectory();
        if (chunkyDir == null) {
            return;
        }
        File versionsDir = new File(chunkyDir, "versions");
        if (!versionsDir.isDirectory()) {
            versionsDir.mkdirs();
        }
        if (!(libDir = new File(chunkyDir, "lib")).isDirectory()) {
            libDir.mkdirs();
        }
        try {
            File versionJson = new File(versionsDir, version.name + ".json");
            version.writeTo(versionJson);
            ClassLoader parentCL = ChunkyDeployer.class.getClassLoader();
            for (VersionInfo.Library lib : version.libraries) {
                if (lib.testIntegrity(libDir) == VersionInfo.LibraryStatus.PASSED) continue;
                ChunkyDeployer.unpackLibrary(parentCL, "lib/" + lib.name, new File(libDir, lib.name));
            }
        }
        catch (IOException | IllegalArgumentException | SecurityException e) {
            e.printStackTrace();
        }
    }

    private static void unpackLibrary(ClassLoader parentCL, String name, File dest) throws IOException {
        int len;
        BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(dest));
        InputStream in = parentCL.getResourceAsStream(name);
        byte[] buffer = new byte[4096];
        while ((len = in.read(buffer)) != -1) {
            out.write(buffer, 0, len);
        }
        out.close();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static VersionInfo embeddedVersion() {
        try {
            ClassLoader parentCL = ChunkyDeployer.class.getClassLoader();
            try (InputStream in = parentCL.getResourceAsStream("version.json");){
                if (in == null) return null;
                JsonParser parser = new JsonParser(in);
                VersionInfo versionInfo = new VersionInfo(parser.parse().object());
                return versionInfo;
            }
            catch (IOException | JsonParser.SyntaxError exception) {
                return null;
            }
        }
        catch (SecurityException securityException) {
            // empty catch block
        }
        return null;
    }

    public static int launchChunky(LauncherSettings settings, VersionInfo version, LaunchMode mode, Consumer<String> failureHandler, LoggerBuilder loggerBuilder) {
        int exitValue;
        List<String> command = ChunkyDeployer.buildCommandLine(version, settings);
        if (settings.verboseLauncher || Log.level == Level.INFO) {
            System.out.println(ChunkyDeployer.commandString(command));
        }
        if ((exitValue = ChunkyDeployer.launchChunky(mode, command, loggerBuilder)) != 0) {
            failureHandler.accept(ChunkyDeployer.commandString(command));
        }
        return exitValue;
    }

    public static int launchChunky(LaunchMode mode, List<String> command, LoggerBuilder loggerBuilder) {
        ProcessBuilder processBuilder = new ProcessBuilder(command);
        final Logger logger = loggerBuilder.build();
        try {
            final Process process = processBuilder.start();
            Runtime.getRuntime().addShutdownHook(new Thread(){

                @Override
                public void run() {
                    process.destroy();
                }
            });
            Thread outputScanner = new Thread("Output Logger"){

                @Override
                public void run() {
                    try (InputStream is = process.getInputStream();){
                        int size;
                        byte[] buffer = new byte[4096];
                        while ((size = is.read(buffer, 0, buffer.length)) != -1) {
                            logger.appendStdout(buffer, size);
                        }
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
            };
            outputScanner.start();
            Thread errorScanner = new Thread("Error Logger"){

                @Override
                public void run() {
                    try (InputStream is = process.getErrorStream();){
                        int size;
                        byte[] buffer = new byte[4096];
                        while ((size = is.read(buffer, 0, buffer.length)) != -1) {
                            logger.appendStderr(buffer, size);
                        }
                    }
                    catch (IOException iOException) {
                        // empty catch block
                    }
                }
            };
            errorScanner.start();
            ShutdownThread shutdownThread = new ShutdownThread(process, logger, outputScanner, errorScanner);
            shutdownThread.start();
            try {
                if (mode == LaunchMode.GUI) {
                    Thread.sleep(3000L);
                    return shutdownThread.exitValue;
                }
                return shutdownThread.exitValue();
            }
            catch (InterruptedException interruptedException) {
                return 0;
            }
        }
        catch (IOException e) {
            logger.appendErrorLine(e.getMessage());
            return 3;
        }
    }

    public static String commandString(List<String> command) {
        StringBuilder sb = new StringBuilder();
        for (String part : command) {
            if (sb.length() > 0) {
                sb.append(" ");
            }
            sb.append(part);
        }
        return sb.toString();
    }

    private static List<String> buildCommandLine(VersionInfo version, LauncherSettings settings) {
        String[] parts;
        LinkedList<String> cmd = new LinkedList<String>();
        cmd.add(JreUtil.javaCommand(settings.javaDir));
        cmd.add("-Xmx" + settings.memoryLimit + "m");
        File settingsDirectory = SettingsDirectory.getSettingsDirectory();
        if (settingsDirectory != null) {
            cmd.add("-Dchunky.home=" + settingsDirectory.getAbsolutePath());
        }
        for (String part : parts = settings.javaOptions.split(" ")) {
            if (part.isEmpty()) continue;
            cmd.add(part);
        }
        cmd.add("-classpath");
        cmd.add(ChunkyDeployer.classpath(version));
        if (settings.verboseLogging) {
            cmd.add("-DlogLevel=INFO");
        }
        cmd.add("se.llbit.chunky.main.Chunky");
        for (String part : parts = settings.chunkyOptions.split(" ")) {
            if (part.isEmpty()) continue;
            cmd.add(part);
        }
        return cmd;
    }

    private static String classpath(VersionInfo version) {
        File chunkyDir = SettingsDirectory.getSettingsDirectory();
        File libDir = new File(chunkyDir, "lib");
        List jars = version.libraries.stream().map(library -> library.getFile(libDir)).collect(Collectors.toList());
        String classpath = "";
        for (File file : jars) {
            if (!classpath.isEmpty()) {
                classpath = classpath + File.pathSeparator;
            }
            classpath = classpath + file.getAbsolutePath();
        }
        return classpath;
    }

    public static VersionInfo resolveVersion(String name) {
        List<VersionInfo> versions = ChunkyDeployer.availableVersions();
        VersionInfo version = VersionInfo.LATEST;
        for (VersionInfo info : versions) {
            if (!info.name.equals(name)) continue;
            version = info;
            break;
        }
        if (version == VersionInfo.LATEST) {
            if (versions.size() > 0) {
                return versions.get(0);
            }
            return VersionInfo.NONE;
        }
        return version;
    }

    public static boolean canLaunch(VersionInfo version, ChunkyLauncherController launcher, boolean reportErrors) {
        if (version == VersionInfo.NONE) {
            System.err.println("Found no installed Chunky version.");
            if (reportErrors) {
                launcher.launcherError("No Chunky Available", "There is no local Chunky version installed. Please try updating.");
            }
            return false;
        }
        if (!ChunkyDeployer.checkVersionIntegrity(version.name)) {
            System.err.println("Version integrity check failed for version " + version.name);
            if (reportErrors) {
                launcher.launcherError("Chunky Version is Corrupt", "Version integrity check failed for version " + version.name + ". Please select another version.");
            }
            return false;
        }
        return true;
    }

    private static class ShutdownThread
    extends Thread {
        public volatile int exitValue = 0;
        private final Thread outputScanner;
        private final Thread errorScanner;
        private final Process proc;
        private final Logger logger;
        private boolean finished = false;

        public ShutdownThread(Process proc, Logger logger, Thread output, Thread error) {
            this.proc = proc;
            this.logger = logger;
            this.outputScanner = output;
            this.errorScanner = error;
        }

        public synchronized int exitValue() throws InterruptedException {
            while (!this.finished) {
                this.wait();
            }
            return this.exitValue;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                this.outputScanner.join();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            try {
                this.errorScanner.join();
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            try {
                this.proc.waitFor();
                this.exitValue = this.proc.exitValue();
                this.logger.processExited(this.exitValue);
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            ShutdownThread shutdownThread = this;
            synchronized (shutdownThread) {
                this.finished = true;
                this.notifyAll();
            }
        }
    }

    public static interface LoggerBuilder {
        public Logger build();
    }
}

