import { fixPre, fixedEncodeURIComponent } from '../utils/utils';
import log, { mark } from '../utils/log';

import config from '../config/config';
import itemNav from '../module/itemNavMixin';
import mixin from '../utils/mixin';
import reduxMixin from '../module/reduxMixin';
import reduxSettingMixin from '../module/reduxSettingMixin';

class Component {
    constructor(options = {}) {
        Object.assign(this, {
            options
        });
        if (this.options.id) {
            this.scope = fixPre(this.options.id, this.options.parentScope);
        } else {
            this.scope = '';
        }
        this.__init();
    }

    /**
   * 初始化
   */
    __init() {
        // mark('setPage');
        this.page = getCurrentPages()[getCurrentPages().length - 1];
        // mark('setPage', 'setPageEnd');
        this.__initState();
    }

    mark(start, end) {
        if (end) {
            mark(`${this.page.route}#${this.scope}#${start}`, `${this.page.route}#${this.scope}#${end}`);
        } else {
            mark(`${this.page.route}#${this.scope}#${start}`);
        }
    }
    /**
   * 初始化组件状态
   */
    __initState() {
        this.mark('initState');

        this.firstShow = true;
        this.data = Object.assign({ visible: true }, this.data, this.options.data);
        delete this.options.data;
        this.methods = Object.assign({}, this.methods, this.options.methods);
        delete this.options.methods;
        this.components = Object.assign({}, this.components, this.options.components);
        delete this.options.components;
        this.__initMethods(this.data);

        this.visible = this.data.visible;
        if (typeof this.data.checkType !== 'undefined') {
            this.checkType = this.data.checkType;
        }
        if (typeof this.checkType !== 'undefined') {
            this.data.checkType = this.checkType;
        }

        this.setData(this.data);

        this.__initOther();

        this.__initComponents();

        this.on('onReady', this.onReady);
        this.on('onLoad', this.onLoad);
        this.on('onShow', this._onShow);
        this.on('onHide', this._onHide);
        this.on('onUnload', this.onUnload);

        // this.mark('initState', 'initStateEnd');
        log(this.scope, this);
    }

    __initComponents() {
        this.coms = {};
        if (this.isEmptyObject(this.components)) {
            return;
        }

        // this.mark(`__initComponents`);

        let keys = Object.keys(this.components);

        keys.forEach((item) => {
            const componentInfo = this.components[item];
            if (componentInfo) {
                const Com = componentInfo.component ? componentInfo.component : componentInfo;
                let options = {};
                if (componentInfo.component) {
                    options =
                        typeof componentInfo.options === 'function'
                            ? componentInfo.options.apply(this, [ this ])
                            : componentInfo.options;
                }
                // this.mark(`create#${item}`);
                this.createComponent(Com, item, options);
                // this.mark(`create#${item}`, `create#${item}#end`);
            }
        });

        // this.mark(`__initComponents`, `__initComponentsEnd`);
    }

    createComponent(type, id, options) {
        this.coms[id] = new type(
            Object.assign({}, this.options, options, {
                parentScope: this.scope,
                id,
                componentInited: this.componentInited || false
            })
        );
        if (this.componentInited) {
            this.coms[id].onLoad({});
            this.coms[id].onReady();
            if (this.coms[id].visible) {
                this.coms[id].onShow();
            }
        }
    }

    callChild(method, filter) {
        const args = [].slice.call(arguments, 2);
        const keys = Object.keys(this.coms);
        keys.forEach((key) => {
            if (!filter || (filter && filter(this.coms[key]))) {
                this.coms[key][method].apply(this.coms[key], args);
            }
        });
    }

    __initMethods(data) {
        if (this.isEmptyObject(this.methods)) {
            return;
        }

        // this.mark(`__initMethods`);
        const methods = this.methods;

        for (let key in methods) {
            if (methods.hasOwnProperty(key) && typeof methods[key] === 'function') {
                this[key] = methods[key] = methods[key].bind(this);

                const name = fixPre(key, this.scope);
                // 将 methods 内的方法重命名并挂载到 page 上面，否则 template 内找不到事件
                this.page[name] = methods[key];

                // 将方法名同步至 page.data 上面，方便在模板内使用 {{ method }} 方式绑定事件
                data[key] = name;
            }
        }
        // this.mark(`__initMethods`, `__initMethodsEnd`);
    }

    __initOther() {
        // this.mark(`__initOther`);
        const options = this.options;
        for (let key in options) {
            if (options.hasOwnProperty(key)) {
                this[key] = options[key];
                if (typeof options[key] === 'function') {
                    delete options[key];
                }
            }
        }
        // this.mark(`__initOther`, `__initOtherEnd`);
    }

    /**
   * 判断 object 是否为空
   */
    isEmptyObject(e) {
        for (let t in e) return false;
        return true;
    }

    getData() {
        if (this.scope) {
            let data = this.page.data;
            let name = this.scope.split('.');

            name.forEach((n) => {
                data = data[n] || {};
            });

            return data;
        }
        return this.page.data;
    }

    setVisible(className) {
        if (!this.visible) {
            if (!className) {
                if (typeof this.defaultShowCss === 'function') {
                    className = this.defaultShowCss();
                } else {
                    className = this.defaultShowCss;
                }
            }

            this.setData({
                animateCss: className,
                visible: !0
            });
            this.visible = true;
            this.onShow();
            return true;
        }
        return false;
    }

    setHidden(className, callback) {
        if (this.visible) {
            if (!className) {
                if (typeof this.defaultHideCss === 'function') {
                    className = this.defaultHideCss();
                } else {
                    className = this.defaultHideCss;
                }
            }
            this.setData({
                animateCss: className
            });
            this.onHide();

            this.setDelayData(
                {
                    visible: false
                },
                config.animate || 200,
                callback
            );
            this.visible = false;
            return true;
        }
        callback && callback();
        return false;
    }

    setDelayData(data, timer = 300, callback) {
        if (timer === 0) {
            this.setData(data);
        } else {
            setTimeout(() => {
                this.setData(data);
                callback && callback();
            }, timer);
        }
    }
    _setData(data, prefix) {
        if (this.isEmptyObject(data)) {
            return data;
        }
        if (this.scope || prefix) {
            const newData = {};
            for (let key in data) {
                if (this.scope) {
                    if (prefix) {
                        newData[`${this.scope}.${prefix}.${key}`] = data[key];
                    } else {
                        newData[`${this.scope}.${key}`] = data[key];
                    }
                } else {
                    newData[`${prefix}.${key}`] = data[key];
                }
            }
            return newData;
        }
        return data;
    }
    setData(data, prefix, forceImmediate) {
        if (this.page.pageData.batch && !forceImmediate) {
            this.page.setPageData(this._setData(data, prefix));
        } else {
            this.page.setDataFix(this._setData(data, prefix));
        }
    }
    on(event, fn) {
        this.page.on(event, fn, this);
    }
    off(event) {
        this.page.off(event);
    }
    emit(event, ctx, ...params) {
        this.page.emit(event, ctx, this.scope, ...params);
    }
    setPageId(pageId) {
        if (pageId && config.router[pageId]) {
            Object.assign(this.options, config.router[pageId], { pageId });
        } else {
            log('setPageId 没有找到pageId，是否忘了配置', pageId);
        }
    }
    onLoad(options) {
        // Do some initialize when page load.
        let newOptions = Object.assign({}, options, this.options);
        if (newOptions.pageId && config.router[newOptions.pageId]) {
            Object.assign(newOptions, config.router[newOptions.pageId]);
        }
        this.setData(newOptions);
        this.options = newOptions;
        this.componentInited = true;
    }
    getOption(option) {
        if (typeof this[option] !== 'undefined') {
            return this[option];
        }
        return this.options[option];
    }
    // 不要直接调用, 区分是否是事件触发（事件是全局的）
    _onShow() {
        this.visible && this.onShow(true);
    }
    // 不要直接调用, 区分是否是事件触发（事件是全局的）
    _onHide() {
        this.visible && this.onHide(true);
    }
    onReady() {
        // Do something when page ready.
    }
    onFirstShow() {}
    onShow(isEvent) {
        // Do something when page show.
        // console.log(isEvent);
        this.isShowed = true;
        if (this.firstShow) {
            this.firstShow = false;
            this.onFirstShow();
        }
        if (!isEvent) {
            this.callChild('onShow', (com) => com.getData().visible);
        }
    }
    // 父节点隐藏会触发子节点隐藏，但是子节点visible还是true，所以有可能onHide调用多次
    onHide(isEvent) {
        this.isShowed = false;
        // Do something when page hide.
        if (!isEvent) {
            this.callChild('onHide', (com) => com.getData().visible);
        }
    }
    onUnload() {
        // Do something when page close.
        // this.coms = {};
    }

    getChildEventScope(scope) {
        if (this.scope) {
            return scope.replace(`${this.scope}.`, '');
        }
        return scope;
    }
    navigateTo(url, params) {
        this.page.navigateTo(url, params);
    }
    redirectTo(url, params) {
        params && (url = `${url}?json=${fixedEncodeURIComponent(JSON.stringify(params))}`);
        wx.redirectTo({
            url
        });
    }
}

export default mixin(Component, [
    reduxMixin,
    reduxSettingMixin,
    itemNav,
    {
        defaultShowCss: 'fadeIn',
        defaultHideCss: 'fadeOut'
    }
]);
