/**
 * @copyright Maichong Software Ltd. 2016 http://maichong.it
 * @date 2016-11-19
 * @author Liang <liang@maichong.it>
 */

import log, { mark } from '../utils/log';

import config from '../config/config';
import storage from './storage';
import stringify from '../lib/qs/lib/stringify';
import { tryJsonToStr } from '../utils/utils';

/**
 * 默认获取SessionID方法
 * @returns {string}
 */
function defaultGetSession() {
  return getApp().sessionId;
}

/**
 * 默认设置SessionID方法
 * @param {string} sessionId
 */
function defaultSetSession(sessionId) {
  getApp().sessionId = sessionId;
}

// 有效HTTP方法列表
const methods = ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS', 'HEAD', 'TRACE', 'CONNECT'];

/**
 * 创建API Request客户端
 * @param {Object} options 选项
 * @returns {Function}
 */
export function create(options) {
  options = options || {};

  /**
   * 通用Alaska RESTFUL风格API请求,如果alaska接口返回错误,则抛出异常
   * @param {string} [method] 请求方法,可选默认GET,有效值：OPTIONS, GET, HEAD, POST, PUT, DELETE, TRACE, CONNECT
   * @param {string} apiName  API名称,必选
   * @param {object} [data]   数据,可选,如果方法为GET或DELETE,则此对象中的所有数据将传入URL query
   * @param {object} [header] HTTP头对象,可选
   * @returns {*}
   */
  function request(method, apiName, data, header) {
    return new Promise((resolve, reject) => {
      const apiRoot = options.apiRoot || {};
      const updateKey = options.updateKey || '_session';
      const headerKey = options.headerKey || 'Session';
      const getSession = options.getSession || defaultGetSession;
      const setSession = options.setSession || defaultSetSession;
      const defaultHeader = options.defaultHeader;

      if (methods.indexOf(method) === -1) {
        header = data;
        data = apiName;
        apiName = method;
        method = 'GET';
      }

      let url = apiName.indexOf('http') === 0 ? apiName : apiRoot   + apiName;


      if (!config.instance){
        let subDomain = getApp().subDomain;
        if (!subDomain) return false;
        url = url.replace(/\/\/([1-9a-z\.]+)\//ig, '//$1/d-'+subDomain+'/');
      }

      if (['POST', 'PUT'].indexOf(method) === -1 && data) {
        let querystring = stringify(data);
        if (url.indexOf('?') > -1) {
          url += '&' + querystring;
        } else {
          url += '?' + querystring;
        }
        data = undefined;
      }

      header = Object.assign({}, defaultHeader, header);

      if (options.session !== false) {
        let sessionId = getSession();
        if (sessionId) {
          header[headerKey] = sessionId;
        }
      }

      mark('beforeRequest');
      wx.request({
        method,
        url,
        data,
        header,
        success(res) {
          mark('beforeRequest', 'endRequest');
          if (options.session !== false && res.data && res.data[updateKey]) {
            if (res.data && res.data[updateKey]) {
              setSession(res.data[updateKey]);
            }
          }

          if (res.statusCode < 200 || res.statusCode > 299) {
            reject({
              code: res.statusCode,
              message: res.errMsg
            });
            return;
          }

          if (res.data && res.data.error) {
            reject(res.data.error);
            return;
          }

          if (res.data && res.data.returnCode && res.data.returnCode !== '000000') {
            reject({
              code: res.data.returnCode,
              message: res.data.message
            });
            return;
          }

          if (res.data && res.data.resp && res.data.resp.respCode && res.data.resp.respCode !== '000000') {
            reject({
              code: res.data.resp.respCode,
              message: res.data.resp.message
            });
            return;
          }

          resolve(res.data);
        },
        fail(err) {
          reject(err);
        }
      });
    });
  }

  methods.forEach((method) => {
    request[method.toLowerCase()] = function(...args) {
      return request(method, ...args);
    };
  });

  request.setOptions = function(newOptions) {
    Object.assign(options, newOptions);
  };

  request.getOptions = function() {
    return options;
  };

  return request;
}

/**
 * 导出默认API客户端
 */
const request = create({
  apiRoot: config.apiRoot
});
export default request;

const cookieKey = config.cookie;
export function api(method, apiName, data, header, options) {
  return new Promise((resolve, reject) => {
    const config = Object.assign({}, request.getOptions(), options);

    const callback = (cookie) => {

      let newHeader = Object.assign({}, header);
      if (cookie) {
        let newCookieKey = cookieKey;
        if(getApp().subDomain)   newCookieKey = '.'+ getApp().subDomain;
        newHeader.Cookie = `${newCookieKey}=${cookie}`;
      }
      const req = create(config);
      log('request method %s, apiName %s, data %o, header %o, options %o ', method, apiName, data, newHeader, config);
      if (options) {
        req.setOptions(options);
      }

      req(method, apiName, data, newHeader).then((result) => {
        log('request result', result); //, JSON.stringify(result));
        resolve(result);
      }).catch((error) => {
        log('request error', error);
        reject(error);
      });
    }

    // console.log('getApp', getApp())
    if (getApp().cookie) {
      callback(getApp().cookie);
    } else {
      storage.getItem('userData').then((userData) => {
        getApp().cookie = userData.cookie;
        callback(userData.cookie);
      }).catch((error) => {
        console.log(error);
        callback(null);
      });
    }
  });
}






export function wxLogin() {
  return new Promise((resolve, reject) => {


     getSubDomain().then((d)=>{

       console.log(d);


       wx.getSystemInfo({
           success: (sysInfo) => {

                 if (sysInfo.environment && sysInfo.environment=='wxwork'){

                   wx.qy.login({
                      success: function(res) {
                        if (res.code) {
                            console.log(res.code);
                            var res2 ={};
                            res2.userInfo ={ nickName:'', avatarUrl:''};
                            res2.code = res.code;
                            res2.wxwork = 1;
                            resolve(res2);


                        } else {
                          log('获取用户登录态失败！' + res.errMsg);
                          reject(res.errMsg);
                        }
                      }
                    });



                 }else{

                   mark('wxLogin');
                   wx.login({
                     success(res) {
                       mark('wxLogin', 'wxLoginEnd');
                       log('wxLogin', res);
                       console.log(res);
                       if (res.code) {
                         mark('getUserInfo');
                         wx.getUserInfo({
                           success(res2) {
                             console.log(res2);

                             mark('getUserInfo', 'getUserInfoEnd');
                             log('getUserInfo', res2);
                             res2.code = res.code;
                             res2.wxwork = 0;
                             resolve(res2);
                             // linkWx2(res2.userInfo.nickName, res2.userInfo.avatarUrl, res.code).then((data) => {
                             //   res2.user = data;
                             //   res2.user.wx = res2.userInfo;
                             // }).catch(error => reject(error));
                           },
                           fail(error) {
                             console.log(error);
                             reject(error);
                           }
                         });
                       } else {
                         log('获取用户登录态失败！' + res.errMsg);
                         reject(res.errMsg);
                       }
                     },
                     fail(error) {
                       reject(error);
                     }
                   });




                 }
              }

         });




     });



  });
}



export function getSubDomain() {

     return new Promise((resolve, reject) => {

        const querySubDomain = () => {

          wx.getSystemInfo({
              success: (sysInfo) => {

                     if (sysInfo.environment && sysInfo.environment=='wxwork'){

                       wx.qy.login({
                          success: function(res) {
                            if (res.code) {
                              console.log(res.code);
                              //发起网络请求
                              wx.request({
                                    url:  config.registerUrl   , //仅为示例，并非真实的接口地址
                                    data: {  code : res.code   },
                                    header: {
                                      'content-type': 'application/json' // 默认值
                                    },
                                    success (resx) {
                                          storage.setItem('subDomain', resx.data.name).then(() => {
                                             console.log(resx);
                                             getApp().subDomain  = resx.data.name;
                                             resolve(resx.data.name);
                                      });
                                    }
                                  });

                            } else {
                              getApp().subDomain  = '';
                              resolve('');
                            }
                          }
                        });





                     }else{
                         getApp().subDomain  = '';
                         resolve('');
                     }

                   }
               });


        };

      return  storage.getItem('subDomain').then((d) => {
        if (d) {
            console.log('kkkk');
            getApp().subDomain =  d;
            resolve(d);
        }
        else {
            console.log('fffksssx');
            return querySubDomain();
        }
      }).catch(querySubDomain);

    });
}



export function linkWx2(nickName, avatarUrl, jsCode, extra) {
  const queryUser = () => {
      console.log('newx');
    return api('POST', 'account/login', {
      UserName: '_',
      Password: '_',
      nickName,
      js_code: jsCode,
      avatarUrl,
      ...extra
    }).then((data) => {
      return storage.setItem('userData', data.resp.data).then(() => {
            getApp().cookie = data.resp.data.cookie;
            return data.resp.data;
      });
    });
  };

  return  storage.getItem('userData').then((userData) => {
    console.log(userData);
    console.log(config.forceLogin);
    if (!config.forceLogin && userData)
    {
        getApp().cookie =  userData.cookie;
        return userData;
    }
    else {
        return queryUser();
    }
  }).catch(queryUser);
}

export function linkWx(nickName, openId) {
  return api('POST', 'list/api', {
    pageId: 3,
    index: 0,
    size: 15,
    nickName,
    js_code: openId
  });
}

export function login(username, password) {
  return api('POST', 'account/login', { username, password }).then((result) => {
    if (result.resp.respCode === '000000') {
      return storage.setItem('cookie', result.resp.data.cookie).then(() => {
        getApp().cookie =  result.resp.data.cookie;
        return result.resp.data;
      })
    } else {
      throw Error('登录失败');
    }
  });
}

export function upload(itemId, uri, uploadKey) {
  return new Promise((resolve, reject) => {
    // const name = uri.substr(uri.lastIndexOf('/') + 1);

    let url = config.apiRoot + 'file/jcreate';

    if (!config.instance){
      let subDomain = getApp().subDomain;
      if (!subDomain) return false;
      url = url.replace(/\/\/([1-9a-z\.]+)\//ig, '//$1/d-'+subDomain+'/');
    }


    wx.uploadFile({
      url:  url ,
      filePath: uri,
      name: 'file',
      formData: {
        key: uploadKey,
        itemId
      },
      success: (res, successCode) => {
        log(res);
        if (res.statusCode === 200 && res.data) {
          resolve(JSON.parse(res.data));
        } else {
          reject(res);
        }
      },
      fail: (error) => {
        log('upload error', error);
        reject(error);
      }
    });
  });
}

export function updateProfile(userId, data) {
  return api('POST', 'account/profile/' + userId, data);
}

export function getProfile() {
  return api('GET', 'account/profile/me');
}

export function restPassword(email) {
  return api('POST', 'account/resetPasswordRequest', { email });
}

export function logout() {
  return api('POST', 'account/logout');
}

export function signUp(username, email, password) {
  return api('POST', 'account/register', {
    username,
    email,
    password
  });
}

export function anonymousLogin() {
  return api('POST', 'account/nodelogin?t=' + Math.random(), {
    companyName: config.companyName,
    UserName: '_',
    Password: '_'
  });
}

export function cache(key, time) {
  return api('POST', 'cache/list?t=' + Math.random(), {
    cachekey: key,
    ttt: time
  });
}

export function cache2(key, time) {
  return api('POST', 'cache/list2?t=' + Math.random(), {
    cachekey: key,
    ttt: time
  });
}

export function cachePage(pageId) {
  return api('POST', 'cache/list_page?t=' + Math.random(), {
    pageId
  });
}

export function fakeDel({ categoryId, ids }) {
  if (Array.isArray(ids)) {
    ids = ids.join(',');
  }
  return api('POST', 'list/fakedel', { ids, categoryId, del: true });
}

export function updateStatus({ cateId,
        ids,
        listPageId,
        pageId,
        patternId,
        statusId }) {
  if (Array.isArray(ids)) {
    ids = ids.join(',');
  }
  return api('POST', 'list/updatestatus', { cateId,
    ids,
    listPageId,
    pageId,
    patternId,
    statusId });
}

export function updateWorkflow(data) {
  return api('POST', 'list/updateworkflow', data);
}

export function search(categoryId, pageId, searchConditions, indx, size) {
  return api('POST', 'list/ajxlist', {
    companyId: config.companyId,
    cid: categoryId,
    id: pageId,
    lsc: searchConditions,
    iindex: indx,
    isize: size
  });
}

export function search2(categoryId, pageId, searchConditions, indx, size) {
  return api('POST', 'list/ajxlist2', {
    companyId: config.companyId,
    cid: categoryId,
    id: pageId,
    lsc: searchConditions,
    iindex: indx,
    isize: size
  });
}

export function search2o({cid, id, lsc, iindex, isize}) {
  return search2(cid, id, lsc, iindex, isize);
}

export function plugin(plugin, data) {
  return api('POST', 'plugin/' + plugin, data);
}

export function delFile(cateId, attachmentId) {
  return api('POST', 'file/jdel', {
    cateId,
    attachmentId
  });
}

export function bi({ pageId, lscStr }) {
  return api('POST', 'report/bi', {
    pageId,
    lscStr: tryJsonToStr(lscStr)
  });
}

export function common({ pageId, lscStr }) {
  return api('POST', 'report/common', {
    pageId,
    lscStr: tryJsonToStr(lscStr)
  });
}


export function bi_server_long({ pageId, lscStr }) {
  return api('POST', 'report/bi_server_long', {
    pageId,
    lscStr: tryJsonToStr(lscStr)
  });
}


export function bi_client_short({ pageId, lscStr }) {
  return api('POST', 'report/bi_client_short', {
    pageId,
    lscStr: tryJsonToStr(lscStr)
  });
}

export function txt(path)
{
  return api('GET', 'file', {path});
}

export function get(url)
{
  return api('GET', url);
}


export function detail(pageId, itemId) {
  return api('POST', 'detail/ajx', {
    pageId,
    itemId
  });
}

export function detail2(pageId, itemId, server_identity) {
  const params = { pageId };
  if (itemId) {
    params.itemId = itemId;
  } else if (server_identity) {
    params.itemId = 0;
    params.server_identity = server_identity;
  }
  return api('POST', 'detail/ajx2', params);
}

export function detail2o({ pageId, itemId, server_identity }) {
  return detail2(pageId, itemId, server_identity);
}

export function fileExist(data) {
  return api('POST', 'file/exist', data);
}

export function workflowLog(data) {
  return api('POST', 'log/workflow', data);
}

export function dataLog(data) {
  return api('POST', 'log/data', data);
}

export function cacheUpdate(data) {
  return api('POST', 'admin/cacheupdate', data);
}

export function checkLoginStatus() {
  return api('POST', 'account/user').then((result) => {
    if (result.XingMing === null || result.XingMing === '匿名') {
      storage.removeItem('sessionKey');
      storage.removeItem('cookie');
    }
    return result;
  });
}

export function bulkCreate({ pageId, jsonStr }) {
  return api('POST', 'e/bulkcreate', {
    jsonStr: tryJsonToStr(jsonStr),
    pageId
  });
}

// export function complexUpdate(data) {
//   return api('POST', 'e/complexupdate', data);
// }

export function complexUpdate({ pageId, itemId, jsonStr, relatedSTR, delSTR }) {
  return api('POST', 'e/complexupdate', {
    jsonStr: tryJsonToStr(jsonStr),
    relatedSTR: relatedSTR ? tryJsonToStr(relatedSTR) : '[]',
    itemId,
    pageId,
    delSTR: tryJsonToStr(delSTR)
  });
}

export function createUpdate({ pageId, itemId, jsonStr, relatedSTR }) {
  return api('POST', 'e/createupdate', {
    jsonStr: tryJsonToStr(jsonStr),
    relatedSTR: relatedSTR ? tryJsonToStr(relatedSTR) : '[]',
    itemId,
    pageId
  });
}

// export function editUpdate(data) {
//   return api('POST', 'e/editupdate', data);
// }

export function changePassword(data) {
  return api('POST', 'account/ChangePassword', data);
}

export function checkUnique(data) {
  return api('POST', 'ajax/checkunique', data);
}
