﻿using System;
using RimWorld;
using Verse;
using System.Collections.Generic;

namespace Markers
{
    abstract class Designator_Marker : Designator
    {
        private List<Pawn> justDesignated = new List<Pawn>();

        // --- Overriding Methods ---
        public override int DraggableDimensions
        {
            get
            {
                return 0;
            }
        }

        public override AcceptanceReport CanDesignateCell(IntVec3 c)
        {
            AcceptanceReport result;
            if (!c.InBounds(base.Map))
            {
                return false;
            }
            if (this.hasNoMarkerDesignation(c))
            {
                foreach (Thing thing in base.Map.thingGrid.ThingsAt(c))
                {
                    if (this.IsHumanAndNotColonist(thing) && !c.Fogged(base.Map))
                    {
                        return AcceptanceReport.WasAccepted;
                    }
                }

                return "MarkPawnReport".Translate();
            }
            result = AcceptanceReport.WasRejected;
            
            return result;
        }

        public override void DesignateSingleCell(IntVec3 loc)
        {
            foreach (Thing current in base.Map.thingGrid.ThingsAt(loc))
            {
                if (IsHumanAndNotColonist(current))
                {
                    this.DesignateThing(current);
                }
            }
        }

        public override void DesignateThing(Thing t)
        {
            base.Map.designationManager.RemoveAllDesignationsOn(t, false);
            base.Map.designationManager.AddDesignation(this.getCurrentDesignation(t));
            this.justDesignated.Add((Pawn)t);
        }

        protected override void FinalizeDesignationSucceeded()
        {
            base.FinalizeDesignationSucceeded();
            this.justDesignated.Clear();
        }

        // --- Custom Methods ---
        protected bool IsHumanAndNotColonist(Thing t)
        {
            if(null == t)
            {
                return false;
            }

            Pawn pawn = t as Pawn;

            if(null == pawn)
            {
                return false;
            }

            return pawn.RaceProps.Humanlike && !pawn.IsColonistPlayerControlled;
        }

        protected bool hasNoMarkerDesignation(object o)
        {
            Type objectType = o.GetType();
            if (typeof(IntVec3) == o.GetType())
            {
                return CellHasNoMarkerDesignation((IntVec3)o);
            }

            return false;
        }

        abstract protected bool CellHasNoMarkerDesignation(IntVec3 c);

        abstract protected Designation getCurrentDesignation(Thing t);
    }
}