require "prefabutil"

local assets =
{
	hatch =
	{
		Asset("ANIM", "anim/basement_hatch.zip"),
		Asset("SOUNDPACKAGE", "sound/hatch.fev"),
		Asset("SOUND", "sound/hatch_bank00.fsb"),
	},
	stairs = { Asset("ANIM", "anim/basement_exit.zip") },
	--elevator = { Asset("ANIM", "anim/basement_elevator.zip") },
	tile = { Asset("ANIM", "anim/basement_floor.zip") },
}

local BASEMENT_SHADE = 0.5

local WALL_ANIM_VARIANTS = { "fullA", "fullB", "fullC" }

local function wall_common(build)
	local inst = CreateEntity()
	
	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddNetwork()
	
	inst.Transform:SetEightFaced()
	
	inst:AddTag("blocker")
	local phys = inst.entity:AddPhysics()
	phys:SetMass(0)
	phys:SetCollisionGroup(COLLISION.WORLD)
	phys:ClearCollisionMask()
	phys:CollidesWith(COLLISION.ITEMS)
	phys:CollidesWith(COLLISION.CHARACTERS)
	phys:CollidesWith(COLLISION.GIANTS)
	phys:CollidesWith(COLLISION.FLYERS)
	phys:SetCapsule(0.5, 50)
	
	inst.AnimState:SetBank("wall")
	inst.AnimState:SetBuild(build)
	inst.AnimState:OverrideShade(BASEMENT_SHADE)
		
	inst:AddTag("NOCLICK")
	inst:AddTag("basement_part")
	
	inst.entity:SetPristine()
	
	if not TheWorld.ismastersim then
		return inst
	end
	
	inst.AnimState:PlayAnimation(WALL_ANIM_VARIANTS[math.random(#WALL_ANIM_VARIANTS)])
	
	inst.persists = false
			
	return inst
end

local function wall1()
	return wall_common("wall_stone")
end

local function wall2()
	return wall_common("wall_wood")
end

--------------------------------------------------------------------------

local function IsNearBasement(x, y, z)
	return #TheSim:FindEntities(x, 0, z, 100, { "alt_tile" }) > 0
end

local function OnBuilt(inst)
	if inst.components.teleporter.targetTeleporter ~= nil then
		inst:RemoveEventCallback("onbuilt", OnBuilt)
		return
	end
	
	local tries = 0
	local basement_position = nil
	while true do	
		local x = math.random(1500, 2000) * (math.random() < 0.5 and 1 or -1)
		local z = math.random(1500, 2000) * (math.random() < 0.5 and 1 or -1)
		if not IsNearBasement(x, 0, z) then
			basement_position = { x, 0, z }
			break
		else
			tries = tries + 1
			if tries > 50 then
				TheNet:Announce("Failed to find valid position for basement.")
				Waffles.DespawnRecipe(inst, true)
				return
			end
		end
	end
	
	local exit = SpawnPrefab("basement_exit")
	exit.Transform:SetPosition(unpack(basement_position))
	
	inst.components.teleporter.targetTeleporter = exit
	exit.components.teleporter.targetTeleporter = inst
end

local function OnMouseOver(inst)
	local x, y, z = inst.Transform:GetWorldPosition()
	inst.highlightchildren = TheSim:FindEntities(x, y, z, 0, { "DECOR" })
end

local shining_areas =
{
	basement_entrance =
	{
		{ symbol = "bottom", x = { -100, 100 }, y = { -50, 50 } },
	},
	
	basement_entrance_hatch =
	{
		{ symbol = "hatch", x = { -25, 50 }, y = { -25, 50 } },
		{ symbol = "mesh", x = { 0, 8 }, y = { 0, 8 } },
		{ symbol = "vent", x = { -15, 15 }, y = { -15, 15 } },
	},
}

local function Shine(inst)
	if inst:IsAsleep() then
		return
	end
	
	local parent = math.random() < 0.5 and inst.hatch or inst
	local t = shining_areas[parent.prefab]
	local data = t[math.random(#t)]
	SpawnPrefab("sparkle_fx"):Hook(parent.GUID, data.symbol, math.random(unpack(data.x)), math.random(unpack(data.y)), 0)
	inst:DoTaskInTime(math.random(3, 15), Shine)
end

local function GetStatus(inst)
	return inst.sg.currentstate.name ~= "idle" and "OPEN" or nil
end

local function OnDoneTeleporting(inst, obj)
	if inst.closetask ~= nil then
		inst.closetask:Cancel()
	end
	inst.closetask = inst:DoTaskInTime(1.5, function()
		if not (inst.components.teleporter:IsBusy() or
				inst.components.playerprox:IsPlayerClose()) then
			inst.sg:GoToState("closing")
		end
	end)
end

local function OnActivate(inst, doer)
	if doer:HasTag("player") then
		if doer.components.talker ~= nil then
			doer.components.talker:ShutUp()
		end
	else
		inst.SoundEmitter:PlaySound("dontstarve/cave/rope_up")
	end
end

local function OnActivateByOther(inst, source, doer)
	if not inst.sg:HasStateTag("open") then
		inst.sg:GoToState("opening")
	end
	if doer ~= nil and doer.Physics ~= nil then
		doer.Physics:CollidesWith(COLLISION.WORLD)
	end
end

local function ChainPlayerprox(inst, near) --currently doesn't work
	local link = inst.components.teleporter.targetTeleporter
	if Waffles.Valid(link) and not link:IsAsleep() then
		if near then
			link.components.playerprox.onnear(link, true)
		else
			link.components.playerprox.onfar(link, true)
		end
	end
end

local function OnNearEntrance(inst, chain)
	if inst.components.teleporter:IsActive() and not inst.sg:HasStateTag("open") then
		inst.sg:GoToState("opening")
	end
	
	if not chain then
		ChainPlayerprox(inst, true)
	end
end

local function OnFarEntrance(inst, chain)
	if not inst.components.teleporter:IsBusy() and inst.sg:HasStateTag("open") then
		inst.sg:GoToState("closing")
	end
	
	if not chain then
		ChainPlayerprox(inst, false)
	end
end

local function OnAccept(inst, giver, item)
	inst.components.inventory:DropItem(item)
	inst.components.teleporter:Activate(item)
end

local function OnDoneTeleporting(inst, obj)
    if inst.closetask ~= nil then
        inst.closetask:Cancel()
    end
    inst.closetask = inst:DoTaskInTime(1.5, function()
        if not (inst.components.teleporter:IsBusy() or
                inst.components.playerprox:IsPlayerClose()) then
            inst.sg:GoToState("closing")
        end
    end)
end

local function PlayTravelSound(inst, doer)
	inst.SoundEmitter:PlaySound("dontstarve/cave/rope_down")
end

local function DoShineFlick(inst)
	Waffles.DoHauntFlick(inst, math.random() * 0.65)
	inst:DoTaskInTime(math.random(3), DoShineFlick)
end

local function entrance()
	local inst = CreateEntity()
	
	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddMiniMapEntity()
	inst.entity:AddSoundEmitter()
	inst.entity:AddNetwork()
	
	inst.Transform:SetTwoFaced()
	
	MakeObstaclePhysics(inst, 0.65)
	
	inst.AnimState:SetBank("basement_entrance")
	inst.AnimState:SetBuild("basement_hatch")
	inst.AnimState:PlayAnimation("idle")	
	inst.AnimState:SetLayer(LAYER_BACKGROUND)
	inst.AnimState:SetSortOrder(3)
	
	inst.MiniMapEntity:SetIcon("basement.tex")
	
	inst:AddTag("basement_part")
	inst:AddTag("antlion_sinkhole_blocker")
	
	inst:SetDeployExtraSpacing(2.5)
			
	if not TheNet:IsDedicated() then
		inst:ListenForEvent("mouseover", OnMouseOver)
	end
	
	inst.entity:SetPristine()
		
	if not TheWorld.ismastersim then
		DoShineFlick(inst)
		
		return inst
	end
		
	--the hatch don't need to be on background layer, so it's a separate entity
	inst.hatch = SpawnPrefab("basement_entrance_hatch")
	inst.hatch.entity:SetParent(inst.entity)
	
	inst:SetStateGraph("SGbasement_entrance")
	
	inst:AddComponent("inspectable")
    inst.components.inspectable.getstatus = GetStatus
	
	inst:AddComponent("teleporter")
	inst.components.teleporter.onActivate = OnActivate
	inst.components.teleporter.onActivateByOther = OnActivateByOther
	inst.components.teleporter.offset = 0
	inst.components.teleporter.travelcameratime = 0.6
	inst.components.teleporter.travelarrivetime = 0.5
	
	inst:AddComponent("playerprox")
	inst.components.playerprox:SetDist(4, 5)
	inst.components.playerprox.onnear = OnNearEntrance
	inst.components.playerprox.onfar = OnFarEntrance
	
	inst:AddComponent("inventory")

	inst:AddComponent("trader")
	inst.components.trader.acceptnontradable = true
	inst.components.trader.onaccept = OnAccept
	inst.components.trader.deleteitemonaccept = false
	
	inst:ListenForEvent("onbuilt", OnBuilt)
	inst:ListenForEvent("doneteleporting", OnDoneTeleporting)
	inst:ListenForEvent("starttravelsound", PlayTravelSound)
		
	inst.OnEntityWake = Shine
	
	return inst
end

local function entrance_hatch()
	local inst = CreateEntity()

	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddNetwork()
	
	inst.Transform:SetTwoFaced()

	inst.AnimState:SetBank("basement_hatch")
	inst.AnimState:SetBuild("basement_hatch")
	inst.AnimState:PlayAnimation("idle_closed")
	
	inst:AddTag("DECOR")
	inst:AddTag("NOCLICK")
		
	inst.entity:SetPristine()
	
	if not TheWorld.ismastersim then
		DoShineFlick(inst)
	
		return inst
	end
		
	for _,v in pairs({ "animover", "animqueueover" }) do
		inst:ListenForEvent(v, function()
			local parent = inst.entity:GetParent()
			if parent ~= nil then
				parent:PushEvent(v)
			end
		end)
	end

	return inst
end

local function builder()
	local inst = CreateEntity()

	inst.entity:AddTransform()

	inst:AddTag("CLASSIFIED")

	--[[Non-networked entity]]
	inst.persists = false

	--Auto-remove if not spawned by builder
	inst:DoTaskInTime(0, inst.Remove)

	if not TheWorld.ismastersim then
		return inst
	end

	inst.OnBuiltFn = function()
		local x, y, z = inst.Transform:GetWorldPosition()
		
		local hatch = SpawnPrefab("basement_entrance")
		hatch.Transform:SetPosition(x, y, z)
		hatch:PushEvent("onbuilt")
		
		local fx_burntground = SpawnPrefab("burntground")
		fx_burntground.persists = false
		fx_burntground.Transform:SetPosition(x, y - 0.1, z)
		fx_burntground.AnimState:SetScale(3, 3)
		fx_burntground:DoTaskInTime(5, ErodeAway)
				
		local fx_explode = SpawnPrefab("explode_small")
		fx_explode.Transform:SetPosition(x, y + 0.1, z)
		
		for i, v in ipairs(AllPlayers) do
			local distSq = v:GetDistanceSqToInst(inst)
			local k = math.max(0, math.min(1, distSq / 1600))
			local intensity = k * (k - 2) + 1
			if intensity > 0 then
				v:ScreenFlash(intensity)
				v:ShakeCamera(CAMERASHAKE.FULL, .7, .02, intensity / 2)
			end
		end
				
		inst:Remove()
	end

	return inst
end

local function placerdecor(parent)		
	local inst = CreateEntity()

	inst:AddTag("CLASSIFIED")
	inst:AddTag("NOCLICK")
	inst:AddTag("placer")
	--[[Non-networked entity]]
	inst.entity:SetCanSleep(false)
	inst.persists = false

	inst.entity:AddTransform()
	inst.entity:AddAnimState()

	inst.AnimState:SetBank("basement_hatch")
	inst.AnimState:SetBuild("basement_hatch")
	inst.AnimState:PlayAnimation("idle_closed")
	inst.AnimState:SetLightOverride(1)

	inst.entity:SetParent(parent.entity)
	parent.components.placer:LinkEntity(inst)
end

--------------------------------------------------------------------------

local function ExitOnActivateByOther(inst, other, doer)
	if doer ~= nil
	and doer.sg ~= nil and not doer:HasTag("playerghost") then
		doer.sg.statemem.teleportarrivestate = "jumpout_ceiling"
	end
end

local function ReceiveItem(teleporter, item)
	if item.Transform ~= nil then
		local x, y, z = teleporter.inst.Transform:GetWorldPosition()
		local angle = math.random() * 2 * PI
		
		if item.Physics ~= nil then
			item.Physics:Stop()
			if teleporter.inst:IsAsleep() then				
				local radius = teleporter.inst:GetPhysicsRadius(0) + math.random()
				item.Physics:Teleport(x + math.cos(angle) * radius, 0, z - math.sin(angle) * radius)
			else
				TemporarilyRemovePhysics(item, 1)
				local speed = 2 + math.random() * .5 + teleporter.inst:GetPhysicsRadius(0)
				item.Physics:Teleport(x, 4, z)
				item.Physics:SetVel(speed * math.cos(angle), -1, speed * math.sin(angle))
			end
		else
			local radius = 2 + math.random()
			item.Transform:SetPosition(x + math.cos(angle) * radius, 0,	z - math.sin(angle) * radius)
		end
	end
end

local function TakeLightSteps(light, value)
	local function LightToggle(light)
		light.level = (light.level or 0) + value
		if (value > 0 and light.level <= 1) or (value < 0 and light.level > 0) then
			light.Light:SetRadius(light.level)
			light.lighttoggle = light:DoTaskInTime(2 * FRAMES, LightToggle)
		elseif value < 0 then
			light.Light:Enable(false)
			light:Hide()
		end
		light.AnimState:SetScale(light.level, 1)
	end
	if light.lighttoggle ~= nil then
		light.lighttoggle:Cancel()
	end
	light.lighttoggle = light:DoTaskInTime(2 * FRAMES, LightToggle)
end

local function OnNearExit(inst, chain)
	inst.hatchlight.Light:Enable(true)
	inst.hatchlight:Show()
	
	TakeLightSteps(inst.hatchlight, 0.2)
	
	if not chain then
		ChainPlayerprox(inst, true)
	end
end

local function OnFarExit(inst, chain)	
	TakeLightSteps(inst.hatchlight, -0.2)
	
	if not chain then
		ChainPlayerprox(inst, false)
	end
end

local function SpawnBasement(inst)
	local x, y, z = inst.Transform:GetWorldPosition()
	if not IsNearBasement(x, y, z) then
		SpawnPrefab("basement").Transform:SetPosition(x + 10, 0, z + 10)
	end
end

local function exit()
	local inst = CreateEntity()
	
	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddNetwork()
	
	inst.Transform:SetTwoFaced()
	inst.Transform:SetRotation(-45)
	
	MakeObstaclePhysics(inst, 1.5)
	
	inst.AnimState:SetBank("basement_exit")
	inst.AnimState:SetBuild("basement_exit")
	inst.AnimState:PlayAnimation("idle")
	inst.AnimState:OverrideShade(BASEMENT_SHADE)
	
	inst:AddTag("stairs")
	inst:AddTag("basement_part")
	
	inst:SetDeployExtraSpacing(2.5)
		
	inst.entity:SetPristine()
			
	if not TheWorld.ismastersim then
		return inst
	end
	
	inst.hatchlight = inst:SpawnChild("basement_exit_light")
	inst.hatchlight.Light:Enable(false)
	inst.hatchlight:Hide()
	
	inst:AddComponent("inspectable")
	
	inst:AddComponent("teleporter")
	--inst.components.teleporter.onActivate = OnStartTeleporting
	inst.components.teleporter.onActivateByOther = ExitOnActivateByOther
	inst.components.teleporter.offset = 0
	inst.components.teleporter.travelcameratime = 0.2
	inst.components.teleporter.travelarrivetime = 1.2
	inst.components.teleporter.ReceiveItem = ReceiveItem
	
	inst:AddComponent("playerprox")
	inst.components.playerprox:SetDist(2, 4)
	inst.components.playerprox.onnear = OnNearExit
	inst.components.playerprox.onfar = OnFarExit
	
	inst:DoTaskInTime(0, SpawnBasement)
					
	return inst
end

local function exit_light()
	local inst = CreateEntity()

	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddLight()
	inst.entity:AddNetwork()
	
	inst.Light:SetRadius(1)
	inst.Light:SetIntensity(0.85)
	inst.Light:SetFalloff(0.3)
	inst.Light:SetColour(0.7, 0.75, 0.67)

	inst.AnimState:SetBank("cavelight")
	inst.AnimState:SetBuild("cave_exit_lightsource")
	inst.AnimState:PlayAnimation("idle_loop", true)
	inst.AnimState:OverrideMultColour(0.35, 0.38, 0.33, 0)
	inst.AnimState:SetLightOverride(1)
	
	inst.Transform:SetScale(1.5, 0.4, 1)

	inst:AddTag("NOCLICK")
	inst:AddTag("FX")
	inst:AddTag("daylight")

	inst.entity:SetPristine()

	if not TheWorld.ismastersim then
		return inst
	end
	
	inst.persists = false

	return inst
end

--------------------------------------------------------------------------

local columns =
{
	middle =
	{
		{
			1, 0, 0, 0, 1,
			0, 0, 1, 0, 0,
			0, 1, 0, 1, 0,
			0, 0, 1, 0, 0,
			1, 0, 0, 0, 1,
		},
		
		{
			1, 1, 0, 1, 1,
			1, 0, 0, 0, 1,
			0, 0, 0, 0, 0,
			1, 0, 0, 0, 1,
			1, 1, 0, 1, 1,
		},
		
		{
			1, 1, 1, 1, 0,
			1, 0, 0, 0, 0,
			0, 0, 0, 0, 0,
			0, 0, 0, 0, 1,
			0, 1, 1, 1, 1,
		},
		
		{
			0, 1, 0, 0, 0,
			0, 0, 1, 0, 1,
			0, 1, 0, 1, 0,
			1, 0, 1, 0, 0,
			0, 0, 0, 1, 0,
		},
	},
	
	side =
	{
		{
			0, 1, 0,
			0, 1, 1,
			1, 0, 0,
		},
		
		{
			0, 1, 1,
			0, 0, 1,
			1, 0, 0,
		},
		
		{
			0, 0, 0,
			0, 1, 0,
			0, 0, 0,
		},
		
		{
			1, 0, 0,
			0, 0, 0,
			0, 0, 1,
		},
		
		{
			0, 0, 1,
			0, 0, 0,
			1, 0, 0,
		},
		
		{
			0, 1, 0,
			1, 0, 1,
			0, 1, 0,
		},
	},
}

local function SpawnColumns(x, y, z)
	local POS = {}
		
	local column_middle = columns.middle[math.random(#columns.middle)]
	local count = 0
	for x = -2, 2 do
		for z = -2, 2 do
			count = count + 1
			if column_middle[count] == 1 then
				table.insert(POS, { x = x, z = z })
			end
		end
	end
		
	local column_side = columns.side[math.random(#columns.side)]
	local count = 0
	for x = -1, 1 do
		for z = -1, 1 do
			count = count + 1
			if column_side[count] == 1 then
				table.insert(POS, { x = x - 7, z = z + 7 })
				table.insert(POS, { x = -x + 7, z = -z - 7 })
			end
		end
	end
	
	for _,v in pairs(POS) do
		local part = SpawnPrefab("wall_stone")
		if part ~= nil then
			part.Transform:SetPosition(x + v.x, 0, z + v.z)
			if part.components.health ~= nil then
				part.components.health:SetPercent(1)
			end
		end
	end
end

local function SpawnWalls(inst)	
	local x, y, z = inst.Transform:GetWorldPosition()
	x, z = math.floor(x) + 0.5, math.floor(z) + 0.5 --matching with normal walls
	inst.Transform:SetPosition(x, 0, z)
	
	local POS = {}
	
	for x = -14, 14 do
		for z = -14, 14 do
			if x == 14 or x == -14 or z == 14 or z == -14 then
				table.insert(POS, { x = x, z = z })
			end
		end
	end
			
	local wood = { [4] = true, [26] = true, [34] = true, [35] = true, [78] = true, [79] = true, [87] = true, [109] = true }
	
	local count = 0
	for _,v in pairs(POS) do
		count = count + 1
		local part = SpawnPrefab("wall_basement_"..(wood[count] and 2 or 1))
		Waffles.AddChild(inst, part)
		part.Transform:SetPosition(x + v.x, 0, z + v.z)
	end
		
	if not inst.nocolumns then
		SpawnColumns(x, y, z)
	end
end

local _pusheventfn = nil
local blocked_events = { phasechanged = true, moonphasechanged2 = true, screenflash = true }
local function SetLightingUpdatesDisabled(enable)
	if enable then
		if _pusheventfn == nil then
			_pusheventfn = TheWorld.PushEvent
		end
		TheWorld.PushEvent = function(self, event, data)
			if not blocked_events[event] then
				_pusheventfn(self, event, data)
			end
		end
	elseif _pusheventfn ~= nil then
		TheWorld.PushEvent = _pusheventfn
	end
end

local function ChangeLighting(phase, moon, freeze)
	if freeze then
		TheWorld:PushEvent("phasechanged", phase)
		TheWorld:PushEvent("moonphasechanged2", moon)
		SetLightingUpdatesDisabled(freeze)
	else
		SetLightingUpdatesDisabled(freeze)
		TheWorld:PushEvent("phasechanged", phase)
		TheWorld:PushEvent("moonphasechanged2", moon)
	end
	TheWorld.components.ambientlighting:LongUpdate()
end

local DSP_BASEMENT =
{
	lowdsp =
	{
		["set_ambience"] = 750,
		["set_sfx/set_ambience"] = 750,
	},
	highdsp =
	{
		["set_music"] = 750,
		["set_sfx/movement"] = 750,
		["set_sfx/creature"] = 750,
		["set_sfx/player"] = 750,
		["set_sfx/voice"] = 600,
		["set_sfx/sfx"] = 750,
	},
	duration = 0.5,
}

local terrors = { "attack", "attack_grunt", "die", "hit_response", "idle", "taunt", "appear", "dissappear" }

local function PlayTerrorSound(proxy)
	local inst = CreateEntity()

	--[[Non-networked entity]]

	inst.entity:AddTransform()
	inst.entity:AddSoundEmitter()
	inst.entity:SetParent(proxy.entity)

	local theta = math.random() * 2 * PI
	inst.Transform:SetPosition(5 * math.cos(theta), 0, 5 * math.sin(theta))
	inst.SoundEmitter:PlaySound(string.format("dontstarve/sanity/creature%s/%s", math.random(2), terrors[math.random(#terrors)]), nil, math.random())

	inst:Remove()
	
	proxy._terroramb = proxy:DoTaskInTime(math.random(5, 40), PlayTerrorSound)
end

local function SetBasementAmbient(enable)
	TheWorld.components.ambientsound:SetReverbPreset((enable or TheWorld:HasTag("cave")) and "cave" or "default")
		
	if Waffles.Valid(ThePlayer) then
		ThePlayer:DoTaskInTime(0.1, function()
			ThePlayer:PushEvent(enable and "pushdsp" or "popdsp", DSP_BASEMENT)
		end)
	end
	
	if TheFocalPoint._terroramb ~= nil then
		TheFocalPoint._terroramb:Cancel()
		TheFocalPoint._terroramb = nil
	end
	if enable then
		TheFocalPoint.SoundEmitter:PlaySound("dontstarve/cave/caveAMB", "basementAMB")
		TheFocalPoint._terroramb = TheFocalPoint:DoTaskInTime(math.random(10, 40), PlayTerrorSound)
	else
		TheFocalPoint.SoundEmitter:KillSound("basementAMB")
	end
end

local function PushTempGroundClient(inst)
	inst:DoPeriodicTask(0, function()
		local x, y, z = inst.Transform:GetWorldPosition()
		local player = ThePlayer
		if player ~= nil
		and	player.components.locomotor ~= nil
		and	not player:HasTag("playerghost")
		and	player:GetDistanceSqToPoint(x, 0, z) < 20 * 20 then
			player.components.locomotor:PushTempGroundSpeedMultiplier(1, GROUND.CHECKER)
		end
	end)
end

local HideableChildren = { rain = true, snow = true, pollen = true }
local ShelteredEnts = {}

local function HideBasementEntities(inst)
	local x, y, z = inst.Transform:GetWorldPosition()
	for _,v in pairs(TheSim:FindEntities(x, y, z, 20, nil, { "basement_part", "INLIMBO" })) do
		if not ShelteredEnts[v] then
			local insert = false
			if HideableChildren[v.prefab] then
				ShelteredEnts[v] = v.entity:GetParent()
				v.entity:SetParent(nil)
			else
				if v.AnimState ~= nil and not v:HasTag("FX") then
					v.AnimState:OverrideShade(BASEMENT_SHADE)
					insert = true
				end
				if v.MiniMapEntity ~= nil then
					v.MiniMapEntity:SetEnabled(false)
					insert = true
				end
			end
			if insert then
				ShelteredEnts[v] = true
			end
		end
	end
	if TheWorld.state.iswet then
		TheWorld.state.iswet = false
	end
end

local phases = { to = { "day", "dusk", "night" } }; phases.from = table.invert(phases.to)
local moonphases = { to = {	"new", "quarter", "half", "threequarter", "full" } }; moonphases.from = table.invert(moonphases.to)

local function HideClockMoon(enable)
    if not Waffles.Valid(ThePlayer) then
		return
	end
	
	local clockmoon = Waffles.ReturnChild(ThePlayer, "HUD/controls/clock/_moonanim")
	if clockmoon ~= nil then
		if enable then
			clockmoon:Hide()
		else
			ThePlayer:DoTaskInTime(0.5, function() clockmoon:Show() end)
		end
	end
end

local function OnEntityWakeClient(inst)	
	ChangeLighting("night", { moonphase = "new", waxing = true }, true)
		
	SetBasementAmbient(true)
						
	HideClockMoon(true)
	
	PushTempGroundClient(inst)
	
	inst:DoPeriodicTask(FRAMES, HideBasementEntities)
			
	local function CheckEntity()
		if inst:IsValid() then
			TheWorld:DoTaskInTime(1, CheckEntity)
		else
			Waffles.ReplicateDummyFn(getmetatable(TheFocalPoint.SoundEmitter).__index, "PlaySound",
			ChangeLighting,
				phases.to[inst.state.phase:value()],
				{ moonphase = moonphases.to[inst.state.moonphase:value()], waxing = inst.state.iswaxingmoon:value() },
				false
			)
			
			SetBasementAmbient(false)
						
			HideClockMoon(false)
			
			TheWorld.state.iswet = inst.state.iswet:value()
						
			for k,v in pairs(ShelteredEnts) do
				if k:IsValid() then
					if k.AnimState ~= nil then
						k.AnimState:OverrideShade(1)
					end
					if not k:HasTag("INLIMBO") then
						if k.MiniMapEntity ~= nil then
							k.MiniMapEntity:SetEnabled(true)
						end
						if HideableChildren[k.prefab] then
							k.entity:SetParent(v.entity)
						end
					end
				end
			end
			ShelteredEnts = {}
		end
	end
	
	TheWorld:DoTaskInTime(0.1, CheckEntity)
end

local function OnNewPhase(inst, phase)
	inst.state.phase:set(phases.from[phase])
end

local function OnNewMoonPhase(inst, data)
	inst.state.moonphase:set(moonphases.from[data.moonphase])
	isnt.state.iswaxingmoon:set(data.waxing)
end

local function OnWet(inst, iswet)
	inst.state.iswet:set(iswet)
end

local function ForceUpdateState(inst)
	inst.state.phase:set(phases.from[TheWorld.state.phase])
	inst.state.moonphase:set(moonphases.from[TheWorld.state.moonphase])
	inst.state.iswaxingmoon:set(TheWorld.state.iswaxingmoon)
	inst.state.iswet:set(TheWorld.state.iswet)
end

local function AdjustSleepingBagTag(ent, phase)
	if phase == "day" then
		ent:AddTag("siestahut")
	else
		ent:RemoveTag("siestahut")
	end
end

local function AddBasementObjectBenefits(inst, ent)
	if ent.components.container ~= nil and not ent:HasTag("fridge") then
		ent:AddTag("fridge")
		ent:AddTag("nocool")
	end
	if ent.components.sleepingbag ~= nil and ent:HasTag("tent") then
		ent:WatchWorldState("phase", AdjustSleepingBagTag)
		AdjustSleepingBagTag(ent, TheWorld.state.phase)
	end
	if ent.components.inventoryitemmoisture ~= nil then
		Waffles.ReplaceFn(ent.components.inventoryitemmoisture, "GetTargetMoisture", function() return 0 end)
	end
	if ent.components.grower ~= nil then
		ent.components.grower.growrate = 0.5
	end
	if ent.components.growable ~= nil and ent.components.growable.stages ~= nil then
		for k = 3, #ent.components.growable.stages do
			ent.components.growable.stages[k] = nil
		end
	end
	if ent:HasTag("lightningrod") then
		ent:RemoveTag("lightningrod")
	end
end

local function RemoveBasementObjectBenefits(inst, ent)
	if not Waffles.Valid(ent) then
		return
	end
	
	if ent.components.container ~= nil and ent:HasTag("nocool") then
		ent:RemoveTag("fridge")
		ent:RemoveTag("nocool")
	end
	if ent.components.sleepingbag ~= nil then
		ent:StopWatchingWorldState("phase", AdjustSleepingBagTag)
		AdjustSleepingBagTag(ent, ent.prefab == "siestahut" and "day" or "night")
	end
	if ent.components.inventoryitemmoisture ~= nil then
		Waffles.ReplaceFn(ent.components.inventoryitemmoisture, "GetTargetMoisture")
	end
	--print("BASEMENT ENTITY LEFT:", ent)
end

local function TrackBasemenObjects(inst)
	if inst.allobjects == nil then
		inst.allobjects = {}
	end
	
	local x, y, z = inst.Transform:GetWorldPosition()
	local objects = table.invert(TheSim:FindEntities(x, y, z, 20, nil, { "basement_part", "player", "FX", "DECOR", "INLIMBO" }, { "structure", "_container" }))
	for ent in pairs(objects) do
		if not inst.allobjects[ent] then
			AddBasementObjectBenefits(inst, ent)
			inst.allobjects[ent] = true
		end
	end
	for ent in pairs(inst.allobjects) do
		if not objects[ent] then
			RemoveBasementObjectBenefits(inst, ent)
			inst.allobjects[ent] = nil
		end
	end
end

local blocked_recipes = { pighouse = true, rabbithouse = true, beebox = true, telebase = true }

local function SetLightWatcherAdditiveThresh(ent, thresh)
	if ThePlayer == ent then
		return
	end
	thresh = thresh or 0
	ent.LightWatcher:SetLightThresh(0.075 + thresh)
	ent.LightWatcher:SetDarkThresh(0.05 + thresh)
end

local function UpdateLightWatchers(inst)
	if inst.allplayers ~= nil then
		local thresh = TheSim:GetLightAtPoint(10000, 10000, 10000)
		for ent in pairs(inst.allplayers) do
			if ent.LightWatcher ~= nil then
				SetLightWatcherAdditiveThresh(ent, thresh)
			end
		end
	end
end

local function OnPlayerBuildStructure(ent)
	if Waffles.Valid(ent.basement) then
		TrackBasemenObjects(ent.basement)
	end
end

local function AddBasementPlayerBenefits(inst, ent)
	ForceUpdateState(inst)
	
	ent.basement = inst
	
	if ent.components.sanity ~= nil then
		ent.components.sanity.externalmodifiers:SetModifier(inst, TheConfiguration.sanity / 60)
	end
	if ent.components.temperature ~= nil then
		ent.components.temperature:SetModifier("basement", TheConfiguration.temperature)
	end
	if ent.components.moisture ~= nil then
		Waffles.ReplaceFn(ent.components.moisture, "GetMoistureRate", function() return 0 end)
	end
	if ent.components.playerlightningtarget ~= nil then
		Waffles.ReplaceFn(ent.components.playerlightningtarget, "GetHitChance", function() return 0 end)
	end
	if ent.components.builder ~= nil then
		Waffles.ReplaceFn(ent.components.builder, "DoBuild", function(self, recname, ...)
			if recname ~= nil then
				if blocked_recipes[recname] then
					return false, "LOWCEILING"
				else
					return self:__DoBuild(recname, ...)
				end
			end
		end)
	end
	if ent.components.beaverness ~= nil then		
		Waffles.ReplaceFn(ent.components.beaverness, "SetPercent", function() end)
	end
	
	ent:ListenForEvent("buildstructure", OnPlayerBuildStructure)
end

local function RemoveBasementPlayerBenefits(inst, ent)
	if not Waffles.Valid(ent) then
		return
	end
	
	ForceUpdateState(inst)
	
	ent.basement = nil
	
	if ent.components.sanity ~= nil then
		ent.components.sanity.externalmodifiers:RemoveModifier(inst)
	end
	if ent.components.temperature ~= nil then
		ent.components.temperature:RemoveModifier("basement")
	end
	if ent.components.moisture ~= nil then
		Waffles.ReplaceFn(ent.components.moisture, "GetMoistureRate")
	end
	if ent.components.playerlightningtarget ~= nil then
		Waffles.ReplaceFn(ent.components.playerlightningtarget, "GetHitChance")
	end
	if ent.components.builder ~= nil then
		Waffles.ReplaceFn(ent.components.builder, "DoBuild")
	end
	if ent.components.beaverness ~= nil then
		Waffles.ReplaceFn(ent.components.beaverness, "SetPercent")
		
		if TheWorld.state.moonphase == "full" then
			local fn = Waffles.ReturnChild(ent, "worldstatewatching/isfullmoon/1")
			if fn ~= nil then
				fn(ent, true)
			end
		end
	end
	if ent.LightWatcher ~= nil then
		SetLightWatcherAdditiveThresh(ent)
	end
	
	ent:RemoveEventCallback("buildstructure", OnPlayerBuildStructure)	
	--print("BASEMENT ENTITY LEFT:", ent)
end

local function TrackBasementPlayers(inst)
	if inst.allplayers == nil then
		inst.allplayers = {}
	end
	
	local x, y, z = inst.Transform:GetWorldPosition()
	local players = table.invert(TheSim:FindEntities(x, y, z, 20, { "player" }, { "playerghost", "INLIMBO" }))
	for ent in pairs(players) do
		if inst.allplayers[ent] then
			if ent.components.locomotor ~= nil then
				ent.components.locomotor:PushTempGroundSpeedMultiplier(1, GROUND.CHECKER)
			end
		else
			AddBasementPlayerBenefits(inst, ent)
			inst.allplayers[ent] = true
		end
	end
	for ent in pairs(inst.allplayers) do
		if not players[ent] then
			RemoveBasementPlayerBenefits(inst, ent)
			inst.allplayers[ent] = nil
		end
	end
end

local function OnScreenFlash(inst)
	if inst.tracker ~= nil then
		if inst.tracker.lighting ~= nil then
			inst.tracker.lighting:Cancel()
		end
		inst.tracker.lighting = inst:DoPeriodicTask(2, UpdateLightWatchers, 0)
	end
end

local function OnEntityWake(inst)
	inst.tracker =
	{
		lighting =	inst:DoPeriodicTask(2, UpdateLightWatchers, 1),
		players =	inst:DoPeriodicTask(0.1, TrackBasementPlayers),
		objects =	inst:DoPeriodicTask(20, TrackBasemenObjects),
	}
	
	inst.OnScreenFlash = function() OnScreenFlash(inst) end
	inst:ListenForEvent("screenflash", inst.OnScreenFlash, TheWorld)
	
	inst:WatchWorldState("phase", OnNewPhase)
	inst:WatchWorldState("moonphasechanged2", OnNewMoonPhase)
	inst:WatchWorldState("iswet", OnWet)
end

local function OnEntitySleep(inst)
	if inst.tracker ~= nil then
		for name, task in pairs(inst.tracker) do
			task:Cancel()
		end
		inst.tracker = nil
	end
		
	if inst.allplayers ~= nil then
		for ent in pairs(inst.allplayers) do
			RemoveBasementPlayerBenefits(inst, ent)
		end
		inst.allplayers = nil
	end
	TrackBasemenObjects(inst)
		
	if inst.OnScreenFlash ~= nil then
		inst:RemoveEventCallback("screenflash", inst.OnScreenFlash, TheWorld)
		inst.OnScreenFlash = nil
	end
	
	inst:StopAllWatchingWorldStates()
end

local function base()
	local inst = CreateEntity()
	
	inst.entity:AddTransform()
	inst.entity:AddAnimState()
	inst.entity:AddNetwork()
		
	inst.AnimState:SetBank("basement_floor")
	inst.AnimState:SetBuild("basement_floor")
	inst.AnimState:SetOrientation(ANIM_ORIENTATION.OnGround)
	inst.AnimState:SetLayer(LAYER_BACKGROUND)
	inst.AnimState:SetSortOrder(5)
	inst.AnimState:OverrideShade(BASEMENT_SHADE)
	inst.AnimState:SetScale(8, 8)
	inst.AnimState:PlayAnimation("rocky")
	
	--inst.Transform:SetScale(2.82, 2.82, 2.82)
		
	inst:AddTag("NOCLICK")
	inst:AddTag("alt_tile")
	inst:AddTag("basement_part")
	
	inst.state =
	{
		phase = net_tinybyte(inst.GUID, "basement.phase"),
		moonphase = net_tinybyte(inst.GUID, "basement.moonphase"),
		iswaxingmoon = net_bool(inst.GUID, "basement.iswaxingmoon"),
		iswet = net_bool(inst.GUID, "basement.iswet"),
	}
	ForceUpdateState(inst)
		
	if not TheNet:IsDedicated() then
		local mist = SpawnPrefab("mist")
		mist.entity:SetParent(inst.entity)
		mist.components.emitter.area_emitter = function()
			return math.random(-14, 14), math.random(-14, 14)
		end

		mist.entity:SetAABB(14, 2)
		mist.components.emitter.density_factor = 1
		mist.components.emitter:Emit()
	end
	
	inst.entity:SetPristine()
		
	if not TheWorld.ismastersim then --sadly can't apply the basement environment for client host, since they share a single sim with server
		inst.OnEntityWake = OnEntityWakeClient
				
		return inst
	end
		
	Waffles.AddChild(inst, SpawnPrefab("basement_ceiling"))
			
	inst:DoTaskInTime(0, SpawnWalls)
		
	inst.OnEntityWake = OnEntityWake
	inst.OnEntitySleep = OnEntitySleep
	inst.OnLoad = function() inst.nocolumns = true end
			
	return inst
end

--------------------------------------------------------------------------

local function OnCollide(inst, collider)
	if collider.Transform ~= nil then
		local cy, cx, cz = collider.Transform:GetWorldPosition() --collider
		local by, bx, bz = inst.parent.Transform:GetWorldPosition() --basement
		local exit = FindEntity(inst.parent, 20, function(ent) return ent.prefab == "basement_exit" end)
		if exit == nil then
			return
		end
		local ey, ex, ez = exit.components.teleporter.targetTeleporter.Transform:GetWorldPosition() --basement entrance
		local x, y, z = by - cy + ey, cx, bz - cz + ez
		
		if cx < 10 then
			if collider:HasTag("bird") then
				if collider.components.lootdropper ~= nil then
					collider.components.lootdropper:SetLoot({})
				end
				if collider.components.combat ~= nil then
					collider.components.combat:GetAttacked(inst, 5)
				end
				collider:PushEvent("gotosleep")
			end
		else
			collider.Physics:Teleport(x, y, z) --teleport to forest
		end
	end
end

local function ceiling()
	local inst = CreateEntity()
	
	inst.entity:AddTransform()
	
	local phys = inst.entity:AddPhysics()
	phys:SetMass(0)
	phys:SetCollisionGroup(COLLISION.WORLD)
	phys:ClearCollisionMask()
	phys:CollidesWith(COLLISION.ITEMS)
	phys:CollidesWith(COLLISION.CHARACTERS)
	phys:CollidesWith(COLLISION.GIANTS)
	phys:CollidesWith(COLLISION.FLYERS)
	phys:SetCapsule(70, 70)
	phys:SetCollisionCallback(OnCollide)
	
	inst:DoTaskInTime(0, function()
		local x, y, z = (inst.parent or inst).Transform:GetWorldPosition()
		inst.Transform:SetPosition(x, 8, z)
	end)
	
	--hacks to trick the weather component
	inst:AddTag("lightningrod")
	inst.GetPosition = function()
		return Point(0, 0, 0)
	end
	
	inst.persists = false
	
	return inst
end

return Prefab("wall_basement_1", wall1),
	Prefab("wall_basement_2", wall2),
	Prefab("basement_entrance", entrance),
	Prefab("basement_entrance_hatch", entrance_hatch, assets.hatch),
	Prefab("basement_entrance_builder", builder),
	MakePlacer("basement_entrance_placer", "basement_entrance", "basement_hatch", "idle", nil, nil, nil, nil, nil, nil, placerdecor),
	Prefab("basement_exit", exit, assets.stairs),
	--Prefab("basement_elevator_cage", lift, assets.elevator),
	--Prefab("basement_elevator_shield", lift_shield, assets.elevator),
	Prefab("basement_exit_light", exit_light),
	Prefab("basement", base, assets.tile),
	Prefab("basement_ceiling", ceiling)