<?php

namespace AppBundle\Component\Export\Course;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Component\Export\Exporter;
use Biz\Course\Service\CourseService;
use Biz\Course\Service\MemberService;
use Biz\Task\Service\TaskResultService;
use CorporateTrainingBundle\Common\OrgToolkit;

class OverviewStudentExporter extends Exporter
{
    public function canExport()
    {
        $user = $this->getUser();
        try {
            $hasPermission = $user->hasPermission('admin_course_manage');
        } catch (\Exception $e) {
            return false;
        }

        return $user->isAdmin() || $hasPermission;
    }

    public function getCount()
    {
        return $this->getCourseMemberService()->countMembers($this->conditions);
    }

    public function getTitles()
    {
        $titles = array(
            'task.learn_data_detail.nickname',
            'task.learn_data_detail.truename',
            'user.profile.org',
            'user_post',
            'course.name',
            'course_statistic_data.export_field.teachers',
            'course_statistic_data.export.total_time',
            'course_statistic_data.export.total_learn_time',
            'task.learn_data_detail.finished_rate',
        );
        $tasks = $this->getAllTaskByCourseId();

        $taskTitles = array();
        foreach ($tasks as $key => $task) {
            $taskTitles[] = '任务'.(++$key);
            $taskTitles[] = 'task.learn_data_detail.start_time';
            $taskTitles[] = 'task.learn_data_detail.end_time';
            $taskTitles[] = 'task.learn_data_detail.learned.seconds';
        }

        return array_merge($titles, $taskTitles);
    }

    public function getContent($start, $limit)
    {
        $course = $this->getCourseService()->getCourse($this->parameter['courseId']);

        $members = $this->getCourseMemberService()->searchMembers(
            $this->conditions,
            $this->parameter['orderBy'],
            $start,
            $limit
        );

        $userIds = ArrayToolkit::column($members, 'userId');
        $taskCount = $this->countTasksByCourseId();

        list($users, $userProfiles, $tasks, $taskResults) = $this->getReportService()->getStudentDetail($course['id'], $userIds, $taskCount);

        list($users, $userProfiles, $orgs, $posts) = $this->buildUsersData($userIds);

        $teacherNames = $this->buildTeachersNameByCourseId($course['id']);
        $videoAndAudioLength = $this->getTaskService()->sumAudioAndVideoLengthByCourseId($course['id']);

        $videoAndAudioLength = round($videoAndAudioLength / 60, 1);

        $datas = array();

        foreach ($members as $member) {
            $userTaskResults = !empty($taskResults[$member['userId']]) ? $taskResults[$member['userId']] : array();

            $user = $users[$member['userId']];
            $userProfile = $userProfiles[$member['userId']];
            $data = array();
            $data[] = $user['nickname'];
            $data[] = $userProfile['truename'];
            $data[] = OrgToolkit::buildOrgsNames($user['orgIds'], $orgs);
            $data[] = empty($posts[$user['postId']]) ? '-' : $posts[$user['postId']]['name'];
            $data[] = $course['title'];
            $data[] = $teacherNames;
            $data[] = $videoAndAudioLength;

            $learnedTime = $this->getTaskResultService()->sumLearnTimeByCourseIdAndUserId($course['id'], $user['id']);
            $data[] = round($learnedTime / 60, 1);

            $learnProccess = (empty($member['learnedCompulsoryTaskNum']) || empty($course['compulsoryTaskNum'])) ? 0 : (int) ($member['learnedCompulsoryTaskNum'] * 100 / $course['compulsoryTaskNum']);
            $data[] = $learnProccess > 100 ? '100%' : $learnProccess.'%';

            foreach ($tasks as $task) {
                $data[] = $task['title'];
                $taskResult = !empty($userTaskResults[$task['id']]) ? $userTaskResults[$task['id']] : array();
                $data[] = empty($taskResult) ? '--' : date('Y-m-d H:i:s', $taskResult['createdTime']);
                $data[] = (empty($taskResult) or empty($taskResult['finishedTime'])) ? '--' : date('Y-m-d H:i:s', $taskResult['finishedTime']);
                $data[] = empty($taskResult) ? '--' : round($taskResult['time'] / 60, 1);
            }

            $datas[] = $data;
        }

        return $datas;
    }

    protected function buildTeachersNameByCourseId($courseId)
    {
        $teachers = $this->getCourseMemberService()->findCourseTeachers($courseId);
        $teacherIds = ArrayToolkit::column($teachers, 'userId');

        $teacherProfiles = $this->getUserService()->findUserProfilesByIds($teacherIds);
        $teachers = $this->getUserService()->findUsersByIds($teacherIds);

        $names = array();
        foreach ($teachers as $teacher) {
            $names[] = empty($teacherProfiles[$teacher['id']]) ? '('.$teacher['nickname'].')' : $teacherProfiles[$teacher['id']]['truename'].'('.$teacher['nickname'].')';
        }

        return implode(',', $names);
    }

    private function getAllTaskByCourseId()
    {
        return $this->getTaskService()->searchTasks(
            array(
                'courseId' => $this->parameter['courseId'],
                'isOptional' => 0,
                'status' => 'published',
            ),
            array('seq' => 'ASC'),
            0,
            PHP_INT_MAX
        );
    }

    private function countTasksByCourseId()
    {
        return $this->getTaskService()->countTasks(
            array(
                'courseId' => $this->parameter['courseId'],
                'isOptional' => 0,
                'status' => 'published',
            )
        );
    }

    public function buildParameter($conditions)
    {
        $parameter = parent::buildParameter($conditions);
        $parameter['courseId'] = $conditions['courseId'];
        $parameter['orderBy'] = $this->getReportService()->buildStudentDetailOrderBy($conditions);

        return $parameter;
    }

    public function buildCondition($conditions)
    {
        return $this->getReportService()->buildStudentDetailConditions($conditions, $conditions['courseId']);
    }

    protected function getReportService()
    {
        return $this->getBiz()->service('Course:ReportService');
    }

    /**
     * @return TaskService
     */
    protected function getTaskService()
    {
        return $this->getBiz()->service('Task:TaskService');
    }

    /**
     * @return CourseService
     */
    protected function getCourseService()
    {
        return $this->getBiz()->service('Course:CourseService');
    }

    /**
     * @return MemberService
     */
    protected function getCourseMemberService()
    {
        return $this->getBiz()->service('Course:MemberService');
    }

    /**
     * @return OrgService
     */
    protected function getOrgService()
    {
        return $this->getBiz()->service('Org:OrgService');
    }

    /**
     * @return TaskResultService
     */
    protected function getTaskResultService()
    {
        return $this->getBiz()->service('Task:TaskResultService');
    }
}
