<?php

namespace CorporateTrainingBundle\Biz\AdvancedMemberSelect;

use AppBundle\Common\Exception\AccessDeniedException;
use Biz\System\Service\SettingService;
use Codeages\Biz\Framework\Context\Biz;
use Codeages\Biz\Framework\Event\Event;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

abstract class AbstractMemberSelect
{
    protected $biz;

    protected $advancedMemberSelectType = '';

    protected $enrollType = '';

    protected $mailNotification = [];

    protected $dingtalkNotification = [];

    abstract public function canSelect($targetId);

    abstract public function becomeMember($targetId, $userIds);

    /**
     * 短信默认发，钉钉走配置
     *
     * @param $userIds
     * @param $targetId
     * @param $dingTalkNotificationSetting
     *
     * @return mixed
     */
    abstract protected function sendNotification($userIds, $targetId, $dingTalkNotificationSetting);

    abstract protected function sendEnrollNotification($userIds, $targetId);

    /**
     * @param $targetId
     * @param $userIds
     *
     * @return mixed
     *
     * 过滤已经存在的用户
     */
    abstract protected function filterMembers($targetId, $userIds);

    public function __construct(Biz $biz)
    {
        $this->biz = $biz;
        $this->mailNotification = $this->getSettingService()->get('mail_notification', []);
        $this->dingtalkNotification = $this->getSettingService()->get('dingtalk_notification', []);
    }

    /**
     * 管理员直接推送
     *
     * @param $targetId
     * @param $userIds
     * @param int $dingTalkNotificationSetting
     *
     * @return bool
     */
    public function selectUserIds($targetId, $userIds, $dingTalkNotificationSetting = 0)
    {
        if (!$this->canSelect($targetId)) {
            throw new AccessDeniedException('Access Denied');
        }
        if (!is_array($userIds)) {
            $userIds = explode(',', $userIds);
        }

        $userIds = $this->filterMembers($targetId, $userIds);
        $userIds = array_values($userIds);
        if (empty($targetId) || empty($userIds)) {
            return false;
        }

        $this->becomeMember($targetId, $userIds);
        if (!empty($this->enrollType)) {
            $this->getEnrollStrategyContext()->createStrategy($this->enrollType)->createEnrollRecordWithAdminAddUser($targetId, $userIds);
        }

        $this->sendNotification($userIds, $targetId, $dingTalkNotificationSetting);

        return true;
    }

    /**
     * 报名成为学员
     *
     * @param int   $targetId
     * @param array $userIds
     * @param int   $notificationSetting
     *
     * @return bool
     */
    public function enrollBecomeMember($targetId, $userIds, $notificationSetting = 0)
    {
        if (empty($targetId) || empty($userIds)) {
            return false;
        }
        $this->becomeMember($targetId, $userIds);

        if ($notificationSetting) {
            $this->sendEnrollNotification($userIds, $targetId);
        }

        return true;
    }

    /**
     * @return EventDispatcherInterface
     */
    private function getDispatcher()
    {
        return $this->biz['dispatcher'];
    }

    /**
     * @param string      $eventName
     * @param Event|mixed $subject
     *
     * @return Event
     */
    protected function dispatchEvent($eventName, $subject, $arguments = [])
    {
        if ($subject instanceof Event) {
            $event = $subject;
        } else {
            $event = new Event($subject, $arguments);
        }

        return $this->getDispatcher()->dispatch($eventName, $event);
    }

    protected function createService($alias)
    {
        return $this->biz->service($alias);
    }

    /**
     * @return SettingService
     */
    protected function getSettingService()
    {
        return $this->biz->service('System:SettingService');
    }

    protected function getEnrollStrategyContext()
    {
        return $this->biz['enroll_strategy_context'];
    }
}
