<?php

namespace CorporateTrainingBundle\Biz\DingTalk\Job;

use Biz\System\Service\LogService;
use Biz\System\Service\SettingService;
use Biz\User\Service\UserService;
use Codeages\Biz\Framework\Scheduler\AbstractJob;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkSyncUserService;
use CorporateTrainingBundle\Biz\Org\Service\OrgService;
use CorporateTrainingBundle\Component\EIMClient\UserFactory;

class SyncDingTalkUserJob extends AbstractJob
{
    const LIMIT = 100;
    protected $maxHandleDepartmentNum = 50;
    protected $maxHandleUserNum = 1000;
    protected $offset = 0;
    protected $handleUserNum = 0;
    protected $lastRequestStatus = true;
    protected $handleDepartmentIds = [];
    protected $lastSyncId = 0;
    protected $nextDepartmentId = 0;

    protected $userClient = null;

    public function execute()
    {
        $magic = $this->getSettingService()->get('magic');
        $syncSetting = $this->getSettingService()->get('sync_department_setting', array());

        $this->maxHandleDepartmentNum = empty($magic['dingtalk_sync_max_department_num']) ? $this->maxHandleDepartmentNum : $magic['dingtalk_sync_max_department_num'];
        $this->maxHandleUserNum = empty($magic['dingtalk_sync_max_handle_user_num']) ? $this->maxHandleUserNum : $magic['dingtalk_sync_max_handle_user_num'];
        $this->lastSyncId = $this->getLastOrgSyncId();
        $this->userClient = UserFactory::create($syncSetting);

        try {
            $this->requestDingTalkUsers();
        } catch (\RuntimeException $e) {
            $this->getLogService()->error('dingtalk', 'dingtalk_sync_user', '钉钉同步用户失败:'.$e->getMessage());
        }
    }

    protected function requestDingTalkUsers()
    {
        if ($this->checkFinishSyncWithStartQuest()) {
            return  true;
        }
        $syncDingTalkUserSetting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
        $this->offset = empty($syncDingTalkUserSetting['offset']) ? 0 : $syncDingTalkUserSetting['offset'];
        $departmentId = empty($syncDingTalkUserSetting['department_id']) ? $this->getNextOrgSyncId(0) : $syncDingTalkUserSetting['department_id'];

        $users = $this->userClient->lists($departmentId, $this->offset, self::LIMIT);

        if (!in_array($departmentId, $this->handleDepartmentIds)) {
            $this->handleDepartmentIds[] = $departmentId;
        }
        if (isset($users['errcode']) && 0 != $users['errcode']) {
            return  $this->requestDingTalkUsersError($departmentId, $users['errmsg']);
        }

        return $this->requestDingTalkUsersSuccess($users, $departmentId);
    }

    /**
     * 请求失败(第一次失败后重试，第二次直接跳过)
     *
     * @param int    $departmentId
     * @param string $errmsg
     *
     * @return bool
     */
    protected function requestDingTalkUsersError($departmentId, $errmsg)
    {
        if ($this->lastRequestStatus) {
            $this->lastRequestStatus = false;
            $setting = ['department_id' => $departmentId, 'offset' => $this->offset, 'syncTime' => time(), 'syncStatus' => false];
            $log = "钉钉同步组织机构用户第一次失败(钉钉部门id-{$departmentId}):{$errmsg}";
        } else {
            if ($this->checkFinishSyncWithEndQuest($departmentId)) {
                return  true;
            }
            $setting = ['department_id' => $this->nextDepartmentId, 'offset' => 0, 'syncTime' => time(), 'syncStatus' => true];
            $log = "钉钉同步组织机构用户第二次失败(钉钉部门id-{$departmentId}):{$errmsg}, 直接跳过";
        }

        $this->getSettingService()->set('dingTalk_sync_user_setting', $setting);
        $this->getLogService()->error('dingtalk', 'dingtalk_sync_user', $log);

        return $this->requestDingTalkUsers();
    }

    /**
     * 请求成功
     *
     * @param array $users
     * @param int   $departmentId
     *
     * @return bool
     */
    protected function requestDingTalkUsersSuccess($users, $departmentId)
    {
        $this->lastRequestStatus = true;
        $setting = ['department_id' => $departmentId, 'offset' => $this->offset + self::LIMIT, 'syncTime' => time(), 'syncStatus' => true];

        //组织机构下还存在多页数据
        if (isset($users['hasMore']) && !$users['hasMore']) {
            if ($this->checkFinishSyncWithEndQuest($departmentId)) {
                return  true;
            }
            $setting = ['department_id' => $this->nextDepartmentId, 'offset' => 0, 'syncTime' => time(), 'syncStatus' => true];
        }

        $this->handleUserNum += count($users['userlist']);
        $this->getSettingService()->set('dingTalk_sync_user_setting', $setting);
        $this->handleRequestResults($users);

        return $this->requestDingTalkUsers();
    }

    /**
     * 处理请求结果
     *
     * @param $requestResult
     *
     * @return bool
     */
    protected function handleRequestResults($requestResult)
    {
        if (empty($requestResult['userlist'])) {
            return true;
        }

        $this->getDingTalkSyncUserService()->syncDingTalkUsers($requestResult['userlist']);

        unset($requestResult);

        return  true;
    }

    protected function checkFinishSyncWithStartQuest()
    {
        $syncOrgSetting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
        if (!empty($syncOrgSetting['finishTime']) && date('Y-m-d', $syncOrgSetting['finishTime']) == date('Y-m-d', time())) {
            return true;
        }

        if (empty($this->userClient)) {
            return true;
        }

        if ($this->handleUserNum >= $this->maxHandleUserNum) {
            return true;
        }

        if (count($this->handleDepartmentIds) >= $this->maxHandleDepartmentNum) {
            return true;
        }

        return false;
    }

    protected function checkFinishSyncWithEndQuest($departmentId)
    {
        if ($this->lastSyncId == $departmentId) {
            $this->getSettingService()->set('dingTalk_sync_user_setting', ['department_id' => 0, 'offset' => 0, 'syncTime' => 0, 'syncStatus' => true, 'syncStartTime' => strtotime(date('Y-m-d', time()).' 02:00:00'), 'finishTime' => time()]);
            $this->getLogService()->info('dingtalk', 'dingtalk_sync_user', date('Y-m-d H:i:s', time()).'同步用户结束');

            return true;
        }

        $this->nextDepartmentId = $this->getNextOrgSyncId($departmentId);
        if (empty($this->nextDepartmentId)) {
            return true;
        }

        return false;
    }

    protected function getNextOrgSyncId($syncId)
    {
        $org = $this->getOrgService()->searchOrgs(['syncId_GT' => $syncId], ['syncId' => 'ASC'], 0, 1);

        return empty($org) ? 0 : $org[0]['syncId'];
    }

    protected function getLastOrgSyncId()
    {
        $org = $this->getOrgService()->searchOrgs([], ['syncId' => 'DESC'], 0, 1);

        return empty($org) ? 0 : $org[0]['syncId'];
    }

    /**
     * @return OrgService
     */
    protected function getOrgService()
    {
        return $this->biz->service('Org:OrgService');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->biz->service('System:LogService');
    }

    protected function getAuthService()
    {
        return $this->biz->service('User:AuthService');
    }

    /**
     * @return SettingService
     */
    protected function getSettingService()
    {
        return $this->biz->service('System:SettingService');
    }

    /**
     * @return UserService
     */
    protected function getUserService()
    {
        return $this->biz->service('User:UserService');
    }

    /**
     * @return DingTalkSyncUserService
     */
    protected function getDingTalkSyncUserService()
    {
        return $this->biz->service('CorporateTrainingBundle:DingTalk:DingTalkSyncUserService');
    }
}
