<?php

namespace CorporateTrainingBundle\Biz\DingTalk\Service\Impl;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\SimpleValidator;
use Biz\BaseService;
use Biz\System\Service\LogService;
use Biz\System\Service\SettingService;
use Biz\User\Dao\UserBindDao;
use Biz\User\Dao\UserDao;
use Biz\User\Dao\UserProfileDao;
use Codeages\Biz\Framework\Scheduler\Service\SchedulerService;
use CorporateTrainingBundle\Biz\DingTalk\Client\DingTalkClientFactory;
use CorporateTrainingBundle\Biz\DingTalk\Dao\DingTalkUserDao;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkSyncUserService;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkUserService;
use CorporateTrainingBundle\Biz\Org\Service\OrgService;
use CorporateTrainingBundle\Biz\User\Dao\UserOrgDao;
use CorporateTrainingBundle\Biz\User\Service\AuthService;
use CorporateTrainingBundle\Biz\User\Service\UserOrgService;
use Symfony\Component\Security\Core\Encoder\MessageDigestPasswordEncoder;

class DingTalkSyncUserServiceImpl extends BaseService implements DingTalkSyncUserService
{
    protected $jobNumberToNickname = 0;

    public function syncDingTalkUsers($dingTalkUsers)
    {
        $setting = $this->getSettingService()->get('sync_department_setting', []);
        if (empty($dingTalkUsers) || empty($setting['enable'])) {
            return true;
        }

        $syncSetting = $this->getSettingService()->get('sync_department_setting', []);
        $this->jobNumberToNickname = empty($syncSetting['dingtalk_jobnumber_to_nickname']) ? 0 : 1;

        list($registerUserData, $updateUserData) = $this->preprocessDingTalkUsers($dingTalkUsers);

        $this->processResisterUsers($registerUserData);
        $this->processExistUsers($updateUserData);

        unset($dingTalkUsers);

        return true;
    }

    /**
     * 创建同步的job
     *
     * @return bool
     */
    public function createSyncDingTalkJobs()
    {
        $setting = $this->getSettingService()->get('sync_department_setting', []);
        if (empty($setting['enable'])) {
            return true;
        }
        $this->registerSyncDingTalkOrgsJob();
        $this->registerSyncDingTalkUserJob();
        $this->registerLockDingTalkUserJob();

        return true;
    }

    /**
     * 删除同步的job
     *
     * @return bool
     */
    public function deleteSyncDingTalkJobs()
    {
        $setting = $this->getSettingService()->get('sync_department_setting', []);
        if (!empty($setting['enable'])) {
            return true;
        }

        if (!empty($this->getJobDao()->getByName('SyncDingTalkUserJob'))) {
            $this->getSchedulerService()->deleteJobByName('SyncDingTalkUserJob');
        }

        if (!empty($this->getJobDao()->getByName('SyncDingTalkOrgJob'))) {
            $this->getSchedulerService()->deleteJobByName('SyncDingTalkOrgJob');
        }

        if (!empty($this->getJobDao()->getByName('LockDingTalkUserJob'))) {
            $this->getSchedulerService()->deleteJobByName('LockDingTalkUserJob');
        }

        return true;
    }

    /**
     * 封禁未同步信息的钉钉用户
     */
    public function lockDingTalkUsers()
    {
        if (!$this->checkFinishSync()) {
            return true;
        }

        $this->beginTransaction();
        try {
            $userBinds = $this->getNeedLockUserBinds();
            foreach ($userBinds as $user) {
                $lockUsers[$user['toId']] = ['locked' => 1];
            }
            if (empty($lockUsers)) {
                $setting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
                $this->getSettingService()->set('dingTalk_sync_user_setting', array_merge($setting, ['lockFinishTime' => time()]));
                $this->commit();

                return true;
            }
            $this->getUserDao()->batchUpdate(array_keys($lockUsers), $lockUsers, 'id');
            $userIds = empty($userBinds) ? [-1] : ArrayToolkit::column($userBinds, 'toId');
            $this->getLogService()->info('dingtalk', 'dingtalk_lock_user', '封禁离职用户', $userIds);
            $this->commit();
        } catch (\Exception $e) {
            $this->rollback();
            $this->getLogService()->error('dingtalk', 'dingtalk_lock_user', '封禁离职用户失败:'.$e->getMessage());
        }

        return true;
    }

    /**
     * 钉钉用户数据预处理(拆分成待创建和已绑定两大类)
     *
     * @param $dingTalkUsers
     *
     * @return array
     */
    protected function preprocessDingTalkUsers($dingTalkUsers)
    {
        $registerUserData = [];
        $updateUserData = [];
        $fromIds = ArrayToolkit::column($dingTalkUsers, 'unionid');
        $userBinds = $this->getUserService()->searchUserBinds(
            ['fromIds' => $fromIds, 'type' => 'dingtalk'],
            [],
            0,
            count($dingTalkUsers)
        );
        $userBinds = ArrayToolkit::index($userBinds, 'fromId');

        foreach ($dingTalkUsers as &$dingTalkUser) {
            $dingTalkUser = $this->filterDingTalkUser($dingTalkUser);

            if (isset($userBinds[$dingTalkUser['unionid']])) {
                $updateUserData[$dingTalkUser['unionid']] = $dingTalkUser;
                continue;
            }

            $registerUserData[$dingTalkUser['unionid']] = $dingTalkUser;
        }

        unset($dingTalkUsers);

        return [$registerUserData, $updateUserData];
    }

    /**
     * 处理待创建的数据
     *
     * @param $userData
     *
     * @return bool
     */
    protected function processResisterUsers($userData)
    {
        if (empty($userData)) {
            return true;
        }
        $userData = $this->preprocessResisterUsersNickname($userData);
        list($needRegisterUsers, $needBindUsers) = $this->hierarchicalProcessRegisterData($userData);

        $this->beginTransaction();
        try {
            $this->registerUsers($needRegisterUsers);
            $needRegisterUsers = $this->createUserProfilesAndUpdateDingTalkUsers($needRegisterUsers);

            $this->bindUsers(array_merge($needRegisterUsers, $needBindUsers));
            $this->batchSetUserData(array_merge($needRegisterUsers, $needBindUsers));

            $this->commit();
        } catch (\Exception $e) {
            $this->rollback();
            $this->getLogService()->error('dingtalk', 'dingtalk_sync_user', '钉钉同步创建用户失败:'.$e->getMessage());
        }

        return true;
    }

    /**
     * 处理已绑定的用户
     *
     * @param $userData
     *
     * @return bool
     */
    protected function processExistUsers($userData)
    {
        if (empty($userData)) {
            return true;
        }

        $userData = $this->preprocessUpdateUsersNickname($userData);

        $this->beginTransaction();
        try {
            if ($this->jobNumberToNickname) {
                list($needUpdateUsers, $updateNicknameToOldUserIds) = $this->hierarchicalProcessUpdateData($userData);
                $this->batchUpdateNicknameToOld($updateNicknameToOldUserIds);
                $userData = $needUpdateUsers;
            }
            $this->batchSetUserData($userData);
            $this->commit();
        } catch (\Exception $e) {
            $this->rollback();
            $this->getLogService()->error('dingtalk', 'dingtalk_sync_user', '钉钉同步创建用户失败:'.$e->getMessage());
        }

        return true;
    }

    /**
     * 待创建数据用户名预处理（工号作为用户名， m+手机号作为用户名，当前处理列表内重复则丢弃等待下次同步处理 ）
     *
     * @param $registerDingTalkUsers
     *
     * @return array
     */
    protected function preprocessResisterUsersNickname($registerDingTalkUsers)
    {
        $registerNicknames = [];
        foreach ($registerDingTalkUsers as &$user) {
            $user['nickname'] = $user['jobnumber'];

            if (empty($user['jobnumber']) || !$this->jobNumberToNickname || !SimpleValidator::nickname($user['jobnumber'])) {
                $user['nickname'] = 'm'.$user['mobile'];
            }

            if (in_array($user['nickname'], $registerNicknames)) {
                unset($registerDingTalkUsers[$user['unionid']]);
            }

            $registerNicknames[] = $user['nickname'];
        }

        return $registerDingTalkUsers;
    }

    /**
     * 已绑定数据用户名处理（工号作为用户名,开启才处理用户名）
     *
     * @param $updateDingTalkUsers
     *
     * @return array
     */
    protected function preprocessUpdateUsersNickname($updateDingTalkUsers)
    {
        if (empty($updateDingTalkUsers)) {
            return $updateDingTalkUsers;
        }
        $userFromIds = ArrayToolkit::column($updateDingTalkUsers, 'unionid');
        $bindUsers = $this->getUserService()->searchUserBinds(
            ['type' => 'dingtalk', 'fromIds' => $userFromIds],
            [],
            0,
            count($userFromIds)
        );
        $userIds = ArrayToolkit::column($bindUsers, 'toId');
        $bindUsers = ArrayToolkit::index($bindUsers, 'fromId');
        $users = $this->getUserService()->findUsersByIds($userIds);

        foreach ($updateDingTalkUsers as &$user) {
            $user['userId'] = $bindUsers[$user['unionid']]['toId'];
            if (empty($user['jobnumber']) || !$this->jobNumberToNickname || !SimpleValidator::nickname(
                    $user['jobnumber']
                )) {
                $user['nickname'] = $users[$user['userId']]['nickname'];
                continue;
            }
            $user['nickname'] = $user['jobnumber'];
        }

        return $updateDingTalkUsers;
    }

    /**
     * 拆分处理待创建数据（分成'待创建'，'待更新用户名为xxx_旧'， '待绑定'）
     * 1.用户名存在系统用户且已绑定钉钉，  跳过
     * 2.手机号存在系统用户且已经绑定钉钉， 跳过
     * 3.用户名存在系统用户且未绑定钉钉，绑定
     * 4.手机号存在系统用户且未绑定钉钉且在当前批次没有与已存在的用户相同的用户名， 绑定
     * 5. 手机号存在系统用户且未绑定钉钉且在当前批次有与已存在的用户相同的用户名， 跳过
     * 6. 新建
     *
     * @param $registerUsers
     *
     * @return array
     */
    protected function hierarchicalProcessRegisterData($registerUsers)
    {
        $needRegisterUsers = [];
        $needBindUsers = [];
        if (empty($registerUsers)) {
            return [$needRegisterUsers, $needBindUsers];
        }
        $registerUserNicknames = ArrayToolkit::column($registerUsers, 'nickname');
        $existUsers = $this->getUserService()->findUsersByNicknames($registerUserNicknames);
        $existUsers = ArrayToolkit::index($existUsers, 'nickname');

        $existUserIds = empty($existUsers) ? [-1] : ArrayToolkit::column($existUsers, 'id');
        $registerUserMobiles = ArrayToolkit::column($registerUsers, 'mobile');
        $existMobileUsers = $this->getUserService()->findByVerifiedMobiles($registerUserMobiles);
        $existMobileUserIds = empty($existMobileUsers) ? [-1] : ArrayToolkit::column($existMobileUsers, 'id');
        $existMobileUsers = ArrayToolkit::index($existMobileUsers, 'verifiedMobile');
        $existUserIds = array_merge($existUserIds, $existMobileUserIds);
        $bindUsers = $this->getUserService()->searchUserBinds(['toIds' => $existUserIds], [], 0, count($existUserIds));
        $bindUsers = ArrayToolkit::index($bindUsers, 'toId');

        foreach ($registerUsers as $user) {
            if (isset($existUsers[$user['nickname']]) && isset($bindUsers[$existUsers[$user['nickname']]['id']])) {
                $this->getLogService()->error('dingtalk', 'dingtalk_can_not_register', "钉钉同步用户失败,{$user['nickname']}已经存在绑定过的用户无法创建用户".json_encode($user));
                continue;
            }

            if (!empty($existMobileUsers[$user['mobile']]) && isset($bindUsers[$existMobileUsers[$user['mobile']]['id']])) {
                $this->getLogService()->error('dingtalk', 'dingtalk_can_not_register', "钉钉同步用户失败,{$user['mobile']}已经存在绑定过的用户无法创建用户,可让用户自行选择绑定".json_encode($user));
                continue;
            }

            if (isset($existUsers[$user['nickname']]) && !isset($bindUsers[$existUsers[$user['nickname']]['id']])) {
                $user['userId'] = $existUsers[$user['nickname']]['id'];
                $needBindUsers[] = $user;
                continue;
            }

            if (!empty($existMobileUsers[$user['mobile']]) && !isset($bindUsers[$existMobileUsers[$user['mobile']]['id']])) {
                if (!isset($existUsers[$existMobileUsers[$user['mobile']]['nickname']])) {
                    $user['userId'] = $existMobileUsers[$user['mobile']]['id'];
                    $needBindUsers[] = $user;
                }
                continue;
            }

            $needRegisterUsers[] = $user;
        }

        unset($bindUsers);
        unset($existUsers);

        return [$needRegisterUsers, $needBindUsers];
    }

    /**
     * 拆分处理已绑定数据（分成'待更新'，'待更新用户名为xxx_旧'）
     *
     * @param $updateUsers
     *
     * @return array
     */
    protected function hierarchicalProcessUpdateData($updateUsers)
    {
        $updateNicknameToOldUserIds = [];
        $needUpdateUsers = [];
        if (empty($updateUsers) || !$this->jobNumberToNickname) {
            return [$needUpdateUsers, $updateNicknameToOldUserIds];
        }

        $userNicknames = ArrayToolkit::column($updateUsers, 'nickname');
        $existUsers = $this->getUserService()->findUsersByNicknames($userNicknames);
        $existUsers = ArrayToolkit::index($existUsers, 'nickname');
        $existUserIds = empty($existUsers) ? [-1] : ArrayToolkit::column($existUsers, 'id');
        $bindUsers = $this->getUserService()->searchUserBinds(['toIds' => $existUserIds], [], 0, count($existUserIds));
        $bindUsers = ArrayToolkit::index($bindUsers, 'toId');

        foreach ($updateUsers as $user) {
            if (!empty($user['nickname']) && isset($existUsers[$user['nickname']]) && isset($bindUsers[$existUsers[$user['nickname']]['id']]) && $existUsers[$user['nickname']]['id'] != $user['userId']) {
                $this->getLogService()->error('dingtalk', 'dingtalk_can_not_update', "钉钉同步用户失败,{$user['nickname']}已经存在绑定过的用户无法更新用户".json_encode($user));
                continue;
            }

            if (!empty($user['nickname']) && isset($existUsers[$user['nickname']]) && !isset($bindUsers[$existUsers[$user['nickname']]['id']])) {
                $updateNicknameToOldUserIds[] = $existUsers[$user['nickname']]['id'];
            }

            $needUpdateUsers[] = $user;
        }

        return [$needUpdateUsers, $updateNicknameToOldUserIds];
    }

    /**
     * 创建user表数据
     *
     * @param $dingTalkUsers
     *
     * @return bool
     */
    protected function registerUsers($dingTalkUsers)
    {
        if (empty($dingTalkUsers)) {
            return true;
        }

        $registerUsers = [];
        foreach ($dingTalkUsers as $user) {
            $registerUsers[] = $this->buildRegisterUser($user);
        }

        if (!empty($registerUsers)) {
            $this->getUserDao()->batchCreate($registerUsers);
        }

        return true;
    }

    /**
     * 绑定用户
     *
     * @param $dingTalkUsers
     *
     * @return bool
     */
    protected function bindUsers($dingTalkUsers)
    {
        $dingTalkUsers = ArrayToolkit::index($dingTalkUsers, 'nickname');
        $userIds = ArrayToolkit::column($dingTalkUsers, 'userId');
        if (empty($userIds)) {
            return true;
        }
        $bindUsers = $this->getUserService()->searchUserBinds(['toIds' => $userIds], [], 0, count($userIds));
        $bindUsers = ArrayToolkit::index($bindUsers, 'toId');
        $createBindUsers = [];
        foreach ($dingTalkUsers as $user) {
            if (isset($bindUsers[$user['userId']])) {
                continue;
            }
            $createBindUsers[] = [
                'fromId' => $dingTalkUsers[$user['nickname']]['unionid'],
                'toId' => $user['userId'],
                'type' => 'dingtalk',
                'createdTime' => time(),
            ];
        }
        if (!empty($createBindUsers)) {
            $this->getUserBindDao()->batchCreate($createBindUsers);
        }

        return true;
    }

    /**
     * 创建user_profile数据
     *
     * @param $dingTalkUsers
     *
     * @return array
     */
    protected function createUserProfilesAndUpdateDingTalkUsers($dingTalkUsers)
    {
        $profiles = [];
        $profileTemplate = $this->buildUserProfileTemplate();
        $nickNames = ArrayToolkit::column($dingTalkUsers, 'nickname');
        $dingTalkUsers = ArrayToolkit::index($dingTalkUsers, 'nickname');
        $users = $this->getUserService()->findUsersByNicknames($nickNames);
        foreach ($users as $user) {
            $dingTalkUsers[$user['nickname']]['userId'] = $user['id'];
            $profiles[$user['nickname']]['truename'] = $dingTalkUsers[$user['nickname']]['name'];
            $profiles[$user['nickname']]['id'] = $user['id'];
            $profiles[$user['nickname']] = array_merge($profiles[$user['nickname']], $profileTemplate);
        }
        if (!empty($profiles)) {
            $this->getUserProfileDao()->batchCreate(array_values($profiles));
        }

        return $dingTalkUsers;
    }

    /**
     * 构建用户数据
     *
     * @param $user
     *
     * @return array
     */
    protected function buildRegisterUser($user)
    {
        $registerUser['type'] = 'dingtalk';
        $registerUser['email'] = $this->getUserService()->generateEmail([]);
        $registerUser['orgIds'] = [1];
        $registerUser['orgCodes'] = ['1.'];
        $registerUser['nickname'] = $user['nickname'];
        $registerUser['roles'] = ['ROLE_USER'];
        $registerUser['createdTime'] = time();
        $registerUser['salt'] = base_convert(sha1(uniqid(mt_rand(), true)), 16, 36);
        $registerUser['password'] = $this->getPasswordEncoder()->encodePassword('pw123456', $registerUser['salt']);
        $registerUser['setup'] = 1;
        $registerUser['uuid'] = $this->getUserService()->generateUUID();

        return $registerUser;
    }

    /**
     * 构建user_profile 用户模板
     *
     * @return array
     */
    protected function buildUserProfileTemplate()
    {
        $profile = [];
        for ($i = 1; $i <= 5; ++$i) {
            $profile['intField'.$i] = null;
            $profile['dateField'.$i] = null;
            $profile['floatField'.$i] = null;
        }

        for ($i = 1; $i <= 10; ++$i) {
            $profile['varcharField'.$i] = '';
            $profile['textField'.$i] = '';
        }

        $profile['quick_entrance'] = '';

        $profile['custom_column'] = '';

        return $profile;
    }

    /**
     * 批量将用户名更新成xxxx_旧
     *
     * @param $userIds
     *
     * @return bool
     */
    protected function batchUpdateNicknameToOld($userIds)
    {
        if (empty($userIds)) {
            return true;
        }
        $users = $this->getUserService()->searchUsers(
            ['userIds' => $userIds],
            [],
            0,
            count($userIds),
            ['id', 'nickname']
        );
        $updateUsers = [];
        foreach ($users as $user) {
            $updateUsers[$user['id']] = [
                'nickname' => "{$user['nickname']}_旧{$user['id']}",
            ];
        }
        if (!empty($updateUsers)) {
            $this->getUserDao()->batchUpdate(array_keys($updateUsers), $updateUsers, 'id');
        }

        return true;
    }

    /**
     * 同步用户信息，设置组织机构，修改dingtalk_user
     *
     * @param $dingTalkUsers
     *
     * @return bool
     */
    protected function batchSetUserData($dingTalkUsers)
    {
        $this->batchUpdateUserInfo($dingTalkUsers);
        $this->batchSetUserOrgs($dingTalkUsers);
        $this->batchSetDingTalkUsers($dingTalkUsers);

        return true;
    }

    /**
     * 批量更新用户信息
     *
     * @param $dingTalkUsers
     *
     * @return bool
     */
    protected function batchUpdateUserInfo($dingTalkUsers)
    {
        $updateUsers = [];
        $updateUserProfiles = [];
        $users = $this->getUserService()->findUsersByIds(ArrayToolkit::column($dingTalkUsers, 'userId'));

        foreach ($dingTalkUsers as $user) {
            $updateUsers[$user['userId']]['hireDate'] = empty($user['hiredDate']) ? $users[$user['userId']]['hireDate'] : $user['hiredDate'] / 1000;
            $updateUsers[$user['userId']]['nickname'] = $user['nickname'];
            $updateUsers[$user['userId']]['email'] = $users[$user['userId']]['email'];
            $updateUsers[$user['userId']]['verifiedMobile'] = $user['mobile'];
            $updateUsers[$user['userId']]['locked'] = 0;
            $updateUserProfiles[$user['userId']]['truename'] = $user['name'];

            $emailIsExist = $this->getUserService()->getUserByEmail($user['email']);
            if (!empty($user['email']) && !$emailIsExist) {
                $updateUsers[$user['userId']]['email'] = $user['email'];
            }
        }

        if (!empty($updateUsers)) {
            $this->getUserDao()->batchUpdate(array_keys($updateUsers), $updateUsers, 'id');
        }

        if (!empty($updateUsers)) {
            $this->getUserProfileDao()->batchUpdate(array_keys($updateUserProfiles), $updateUserProfiles, 'id');
        }

        return true;
    }

    /**
     * 设置用户组织机构
     *
     * @param $dingTalkUsers
     *
     * @return bool
     */
    protected function batchSetUserOrgs($dingTalkUsers)
    {
        $userIds = ArrayToolkit::column($dingTalkUsers, 'userId');
        if (empty($userIds)) {
            return true;
        }
        $this->getUserOrgDao()->batchDelete(['userIds' => $userIds]);

        $updateUsers = [];
        $userOrgData = [];
        foreach ($dingTalkUsers as $user) {
            $orgs = $this->getOrgService()->findOrgsBySyncIds($user['department']);
            if (empty($orgs)) {
                continue;
            }
            foreach ($orgs as $org) {
                $userOrgData[] = [
                    'userId' => $user['userId'],
                    'orgId' => $org['id'],
                    'orgCode' => $org['orgCode'],
                    'createdTime' => time(),
                ];
            }
            $updateUsers[$user['userId']]['orgIds'] = ArrayToolkit::column($orgs, 'id');
            $updateUsers[$user['userId']]['orgCodes'] = ArrayToolkit::column($orgs, 'orgCode');
        }

        if (!empty($updateUsers)) {
            $this->getUserDao()->batchUpdate(array_keys($updateUsers), $updateUsers, 'id');
        }

        if (!empty($userOrgData)) {
            $this->getUserOrgDao()->batchCreate($userOrgData);
        }

        return true;
    }

    /**
     * 修改dingtalk_user表
     *
     * @param $dingTalkUsers
     *
     * @return bool
     */
    protected function batchSetDingTalkUsers($dingTalkUsers)
    {
        $unionids = ArrayToolkit::column($dingTalkUsers, 'unionid');
        $users = $this->getDingTalkUserService()->findUsersByUnionids($unionids);
        $users = ArrayToolkit::index($users, 'unionid');
        $createData = [];
        $updateDate = [];
        foreach ($dingTalkUsers as $user) {
            if (isset($users[$user['unionid']])) {
                $updateDate[$users[$user['unionid']]['id']] = [
                    'unionid' => $user['unionid'],
                    'userid' => $user['userid'],
                    'syncTime' => time(),
                ];
                continue;
            }
            $createData[] = ['unionid' => $user['unionid'], 'userid' => $user['userid'], 'syncTime' => time()];
        }
        if (!empty($createData)) {
            $this->getDingTalkUserDao()->batchCreate($createData);
        }
        if (!empty($updateDate)) {
            $this->getDingTalkUserDao()->batchUpdate(array_keys($updateDate), $updateDate, 'id');
        }

        return true;
    }

    /**
     * @param array $dingTalkUser
     *
     * @return array
     */
    protected function filterDingTalkUser($dingTalkUser)
    {
        $dingTalkUser['jobnumber'] = isset($dingTalkUser['jobnumber']) ? str_replace(' ', '', $dingTalkUser['jobnumber']) : '';
        $dingTalkUser['email'] = isset($dingTalkUser['email']) ? str_replace(' ', '', $dingTalkUser['email']) : '';
        $dingTalkUser['name'] = isset($dingTalkUser['name']) ? str_replace(' ', '', $dingTalkUser['name']) : '';

        return $dingTalkUser;
    }

    protected function getNeedLockUserBinds()
    {
        $setting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
        $needLockDingTalkUsers = $this->getDingTalkUserService()->searchDingTalkUsers(
            ['syncTime_LT' => empty($setting['syncStartTime']) ? strtotime(date('Y-m-d', time()).' 02:00:00') : $setting['syncStartTime']],
            [],
            0,
            200
        );
        if (empty($needLockDingTalkUsers)) {
            return [];
        }
        $dingTalkClient = DingTalkClientFactory::create();
        $lockUserUnionids = [];
        $updateDingTalkUsers = [];
        foreach ($needLockDingTalkUsers as $user) {
            $result = $dingTalkClient->getUserInfoByUserid($user['userid']);
            if (!empty($result)) {
                $updateDingTalkUsers[$user['id']] = ['syncTime' => time()];
                continue;
            }
            $lockUserUnionids[] = $user['unionid'];
        }

        if (!empty($updateDingTalkUsers)) {
            $this->getDingTalkUserDao()->batchUpdate(array_keys($updateDingTalkUsers), $updateDingTalkUsers, 'id');
        }

        if (!empty($lockUserUnionids)) {
            $this->getDingTalkUserDao()->batchDelete(['unionids' => $lockUserUnionids]);
        }

        return $this->getUserService()->searchUserBinds(
            ['fromIds' => empty($lockUserUnionids) ? [-1] : $lockUserUnionids, 'type' => 'dingtalk'],
            [],
            0,
            count($lockUserUnionids)
        );
    }

    protected function checkFinishSync()
    {
        $setting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);

        if (time() < $setting['finishTime'] || date('Y-m-d', $setting['finishTime']) != date('Y-m-d', time())) {
            return false;
        }

        return true;
    }

    /**
     * @return array
     */
    protected function registerSyncDingTalkUserJob()
    {
        if (!empty($this->getJobDao()->getByName('SyncDingTalkUserJob'))) {
            return true;
        }
        $job = [
            'name' => 'SyncDingTalkUserJob',
            'source' => 'TRAININGMAIN',
            'expression' => '*/7 1-7 * * *',
            'class' => 'CorporateTrainingBundle\Biz\DingTalk\Job\SyncDingTalkUserJob',
            'args' => [],
            'misfire_policy' => 'executing',
        ];

        $this->getSchedulerService()->register($job);
    }

    /**
     * @return array
     */
    protected function registerSyncDingTalkOrgsJob()
    {
        if (!empty($this->getJobDao()->getByName('SyncDingTalkOrgJob'))) {
            return true;
        }

        $job = [
            'name' => 'SyncDingTalkOrgJob',
            'source' => 'TRAININGMAIN',
            'expression' => '17 0 * * *',
            'class' => 'CorporateTrainingBundle\Biz\Org\Job\SyncDingTalkOrgJob',
            'args' => [],
            'misfire_policy' => 'executing',
        ];
        $this->getSchedulerService()->register($job);
    }

    /**
     * @return array
     */
    protected function registerLockDingTalkUserJob()
    {
        if (!empty($this->getJobDao()->getByName('LockDingTalkUserJob'))) {
            return true;
        }
        $job = [
            'name' => 'LockDingTalkUserJob',
            'source' => 'TRAININGMAIN',
            'expression' => '17 7 * * *',
            'class' => 'CorporateTrainingBundle\Biz\DingTalk\Job\LockDingTalkUserJob',
            'args' => [],
            'misfire_policy' => 'executing',
        ];
        $this->getSchedulerService()->register($job);
    }

    /**
     * @return UserDao
     */
    protected function getUserDao()
    {
        return $this->createDao('CorporateTrainingBundle:User:UserDao');
    }

    /**
     * @return UserProfileDao
     */
    protected function getUserProfileDao()
    {
        return $this->createDao('User:UserProfileDao');
    }

    /**
     * @return SettingService
     */
    protected function getSettingService()
    {
        return $this->createService('System:SettingService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\User\Service\UserService
     */
    protected function getUserService()
    {
        return $this->createService('CorporateTrainingBundle:User:UserService');
    }

    /**
     * @return UserOrgService
     */
    protected function getUserOrgService()
    {
        return $this->createService('User:UserOrgService');
    }

    /**
     * @return OrgService
     */
    protected function getOrgService()
    {
        return $this->createService('Org:OrgService');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->createService('System:LogService');
    }

    /**
     * @return AuthService
     */
    protected function getAuthService()
    {
        return $this->createService('User:AuthService');
    }

    /**
     * @return DingTalkUserService
     */
    protected function getDingTalkUserService()
    {
        return $this->createService('CorporateTrainingBundle:DingTalk:DingTalkUserService');
    }

    protected function getPasswordEncoder()
    {
        return new MessageDigestPasswordEncoder('sha256');
    }

    /**
     * @return UserBindDao
     */
    protected function getUserBindDao()
    {
        return $this->createDao('User:UserBindDao');
    }

    /**
     * @return UserOrgDao
     */
    protected function getUserOrgDao()
    {
        return $this->createDao('User:UserOrgDao');
    }

    /**
     * @return DingTalkUserDao
     */
    protected function getDingTalkUserDao()
    {
        return $this->createDao('CorporateTrainingBundle:DingTalk:DingTalkUserDao');
    }

    /**
     * @return SchedulerService
     */
    protected function getSchedulerService()
    {
        return $this->createService('Scheduler:SchedulerService');
    }

    protected function getJobDao()
    {
        return $this->biz->dao('Scheduler:JobDao');
    }
}
