<?php

namespace CorporateTrainingBundle\Biz\Enroll\Service\Impl;

use AppBundle\Common\ArrayToolkit;
use Biz\BaseService;
use CorporateTrainingBundle\Biz\Enroll\Dao\EnrollRecordDao;
use CorporateTrainingBundle\Biz\Enroll\Service\EnrollRecordService;

class EnrollRecordServiceImpl extends BaseService implements EnrollRecordService
{
    public function createEnrollRecord($record)
    {
        if (!ArrayToolkit::requireds($record, array('userId', 'targetId', 'targetType'))) {
            throw $this->createInvalidArgumentException('parameter is invalid!');
        }

        $record = $this->filterFields($record);
        $record['submittedTime'] = time();
        $record['status'] = 'submitted';

        return $this->getEnrollRecordDao()->create($record);
    }

    public function updateEnrollRecord($id, $fields)
    {
        $fields = $this->filterFields($fields);

        return $this->getEnrollRecordDao()->update($id, $fields);
    }

    public function batchUpdateEnrollRecord($records)
    {
        if (empty($records)) {
            return true;
        }

        return $this->getEnrollRecordDao()->batchUpdate(array_keys($records), $records, 'id');
    }

    public function submitEnrollment($targetId, $userId, $targetType)
    {
        $record = $this->getEnrollRecordByTargetIdAndUserIdAndTargetType($targetId, $userId, $targetType);
        if (empty($record)) {
            return $this->createEnrollRecord([
                'userId' => $userId,
                'targetId' => $targetId,
                'targetType' => $targetType,
            ]);
        }

        return $this->getEnrollRecordDao()->update($record['id'], ['status' => 'submitted', 'submittedTime' => time()]);
    }

    public function findQueueRecordsByTargetIdAndTargetType($targetId, $targetType, $count = 1)
    {
        return $this->getEnrollRecordDao()->findQueueRecordsByTargetIdAndTargetType($targetId, $targetType, $count);
    }

    public function findEnrollRecordsByIds($ids)
    {
        return $this->getEnrollRecordDao()->findByIds($ids);
    }

    public function findEnrollRecordsByTargetIdAndTargetTypeAndUserIds($targetId, $targetType, $userIds)
    {
        return $this->getEnrollRecordDao()->findByTargetIdAndTargetTypeAndUserIds($targetId, $targetType, $userIds);
    }

    public function getEnrollRecord($id)
    {
        return $this->getEnrollRecordDao()->get($id);
    }

    public function getEnrollRecordByTargetIdAndUserIdAndTargetType($targetId, $userId, $targetType)
    {
        return $this->getEnrollRecordDao()->getByTargetIdAndUserIdAndTargetType($targetId, $userId, $targetType);
    }

    /**
     * 获取当前排队的位置
     *
     * @param $userId
     * @param $targetId
     * @param $targetType
     *
     * @return int
     */
    public function getUserQueueCountByTargetIdAndTargetType($userId, $targetId, $targetType)
    {
        $record = $this->getEnrollRecordByTargetIdAndUserIdAndTargetType($targetId, $userId, $targetType);
        if (empty($record) || 'queue' != $record['status']) {
            return  0;
        }

        return $this->getEnrollRecordDao()->count([
            'status' => 'queue',
            'targetId' => $targetId,
            'targetType' => $targetType,
            'queueTime_LE' => $record['queueTime'],
        ]);
    }

    public function getQueueCountByTargetIdAndTargetType($targetId, $targetType)
    {
        return $this->countEnrollRecords(['targetId' => $targetId, 'targetType' => $targetType, 'status' => 'queue']);
    }

    public function countEnrollRecords(array $conditions)
    {
        return $this->getEnrollRecordDao()->count($conditions);
    }

    public function searchEnrollRecords(array $conditions, array $orderBys, $start, $limit)
    {
        return $this->getEnrollRecordDao()->search($conditions, $orderBys, $start, $limit);
    }

    public function selectEnrollRecordCountGroupByStatus($conditions)
    {
        return $this->getEnrollRecordDao()->selectRecordCountGroupByStatus($conditions);
    }

    protected function filterFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            array(
                'userId',
                'targetId',
                'submittedTime',
                'status',
                'approvedTime',
                'rejectedReason',
                'remark',
                'targetType',
                'queueTime',
            )
        );
    }

    /**
     * @return EnrollRecordDao
     */
    protected function getEnrollRecordDao()
    {
        return $this->createDao('CorporateTrainingBundle:Enroll:EnrollRecordDao');
    }

    /**
     * @return \Biz\User\Service\Impl\UserServiceImpl
     */
    protected function getUserService()
    {
        return $this->createService('User:UserService');
    }
}
