<?php

namespace CorporateTrainingBundle\Biz\Enroll\Strategy;

use AppBundle\Common\ArrayToolkit;
use Biz\System\Service\LogService;
use Biz\System\Service\SettingService;
use Codeages\Biz\Framework\Event\Event;
use CorporateTrainingBundle\Biz\Enroll\Dao\EnrollRecordDao;
use CorporateTrainingBundle\Biz\Enroll\Service\EnrollRecordService;
use CorporateTrainingBundle\Biz\ResourceScope\Service\ResourceAccessScopeService;
use CorporateTrainingBundle\Biz\ResourceScope\Service\ResourceVisibleScopeService;
use Exception;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

abstract class BaseStrategy implements EnrollStrategyInterface
{
    protected $biz;

    protected $targetType = '';

    protected $advancedMemberSelectType = '';

    protected $mailNotification = [];

    protected $dingtalkNotification = [];

    public function __construct($biz)
    {
        $this->biz = $biz;
        $this->mailNotification = $this->getSettingService()->get('mail_notification', []);
        $this->dingtalkNotification = $this->getSettingService()->get('dingtalk_notification', []);
    }

    /**
     * 提交报名
     *
     * @param int   $targetId
     * @param int   $userId
     * @param array $fields
     *
     * @return array
     *
     * @throws Exception
     */
    public function submitEnrollment($targetId, $userId, $fields)
    {
        if (!$this->canEnroll($targetId, $userId)) {
            return [];
        }
        $fields = ArrayToolkit::parts($fields, ['rejectedReason', 'remark']);

        $this->beginTransaction();
        try {
            $record = $this->getEnrollRecordService()->submitEnrollment($targetId, $userId, $this->targetType);

            if (!empty($fields)) {
                $record = $this->getEnrollRecordService()->updateEnrollRecord($record['id'], $fields);
            }

            if ($this->isNeedReview($targetId)) {
                $this->commit();

                return $record;
            }

            $fields = ['status' => 'queue', 'queueTime' => time()];
            if ($this->canBecomeMember($targetId)) {
                $selector = $this->getMemberSelector($this->advancedMemberSelectType);
                $result = $selector->enrollBecomeMember($targetId, [$userId], 1);
                $fields = $result ? ['status' => 'approved', 'approvedTime' => time()] : $fields;
            }

            $record = $this->getEnrollRecordService()->updateEnrollRecord($record['id'], $fields);
            if ('queue' == $record['status']) {
                $this->enrollToQueueList($targetId, [$record['id']], 'submit');
            }

            $this->getLogService()->info('resource_enroll', 'submit_enroll', '提交报名申请!', $record);

            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'submit_enroll', "{$this->targetType}/$targetId/用户{$userId}报名失败!".$e->getMessage());

            throw $e;
        }

        return $record;
    }

    /**
     * 补位
     *
     * @param int $targetId
     * @param int $count    //补位人数
     *
     * @return bool
     *
     * @throws Exception
     */
    public function batchQueueListBecomeMember($targetId, $count)
    {
        if (!$this->canBecomeMember($targetId)) {
            return false;
        }

        $records = $this->getEnrollRecordService()->findQueueRecordsByTargetIdAndTargetType($targetId, $this->targetType, $this->getCanBecomeMemberCount($targetId, $count));

        if (empty($records)) {
            return true;
        }

        $selector = $this->getMemberSelector($this->advancedMemberSelectType);
        $updateRecords = [];
        foreach ($records as $record) {
            $updateRecords[$record['id']] = ['status' => 'approved', 'approvedTime' => time(), 'queueTime' => 0];
        }

        $this->beginTransaction();
        try {
            $this->getEnrollRecordService()->batchUpdateEnrollRecord($updateRecords);
            $userIds = ArrayToolkit::column($records, 'userId');
            $selector->enrollBecomeMember($targetId, $userIds, 1);
            $this->getLogService()->info('resource_enroll', 'queue_enroll', "{$this->targetType}/{$targetId}/补位数量({$count}),自动报名补位!", $userIds);

            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'queue_enroll', "{$this->targetType}/{$targetId}/补位数量({$count}),自动补位失败!".$e->getMessage());
            throw $e;
        }

        return true;
    }

    /**
     * 资源移除成员
     *
     * @param int   $targetId
     * @param array $userIds
     *
     * @return bool
     *
     * @throws Exception
     */
    public function deleteMembers($targetId, $userIds)
    {
        $records = $this->getEnrollRecordService()->findEnrollRecordsByTargetIdAndTargetTypeAndUserIds($targetId, $this->targetType, $userIds);

        $updateRecords = [];
        foreach ($records as $record) {
            $updateRecords[$record['id']] = ['status' => 'deleted', 'queueTime' => 0, 'rejectedReason' => ''];
        }

        $this->beginTransaction();
        try {
            $count = $this->deleteResourceMembers($targetId, $userIds);
            $this->getEnrollRecordService()->batchUpdateEnrollRecord($updateRecords);
            $this->getLogService()->info('resource_enroll', 'delete_member', "{$this->targetType}/{$targetId}/用户删除成功！".json_encode($userIds), $userIds);
            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'delete_member', "{$this->targetType}/{$targetId}/用户删除失败！".json_encode($userIds).$e->getMessage());

            throw $e;
        }

        $this->batchQueueListBecomeMember($targetId, $count);

        return true;
    }

    /**
     * 取消报名
     *
     * @param int $targetId
     * @param int $userId
     *
     * @return bool
     *
     * @throws Exception
     */
    public function cancelEnroll($targetId, $userId)
    {
        $this->beginTransaction();
        try {
            $count = $this->deleteResourceMembers($targetId, [$userId]);
            $record = $this->getEnrollRecordService()->getEnrollRecordByTargetIdAndUserIdAndTargetType($targetId, $userId, $this->targetType);
            if (!empty($record)) {
                $this->getEnrollRecordService()->updateEnrollRecord($record['id'], ['status' => 'cancel', 'rejectedReason' => '']);
                $this->getLogService()->info('resource_enroll', 'cancel_enroll', "{$this->targetType}/{$targetId}/用户{$userId}删除成功！", $record);
            }

            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'cancel_enroll', "{$this->targetType}/{$targetId}/用户{$userId}删除失败！".$e->getMessage());

            throw $e;
        }

        $this->batchQueueListBecomeMember($targetId, $count);

        return true;
    }

    /**
     * 审核报名申请
     *
     * @param int    $targetId
     * @param array  $recordIs
     * @param string $status
     * @param array  $fields
     *
     * @return bool
     *
     * @throws Exception
     */
    public function verifyEnrollment($targetId, $recordIs, $status, $fields = [])
    {
        $records = $this->getEnrollRecordService()->searchEnrollRecords(['ids' => array_values($recordIs)], ['submittedTime' => 'ASC'], 0, count($recordIs));

        if (empty($records) || !in_array($status, ['approved', 'rejected'])) {
            return  true;
        }
        $enrollRecords = [];
        $queueEnrollRecords = [];
        $rejectedReason = isset($fields['rejectedReason']) ? $fields['rejectedReason'] : '';
        $approvedNum = 0;
        foreach ($records as $record) {
            if ('rejected' == $status) {
                $enrollRecords[$record['id']] = ['userId' => $record['userId'], 'status' => $status, 'approvedTime' => 0, 'queueTime' => 0, 'rejectedReason' => $rejectedReason];
                continue;
            }
            if ($this->canBecomeMember($targetId) && $this->getCanBecomeMemberCount($targetId, $approvedNum + 1) >= $approvedNum + 1) {
                $enrollRecords[$record['id']] = ['userId' => $record['userId'], 'status' => $status, 'approvedTime' => time(), 'queueTime' => 0, 'rejectedReason' => $rejectedReason];
                ++$approvedNum;
            } else {
                $queueEnrollRecords[$record['id']] = ['userId' => $record['userId'], 'status' => 'queue', 'queueTime' => time()];
            }
        }

        $selector = $this->getMemberSelector($this->advancedMemberSelectType);

        $this->beginTransaction();
        try {
            $this->getEnrollRecordService()->batchUpdateEnrollRecord($enrollRecords);
            $this->getEnrollRecordService()->batchUpdateEnrollRecord($queueEnrollRecords);

            if ('approved' == $status && !empty($enrollRecords)) {
                $selector->enrollBecomeMember($targetId, ArrayToolkit::column($enrollRecords, 'userId'), 1);
            }

            if ('approved' == $status && !empty($queueEnrollRecords)) {
                $this->enrollToQueueList($targetId, array_keys($queueEnrollRecords), 'review');
            }

            if ('rejected' == $status) {
                $this->rejectEnroll($targetId, $recordIs, $fields);
            }
            $this->getLogService()->info('resource_enroll', 'verify_enroll', "{$this->targetType}/{$targetId}/审核成功{$status}！".json_encode($recordIs), $recordIs);

            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'verify_enroll', "{$this->targetType}/{$targetId}/审核失败{$status}！".json_encode($recordIs).$e->getMessage());
            throw $e;
        }

        return true;
    }

    /**
     * 补位直接添加成成员
     *
     * @param $targetId
     * @param $recordIs
     *
     * @return bool
     *
     * @throws Exception
     */
    public function queueListBecomeMember($targetId, $recordIs)
    {
        $records = $this->getEnrollRecordService()->searchEnrollRecords(['ids' => array_values($recordIs)], ['submittedTime' => 'ASC'], 0, count($recordIs));
        if (empty($records)) {
            return  true;
        }
        foreach ($records as $record) {
            $enrollRecords[$record['id']] = ['status' => 'approved', 'approvedTime' => time(), 'queueTime' => 0, 'rejectedReason' => ''];
        }

        $selector = $this->getMemberSelector($this->advancedMemberSelectType);

        $this->beginTransaction();
        try {
            $this->getEnrollRecordService()->batchUpdateEnrollRecord($enrollRecords);
            $selector->enrollBecomeMember($targetId, ArrayToolkit::column($records, 'userId'), 1);
            $this->getLogService()->info('resource_enroll', 'queue_to_member', '补位直接变成成员'.json_encode($recordIs), $recordIs);

            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'queue_to_member', '补位直接变成成员失败'.json_encode($recordIs).$e->getMessage());

            throw $e;
        }

        return true;
    }

    /**
     * 管理员添加成员创建审核记录
     *
     * @param $targetId
     * @param $userIds
     *
     * @return bool
     *
     * @throws Exception
     */
    public function createEnrollRecordWithAdminAddUser($targetId, $userIds)
    {
        if (empty($userIds)) {
            return  true;
        }

        $records = $this->getEnrollRecordService()->findEnrollRecordsByTargetIdAndTargetTypeAndUserIds($targetId, $this->targetType, $userIds);
        $records = ArrayToolkit::index($records, 'userId');
        $createEnrollRecords = [];
        $updateEnrollRecords = [];

        foreach ($userIds as $userId) {
            if (!empty($records[$userId])) {
                $updateEnrollRecords[$records[$userId]['id']] = [
                     'status' => 'approved',
                     'approvedTime' => time(),
                     'queueTime' => 0,
                 ];
                continue;
            }
            $createEnrollRecords[] = [
                 'userId' => $userId,
                 'targetId' => $targetId,
                 'targetType' => $this->targetType,
                 'status' => 'approved',
                 'approvedTime' => time(),
             ];
        }

        $this->beginTransaction();
        try {
            $this->getEnrollRecordDao()->batchCreate($createEnrollRecords);
            $this->getEnrollRecordService()->batchUpdateEnrollRecord($updateEnrollRecords);
            $this->getLogService()->info('resource_enroll', 'admin_add_member', "{$this->targetType}/{$targetId}/添加成员成功！".json_encode($userIds), $userIds);

            $this->commit();
        } catch (Exception $e) {
            $this->rollback();
            $this->getLogService()->error('resource_enroll', 'admin_add_member', "{$this->targetType}/{$targetId}/添加成员失败！".json_encode($userIds));

            throw $e;
        }

        return true;
    }

    protected function canEnroll($id, $userId)
    {
        $status = $this->getUserEnrollStatus($id, $userId);

        if (in_array($status, ['reset', 'enrollAble', 'queueAble'])) {
            return true;
        }

        return false;
    }

    /**
     * 判断管理权限
     *
     * @param $targetId
     *
     * @return bool
     */
    abstract public function canManageResource($targetId);

    /**
     * 获取用户报名状态
     *
     * @param $targetId
     * @param $userId
     *
     * @return string ('reset'重新报名, 'enrollAble'报名条件允许, 'queueAble'补位条件允许,.......)
     */
    abstract public function getUserEnrollStatus($targetId, $userId);

    /**
     * 是否需要审核
     *
     * @param $targetId
     *
     * @return bool
     */
    abstract protected function isNeedReview($targetId);

    /**
     * 是否可以直接添加为成员
     *
     * @param $targetId
     *
     * @return bool
     */
    abstract protected function canBecomeMember($targetId);

    /**
     * @param int $targetId
     * @param int $needCount
     *
     * @return int
     */
    abstract protected function getCanBecomeMemberCount($targetId, $needCount);

    /**
     * 批量移除成员
     *
     * @param $targetId
     * @param array $userIds
     *
     * @return int //删除个数
     */
    abstract protected function deleteResourceMembers($targetId, $userIds);

    /**
     * 拒绝报名审核（主要用于资源本身的业务处理）
     *
     * @param $targetId
     * @param $recordIds
     * @param array $info
     *
     * @return mixed
     */
    abstract protected function rejectEnroll($targetId, $recordIds, $info = []);

    /**
     * 进入补位队列（主要用于资源本身的业务处理, 如通知等）
     *
     * @param $targetId
     * @param $recordIds
     * @param string $type
     */
    abstract protected function enrollToQueueList($targetId, $recordIds, $type = 'submit');

    abstract protected function getResourceDao();

    abstract protected function getResourceService();

    protected function getCurrentUser()
    {
        return $this->biz['user'];
    }

    protected function getLogger()
    {
        return $this->biz['logger'];
    }

    protected function beginTransaction()
    {
        $this->biz['db']->beginTransaction();
    }

    protected function commit()
    {
        $this->biz['db']->commit();
    }

    protected function rollback()
    {
        $this->biz['db']->rollback();
    }

    protected function createService($alias)
    {
        return $this->biz->service($alias);
    }

    protected function createDao($alias)
    {
        return $this->biz->Dao($alias);
    }

    /**
     * @return EventDispatcherInterface
     */
    private function getDispatcher()
    {
        return $this->biz['dispatcher'];
    }

    /**
     * @param string      $eventName
     * @param Event|mixed $subject
     *
     * @return Event
     */
    protected function dispatchEvent($eventName, $subject, $arguments = [])
    {
        if ($subject instanceof Event) {
            $event = $subject;
        } else {
            $event = new Event($subject, $arguments);
        }

        return $this->getDispatcher()->dispatch($eventName, $event);
    }

    /**
     * @return EnrollRecordService
     */
    protected function getEnrollRecordService()
    {
        return $this->createService('CorporateTrainingBundle:Enroll:EnrollRecordService');
    }

    /**
     * @return EnrollRecordDao
     */
    protected function getEnrollRecordDao()
    {
        return $this->createDao('CorporateTrainingBundle:Enroll:EnrollRecordDao');
    }

    /**
     * @return ResourceVisibleScopeService
     */
    protected function getResourceVisibleService()
    {
        return $this->createService('CorporateTrainingBundle:ResourceScope:ResourceVisibleScopeService');
    }

    /**
     * @return ResourceAccessScopeService
     */
    protected function getResourceAccessService()
    {
        return $this->createService('CorporateTrainingBundle:ResourceScope:ResourceAccessScopeService');
    }

    protected function getMemberSelector($targetType)
    {
        $memberSelectFactory = $this->biz->offsetGet('advanced_member_select_factory');

        return $memberSelectFactory->create($targetType);
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->createService('System:LogService');
    }

    /**
     * @return SettingService
     */
    protected function getSettingService()
    {
        return $this->biz->service('System:SettingService');
    }
}
