<?php

namespace CorporateTrainingBundle\Biz\Enroll\Strategy\Impl;

use AppBundle\Common\ArrayToolkit;
use CorporateTrainingBundle\Biz\Enroll\Strategy\BaseStrategy;
use CorporateTrainingBundle\Biz\OfflineActivity\Dao\MemberDao;
use CorporateTrainingBundle\Biz\OfflineActivity\Dao\OfflineActivityDao;
use CorporateTrainingBundle\Biz\OfflineActivity\Service\MemberService;
use CorporateTrainingBundle\Biz\OfflineActivity\Service\OfflineActivityService;

class OfflineActivityStrategy extends BaseStrategy
{
    protected $targetType = 'offlineActivity';

    protected $advancedMemberSelectType = 'offlineActivity_member';

    public function getUserEnrollStatus($targetId, $userId)
    {
        $member = $this->getOfflineActivityMemberService()->getMemberByActivityIdAndUserId($targetId, $userId);

        if (!empty($member)) {
            return 'join';
        }

        $activity = $this->getResourceService()->getOfflineActivity($targetId);

        if (time() < $activity['enrollmentStartDate']) {
            return 'enrollmentUnStart';
        }

        if (time() > $activity['enrollmentEndDate']) {
            return 'enrollmentEnd';
        }

        if ($activity['conditionalAccess']) {
            if (!$this->getResourceAccessService()->canUserAccessResource('offlineActivity', $activity['id'], $this->getCurrentUser()->getId())) {
                return 'notAvailableForYou';
            }
        } else {
            if (!$this->getResourceVisibleService()->canUserVisitResource('offlineActivity', $activity['id'], $userId)) {
                return 'notAvailableForYou';
            }
        }

        $record = $this->getEnrollRecordService()->getEnrollRecordByTargetIdAndUserIdAndTargetType($targetId, $userId, $this->targetType);
        if (!empty($record) && in_array($record['status'], ['queue', 'submitted'])) {
            return $record['status'];
        }

        $memberCount = $this->getOfflineActivityMemberService()->countMembers(['activityId' => $targetId]);

        if ($memberCount >= $activity['maxStudentNum'] && $activity['maxStudentNum'] > 0) {
            return 'queueAble';
        }

        if ((!empty($record) && in_array($record['status'], ['deleted', 'cancel', 'rejected']))) {
            return 'reset';
        }

        return 'enrollAble';
    }

    public function canManageResource($targetId)
    {
        return $this->getResourceService()->canManageOfflineActivity($targetId);
    }

    protected function isNeedReview($targetId)
    {
        $activity = $this->getResourceService()->getOfflineActivity($targetId);

        return $activity['requireAudit'];
    }

    protected function deleteResourceMembers($targetId, $userIds)
    {
        $members = $this->getOfflineActivityMemberService()->searchMembers(['activityId' => $targetId, 'userIds' => $userIds], [], 0, count($userIds));

        foreach ($members as $member) {
            $this->getOfflineActivityMemberDao()->delete($member['id']);
        }
        $this->getResourceService()->refreshMemberCount($targetId);

        return count($members);
    }

    public function canBecomeMember($targetId)
    {
        $activity = $this->getResourceService()->getOfflineActivity($targetId);

        $memberCount = $this->getOfflineActivityMemberService()->countMembers(['activityId' => $targetId]);

        if ($memberCount < $activity['maxStudentNum'] || 0 == $activity['maxStudentNum']) {
            return true;
        }

        return false;
    }

    protected function getCanBecomeMemberCount($targetId, $needCount)
    {
        $activity = $this->getResourceService()->getOfflineActivity($targetId);

        if (0 == $activity['maxStudentNum']) {
            return $needCount;
        }

        $memberCount = $this->getOfflineActivityMemberService()->countMembers(['activityId' => $targetId]);

        $studentNum = $activity['maxStudentNum'] - $memberCount;

        if ($studentNum <= 0) {
            return 0;
        }

        if ($needCount >= $studentNum) {
            return $studentNum;
        }

        return $needCount;
    }

    protected function rejectEnroll($targetId, $recordIds, $info = [])
    {
        $records = $this->getEnrollRecordService()->findEnrollRecordsByIds($recordIds);

        if (empty($records)) {
            return true;
        }

        $userIds = ArrayToolkit::column($records, 'userId');
        $resource = $this->biz->offsetGet('resource_notification.offline_activity');
        $resource->onOfflineActivityBatchReject($targetId, $userIds, empty($info['rejectedReason']) ? '' : $info['rejectedReason']);
    }

    protected function enrollToQueueList($targetId, $recordIds, $type = 'submit')
    {
        $records = $this->getEnrollRecordService()->findEnrollRecordsByIds($recordIds);
        if (empty($records)) {
            return true;
        }
        $userIds = ArrayToolkit::column($records, 'userId');
        $resource = $this->biz->offsetGet('resource_notification.offline_activity');
        $resource->onEnrollToQueueList($targetId, $userIds, $type);
    }

    /**
     * @return OfflineActivityService
     */
    protected function getResourceService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineActivity:OfflineActivityService');
    }

    /**
     * @return MemberService
     */
    protected function getOfflineActivityMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineActivity:MemberService');
    }

    /**
     * @return OfflineActivityDao
     */
    protected function getResourceDao()
    {
        return $this->createDao('CorporateTrainingBundle:OfflineActivity:OfflineActivityDao');
    }

    /**
     * @return MemberDao
     */
    protected function getOfflineActivityMemberDao()
    {
        return $this->createDao('CorporateTrainingBundle:OfflineActivity:MemberDao');
    }
}
