<?php

namespace CorporateTrainingBundle\Biz\Enroll\Strategy\Impl;

use AppBundle\Common\ArrayToolkit;
use CorporateTrainingBundle\Biz\Enroll\Strategy\BaseStrategy;
use CorporateTrainingBundle\Biz\ProjectPlan\Dao\ProjectPlanDao;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\MemberService;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\ProjectPlanService;

class ProjectPlanStrategy extends BaseStrategy
{
    protected $targetType = 'projectPlan';

    protected $advancedMemberSelectType = 'project_plan_member';

    public function getUserEnrollStatus($targetId, $userId)
    {
        $projectPlan = $this->getResourceService()->getProjectPlan($targetId);

        $member = $this->getProjectPlanMemberService()->getProjectPlanMemberByUserIdAndProjectPlanId($userId, $targetId);

        if (!empty($member)) {
            return 'success';
        }

        if (empty($projectPlan['requireEnrollment'])) {
            return 'enrollmentUnOpen';
        }

        if (time() > $projectPlan['enrollmentEndDate']) {
            return 'enrollmentEnd';
        }

        if (time() < $projectPlan['enrollmentStartDate']) {
            return 'enrollmentUnStart';
        }

        if ($projectPlan['conditionalAccess']) {
            if (!$this->getResourceAccessService()->canUserAccessResource('projectPlan', $projectPlan['id'], $userId)) {
                return 'notAvailableForYou';
            }
        } else {
            if (!$this->getResourceVisibleService()->canUserVisitResource('projectPlan', $projectPlan['id'], $userId)) {
                return 'notAvailableForYou';
            }
        }
        $record = $this->getEnrollRecordService()->getEnrollRecordByTargetIdAndUserIdAndTargetType($targetId, $userId, $this->targetType);
        if (!empty($record) && in_array($record['status'], ['queue', 'submitted'])) {
            return $record['status'];
        }

        $memberCount = $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $targetId]);

        if ($memberCount >= $projectPlan['maxStudentNum'] && $projectPlan['maxStudentNum'] > 0) {
            return 'queueAble';
        }

        if ((!empty($record) && in_array($record['status'], ['deleted', 'cancel', 'rejected']))) {
            return 'reset';
        }

        return 'enrollAble';
    }

    public function canManageResource($targetId)
    {
        return $this->getResourceService()->canManageProjectPlan($targetId);
    }

    protected function isNeedReview($targetId)
    {
        $projectPlan = $this->getResourceService()->getProjectPlan($targetId);

        return $projectPlan['requireAudit'];
    }

    protected function deleteResourceMembers($targetId, $userIds)
    {
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $targetId, 'userIds' => $userIds], [], 0, count($userIds));
        foreach ($members as $member) {
            $this->getProjectPlanMemberDao()->delete($member['id']);
        }

        return count($members);
    }

    protected function canBecomeMember($targetId)
    {
        $projectPlan = $this->getResourceService()->getProjectPlan($targetId);

        $memberCount = $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $targetId]);

        if ($memberCount < $projectPlan['maxStudentNum'] || 0 == $projectPlan['maxStudentNum']) {
            return true;
        }

        return false;
    }

    protected function getCanBecomeMemberCount($targetId, $needCount)
    {
        $projectPlan = $this->getResourceService()->getProjectPlan($targetId);

        if (0 == $projectPlan['maxStudentNum']) {
            return $needCount;
        }

        $memberCount = $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $targetId]);

        $studentNum = $projectPlan['maxStudentNum'] - $memberCount;

        if ($studentNum <= 0) {
            return 0;
        }

        if ($needCount >= $studentNum) {
            return $studentNum;
        }

        return $needCount;
    }

    protected function rejectEnroll($targetId, $recordIds, $info = [])
    {
        $records = $this->getEnrollRecordService()->findEnrollRecordsByIds($recordIds);

        if (empty($records)) {
            return true;
        }
        $resource = $this->biz->offsetGet('resource_notification.project_plan');
        $userIds = ArrayToolkit::column($records, 'userId');
        $resource->onRejectProjectPlanApplies($targetId, $userIds, empty($info['rejectedReason']) ? '' : $info['rejectedReason']);
    }

    protected function enrollToQueueList($targetId, $recordIds, $type = 'submit')
    {
        $records = $this->getEnrollRecordService()->findEnrollRecordsByIds($recordIds);

        if (empty($records)) {
            return true;
        }

        $resource = $this->biz->offsetGet('resource_notification.project_plan');
        $userIds = ArrayToolkit::column($records, 'userId');
        $resource->onEnrollToQueueList($targetId, $userIds, $type);
    }

    /**
     * @return ProjectPlanService
     */
    protected function getResourceService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:ProjectPlanService');
    }

    /**
     * @return MemberService
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return ProjectPlanDao
     */
    protected function getResourceDao()
    {
        return $this->createDao('CorporateTrainingBundle:ProjectPlan:ProjectPlanDao');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Dao\Impl\MemberDaoImpl
     */
    protected function getProjectPlanMemberDao()
    {
        return $this->createDao('CorporateTrainingBundle:ProjectPlan:MemberDao');
    }
}
