<?php

namespace CorporateTrainingBundle\Biz\Exporter;

use AppBundle\Common\ArrayToolkit;
use CorporateTrainingBundle\Biz\Task\Service\TaskResultService;
use CorporateTrainingBundle\Biz\Task\Service\TaskService;
use CorporateTrainingBundle\Common\OrgToolkit;

class ProjectPlanMemberStatisticDataExporter extends AbstractExporter
{
    protected $attendStatus = [
        'none' => 'offline_course.attend_status.none',
        'attended' => 'offline_course.attend_status.attended',
        'partial' => 'offline_course.attend_status.partial',
        'absenteeism' => 'offline_course.attend_status.absenteeism',
        'leave' => 'offline_course.attend_status.leave',
    ];

    public function canExport($parameters)
    {
        return $this->biz['user']->hasPermission('admin_project_plan_data_export');
    }

    public function getExportFileName()
    {
        return 'project_plan_member_statistic_data.xls';
    }

    public function getSortedHeadingRow($parameters)
    {
        $headingRows = [
            ['code' => 'nickname', 'title' => $this->trans('student.user_name')],
            ['code' => 'truename', 'title' => $this->trans('student.profile.truename')],
            ['code' => 'org', 'title' => $this->trans('project_plan.export.department')],
            ['code' => 'post', 'title' => $this->trans('student.profile.post')],
            ['code' => 'projectName', 'title' => $this->trans('project_plan.name')],
            ['code' => 'startTime', 'title' => $this->trans('project_plan.export.start_time')],
            ['code' => 'endTime', 'title' => $this->trans('project_plan.export.end_time')],
            ['code' => 'onlineCourseNum', 'title' => $this->trans('project_plan.export.online_course.total_num')],
            ['code' => 'finishedOnlineCourseNum', 'title' => $this->trans('project_plan.export.online_course.finish_number')],
            ['code' => 'onlineCourseTime', 'title' => $this->trans('project_plan.export.online_course.total_learn_time')],
            ['code' => 'onlineCourseLearnTime', 'title' => $this->trans('project_plan.export.online_learn_time')],
            ['code' => 'offlineCourseNum', 'title' => $this->trans('project_plan.export.offline_course.total_num')],
            ['code' => 'offlineCourseAttendance', 'title' => $this->trans('project_plan.export.offline_course.attendance_rate')],
            ['code' => 'homeworkPass', 'title' => $this->trans('project_plan.study_data.homework_passing_rate')],
        ];

        $projectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanId($parameters['projectPlanId']);
        $projectPlanItems = ArrayToolkit::group($projectPlanItems, 'targetType');
        if (!empty($projectPlanItems['course'])) {
            foreach ($projectPlanItems['course'] as $course) {
                $headingRows[] = ['code' => 'onlineCourseTitle'.$course['targetId'], 'title' => $this->trans('project_plan.export.online_course.title')];
                $headingRows[] = ['code' => 'onlineCourseLength'.$course['targetId'], 'title' => $this->trans('project_plan.export.online_course.length')];
                $headingRows[] = ['code' => 'onlineCourseLearnTime'.$course['targetId'], 'title' => $this->trans('project_plan.export.online_course.learn_time')];
                $headingRows[] = ['code' => 'onlineCourseFinishTime'.$course['targetId'], 'title' => $this->trans('project_plan.export.online_course.finish_time')];
            }
        }
        if (!empty($projectPlanItems['exam'])) {
            foreach ($projectPlanItems['exam'] as $exam) {
                $headingRows[] = ['code' => 'onlineExamTitle'.$exam['targetId'], 'title' => $this->trans('project_plan.export.exam.title')];
                $headingRows[] = ['code' => 'onlineExamBestScore'.$exam['targetId'], 'title' => $this->trans('project_plan.export.exam.best_score')];
                $headingRows[] = ['code' => 'onlineExamResult'.$exam['targetId'], 'title' => $this->trans('project_plan.export.exam.pass_result')];
            }
        }
        if (!empty($projectPlanItems['offline_course'])) {
            foreach ($projectPlanItems['offline_course'] as $course) {
                $headingRows[] = ['code' => 'offlineCourseTitle'.$course['targetId'], 'title' => $this->trans('project_plan.export.offline_course.title')];
                $headingRows[] = ['code' => 'offlineCourseLength'.$course['targetId'], 'title' => $this->trans('project_plan.export.offline_course.length')];
                $headingRows[] = ['code' => 'offlineCourseSign'.$course['targetId'], 'title' => $this->trans('project_plan.export.offline_course.sign_rate')];
                $headingRows[] = ['code' => 'offlineCourseSignStatus'.$course['targetId'], 'title' => $this->trans('offline_course.attendance_status')];
                $headingRows[] = ['code' => 'offlineCourseResult'.$course['targetId'], 'title' => $this->trans('project_plan.export.offline_course.pass_rate')];
            }
        }
        if (!empty($projectPlanItems['offline_exam'])) {
            foreach ($projectPlanItems['offline_exam'] as $exam) {
                $headingRows[] = ['code' => 'offlineExamTitle'.$exam['targetId'], 'title' => $this->trans('project_plan.export.exam.title')];
                $headingRows[] = ['code' => 'offlineExamBestScore'.$exam['targetId'], 'title' => $this->trans('project_plan.export.exam.score')];
                $headingRows[] = ['code' => 'offlineExamResult'.$exam['targetId'], 'title' => $this->trans('project_plan.export.exam.pass_result')];
            }
        }

        return $headingRows;
    }

    public function buildExportData($parameters)
    {
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($parameters['projectPlanId']);
        $projectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanId($projectPlan['id']);
        $projectPlanItems = ArrayToolkit::group($projectPlanItems, 'targetType');

        $conditions = $this->prepareStudyDataListSearchConditions($parameters, $parameters['projectPlanId']);

        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers($conditions, [], 0, PHP_INT_MAX);
        $userIds = ArrayToolkit::column($members, 'userId');
        list($users, $userProfiles, $orgs, $posts) = $this->buildUsersData($userIds);

        $onlineCourseLearnTime = $this->getMemberOnlineCourseLearnTime($parameters['projectPlanId'], $userIds);

        $memberData = [];
        $onlineProjectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($parameters['projectPlanId'], 'course');
        $onlineCourseIds = !empty($projectPlanItems['course']) ? ArrayToolkit::column($onlineProjectPlanItems, 'targetId') : [-1];

        $offlineCourseIds = !empty($projectPlanItems['offline_course']) ? ArrayToolkit::column($projectPlanItems['offline_course'], 'targetId') : [-1];
        $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(['offlineCourseIds' => $offlineCourseIds, 'type' => 'homework'], [], 0, PHP_INT_MAX);
        $homeworkTaskIds = empty($homeworkTasks) ? [-1] : ArrayToolkit::column($homeworkTasks, 'id');
        $onlineCourseTotalTime = $this->sumOnlineCourseTotalTime($onlineCourseIds);
        foreach ($members as $member) {
            $attendRateData = $this->getProjectPlanService()->getUserProjectPlanAttendRateData($projectPlan, $member['userId']);
            $member['offlineCoursePassedHomeworkNum'] = $this->getOfflineCourseTaskService()->countTaskResults(['taskIds' => $homeworkTaskIds, 'status' => 'finished', 'userId' => $member['userId']]);
            $member['offlineCoursePassedHomeworkRate'] = empty($member['offlineCoursePassedHomeworkNum']) ? 0 : round(($member['offlineCoursePassedHomeworkNum'] / count($homeworkTasks)) * 100, 2);
            $user = $users[$member['userId']];
            $singleMemberData = [
                'nickname' => empty($user) ? '-' : $user['nickname'],
                'truename' => empty($userProfiles[$user['id']]) ? '-' : $userProfiles[$member['userId']]['truename'],
                'org' => OrgToolkit::buildOrgsNames($user['orgIds'], $orgs),
                'post' => empty($user['postId']) ? '-' : $posts[$user['postId']]['name'],
                'projectName' => $projectPlan['name'],
                'startTime' => date('Y-m-d', $projectPlan['startTime']),
                'endTime' => date('Y-m-d', $projectPlan['endTime']),
                'onlineCourseNum' => empty($projectPlanItems['course']) ? 0 : count($onlineCourseIds),
                'finishedOnlineCourseNum' => $this->getCourseMemberService()->countLearnedMember(['m.userId' => $member['userId'], 'courseId' => $onlineCourseIds]),
                'onlineCourseTime' => substr(sprintf('%.3f', $onlineCourseTotalTime / 3600), 0, -1),
                'onlineCourseLearnTime' => empty($onlineCourseLearnTime[$user['id']]['totalLearnTime']) ? '--' : substr(sprintf('%.3f', $onlineCourseLearnTime[$user['id']]['totalLearnTime'] / 3600), 0, -1),
                'offlineCourseNum' => empty($projectPlanItems['offline_course']) ? 0 : count($offlineCourseIds),
                'offlineCourseAttendance' => $attendRateData['rate'].'%',
                'homeworkPass' => $member['offlineCoursePassedHomeworkRate'].'%',
            ];

            $singleMemberData = $this->appendOnlineCourseData($singleMemberData, $onlineCourseIds, $member['userId']);
            $examIds = (isset($projectPlanItems['exam'])) ? ArrayToolkit::column($projectPlanItems['exam'], 'targetId') : [-1];
            $singleMemberData = $this->appendOnlineExamData($singleMemberData, $examIds, $member['userId']);
            $singleMemberData = $this->appendOfflineCourseData($singleMemberData, $offlineCourseIds, $member['userId']);

            $offlineExamIds = (isset($projectPlanItems['offline_exam'])) ? ArrayToolkit::column($projectPlanItems['offline_exam'], 'targetId') : [-1];
            $singleMemberData = $this->appendOfflineExamData($singleMemberData, $offlineExamIds, $member['userId']);

            $memberData[] = $singleMemberData;
            unset($singleMemberData);
        }
        $exportData[] = [
            'sheetName' => $projectPlan['name'].$this->trans('project_plan.summary.study_data'),
            'data' => $memberData,
        ];

        return $exportData;
    }

    protected function getMemberOnlineCourseLearnTime($projectPlanId, $userIds)
    {
        $projectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'course');
        $onlineCourseIds = isset($projectPlanItems) ? ArrayToolkit::column($projectPlanItems, 'targetId') : [-1];
        $onlineCourseLearnTime = $this->getTaskResultService()->sumLearnTimeByCourseIdsAndUserIdsGroupByUserId($onlineCourseIds, $userIds);

        return ArrayToolkit::index($onlineCourseLearnTime, 'userId');
    }

    protected function appendOfflineExamData($member, $examIds, $userId)
    {
        if ($examIds == [-1]) {
            return $member;
        }

        $offlineExams = $this->getOfflineExamService()->findOfflineExamByIds($examIds);
        foreach ($offlineExams as $exam) {
            $member['offlineExamTitle'.$exam['id']] = $exam['title'];
            $examMember = $this->getOfflineExamMemberService()->getMemberByOfflineExamIdAndUserId($exam['id'], $userId);
            if (empty($examMember)) {
                $member['offlineExamBestScore'.$exam['id']] = '--';
                $member['offlineExamResult'.$exam['id']] = '--';
            } else {
                $member['offlineExamBestScore'.$exam['id']] = $examMember['score'];
                $member['offlineExamResult'.$exam['id']] = $examMember['status'];
            }
        }

        return $member;
    }

    protected function appendOfflineCourseData($member, $courseIds, $userId)
    {
        if ($courseIds == [-1]) {
            return $member;
        }
        $courses = $this->getOfflineCourseService()->findOfflineCoursesByIds($courseIds);
        $courses = ArrayToolkit::index($courses, 'id');
        $offlineCourseMembers = $this->getOfflineCourseMemberService()->searchMembers(['offlineCourseIds' => $courseIds, 'userId' => $userId], [], 0, PHP_INT_MAX, ['id', 'offlineCourseId', 'status']);
        $offlineCourseMembers = ArrayToolkit::index($offlineCourseMembers, 'offlineCourseId');

        foreach ($courseIds as $courseId) {
            $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(['offlineCourseId' => $courseId, 'type' => 'sign'], [], 0, PHP_INT_MAX, ['id']);
            $homeworkTaskIds = empty($homeworkTasks) ? [-1] : ArrayToolkit::column($homeworkTasks, 'id');
            $signTasks = $this->getOfflineCourseTaskService()->searchTasks(['offlineCourseId' => $courseId, 'type' => 'homework'], [], 0, PHP_INT_MAX, ['id']);
            $signTaskIds = empty($signTasks) ? [-1] : ArrayToolkit::column($signTasks, 'id');
            $member['offlineCourseTitle'.$courseId] = $courses[$courseId]['title'];
            $member['offlineCourseLength'.$courseId] = $courses[$courseId]['time'];
            $singedTaskNum = $this->getOfflineCourseTaskService()->countTaskResults([
                'offlineCourseId' => $courseId,
                'userId' => $userId,
                'status' => 'finished',
                'taskIds' => $signTaskIds,
            ]);
            $member['offlineCourseSign'.$courseId] = empty($signTasks) ? 0 : sprintf('%.4f', $singedTaskNum / count($signTasks)) * 100 .'%';
            $member['offlineCourseSignStatus'.$courseId] = empty($offlineCourseMembers[$courseId]) ? $this->trans($this->attendStatus['none']) : $this->trans($this->attendStatus[$offlineCourseMembers[$courseId]['status']]);
            $passedNum = $this->getOfflineCourseTaskService()->countTaskResults([
                'offlineCourseId' => $courseId,
                'userId' => $userId,
                'status' => 'finished',
                'taskIds' => $homeworkTaskIds,
            ]);
            $member['offlineCourseResult'.$courseId] = empty($homeworkTasks) ? 0 : sprintf('%.4f', $passedNum / count($homeworkTasks)) * 100 .'%';
        }

        return $member;
    }

    protected function appendOnlineExamData($member, $examIds, $userId)
    {
        if ($examIds == [-1]) {
            return $member;
        }
        if (!$this->isPluginInstalled('Exam')) {
            return $member;
        }

        $onlineExams = $this->getExamService()->findExamsByIds($examIds);
        foreach ($onlineExams as $exam) {
            $member['onlineExamTitle'.$exam['id']] = $exam['name'];
            $bestScoreResult = $this->getExamService()->getUserBestExamResult($userId, $exam['id']);
            $member['onlineExamBestScore'.$exam['id']] = $bestScoreResult['score'];
            $member['onlineExamResult'.$exam['id']] = $bestScoreResult['passStatus'];
        }

        return $member;
    }

    protected function appendOnlineCourseData($member, $onlineCourseIds, $userId)
    {
        if ($onlineCourseIds == [-1]) {
            return $member;
        }

        $onlineCourses = $this->getCourseService()->findCoursesByIds($onlineCourseIds);
        $courseMembers = $this->getCourseMemberService()->findCoursesByStudentIdAndCourseIds($userId, $onlineCourseIds);
        $courseMembers = ArrayToolkit::index($courseMembers, 'courseId');
        foreach ($onlineCourses as $course) {
            $member['onlineCourseTitle'.$course['id']] = $course['title'];
            $courseLenth = $this->sumOnlineCourseTotalTime([$course['id']]);
            $member['onlineCourseLength'.$course['id']] = sprintf('%.2f', $courseLenth / 3600);
            $learnTime = $this->getTaskResultService()->sumLearnTimeByCourseIdAndUserId($course['id'], $userId);
            $member['onlineCourseLearnTime'.$course['id']] = sprintf('%.2f', $learnTime / 3600);
            $finishTime = empty($courseMembers[$course['id']]) ? 0 : $courseMembers[$course['id']]['finishedTime'];
            $member['onlineCourseFinishTime'.$course['id']] = (0 == $finishTime) ? '-' : date('m/d/Y H:i', $finishTime);
        }

        return $member;
    }

    protected function getProjectPlanOrgUserIds($projectPlanId, $orgIds)
    {
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $projectPlanId], [], 0, PHP_INT_MAX, ['id', 'userId']);

        $userIds = ArrayToolkit::column($members, 'userId');

        $users = $this->getUserOrgService()->searchUserOrgs(
            ['orgIds' => $orgIds, 'userIds' => $userIds],
            [],
            0,
            PHP_INT_MAX
        );

        return ArrayToolkit::column($users, 'userId');
    }

    protected function prepareStudyDataListSearchConditions($conditions, $projectPlanId)
    {
        if (!empty($conditions['orgIds'])) {
            $conditions['orgIds'] = empty($orgIds) ? explode(',', $conditions['orgIds']) : $orgIds;

            $orgUserIds = $this->getAssignmentOrgUserIds($conditions['projectPlanId'], $conditions['orgIds']);
            $conditions['userIds'] = $orgUserIds;
        }

        $orgUserIds = $this->getProjectPlanOrgUserIds($projectPlanId, $conditions['orgIds']);
        $conditions['userIds'] = $orgUserIds;

        if (!empty($conditions['postId'])) {
            $users = $this->getUserService()->searchUsers(
                ['postId' => $conditions['postId']],
                ['id' => 'DESC'],
                0,
                PHP_INT_MAX
            );

            $userIds = ArrayToolkit::column($users, 'id');
            if (empty($conditions['userIds']) || empty($userIds)) {
                $conditions['userIds'] = [];
            } else {
                $conditions['userIds'] = array_intersect($conditions['userIds'], $userIds);
            }
        }

        if (!empty($conditions['username'])) {
            $userIds = $this->getUserService()->findUserIdsByNickNameOrTrueName($conditions['username']);
            if (empty($conditions['userIds']) || empty($userIds)) {
                $conditions['userIds'] = [];
            } else {
                $conditions['userIds'] = array_intersect($conditions['userIds'], $userIds);
            }
            unset($conditions['username']);
        }

        $conditions['userIds'] = empty($conditions['userIds']) ? [-1] : $conditions['userIds'];

        return $conditions;
    }

    protected function getAssignmentOrgUserIds($projectPlanId, $orgIds)
    {
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $projectPlanId], [], 0, PHP_INT_MAX, ['id', 'userId']);

        $userIds = ArrayToolkit::column($members, 'userId');

        $users = $this->getUserOrgService()->searchUserOrgs(
            ['orgIds' => $orgIds, 'userIds' => $userIds],
            [],
            0,
            PHP_INT_MAX
        );

        return ArrayToolkit::column($users, 'userId');
    }

    protected function sumOnlineCourseTotalTime($onlineCourseIds)
    {
        $tasks = $this->getTaskService()->searchTasks(
            ['courseIds' => $onlineCourseIds, 'types' => ['video', 'audio', 'live']],
            [],
            0,
            PHP_INT_MAX,
            ['type', 'length']
        );
        $secondsMaps = [
            'video' => 1,
            'audio' => 1,
            'live' => 60,
        ];
        $totalTime = 0;
        foreach ($tasks as $task) {
            $totalTime += $task['length'] * $secondsMaps[$task['type']];
        }

        return $totalTime;
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\Activity\Service\Impl\ActivityServiceImpl
     */
    protected function getActivityService()
    {
        return $this->createService('Activity:ActivityService');
    }

    /**
     * @return ExamService
     */
    protected function getExamService()
    {
        return $this->createService('ExamPlugin:Exam:ExamService');
    }

    /**
     * @return TaskService
     */
    protected function getTaskService()
    {
        return $this->createService('Task:TaskService');
    }

    /**
     * @return TaskResultService
     */
    protected function getTaskResultService()
    {
        return $this->createService('CorporateTrainingBundle:Task:TaskResultService');
    }

    protected function getTestPaperService()
    {
        return $this->createService('ExamPlugin:TestPaper:TestPaperService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\Course\Service\Impl\MemberServiceImpl
     */
    protected function getCourseMemberService()
    {
        return $this->createService('Course:MemberService');
    }

    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return OfflineExamService
     */
    protected function getOfflineExamService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:OfflineExamService');
    }

    protected function getOfflineExamMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }
}
