<?php

namespace CorporateTrainingBundle\Biz\Exporter;

use AppBundle\Common\ArrayToolkit;
use CorporateTrainingBundle\Biz\OfflineExam\Service\MemberService;
use CorporateTrainingBundle\Biz\OfflineExam\Service\OfflineExamService;
use CorporateTrainingBundle\Biz\Task\Service\TaskResultService;
use CorporateTrainingBundle\Biz\Task\Service\TaskService;
use CorporateTrainingBundle\Biz\UserDailyLearnRecord\Service\UserDailyLearnRecordService;
use ExamPlugin\Biz\Exam\Service\ExamService;

class ProjectPlanMemberStudyRecordExporter extends AbstractExporter
{
    protected $attendStatus = [
        'none' => 'offline_course.attend_status.none',
        'attended' => 'offline_course.attend_status.attended',
        'partial' => 'offline_course.attend_status.partial',
        'absenteeism' => 'offline_course.attend_status.absenteeism',
        'leave' => 'offline_course.attend_status.leave',
    ];
    protected $homeworkStatus = [
        'none' => 'project_plan.status.not_submitted',
        'unpassed' => 'project_plan.status.unpass',
        'finished' => 'project_plan.status.passed',
        'submitted' => 'project_plan.status.not_review',
    ];

    protected $exportFileName;

    public function canExport($parameters)
    {
        return $this->biz['user']->hasPermission('admin_project_plan_data_export') || $this->biz['user']['id'] == $parameters['userId'];
    }

    public function getExportFileName()
    {
        return $this->exportFileName;
    }

    protected function getSortedHeadingRow($parameters)
    {
    }

    protected function buildExportData($parameters)
    {
    }

    public function writeToExcel($parameters)
    {
        $excel = new Excel();
        $PHPExcel = $excel->createPHPExcelObject();
        $PHPExcel->setActiveSheetIndex(0);
        $sheet = $PHPExcel->getActiveSheet();

        $this->writeBaseInfo($parameters, $sheet);

        $projectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanId(
            $parameters['projectPlanId']
        );
        $projectPlanItems = ArrayToolkit::group($projectPlanItems, 'targetType');
        $this->writeOnlineCourse($parameters['userId'], $projectPlanItems, $sheet);
        $this->writeOfflineCourse($parameters['userId'], $projectPlanItems, $sheet);
        $this->writeOnlineExam($parameters['userId'], $projectPlanItems, $sheet);
        $this->writeOfflineExam($parameters['userId'], $projectPlanItems, $sheet);

        $this->setExportFileName($parameters);

        return $excel->createWriter($PHPExcel);
    }

    protected function writeBaseInfo($parameters, \PHPExcel_Worksheet $sheet)
    {
        $user = $this->getUserService()->getUser($parameters['userId']);
        $userProfile = $this->getUserService()->getUserProfile($parameters['userId']);
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($parameters['projectPlanId']);
        $baseInfo = [
            [
                'name' => $this->trans('student.user_name'),
                'value' => $user['nickname'],
                'coordinate' => ['col' => 0, 'row' => 1],
            ],
            [
                'name' => $this->trans('student.profile.truename'),
                'value' => $userProfile['truename'],
                'coordinate' => ['col' => 0, 'row' => 2],
            ],
            [
                'name' => $this->trans('project_plan.name'),
                'value' => $projectPlan['name'],
                'coordinate' => ['col' => 0, 'row' => 3],
            ],
            [
                'name' => $this->trans('project_plan.export.start_time'),
                'value' => date('m/d/Y', $projectPlan['startTime']),
                'coordinate' => ['col' => 0, 'row' => 4],
            ],
            [
                'name' => $this->trans('project_plan.export.end_time'),
                'value' => date('m/d/Y', $projectPlan['endTime']),
                'coordinate' => ['col' => 0, 'row' => 5],
            ],
        ];

        foreach ($baseInfo as $field) {
            $nameCoordinate = \PHPExcel_Cell::stringFromColumnIndex($field['coordinate']['col']).$field['coordinate']['row'];
            $valueCoordinate = \PHPExcel_Cell::stringFromColumnIndex($field['coordinate']['col'] + 1).$field['coordinate']['row'];
            $sheet->getCell($nameCoordinate)->setValue($field['name'])->getStyle()->getFont()->setBold(true);
            $sheet->getCell($valueCoordinate)->setValue($this->filter_Emoji($field['value']));
        }
    }

    protected function writeOnlineCourse($userId, $projectPlanItems, \PHPExcel_Worksheet $sheet)
    {
        if (empty($projectPlanItems['course'])) {
            return;
        }
        $onlineCourseIds = ArrayToolkit::column($projectPlanItems['course'], 'targetId');
        $onlineCourses = $this->getCourseService()->findCoursesByIds($onlineCourseIds);
        $onlineCourseTotalTime = $this->sumOnlineCourseTotalTime($onlineCourseIds);
        foreach ($onlineCourses as $onlineCourse) {
            $startRow = (int) $sheet->getHighestDataRow() + 2;
            $onlineCourseLearnTime = $this->getTaskResultService()->sumLearnTimeByCourseIdAndUserId(
                $onlineCourse['id'],
                $userId
            );
            $learnRecords = $this->getUserDailyLearnRecordService()->findRecordsByUserIdAndCourseId(
                $userId,
                $onlineCourse['id']
            );
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.online_course'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($onlineCourse['title']);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.online_course.total_time'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(substr(sprintf('%.3f', $onlineCourseTotalTime[$onlineCourse['id']] / 3600), 0, -1).'小时');
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.online_course.user_total_learn_time'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(substr(sprintf('%.3f', $onlineCourseLearnTime / 3600), 0, -1).'小时');
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.online_course.daily_start_learn_time'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($this->trans('project_plan.export.online_course.daily_learn_time'))->getStyle()->getFont()->setBold(true);
            foreach ($learnRecords as $learnRecord) {
                $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue(date('m/d/Y H:i', $learnRecord['createdTime']));
                $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(substr(sprintf('%.3f', $learnRecord['learnTime'] / 3600), 0, -1).'小时');
            }
        }
    }

    protected function writeOfflineCourse($userId, $projectPlanItems, \PHPExcel_Worksheet $sheet)
    {
        if (empty($projectPlanItems['offline_course'])) {
            return;
        }
        $courseIds = ArrayToolkit::column($projectPlanItems['offline_course'], 'targetId');
        $courses = $this->getOfflineCourseService()->findOfflineCoursesByIds($courseIds);
        $courses = ArrayToolkit::index($courses, 'id');
        $offlineCourseMembers = $this->getOfflineCourseMemberService()->searchMembers(
            ['offlineCourseIds' => $courseIds, 'userId' => $userId],
            [],
            0,
            PHP_INT_MAX,
            ['id', 'offlineCourseId', 'status']
        );
        $offlineCourseMembers = ArrayToolkit::index($offlineCourseMembers, 'offlineCourseId');
        foreach ($courses as $course) {
            $startRow = (int) $sheet->getHighestDataRow() + 2;
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.offline_course'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($course['title']);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.offline_course.time'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($course['time'].'小时');
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('offline_course.attendance_status'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(empty($offlineCourseMembers[$course['id']]) ? $this->trans($this->attendStatus['none']) : $this->trans($this->attendStatus[$offlineCourseMembers[$course['id']]['status']]));
            $startRow = $this->writeOfflineCourseSignTasks($userId, $course, $sheet, $startRow);
            $this->writeOfflineCourseHomeworkTasks($userId, $course, $sheet, $startRow);
        }
    }

    protected function writeOfflineCourseSignTasks($userId, $course, \PHPExcel_Worksheet $sheet, $startRow)
    {
        $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.offline_course.attendance'))->getStyle()->getFont()->setBold(true);
        $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($this->trans('project_plan.export.online_course.daily_start_learn_time'));
        $signTasks = $this->getOfflineCourseTaskService()->searchTasks(
        ['offlineCourseId' => $course['id'], 'type' => 'sign'],
        ['seq' => 'ASC'],
        0,
        PHP_INT_MAX,
        ['id']
        );
        $signTaskIds = empty($signTasks) ? [-1] : ArrayToolkit::column($signTasks, 'id');
        $results = $this->getOfflineCourseTaskService()->searchTaskResults(['taskIds' => $signTaskIds, 'userId' => $userId], [], 0, count($signTaskIds));
        $results = ArrayToolkit::index($results, 'taskId');
        foreach ($signTasks as $key => $signTask) {
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('offline_course.sign_num', array('%seq%' => $key + 1)));
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(
                empty($results[$signTask['id']]) ? '--' : date('m/d/Y H:i', $results[$signTask['id']]['finishedTime'])
            );
        }

        return $startRow;
    }

    protected function writeOfflineCourseHomeworkTasks($userId, $course, \PHPExcel_Worksheet $sheet, $startRow)
    {
        $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(
            ['offlineCourseId' => $course['id'], 'type' => 'homework'],
            ['seq' => 'ASC'],
            0,
            PHP_INT_MAX,
            ['id']
        );
        if (empty($homeworkTasks)) {
            return  $startRow;
        }
        $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('offline_course.tasks.homework'))->getStyle()->getFont()->setBold(true);
        $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($this->trans('offline_course.homework_list.review_result'));
        $homeworkTaskIds = empty($homeworkTasks) ? [-1] : ArrayToolkit::column($homeworkTasks, 'id');
        $results = $this->getOfflineCourseTaskService()->searchTaskResults(['taskIds' => $homeworkTaskIds, 'userId' => $userId], [], 0, count($homeworkTaskIds));
        $results = ArrayToolkit::index($results, 'taskId');
        foreach ($homeworkTasks as $key => $homeworkTask) {
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('offline_course.homework_num', array('%seq%' => $key + 1)));
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(
                empty($results[$homeworkTask['id']]) ? '' : $this->trans($this->homeworkStatus[$results[$homeworkTask['id']]['status']])
            );
        }

        return $startRow;
    }

    protected function writeOnlineExam($userId, $projectPlanItems, \PHPExcel_Worksheet $sheet)
    {
        if (empty($projectPlanItems['exam'])) {
            return;
        }
        if (!$this->isPluginInstalled('Exam')) {
            return;
        }
        $onlineExamIds = ArrayToolkit::column($projectPlanItems['exam'], 'targetId');
        $onlineExams = $this->getExamService()->findExamsByIds($onlineExamIds);
        foreach ($onlineExams as $onlineExam) {
            $startRow = (int) $sheet->getHighestDataRow() + 2;
            $examResults = $this->getExamService()->findExamResultsByExamIdAndUserId($onlineExam['id'], $userId);
            $scores = ArrayToolkit::column($examResults, 'score');
            rsort($scores);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.online_exam'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($onlineExam['name']);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.exam.best_score'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue(empty($scores) ? 0 : $scores[0]);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.exam.start_time'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow)->setValue($this->trans('project_plan.export.exam.score'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(2).$startRow)->setValue($this->trans('project_plan.export.exam.pass_status'))->getStyle()->getFont()->setBold(true);
            foreach ($examResults as $examResult) {
                $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().++$startRow)->setValue(date('m/d/Y H:i', $examResult['beginTime']));
                $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow)->setValue(empty($examResult['score']) ? '-' : $examResult['score']);
                $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(2).$startRow)->setValue(empty($examResult['status']) ? '-' : $this->trans('project_plan.export.exam.pass_status.'.$examResult['passStatus'])
                );
            }
        }
    }

    protected function writeOfflineExam($userId, $projectPlanItems, \PHPExcel_Worksheet $sheet)
    {
        if (empty($projectPlanItems['offline_exam'])) {
            return;
        }
        $offlineExamIds = ArrayToolkit::column($projectPlanItems['offline_exam'], 'targetId');
        $offlineExams = $this->getOfflineExamService()->findOfflineExamByIds($offlineExamIds);
        foreach ($offlineExams as $offlineExam) {
            $startRow = (int) $sheet->getHighestDataRow() + 2;
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.offline_exam'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow++)->setValue($offlineExam['title']);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue($this->trans('project_plan.export.exam.start_time'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow)->setValue($this->trans('project_plan.export.exam.score'))->getStyle()->getFont()->setBold(true);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(2).$startRow++)->setValue($this->trans('project_plan.export.exam.pass_status'))->getStyle()->getFont()->setBold(true);
            $member = $this->getOfflineExamMemberService()->getMemberByOfflineExamIdAndUserId(
                $offlineExam['id'],
                $userId
            );
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex().$startRow)->setValue(date('m/d/Y H:i', $offlineExam['startTime']));
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(1).$startRow)->setValue(empty($member['score']) ? '-' : $member['score']);
            $sheet->getCell(\PHPExcel_Cell::stringFromColumnIndex(2).$startRow)->setValue(empty($member['status']) ? '-' : $this->trans('project_plan.export.exam.pass_status.'.$member['status']));
        }
    }

    protected function setExportFileName($parameters)
    {
        $user = $this->getUserService()->getUser($parameters['userId']);
        $userProfile = $this->getUserService()->getUserProfile($parameters['userId']);
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($parameters['projectPlanId']);
        $today = date('Y-m-d');

        $this->exportFileName = "{$userProfile['truename']}+{$user['nickname']}+{$today}+{$projectPlan['name']}.xls";
    }

    protected function sumOnlineCourseTotalTime($onlineCourseIds)
    {
        $tasks = $this->getTaskService()->searchTasks(
            ['courseIds' => $onlineCourseIds, 'types' => ['video', 'audio', 'live']],
            [],
            0,
            PHP_INT_MAX,
            ['courseId', 'type', 'length']
        );
        $tasks = ArrayToolkit::group($tasks, 'courseId');
        $secondsMaps = [
            'video' => 1,
            'audio' => 1,
            'live' => 60,
        ];
        $totalTimes = [];
        foreach ($onlineCourseIds as $onlineCourseId) {
            $totalTimes[$onlineCourseId] = 0;
            if (empty($tasks[$onlineCourseId])) {
                continue;
            }
            $totalTime = 0;
            foreach ($tasks[$onlineCourseId] as $task) {
                $totalTime += $task['length'] * $secondsMaps[$task['type']];
            }
            $totalTimes[$onlineCourseId] = $totalTime;
        }

        return $totalTimes;
    }

    /**
     * @return TaskService
     */
    protected function getTaskService()
    {
        return $this->createService('CorporateTrainingBundle:Task:TaskService');
    }

    /**
     * @return TaskResultService
     */
    protected function getTaskResultService()
    {
        return $this->createService('CorporateTrainingBundle:Task:TaskResultService');
    }

    /**
     * @return UserDailyLearnRecordService
     */
    protected function getUserDailyLearnRecordService()
    {
        return $this->createService('CorporateTrainingBundle:UserDailyLearnRecord:UserDailyLearnRecordService');
    }

    /**
     * @return ExamService
     */
    protected function getExamService()
    {
        return $this->createService('ExamPlugin:Exam:ExamService');
    }

    /**
     * @return OfflineExamService
     */
    protected function getOfflineExamService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:OfflineExamService');
    }

    /**
     * @return MemberService
     */
    protected function getOfflineExamMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }
}
