<?php

namespace CorporateTrainingBundle\Biz\Exporter;

use AppBundle\Common\ArrayToolkit;
use OfflineCourseProPlugin\Biz\OfflineClass\Service\MemberService;

class TeacherOfflineCourseProfileExporter extends AbstractExporter
{
    public function canExport($parameters)
    {
        return true;
    }

    public function getExportFileName()
    {
        return ' teacher_offline_course_profile.xls';
    }

    public function getSortedHeadingRow($parameters)
    {
        $row = [];
        if (!empty($parameters['choices'])) {
            $choices = $parameters['choices'];

            if (in_array('课程名', $choices)) {
                $row[] = ['code' => 'title', 'title' => $this->trans('offline_course.course_title')];
                $row[] = ['code' => 'belong', 'title' => $this->trans('admin.offline_course.belong')];
            }
            if (in_array('课程时间', $choices)) {
                $row[] = ['code' => 'createdTime', 'title' => $this->trans('admin.teacher.teacher_archives.course.time')];
            }
            if (in_array('上课地点', $choices)) {
                $row[] = ['code' => 'place', 'title' => $this->trans('admin.teacher.teacher_archives.course.place')];
            }
            if (in_array('学员数', $choices)) {
                $row[] = ['code' => 'studentNum', 'title' => $this->trans('teaching.project_plan.course.student_num')];
            }
            if (in_array('出勤率', $choices)) {
                $row[] = ['code' => 'attend', 'title' => $this->trans('offline_activity.statistics.attendance_rate')];
            }
            if (in_array('作业通过率', $choices)) {
                $row[] = ['code' => 'passHomework', 'title' => $this->trans('teaching_record.course.passing_rate')];
            }
            if (in_array('讲师评价', $choices)) {
                $row[] = ['code' => 'courseSurveyScore', 'title' => $this->trans('admin.teacher.teacher_archives.teacher.survey')];
            }
        }

        return $row;
    }

    public function buildExportData($parameters)
    {
        $memberData = [];
        $conditions = json_decode($parameters['conditions'], true);
        $user = $this->getUserService()->getUser($conditions['userId']);
        unset($conditions['userId']);
        if (!empty($parameters['choices'])) {
            $offlineCourses = $this->getOfflineCourseService()->searchOfflineCourses(
                $conditions,
                ['createdTime' => 'DESC'],
                0,
                PHP_INT_MAX
            );

            foreach ($offlineCourses as &$offlineCourse) {
                if ($parameters['surveyPlugin']) {
                    $offlineCourse['surveyScore'] = $this->getSurveyResultService()->getOfflineCourseSurveyScoreByCourseId($offlineCourse['id']);
                }
                $offlineCourse = $this->buildOfflineCourseStatistic($offlineCourse);

                $offlineCourse['startTimeCustom'] = empty($offlineCourse['startTime']) ? '--' : date('Y-m-d', $offlineCourse['startTime']);
                $offlineCourse['endTimeCustom'] = empty($offlineCourse['endTime']) ? '--' : date('Y-m-d', $offlineCourse['endTime']);

                $memberData[] = [
                    'title' => $offlineCourse['title'],
                    'belong' => 'offlineClass' == $offlineCourse['targetType'] ? '线下课程开班' : '培训项目',
                    'createdTime' => $this->isSameDay($offlineCourse['startTime'], $offlineCourse['endTime']) ? $offlineCourse['startTimeCustom'] : $offlineCourse['startTimeCustom'].'/'.$offlineCourse['endTimeCustom'],
                    'place' => empty($offlineCourse['place']) ? '' : $offlineCourse['place'],
                    'studentNum' => empty($offlineCourse['memberCount']) ? 0 : $offlineCourse['memberCount'],
                    'attend' => empty($offlineCourse['attendTaskCount']) ? 0 : floor($offlineCourse['attendCount'] * 100 / $offlineCourse['attendTaskCount']),
                    'passHomework' => empty($offlineCourse['hasHomeTaskCount']) ? 0 : floor($offlineCourse['passHomeworkCount'] * 100 / $offlineCourse['hasHomeTaskCount']),
                    'courseSurveyScore' => empty($offlineCourse['surveyScore']) ? 0 : $offlineCourse['surveyScore'],
                ];
            }
        }

        $exportData[] = [
            'sheetName' => $user['nickname'].$this->trans('admin.teacher.list.exporter.offline_course_num'),
            'data' => $memberData,
        ];

        return $exportData;
    }

    protected function isSameDay($day, $otherDay)
    {
        $day = date('Y-m-d', $day);
        $otherDay = date('Y-m-d', $otherDay);
        $day = getdate(strtotime($day));
        $otherDay = getdate(strtotime($otherDay));

        if (($day['year'] === $otherDay['year']) && ($day['yday'] === $otherDay['yday'])) {
            return true;
        } else {
            return false;
        }
    }

    protected function buildOfflineCourseStatistic($course)
    {
        $members = [];
        if ('projectPlan' == $course['targetType']) {
            $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(
                ['projectPlanId' => $course['targetId']],
                [],
                0,
                PHP_INT_MAX
            );
        }

        if ('offlineClass' == $course['targetType']) {
            $members = $this->getOfflineClassMemberService()->searchMembers(
                ['offlineClassId' => $course['targetId']],
                [],
                0,
                PHP_INT_MAX
            );
        }
        $course['memberCount'] = count($members);
        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');

        $userOfflineCoursesStatisticData = $this->getOfflineCourseService()->getUserOfflineCoursesStatisticData($userIds, [$course['id']]);

        $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(
            ['offlineCourseId' => $course['id'], 'type' => 'homework'],
            [],
            0,
            PHP_INT_MAX,
            ['id']
        );

        $course['hasHomeTaskCount'] = count($homeworkTasks) * $course['memberCount'];
        $course['passHomeworkCount'] = $userOfflineCoursesStatisticData['offlineCoursePassedHomeworkNum'];
        $course['attendTaskCount'] = $course['memberCount'];
        $course['attendCount'] = $userOfflineCoursesStatisticData['offlineCourseAttendNum'];

        return $course;
    }

    /**
     * @return MemberService
     */
    protected function getOfflineClassMemberService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineClass:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    protected function getSurveyResultService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyResultService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }
}
