<?php

namespace CorporateTrainingBundle\Biz\NotificationCenter\ResourceNotification;

use CorporateTrainingBundle\Biz\OfflineActivity\Service\OfflineActivityService;

class OfflineActivity extends BaseResourceNotification
{
    /**
     * 管理员直接添加
     *
     * @param $offlineActivityId
     * @param $userIds
     * @param $dingTalkNotificationSetting
     */
    public function onAdminSelectUsers($offlineActivityId, $userIds, $dingTalkNotificationSetting)
    {
        $activity = $this->getOfflineActivityService()->getOfflineActivity($offlineActivityId);
        if (empty($userIds) || empty($activity)) {
            return;
        }
        $this->sendEnrollSuccessSmsNotification($userIds, "线下活动《{$activity['title']}》");

        if (!$this->checkDingTalkSetting('offline_activity_add_member') || !$dingTalkNotificationSetting) {
            return;
        }
        list($to, $content) = $this->buildNotificationData($activity, $userIds, 'offline_activity_add_member');
        $this->biz->offsetGet('notification_default')->send(['dingtalk'], $to, $content);
    }

    /**
     * 线下活动通过报名审核
     *
     * @param $offlineActivityId
     * @param $userIds
     */
    public function onOfflineActivityBatchPass($offlineActivityId, $userIds)
    {
        $activity = $this->getOfflineActivityService()->getOfflineActivity($offlineActivityId);
        if (empty($userIds) || empty($activity)) {
            return;
        }
        $this->sendEnrollSuccessSmsNotification($userIds, "线下活动《{$activity['title']}》");

        if ($this->checkEmailSetting('enroll')) {
            $types[] = 'email';
        }

        if ($this->checkDingTalkSetting('offline_activity_apply')) {
            $types[] = 'dingtalk';
        }

        if (empty($types)) {
            return;
        }
        list($to, $content) = $this->buildNotificationData($activity, $userIds, 'offline_activity_apply');
        $this->biz->offsetGet('notification_default')->send($types, $to, $content);
    }

    /**
     * 线下活动拒绝报名审核
     *
     * @param $offlineActivityId
     * @param $userIds
     * @param string $rejectedReason
     */
    public function onOfflineActivityBatchReject($offlineActivityId, $userIds, $rejectedReason = '')
    {
        $activity = $this->getOfflineActivityService()->getOfflineActivity($offlineActivityId);
        if (empty($userIds) || empty($activity)) {
            return;
        }
        $this->sendEnrollRejectSmsNotification($userIds, "线下活动《{$activity['title']}》", $rejectedReason);

        if ($this->checkEmailSetting('enroll')) {
            $types[] = 'email';
        }
        if ($this->checkDingTalkSetting('offline_activity_apply')) {
            $types[] = 'dingtalk';
        }

        if (empty($types)) {
            return;
        }

        list($to, $content) = $this->buildNotificationData($activity, $userIds, 'offline_activity_approve_reject');
        $content['params']['rejectedReason'] = $rejectedReason;
        $this->biz->offsetGet('notification_default')->send($types, $to, $content);
    }

    /**
     * 活动报名审核进入补位
     *
     * @param $offlineActivityId
     * @param $userIds
     * @param string $type
     */
    public function onEnrollToQueueList($offlineActivityId, $userIds, $type = 'submit')
    {
        if ($this->checkEmailSetting('enroll')) {
            $types[] = 'email';
        }

        if ($this->checkDingTalkSetting('offline_activity_add_member')) {
            $types[] = 'dingtalk';
        }
        $activity = $this->getOfflineActivityService()->getOfflineActivity($offlineActivityId);
        if (empty($userIds) || empty($activity)) {
            return;
        }

        list($to, $content) = $this->buildNotificationData($activity, $userIds, 'resource_enroll_queue');

        $params = [
            'resourceType' => '线下活动',
            'title' => '活动名称',
            'targetType' => 'offlineActivity',
            'resourceTitle' => $activity['title'],
            'content' => 'review' == $type ? '通过审核，' : '',
            'type' => $type,
        ];

        $content['params'] = array_merge($content['params'], $params);

        $smsContent = "线下活动《{$activity['title']}》".$params['content'];
        foreach ($userIds as $userId) {
            $queueNum = $this->getEnrollRecordService()->getUserQueueCountByTargetIdAndTargetType($userId, $offlineActivityId, 'offlineActivity');
            $this->sendEnrollQueueSmsNotification($userId, $smsContent, $queueNum);
            if (empty($types)) {
                continue;
            }
            $to['userIds'] = [$userId];
            $content['params']['queueNum'] = $queueNum;
            $this->biz->offsetGet('notification_default')->send($types, $to, $content);
        }
    }

    protected function buildNotificationData($offlineActivity, $userIds, $template)
    {
        global $kernel;
        $url = $kernel->getContainer()->get('router')->generate('offline_activity_detail', ['id' => $offlineActivity['id']], true);

        $to = [
            'type' => 'user',
            'userIds' => $userIds,
            'startNum' => 0,
            'perPageNum' => 20,
        ];

        $content = [
            'template' => $template,
            'params' => [
                'batch' => $template.$offlineActivity['id'].time(),
                'targetId' => $offlineActivity['id'],
                'offlineActivityTitle' => $offlineActivity['title'],
                'url' => $url,
                'imagePath' => empty($offlineActivity['cover']['large']) ? '' : $offlineActivity['cover']['large'],
            ],
        ];

        return [$to, $content];
    }

    /**
     * @return OfflineActivityService
     */
    protected function getOfflineActivityService()
    {
        return $this->biz->service('CorporateTrainingBundle:OfflineActivity:OfflineActivityService');
    }
}
