<?php

namespace CorporateTrainingBundle\Biz\NotificationCenter\ResourceNotification;

class OfflineExam extends BaseResourceNotification
{
    public function onOfflineExamCreate($exam)
    {
        $this->deleteJobByExam($exam);
        $this->registerSendNotificationJob($exam);
    }

    public function onOfflineExamDelete($exam)
    {
        $this->deleteJobByExam($exam);
    }

    public function onOfflineExamMarkPass($member)
    {
        $this->sendNotification($member);
    }

    public function onOfflineExamMarkUnPass($member)
    {
        $this->sendNotification($member);
    }

    protected function deleteJobByExam($exam)
    {
        $this->getSchedulerService()->deleteJobByName('OfflineExamMailNotificationJob_'.$exam['id']);
        $this->getSchedulerService()->deleteJobByName('OfflineExamOneDayDingtalkRemindNotificationJob_'.$exam['id']);
        $this->getSchedulerService()->deleteJobByName('OfflineExamOneHourDingtalkRemindNotificationJob_'.$exam['id']);
    }

    protected function registerSendNotificationJob($exam)
    {
        if ($this->checkEmailSetting('project_plan_progress')) {
            $this->registerMailNotificationJob($exam);
        }

        $this->registerOneDayDingtalkRemindJob($exam);
        $this->registerOneHourDingtalkRemindJob($exam);
    }

    protected function registerMailNotificationJob($exam)
    {
        $currentTime = time();
        $executeTime = strtotime('-1 day 08:00:00', $exam['startTime']);
        $todayExecuteTime = strtotime('08:00:00', $currentTime);

        if ($executeTime <= $todayExecuteTime && $currentTime > $todayExecuteTime) {
            return;
        }

        $job = [
            'name' => 'OfflineExamRemindMailNotificationJob_'.$exam['id'],
            'source' => 'TRAININGMAIN',
            'expression' => $executeTime,
            'class' => 'CorporateTrainingBundle\Biz\NotificationCenter\Job\OfflineExamRemindMailNotificationJob',
            'args' => [
                'exam' => $exam,
            ],
            'misfire_policy' => 'executing',
        ];

        $this->getSchedulerService()->register($job);
    }

    protected function registerOneDayDingtalkRemindJob($exam)
    {
        $currentTime = time();
        $executeTime = strtotime('-1 day 08:00:00', $exam['startTime']);
        $todayExecuteTime = strtotime('08:00:00', $currentTime);

        if ($executeTime <= $todayExecuteTime && $currentTime > $todayExecuteTime) {
            return;
        }

        $job = [
            'name' => 'OfflineExamOneDayDingtalkRemindNotificationJob_'.$exam['id'],
            'source' => 'TRAININGMAIN',
            'expression' => $executeTime,
            'class' => 'CorporateTrainingBundle\Biz\NotificationCenter\Job\OfflineExamDingTalkNotificationJob',
            'args' => [
                'notificationType' => 'offline_exam_day_remind',
                'exam' => $exam,
                'template' => 'offline_exam_remind',
                'url' => $this->buildDingTalkLinkUrl($exam),
            ],
            'misfire_policy' => 'executing',
        ];

        $this->getSchedulerService()->register($job);
    }

    protected function registerOneHourDingtalkRemindJob($exam)
    {
        if ($exam['startTime'] - time() < 1 * 60 * 60) {
            return;
        }

        $job = [
            'name' => 'OfflineExamOneHourDingtalkRemindNotificationJob_'.$exam['id'],
            'source' => 'TRAININGMAIN',
            'expression' => $exam['startTime'] - 60 * 60,
            'class' => 'CorporateTrainingBundle\Biz\NotificationCenter\Job\OfflineExamDingTalkNotificationJob',
            'args' => [
                'notificationType' => 'offline_exam_hour_remind',
                'exam' => $exam,
                'template' => 'offline_exam_remind',
                'url' => $this->buildDingTalkLinkUrl($exam),
            ],
            'misfire_policy' => 'executing',
        ];

        $this->getSchedulerService()->register($job);
    }

    protected function buildDingTalkLinkUrl($exam)
    {
        if ('projectPlan' == $exam['targetType']) {
            $projectPlan = $this->getProjectPlanService()->getProjectPlan($exam['targetId']);

            return $this->generateUrl('project_plan_study_record_user_detail', ['projectPlanId' => $projectPlan['id']], true);
        }
        if ('offlineClass' == $exam['targetType']) {
            $offlineClass = $this->getOfflineClassService()->getClass($exam['targetId']);

            return $this->generateUrl('offline_class_study_record_user_detail', ['offlineClassId' => $offlineClass['id']], true);
        }
    }

    protected function sendNotification($member)
    {
        if ('none' == $member['status']) {
            return;
        }

        $types = [];

        if ($this->checkEmailSetting('exam')) {
            $types[] = 'email';
        }

        if ($this->checkDingTalkSetting('offline_exam_result')) {
            $types[] = 'dingtalk';
        }

        if (empty($types)) {
            return;
        }

        foreach ($types as $type) {
            list($to, $content) = $this->buildNotificationData($member, $type);
            $this->biz->offsetGet('notification_'.$type)->send($to, $content);
        }
    }

    protected function buildNotificationData($member, $type)
    {
        $exam = $this->getOfflineExamService()->getOfflineExam($member['offlineExamId']);

        $to = [
            'type' => 'user',
            'userIds' => [$member['userId']],
            'startNum' => 0,
            'perPageNum' => 20,
        ];

        $template = ('email' === $type) ? 'exam_result' : 'offline_exam_result';

        $content = [
            'template' => $template,
            'params' => [
                'targetId' => $exam['id'],
                'batch' => 'offline_exam_result'.$exam['id'].time(),
                'url' => $this->buildDingTalkLinkUrl($exam),
                'examName' => $exam['title'],
                'score' => $member['score'],
                'totalScore' => $exam['score'],
                'status' => ('passed' == $member['status']) ? '通过' : '未通过',
            ],
        ];

        return [$to, $content];
    }
}
