<?php

namespace CorporateTrainingBundle\Biz\OfflineActivity\Service\Impl;

use AppBundle\Common\ArrayToolkit;
use Biz\BaseService;
use Biz\Content\Service\FileService;
use Biz\User\Service\UserService;
use Codeages\Biz\Framework\Event\Event;
use CorporateTrainingBundle\Biz\ManagePermission\Service\ManagePermissionOrgService;
use CorporateTrainingBundle\Biz\OfflineActivity\Dao\OfflineActivityDao;
use CorporateTrainingBundle\Biz\OfflineActivity\Service\MemberService;
use CorporateTrainingBundle\Biz\OfflineActivity\Service\OfflineActivityService;

class OfflineActivityServiceImpl extends BaseService implements OfflineActivityService
{
    public function findOfflineActivitiesByIds($ids)
    {
        return $this->getOfflineActivityDao()->findOfflineActivitiesByIds($ids);
    }

    public function createOfflineActivity($offlineActivity)
    {
        $this->validateOfflineActivityFields($offlineActivity);
        $offlineActivity = $this->filterOfflineActivityFields($offlineActivity);
        if (!$this->hasActivityCreateRole()) {
            throw $this->createAccessDeniedException('You have no access to Offline Activity Management');
        }
        $offlineActivity['status'] = 'draft';
        $offlineActivity['creator'] = $this->getCurrentUser()->getId();
        $offlineActivity = $this->filterNull($offlineActivity);

        return $this->getOfflineActivityDao()->create($offlineActivity);
    }

    public function searchOfflineActivities($conditions, $orderBy, $start, $limit)
    {
        $conditions = $this->prepareSearchConditions($conditions);

        return $this->getOfflineActivityDao()->search($conditions, $orderBy, $start, $limit);
    }

    public function countOfflineActivities($conditions)
    {
        $conditions = $this->prepareSearchConditions($conditions);

        return $this->getOfflineActivityDao()->count($conditions);
    }

    public function getOfflineActivity($id)
    {
        return $this->getOfflineActivityDao()->get($id);
    }

    public function changeActivityCover($id, $coverArray)
    {
        if (empty($coverArray)) {
            throw $this->createInvalidArgumentException('Invalid Param: cover');
        }
        $offlineActivity = $this->getOfflineActivity($id);
        $covers = [];
        foreach ($coverArray as $cover) {
            $file = $this->getFileService()->getFile($cover['id']);
            $covers[$cover['type']] = $file['uri'];
        }

        $offlineActivity = $this->getOfflineActivityDao()->update($offlineActivity['id'], ['cover' => $covers]);

        $this->getLogService()->info(
            'offlineActivity',
            'update_picture',
            "更新线下活动《{$offlineActivity['title']}》(#{$offlineActivity['id']})图片",
            $covers
        );

        return $offlineActivity;
    }

    public function updateOfflineActivity($id, $fields)
    {
        if (!ArrayToolkit::requireds($fields, ['title', 'startTime', 'endTime', 'address', 'maxStudentNum', 'enrollmentEndDate', 'enrollmentStartDate', 'orgId', 'orgCode'])) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }

        $offlineActivity = $this->getOfflineActivity($id);
        $fields = $this->fieldFields($fields);
        $fields = $this->filterNull($fields);

        if (isset($fields['summary'])) {
            $fields['summary'] = $this->purifyHtml($fields['summary'], true);
        }

        $offlineActivity = $this->getOfflineActivityDao()->update($offlineActivity['id'], $fields);
        $this->dispatchEvent('offline_activity.update', new Event($offlineActivity));

        $this->getLogService()->info('offlineActivity', 'update', "修改线下活动《{$offlineActivity['title']}》(#{$offlineActivity['id']})");

        return $offlineActivity;
    }

    public function publishOfflineActivity($activityId)
    {
        $canManage = $this->canManageOfflineActivity($activityId);

        if (!$canManage) {
            throw $this->createServiceException('can not publish offline activity');
        }

        $offlineActivity = $this->getOfflineActivityDao()->update($activityId, ['status' => 'published']);
        $this->dispatchEvent('offline_activity.publish', new Event($offlineActivity));

        return $offlineActivity;
    }

    public function closeOfflineActivity($activityId)
    {
        if (!$this->canManageOfflineActivity($activityId)) {
            throw $this->createAccessDeniedException('You can not close offline activity ');
        }

        $offlineActivity = $this->getOfflineActivityDao()->get($activityId);
        if ('published' !== $offlineActivity['status']) {
            throw $this->createAccessDeniedException('$offlineActivity has not bean published');
        }
        $offlineActivity = $this->getOfflineActivityDao()->update($activityId, ['status' => 'closed']);

        $this->dispatchEvent('offline_activity.close', new Event($offlineActivity));

        return $offlineActivity;
    }

    public function deleteOfflineActivity($activityId)
    {
        if (!$this->canManageOfflineActivity($activityId)) {
            throw $this->createAccessDeniedException('You can not delete offline activity ');
        }

        $offlineActivity = $this->getOfflineActivityDao()->get($activityId);
        if (empty($offlineActivity)) {
            throw $this->createAccessDeniedException('$offlineActivity has not existed');
        }

        return $this->getOfflineActivityDao()->delete($activityId);
    }

    public function getUserApplyStatus($activityId, $userId)
    {
        return $this->getEnrollStrategyContext()->createStrategy('offlineActivity')->getUserEnrollStatus($activityId, $userId);
    }

    public function hasActivityManageRole()
    {
        $user = $this->getCurrentUser();
        if (!$user->isLogin()) {
            return false;
        }

        return $user->hasPermission('admin_offline_activity_manage') || $this->hasTrainingAdminRole();
    }

    public function hasActivityCreateRole()
    {
        $user = $this->getCurrentUser();
        if (!$user->isLogin()) {
            return false;
        }

        return $user->hasPermission('admin_offline_activity_create') || $this->hasTrainingAdminRole();
    }

    public function hasTrainingAdminRole()
    {
        $currentUser = $this->getCurrentUser();
        if (!$currentUser->isLogin()) {
            return false;
        }

        return in_array('ROLE_TRAINING_ADMIN', $currentUser['roles']);
    }

    public function initOrgsRelation()
    {
        return $this->getOfflineActivityDao()->initOrgsRelation();
    }

    public function refreshMemberCount($activityId)
    {
        $conditions = [
            'activityId' => $activityId,
            'joinStatus' => 'join',
        ];

        $count = $this->getMemberService()->countMembers($conditions);

        return $this->getOfflineActivityDao()->update($activityId, ['studentNum' => $count]);
    }

    public function findOfflineActivitiesByCategoryId($categoryId)
    {
        return $this->getOfflineActivityDao()->findByCategoryId($categoryId);
    }

    public function canManageOfflineActivity($offlineActivityId)
    {
        $user = $this->getCurrentUser();
        if (!$user->isLogin()) {
            return false;
        }

        $offlineActivity = $this->getOfflineActivity($offlineActivityId);

        if (!$this->getCurrentUser()->hasManagePermissionWithOrgCode($offlineActivity['orgCode'])) {
            return false;
        }

        if ($user->hasPermission('admin_offline_activity_manage') || $this->hasTrainingAdminRole()) {
            return true;
        }

        return false;
    }

    protected function validateOfflineActivityFields($fields)
    {
        if (!ArrayToolkit::requireds($fields, ['title', 'categoryId'])) {
            throw $this->createInvalidArgumentException('parameters is invalid');
        }
        $category = $this->getCategoryService()->getCategory($fields['categoryId']);
        if (empty($category)) {
            throw $this->createInvalidArgumentException('parameters is invalid');
        }
    }

    protected function filterOfflineActivityFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'title',
                'categoryId',
                'orgId',
                'orgCode',
                'enrollLockDays',
            ]
        );
    }

    public function canUserVisitResource($activityId)
    {
        $member = $this->getMemberService()->getMemberByActivityIdAndUserId($activityId, $this->getCurrentUser()->getId());
        if (!empty($member)) {
            return true;
        }

        $canUserAccessResource = $this->getResourceVisibleService()->canUserVisitResource('offlineActivity', $activityId, $this->getCurrentUser()->getId());
        if ($canUserAccessResource) {
            return true;
        }

        return false;
    }

    protected function prepareSearchConditions($conditions)
    {
        if (!empty($conditions['searchType']) && 'ongoing' === $conditions['searchType']) {
            $conditions['endTime_GE'] = time();
        }

        if (!empty($conditions['searchType']) && 'end' === $conditions['searchType']) {
            $conditions['endTime_LE'] = time();
        }

        if (!empty($conditions['searchType']) && 'all' === $conditions['searchType']) {
            unset($conditions['searchType']);
        }

        return $conditions;
    }

    /**
     * @return FileService
     */
    protected function getFileService()
    {
        return $this->createService('Content:FileService');
    }

    protected function getLogService()
    {
        return $this->createService('System:LogService');
    }

    protected function fieldFields($fields)
    {
        $fields = ArrayToolkit::parts(
            $fields,
            [
                'title',
                'categoryId',
                'startTime',
                'endTime',
                'address',
                'maxStudentNum',
                'enrollmentStartDate',
                'enrollmentEndDate',
                'summary',
                'requireAudit',
                'orgId',
                'orgCode',
                'conditionalAccess',
                'showable',
                'enrollLockDays',
            ]
        );

        if (!empty($fields['summary'])) {
            $fields['summary'] = $this->purifyHtml($fields['summary'], true);
        }

        return $fields;
    }

    protected function filterNull($fields)
    {
        return array_filter(
            $fields,
            function ($value) {
                if ('' === $value || null === $value) {
                    return false;
                }

                return true;
            }
        );
    }

    /**
     * @return MemberService
     */
    protected function getMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineActivity:MemberService');
    }

    /**
     * @return UserService
     */
    protected function getUserService()
    {
        return $this->createService('User:UserService');
    }

    /**
     * @return OfflineActivityDao
     */
    protected function getOfflineActivityDao()
    {
        return $this->createDao('CorporateTrainingBundle:OfflineActivity:OfflineActivityDao');
    }

    protected function getCategoryService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineActivity:CategoryService');
    }

    protected function getFileUsedDao()
    {
        return $this->createDao('File:FileUsedDao');
    }

    protected function getUploadFileService()
    {
        return $this->createService('File:UploadFileService');
    }

    /**
     * @return ManagePermissionOrgService
     */
    protected function getManagePermissionOrgService()
    {
        return $this->createService('CorporateTrainingBundle:ManagePermission:ManagePermissionOrgService');
    }

    protected function getResourceVisibleService()
    {
        return $this->createService('ResourceScope:ResourceVisibleScopeService');
    }

    protected function getResourceAccessService()
    {
        return $this->createService('CorporateTrainingBundle:ResourceScope:ResourceAccessScopeService');
    }

    protected function getResourceVisibleScopeService()
    {
        return $this->createService('ResourceScope:ResourceVisibleScopeService');
    }

    protected function getEnrollStrategyContext()
    {
        return $this->biz['enroll_strategy_context'];
    }
}
