<?php

namespace CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl;

use AppBundle\Common\ArrayToolkit;
use Biz\BaseService;
use Codeages\Biz\Framework\Event\Event;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\MemberService;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\OfflineCourseService;
use OfflineCourseProPlugin\Biz\OfflineClass\Service\OfflineClassService;

class OfflineCourseServiceImpl extends BaseService implements OfflineCourseService
{
    public function createOfflineCourse($offlineCourse)
    {
        $this->validateFields($offlineCourse);
        $offlineCourse = $this->filterFields($offlineCourse);
        $offlineCourse['creator'] = $this->getCurrentUser()->getId();
        $offlineCourse['status'] = 'published';

        $offlineCourse = $this->getOfflineCourseDao()->create($offlineCourse);
        $this->dispatchEvent('offline_course.create', new Event($offlineCourse));

        return $offlineCourse;
    }

    public function updateOfflineCourse($id, $fields)
    {
        $fields = $this->filterFields($fields);
        $oldOfflineCourse = $this->getOfflineCourse($id);
        $updateOfflineCourse = $this->getOfflineCourseDao()->update($id, $fields);

        $this->dispatchEvent('offline_course.update', new Event($updateOfflineCourse, ['oldOfflineCourse' => $oldOfflineCourse]));

        return $updateOfflineCourse;
    }

    public function deleteOfflineCourse($id)
    {
        $offlineCourse = $this->getOfflineCourse($id);

        $result = $this->getOfflineCourseDao()->delete($id);
        $this->dispatchEvent('offline_course.delete', new Event($offlineCourse));

        return $result;
    }

    public function getOfflineCourse($id)
    {
        return $this->getOfflineCourseDao()->get($id);
    }

    public function findOfflineCoursesByIds($ids)
    {
        return $this->getOfflineCourseDao()->findByIds($ids);
    }

    public function findTeachingOfflineCourseByUserId($userId)
    {
        return $this->getOfflineCourseDao()->search(['teacherId' => $userId], ['createdTime' => 'ASC'], 0, PHP_INT_MAX);
    }

    public function findRecentlyOfflineCourses($limit = PHP_INT_MAX)
    {
        return $this->getOfflineCourseDao()->findRecentlyCourses($limit);
    }

    public function findOfflineCoursesByTemplateId($templateId)
    {
        return $this->getOfflineCourseDao()->findByTemplateId($templateId);
    }

    public function searchOfflineCourses($conditions, $orderBys, $start, $limit, $columns = [])
    {
        return $this->getOfflineCourseDao()->search($conditions, $orderBys, $start, $limit, $columns);
    }

    public function countOfflineCourses($conditions)
    {
        return $this->getOfflineCourseDao()->count($conditions);
    }

    public function countPublishedOfflineClassOrProjectPlanOfflineCourses($conditions)
    {
        return $this->getOfflineCourseDao()->countPublishedOfflineClassOrProjectPlanOfflineCourses($conditions);
    }

    public function searchPublishedOfflineClassOrProjectPlanOfflineCourses($conditions, $start, $limit)
    {
        return $this->getOfflineCourseDao()->searchPublishedOfflineClassOrProjectPlanOfflineCourses($conditions, $start, $limit);
    }

    public function publishOfflineCourse($id)
    {
        return $this->updateOfflineCourse($id, ['status' => 'published']);
    }

    public function closeOfflineCourse($id)
    {
        $offlineCourse = $this->updateOfflineCourse($id, ['status' => 'closed']);
        $this->dispatchEvent('offline_course.close', new Event($offlineCourse));

        return $offlineCourse;
    }

    public function findPublishedOfflineCoursesByTeacherIdAndTimeRange($teacherId, $timeRange)
    {
        return $this->getOfflineCourseDao()->findPublishedOfflineCoursesByTeacherIdAndTimeRange($teacherId, $timeRange);
    }

    public function setTeacher($id, $teacherId)
    {
        $oldOfflineCourse = $this->getOfflineCourse($id);
        $offlineCourse = $this->updateOfflineCourse($id, ['teacherId' => $teacherId]);
        $this->dispatchEvent('offline_course.teacher.set', new Event($offlineCourse, ['oldOfflineCourse' => $oldOfflineCourse]));

        return $offlineCourse;
    }

    public function findOfflineCourseItems($id)
    {
        $offlineCourse = $this->getOfflineCourse($id);
        if (empty($offlineCourse)) {
            throw $this->createNotFoundException("OfflineCourse#{$id} Not Found");
        }

        $tasks = $this->findTasksByOfflineCourseId($id);

        return $this->prepareCourseItems($tasks);
    }

    public function prepareCourseItems($tasks)
    {
        $items = [];
        foreach ($tasks as $task) {
            $task['itemType'] = 'task';
            $items["task-{$task['id']}"] = $task;
        }

        uasort(
            $items,
            function ($item1, $item2) {
                return $item1['seq'] > $item2['seq'];
            }
        );

        return $items;
    }

    public function statisticsOfflineCourseTimeByTimeRangeAndCourseIds($timeRange, $courseIds)
    {
        return $this->getOfflineCourseDao()->statisticsOfflineCourseTimeByTimeRangeAndCourseIds($timeRange, $courseIds);
    }

    protected function findTasksByOfflineCourseId($id)
    {
        return $this->getOfflineCourseTaskService()->findTasksByOfflineCourseId($id);
    }

    public function tryManageOfflineCourse($id)
    {
        $user = $this->getCurrentUser();
        if (!$user->isLogin()) {
            throw $this->createAccessDeniedException('user not login');
        }

        $offlineCourse = $this->getOfflineCourseDao()->get($id);

        if (empty($offlineCourse)) {
            throw $this->createNotFoundException("OfflineCourse#{$id} Not Found");
        }

        if ('projectPlan' == $offlineCourse['targetType'] && !$this->getProjectPlanService()->canManageProjectPlan($offlineCourse['targetId'])) {
            throw $this->createAccessDeniedException();
        }

        if ('offlineClass' == $offlineCourse['targetType'] && !$this->getOfflineClassService()->canManageClass($offlineCourse['targetId'])) {
            throw $this->createAccessDeniedException();
        }

        return $offlineCourse;
    }

    public function findTeachersByOfflineCourseId($id)
    {
        $offlineCourse = $this->getOfflineCourse($id);
        $teacher = $this->getUserService()->getUser($offlineCourse['teacherId']);
        $teachersMap = [];
        if (!empty($teacher)) {
            $teachersMap[] = [
                'id' => $teacher['id'],
                'nickname' => $teacher['nickname'],
                'smallAvatar' => $teacher['smallAvatar'],
            ];
        }

        return $teachersMap;
    }

    public function findCourseByUserIds($userIds)
    {
        $offlineCourses = $this->getOfflineCourseDao()->findCourseByUserIds($userIds);

        return ArrayToolkit::index($offlineCourses, 'id');
    }

    /**
     * @param $course
     *
     * @return mixed
     *               构建线下课程统计数据   todo
     */
    public function buildOfflineCourseStatistic($course)
    {
        $members = [];
        if ('projectPlan' == $course['targetType']) {
            $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(
                ['projectPlanId' => $course['targetId']],
                [],
                0,
                PHP_INT_MAX
            );
        }

        if ('offlineClass' == $course['targetType']) {
            $members = $this->getOfflineClassMemberService()->searchMembers(
                ['offlineClassId' => $course['targetId']],
                [],
                0,
                PHP_INT_MAX
            );
        }
        $course['memberCount'] = count($members);
        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');

        $userOfflineCoursesStatisticData = $this->getUserOfflineCoursesStatisticData($userIds, [$course['id']]);

        $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(
            ['offlineCourseId' => $course['id'], 'type' => 'homework'],
            [],
            0,
            PHP_INT_MAX,
            ['id']
        );

        $course['hasHomeTaskCount'] = count($homeworkTasks) * $course['memberCount'];
        $course['passHomeworkCount'] = $userOfflineCoursesStatisticData['offlineCoursePassedHomeworkNum'];
        $course['attendTaskCount'] = $course['memberCount'];
        $course['attendCount'] = $userOfflineCoursesStatisticData['offlineCourseAttendNum'];

        return $course;
    }

    public function getUserOfflineCoursesStatisticData($userIds, $offlineCourseIds)
    {
        $offlineCourseIds = empty($offlineCourseIds) ? [-1] : $offlineCourseIds;
        $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(
            ['offlineCourseIds' => $offlineCourseIds, 'type' => 'homework'],
            [],
            0,
            PHP_INT_MAX,
            ['id']
        );
        $homeworkTaskIds = empty($homeworkTasks) ? [-1] : ArrayToolkit::column($homeworkTasks, 'id');
        $offlineCourseMembers = $this->getOfflineCourseMemberService()->searchMembers(
            ['offlineCourseIds' => $offlineCourseIds, 'userIds' => $userIds],
            [],
            0,
            PHP_INT_MAX,
            ['id', 'offlineCourseId', 'status']
        );
        $offlineCourseMembers = ArrayToolkit::group($offlineCourseMembers, 'status');

        $member['offlineCourseAttendNum'] = (empty($offlineCourseMembers['attended']) ? 0 : count($offlineCourseMembers['attended'])) + (empty($offlineCourseMembers['partial']) ? 0 : count($offlineCourseMembers['partial']));

        $member['offlineCoursePassedHomeworkNum'] = $this->getOfflineCourseTaskService()->countTaskResults(['taskIds' => $homeworkTaskIds, 'status' => 'finished', 'userIds' => $userIds]);
        $member['offlineCourseAttendRate'] = empty($offlineCourseIds) ? 0 : round(($member['offlineCourseAttendNum'] / count($offlineCourseIds) * count($userIds)) * 100, 2);
        $member['offlineCoursePassedHomeworkRate'] = empty($homeworkTasks) ? 0 : round(($member['offlineCoursePassedHomeworkNum'] / count($homeworkTasks) * count($userIds)) * 100, 2);

        return $member;
    }

    /**
     * @return MemberService
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }

    protected function validateFields($fields)
    {
        if (!ArrayToolkit::requireds($fields, ['title'], true)) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }
    }

    protected function filterFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'title',
                'summary',
                'cover',
                'categoryId',
                'taskNum',
                'status',
                'startTime',
                'endTime',
                'teacherId',
                'creator',
                'templateId',
                'place',
                'time',
                'targetId',
                'targetType',
            ]
        );
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\User\Service\Impl\UserServiceImpl
     */
    protected function getUserService()
    {
        return $this->createService('CorporateTrainingBundle:User:UserService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Dao\Impl\OfflineCourseDaoImpl
     */
    public function getOfflineCourseDao()
    {
        return $this->createDao('CorporateTrainingBundle:OfflineCourse:OfflineCourseDao');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\ProjectPlanServiceImpl
     */
    protected function getProjectPlanService()
    {
        return $this->createService('ProjectPlan:ProjectPlanService');
    }

    /**
     * @return OfflineClassService
     */
    protected function getOfflineClassService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineClass:OfflineClassService');
    }

    /**
     * @return \OfflineCourseProPlugin\Biz\OfflineClass\Service\MemberService
     */
    protected function getOfflineClassMemberService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineClass:MemberService');
    }
}
