<?php

namespace CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl;

use AppBundle\Common\ArrayToolkit;
use Biz\BaseService;
use Biz\System\Service\LogService;
use Codeages\Biz\Framework\Event\Event;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\TaskService;

class TaskServiceImpl extends BaseService implements TaskService
{
    public function createTask($task)
    {
        $this->validateTaskFields($task);
        $task['seq'] = $this->countTasks(['offlineCourseId' => $task['offlineCourseId']]) + 1;
        $task = $this->filterTaskFields($task);
        $task = $this->getTaskDao()->create($task);
        $this->dispatchEvent('offline_course.task.create', $task);

        return $task;
    }

    public function updateTask($id, $fields)
    {
        $fields = $this->filterTaskFields($fields);
        $task = $this->getTaskDao()->update($id, $fields);
        $this->dispatchEvent('offline_course.task.update', $task);

        return $task;
    }

    public function deleteTask($id)
    {
        $task = $this->getTask($id);
        $result = $this->getTaskDao()->delete($id);

        $this->dispatchEvent('offline_course.task.delete', $task);

        return $result;
    }

    public function getTask($id)
    {
        return $this->getTaskDao()->get($id);
    }

    public function findTasksByIds($ids)
    {
        return $this->getTaskDao()->findByIds($ids);
    }

    public function findTasksByOfflineCourseId($offlineCourseId)
    {
        return $this->getTaskDao()->findByOfflineCourseId($offlineCourseId);
    }

    public function findTasksByOfflineCourseIdAndTimeRange($offlineCourseId, $timeRange)
    {
        return $this->getTaskDao()->findByOfflineCourseIdAndTimeRange($offlineCourseId, $timeRange);
    }

    public function searchTasks($conditions, $orderBys, $start, $limit, $columns = [])
    {
        return $this->getTaskDao()->search($conditions, $orderBys, $start, $limit, $columns);
    }

    public function countTasks($conditions)
    {
        return $this->getTaskDao()->count($conditions);
    }

    public function sortTasks($ids)
    {
        foreach ($ids as $index => $id) {
            $this->getTaskDao()->update($id, ['seq' => $index + 1]);
        }
    }

    public function createTaskResult($taskResult)
    {
        $this->validateResultFields($taskResult);
        $taskResult = $this->filterResultFields($taskResult);

        return $this->getTaskResultDao()->create($taskResult);
    }

    public function updateTaskResult($id, $fields)
    {
        $fields = $this->filterResultFields($fields);

        return $this->getTaskResultDao()->update($id, $fields);
    }

    public function deleteTaskResult($id)
    {
        return $this->getTaskResultDao()->delete($id);
    }

    public function deleteTaskResultByTaskId($taskId)
    {
        return $this->getTaskResultDao()->deleteByTaskId($taskId);
    }

    public function getTaskResult($id)
    {
        return $this->getTaskResultDao()->get($id);
    }

    public function getTaskResultByTaskIdAndUserId($taskId, $userId)
    {
        return $this->getTaskResultDao()->getByTaskIdAndUserId($taskId, $userId);
    }

    public function findTaskResultsByIds($ids)
    {
        return $this->getTaskResultDao()->findByIds($ids);
    }

    public function findTaskResultsByOfflineCourseId($offlineCourseId)
    {
        return $this->getTaskResultDao()->findByOfflineCourseId($offlineCourseId);
    }

    public function findTaskResultsByUserId($userId)
    {
        return $this->getTaskResultDao()->findByUserId($userId);
    }

    public function searchTaskResults($conditions, $orderBys, $start, $limit, $columns = [])
    {
        return $this->getTaskResultDao()->search($conditions, $orderBys, $start, $limit, $columns);
    }

    public function selectResultCountGroupByUserId($conditions)
    {
        return $this->getTaskResultDao()->selectResultCountGroupByUserId($conditions);
    }

    public function selectResultCountGroupByStatus($conditions)
    {
        return $this->getTaskResultDao()->selectResultCountGroupByStatus($conditions);
    }

    public function countTaskResults($conditions)
    {
        return $this->getTaskResultDao()->count($conditions);
    }

    public function signIn($userId, $offlineCourse)
    {
        $member = $this->getOfflineCourseMemberService()->getMemberByOfflineCourseIdAndUserId($offlineCourse['id'], $userId);
        if (empty($member)) {
            $member = [
                'offlineCourseId' => $offlineCourse['id'],
                'userId' => $userId,
                'learnedNum' => 0,
            ];
            $member = $this->getOfflineCourseMemberService()->createMember($member);
        }

        $tasks = $this->searchTasks(['offlineCourseId' => $offlineCourse['id'], 'type' => 'sign', 'startTime_LE' => time(), 'endTime_GE' => time()], [], 0, PHP_INT_MAX);
        $taskIds = empty($tasks) ? [-1] : ArrayToolkit::column($tasks, 'id');
        $taskResults = $this->searchTaskResults(['taskIds' => $taskIds, 'offlineCourseId' => $offlineCourse['id'], 'userId' => $this->getCurrentUser()->getId()], [], 0, count($taskIds));
        $taskResults = ArrayToolkit::index($taskResults, 'taskId');
        $createTaskResults = [];
        foreach ($tasks as $task) {
            if (!empty($taskResults[$task['id']])) {
                continue;
            }
            $createTaskResults[] = [
                'offlineCourseId' => $task['offlineCourseId'],
                'userId' => $userId,
                'taskId' => $task['id'],
                'status' => 'finished',
                'finishedTime' => time(),
            ];
        }
        if (!empty($createTaskResults)) {
            $this->getTaskResultDao()->batchCreate($createTaskResults);
        }

        $this->dispatchEvent('offline.course.sign_in', new Event($offlineCourse, ['userId' => $userId]));

        return $this->getOfflineCourseMemberService()->batchUpdateMembersLearnedNum([$member['id']]);
    }

    public function batchCheckHomework($taskId, $userIds, $status)
    {
        $task = $this->getTask($taskId);
        $offlineCourse = $this->getOfflineCourseService()->tryManageOfflineCourse($task['offlineCourseId']);
        if (empty($offlineCourse)) {
            throw $this->createNotFoundException('course not found');
        }

        $taskResults = $this->searchTaskResults(['taskId' => $taskId, 'userIds' => $userIds], [], 0, count($userIds));
        $userIds = empty($taskResultIds) ? [-1] : ArrayToolkit::column($taskResultIds, 'userId');
        $status = 'finished' == $status ? 'finished' : 'unpassed';
        foreach ($taskResults as $taskResult) {
            $updateResults[$taskResult['id']] = ['status' => $status, 'finishedTime' => 'finished' == $status ? time() : 0];
        }
        if (!empty($updateResults)) {
            $this->getTaskResultDao()->batchUpdate(array_keys($updateResults), $updateResults, 'id');
        }
        $members = $this->getOfflineCourseMemberService()->searchMembers(['offlineCourseId' => $offlineCourse['id'], 'userId' => $userIds], [], 0, PHP_INT_MAX);

        return $this->getOfflineCourseMemberService()->batchUpdateMembersLearnedNum(ArrayToolkit::column($members, 'id'));
    }

    public function submitHomework($taskId, $attachment)
    {
        $task = $this->getTask($taskId);
        $taskResult = $this->getTaskResultByTaskIdAndUserId($taskId, $this->getCurrentUser()->getId());
        $offlineCourseMember = $this->getofflineCourseMemberService()->getMemberByOfflineCourseIdAndUserId($taskResult['offlineCourseId'], $taskResult['userId']);
        if (empty($offlineCourseMember)) {
            $this->getOfflineCourseMemberService()->createMember(array('offlineCourseId' => $task['offlineCourseId'], 'userId' => $this->getCurrentUser()->getId()));
        }
        if (empty($taskResult)) {
            $taskResult = $this->createTaskResult(['offlineCourseId' => $task['offlineCourseId'], 'taskId' => $task['id'], 'userId' => $this->getCurrentUser()->getId()]);
        }
        $this->uploadMaterialAttachments($attachment, $taskResult['id']);
        $newTaskResult = $this->updateTaskResult($taskResult['id'], ['status' => 'submitted', 'finishedTime' => 0]);

        $this->dispatchEvent('offline.course.submit.homework', $newTaskResult);

        return $newTaskResult;
    }

    protected function uploadMaterialAttachments($attachment, $taskResultId)
    {
        $file = $this->getUploadFileService()->findUseFilesByTargetTypeAndTargetIdAndType('projectPlaning.offline.homework',
            $taskResultId, 'attachment');

        if (!empty($attachment) && empty($file)) {
            $user = $this->getCurrentUser();
            $orgs = $this->getOrgService()->findOrgsByIds($user['orgIds']);
            $org = reset($orgs);
            $userProfile = $this->getUserService()->getUserProfile($user['id']);
            $orgName = empty($org) ? '' : $org['name'].'+';
            $trueName = empty($userProfile['truename']) ? '' : $userProfile['truename'].'+';
            $mobile = empty($userProfile['mobile']) ? '' : $userProfile['mobile'].'+';

            $file = $this->getUploadFileService()->getFile($attachment['fileIds']);
            $filename = $orgName.$trueName.$mobile.$file['filename'];
            $this->getUploadFileService()->update($attachment['fileIds'], ['name' => $filename]);
            $this->getUploadFileService()->createUseFiles($attachment['fileIds'], $taskResultId,
                $attachment['targetType'], $attachment['type']);
        }
    }

    public function getOfflineStudyHoursLearnDataForUserLearnDataExtension($conditions)
    {
        $usersOfflineLearnTime = $this->calculateUsersOfflineLearnTime($conditions);
        $usersOfflineLearnHours = [];

        foreach ($usersOfflineLearnTime as $key => $userOfflineLearnTime) {
            $usersOfflineLearnHours[$key] = round($userOfflineLearnTime, 1);
        }

        return $usersOfflineLearnHours;
    }

    protected function calculateUsersOfflineLearnTime($conditions)
    {
        $offlineCourseIds = $this->getOfflineCourseService()->findCourseByUserIds($conditions['userIds']);

        $offlineCourseIds = ArrayToolkit::column($offlineCourseIds, 'id');
        if (empty($offlineCourseIds)) {
            return [];
        }

        $usersOfflineLearnTime = $this->getTaskResultDao()->calculateUsersOfflineLearnTime($conditions['userIds'],
            $offlineCourseIds, $conditions['date']);
        $usersOfflineLearnTime = ArrayToolkit::index($usersOfflineLearnTime, 'userId');

        $usersOfflineLearnTimeData = [];
        foreach ($conditions['userIds'] as $userId) {
            $usersOfflineLearnTimeData[$userId] = isset($usersOfflineLearnTime[$userId]['offlineStudyTime']) ? $usersOfflineLearnTime[$userId]['offlineStudyTime'] : 0;
        }

        return $usersOfflineLearnTimeData;
    }

    protected function validateTaskFields($fields)
    {
        if (!ArrayToolkit::requireds($fields, ['title', 'offlineCourseId'], true)) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }
    }

    protected function filterTaskFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'offlineCourseId',
                'title',
                'seq',
                'place',
                'homeworkDeadline',
                'homeworkDemand',
                'creator',
                'orgId',
                'startTime',
                'endTime',
                'type',
            ]
        );
    }

    protected function validateResultFields($fields)
    {
        if (!ArrayToolkit::requireds($fields, ['offlineCourseId', 'userId', 'taskId'], true)) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }
    }

    protected function filterResultFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'offlineCourseId',
                'userId',
                'taskId',
                'status',
                'attendStatus',
                'homeworkStatus',
                'finishedTime',
            ]
        );
    }

    /**
     * @return \Biz\File\Service\Impl\UploadFileServiceImpl
     */
    protected function getUploadFileService()
    {
        return $this->createService('File:UploadFileService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Dao\Impl\TaskDaoImpl
     */
    protected function getTaskDao()
    {
        return $this->createDao('CorporateTrainingBundle:OfflineCourse:TaskDao');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Dao\Impl\TaskResultDaoImpl
     */
    protected function getTaskResultDao()
    {
        return $this->createDao('CorporateTrainingBundle:OfflineCourse:TaskResultDao');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->createService('System:LogService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }

    protected function getOrgService()
    {
        return $this->createService('Org:OrgService');
    }

    protected function getUserService()
    {
        return $this->createService('User:UserService');
    }
}
