<?php

namespace CorporateTrainingBundle\Biz\Org\Event;

use AppBundle\Common\Exception\InvalidArgumentException;
use Biz\Org\Dao\OrgDao;
use Biz\System\Service\LogService;
use Codeages\Biz\Framework\Event\Event;
use Codeages\PluginBundle\Event\EventSubscriber;
use CorporateTrainingBundle\Biz\Org\Service\OrgService;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

class OrgEventSubscriber extends EventSubscriber implements EventSubscriberInterface
{
    public static function getSubscribedEvents()
    {
        return [
            'org.delete' => 'orgDelete',
            'org.update' => 'orgUpdate',
        ];
    }

    public function orgDelete(Event $event)
    {
        $org = $event->getSubject();
        if (empty($org['parentId'])) {
            return;
        }
        $childOrgs = $event->getArguments();
        $org['children'] = $childOrgs;
        $parentOrg = $this->getOrgService()->getOrg($org['parentId']);

        $orgModules = $this->getOrgModules();
        $biz = $this->getBiz();
        foreach ($orgModules as $key => $module) {
            $module = 'org.module.'.$key;
            $biz[$module]->verticalMigrate($org, $parentOrg);
        }

        $this->getLogService()->info('org', 'delete_org_migrate_resource', '删除组织机构迁移数据', ['fromOrg' => $org['name'], 'toOrg' => $parentOrg]);
    }

    public function orgUpdate(Event $event)
    {
        $org = $event->getSubject();
        $oldOrg = $event->getArguments();
        if ($org['parentId'] == $oldOrg['parentId']) {
            return;
        }
        $oldParentOrg = $this->getOrgService()->getOrg($oldOrg['parentId']);
        $this->getOrgService()->waveOrgChildrenNum($oldParentOrg['id'], -1);

        $parentOrg = $this->getOrgService()->getOrg($org['parentId']);
        $this->getOrgService()->waveOrgChildrenNum($parentOrg['id'], 1);

        $orgs = $this->updateChildOrgs($org['orgCode'], $oldParentOrg['orgCode'], $parentOrg['orgCode']);
        $this->updateResourceOrgCode($orgs);
        $this->getLogService()->info('org', 'update_org_parent', '更改组织机构父级迁移数据', ['oldParent' => $oldParentOrg, 'newParent' => $parentOrg]);
    }

    /**
     * @param $currentOrgCode
     * @param $oldParentOrgCode
     * @param $parentOrgCode
     *
     * @return array
     */
    protected function updateChildOrgs($currentOrgCode, $oldParentOrgCode, $parentOrgCode)
    {
        $orgs = $this->getOrgService()->findOrgsByPrefixOrgCode($currentOrgCode);
        $depths = array_column($orgs, 'depth');
        array_multisort($depths, SORT_ASC, $orgs);

        foreach ($orgs as &$pendingOrg) {
            $oldOrgCode = $pendingOrg['orgCode'];
            $pendingOrg['orgCode'] = $parentOrgCode.substr($pendingOrg['orgCode'], strlen($oldParentOrgCode));
            $pendingOrg['depth'] = count(explode('.', $pendingOrg['orgCode'])) - 1;
            if ($pendingOrg['depth'] > 15) {
                throw new InvalidArgumentException('超出最大层级!');
            }
            $this->getOrgDao()->update($pendingOrg['id'], $pendingOrg);
            $pendingOrg['oldOrgCode'] = $oldOrgCode;
        }

        return $orgs;
    }

    /**
     * @param array $orgs
     */
    protected function updateResourceOrgCode(array $orgs)
    {
        $orgModules = $this->getOrgModules();
        $biz = $this->getBiz();
        foreach ($orgs as $org) {
            foreach ($orgModules as $key => $module) {
                $module = 'org.module.'.$key;
                $biz[$module]->horizontalMigrate($org['id'], $org['orgCode'], $org['oldOrgCode']);
            }
        }
    }

    protected function getOrgModules()
    {
        global $kernel;

        return $kernel->getContainer()->get('corporatetraining.extension.manager')->getOrgModules();
    }

    /**
     * @return OrgService
     */
    protected function getOrgService()
    {
        return $this->getBiz()->service('Org:OrgService');
    }

    /**
     * @return OrgDao
     */
    public function getOrgDao()
    {
        return $this->getBiz()->dao('Org:OrgDao');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->getBiz()->service('System:LogService');
    }
}
