<?php

namespace CorporateTrainingBundle\Biz\Org\Module;

use AppBundle\Common\ArrayToolkit;
use Biz\System\Service\LogService;
use Codeages\Biz\Framework\Context\Biz;
use CorporateTrainingBundle\Biz\ResourceScope\Dao\ResourceAccessScopeOrgDao;
use CorporateTrainingBundle\Biz\ResourceScope\Dao\ResourceVisibleScopeOrgDao;
use CorporateTrainingBundle\Biz\ResourceScope\Service\ResourceVisibleScopeService;

abstract class Module
{
    protected $biz;

    protected $type = '';

    protected $resourceType = '';

    protected $resourceName = '';

    //该资源是否含有加入范围或者发布范围
    protected $hasVisibleOrAccessScope = 1;

    final public function __construct(Biz $biz)
    {
        $this->biz = $biz;
    }

    /**
     * 删除组织机构进行资源挂载部门的变更和发布范围的变更
     *
     * @param $fromOrg
     * @param $toOrg
     */
    public function verticalMigrate($fromOrg, $toOrg)
    {
        $orgProperty = [
            'orgId' => $toOrg['id'],
            'orgCode' => $toOrg['orgCode'],
        ];
        $this->updateOrgByOrgCodeAndOrgId($fromOrg['orgCode'], $orgProperty);

        if (!empty($this->hasVisibleOrAccessScope)) {
            $this->deleteUnexistScopes($fromOrg);
        }
    }

    /**
     * 变更父级后，orgCode产生变更
     *
     * @param $orgId
     * @param $newOrgCode
     * @param $oldOrgCode
     *
     * @return mixed
     */
    public function horizontalMigrate($orgId, $newOrgCode, $oldOrgCode)
    {
        return $this->getResourceDao()->update(['orgId' => $orgId], ['orgCode' => $newOrgCode]);
    }

    /**
     * 修改资源showable(带有发布范围的资源在浏览页的action添加，重置showable),组织机构删除或修改父级后发布范围相应的改动
     *
     * @param $resourceId
     */
    public function migrateResourceShowable($resourceId)
    {
        $resource = $this->getResourceDao()->get($resourceId);
        $records = $this->getResourceVisibleScopeOrgDao()->findByResourceTypeAndResourceId($this->resourceType, $resourceId);

        if (!empty($resource['showable']) && empty($records)) {
            $this->getResourceDao()->update($resourceId, ['showable' => 0]);
        }
    }

    /**
     * 初始化资源挂载部门，把所有资源挂载到全站底下,主要用于首次同步组织机构
     *
     * @return mixed
     */
    public function initOrgsRelation()
    {
        return $this->getResourceDao()->initOrgsRelation();
    }

    /**
     * 当资源表同时存在orgId和orgCode字段时，批量变更组织机构
     *
     * @param $orgCode
     * @param $orgProperty
     *
     * @return mixed
     */
    protected function updateOrgByOrgCodeAndOrgId($orgCode, $orgProperty)
    {
        $conditions['likeOrgCode'] = $orgCode;
        $orgProperty = ArrayToolkit::parts($orgProperty, ['orgId', 'orgCode']);

        return $this->getResourceDao()->update($conditions, $orgProperty);
    }

    /**
     * 当资源表只有orgId字段时，批量变更组织机构
     *
     * @param $fromOrg
     * @param $orgProperty
     */
    protected function updateOrgIdWithFromOrg($fromOrg, $orgProperty)
    {
        $deletedOrgIds = [$fromOrg['id']];
        if (!empty($fromOrg['children'])) {
            $childIds = ArrayToolkit::column($fromOrg['children'], 'id');
            $deletedOrgIds = array_merge($childIds, $deletedOrgIds);
        }

        $orgProperty = ArrayToolkit::parts($orgProperty, ['orgId']);

        $updateQuestionnaire = [];
        foreach ($deletedOrgIds as $orgId) {
            $updateQuestionnaire[$orgId] = $orgProperty;
        }

        $this->getResourceDao()->batchUpdate($deletedOrgIds, $updateQuestionnaire, 'orgId');
    }

    /**
     *  删除发布范围和加入条件组织机构设置
     *
     * @param $fromOrg
     *
     * @return bool
     */
    protected function deleteUnexistScopes($fromOrg)
    {
        if (empty($fromOrg) || empty($this->hasVisibleOrAccessScope)) {
            return true;
        }

        $deletedScopeIds = [$fromOrg['id']];
        if (!empty($fromOrg['children'])) {
            $childIds = ArrayToolkit::column($fromOrg['children'], 'id');
            $deletedScopeIds = array_merge($childIds, $deletedScopeIds);
        }
        $conditions = [
            'scopes' => $deletedScopeIds,
            'resourceType' => $this->resourceType,
        ];
        $this->getResourceVisibleScopeOrgDao()->batchDelete($conditions);
        $this->getResourceAccessScopeOrgDao()->batchDelete($conditions);
        $this->getLogService()->info('org', 'delete_org_migrate_resource', '删除组织机构迁移'.$this->resourceName.'发布范围加入范围', ['conditions' => $conditions]);
    }

    abstract protected function getResourceDao();

    /**
     * @return ResourceVisibleScopeOrgDao
     */
    protected function getResourceVisibleScopeOrgDao()
    {
        return $this->biz->dao('ResourceScope:ResourceVisibleScopeOrgDao');
    }

    /**
     * @return ResourceAccessScopeOrgDao
     */
    protected function getResourceAccessScopeOrgDao()
    {
        return $this->biz->dao('ResourceScope:ResourceAccessScopeOrgDao');
    }

    /**
     * @return ResourceVisibleScopeService
     */
    protected function getResourceVisibleScopeService()
    {
        return $this->biz->service('ResourceScope:ResourceVisibleScopeService');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->biz->service('System:LogService');
    }
}
