<?php

namespace CorporateTrainingBundle\Biz\Org\Module;

use AppBundle\Common\ArrayToolkit;
use CorporateTrainingBundle\Biz\ManagePermission\Dao\ManagePermissionOrgDao;
use CorporateTrainingBundle\Biz\Org\Service\OrgService;
use CorporateTrainingBundle\Biz\User\Dao\UserDao;
use CorporateTrainingBundle\Biz\User\Dao\UserOrgDao;
use CorporateTrainingBundle\Biz\User\Service\UserOrgService;

class UserModule extends Module
{
    protected $type = 'user';

    protected $hasVisibleOrAccessScope = 0;

    public function verticalMigrate($fromOrg, $toOrg)
    {
        $result = $this->verticalMigrateUser($fromOrg, $toOrg);
        if ($result) {
            $this->deleteUnexistScopes($fromOrg);
            $this->deleteUnexistPermission($fromOrg);
        }
    }

    public function horizontalMigrate($orgId, $newOrgCode, $oldOrgCode)
    {
        $conditions['orgCode'] = $oldOrgCode;
        $count = $this->getResourceService()->countUserOrgs($conditions);
        if ($count > 0) {
            $records = $this->getResourceService()->searchUserOrgs($conditions, [], 0, 2000);
            $this->batchUpdateUserWithHorizontalMigrate($records, $oldOrgCode, $newOrgCode);
            $this->horizontalMigrate($orgId, $newOrgCode, $oldOrgCode);
        }

        return true;
    }

    public function initOrgsRelation()
    {
        $this->getResourceDao()->initOrgsRelation();

        return $this->getUserDao()->initOrgsRelation();
    }

    private function verticalMigrateUser($fromOrg, $toOrg)
    {
        $conditions['likeOrgCode'] = $fromOrg['orgCode'];
        $count = $this->getResourceService()->countUserOrgs($conditions);
        if ($count > 0) {
            $records = $this->getResourceService()->searchUserOrgs($conditions, [], 0, 2000);
            $recordIds = ArrayToolkit::column($records, 'id');
            $this->batchUpdateUserWithVerticalMigrate($records, $toOrg);
            $this->getResourceDao()->batchDelete(['ids' => $recordIds]);
            $this->verticalMigrateUser($fromOrg, $toOrg);
        }

        return true;
    }

    /**
     * @param array $records
     * @param $oldOrgCode
     * @param $newOrgCode
     */
    private function batchUpdateUserWithHorizontalMigrate($records, $oldOrgCode, $newOrgCode)
    {
        $userIds = ArrayToolkit::column($records, 'userId');
        $uers = $this->getUserService()->findUsersByIds($userIds);
        $updateUsers = [];
        foreach ($records as $record) {
            if (isset($uers[$record['userId']])) {
                $orgCodes = $uers[$record['userId']]['orgCodes'];
                $key = array_search($oldOrgCode, $orgCodes);
                if (false !== $key) {
                    $orgCodes[$key] = $newOrgCode;
                    $updateUsers[$record['userId']] = ['orgCodes' => $orgCodes];
                }
            }
        }
        if (empty($updateUsers)) {
            return;
        }
        $this->getUserDao()->batchUpdate(array_keys($updateUsers), $updateUsers, 'id');
        $recordIds = ArrayToolkit::column($records, 'id');
        $this->getResourceDao()->update(['ids' => $recordIds], ['orgCode' => $newOrgCode]);
    }

    /**
     * @param $records
     * @param $toOrg
     */
    private function batchUpdateUserWithVerticalMigrate($records, $toOrg)
    {
        $userIds = ArrayToolkit::column($records, 'userId');
        $uers = $this->getUserService()->findUsersByIds($userIds);

        $updateUsers = [];
        $createUserOrgs = [];
        foreach ($records as $record) {
            if (isset($uers[$record['userId']])) {
                $orgCodes = array_diff($uers[$record['userId']]['orgCodes'], [$record['orgCode']]);
                $orgIds = array_diff($uers[$record['userId']]['orgIds'], [$record['orgId']]);
                if (empty($orgIds)) {
                    $orgIds = [$toOrg['id']];
                    $orgCodes = [$toOrg['orgCode']];
                    $createUserOrgs[] = ['userId' => $record['userId'], 'orgId' => $toOrg['id'], 'orgCode' => $toOrg['orgCode']];
                }
                $uers[$record['userId']]['orgIds'] = $orgIds;
                $uers[$record['userId']]['orgCodes'] = $orgCodes;
                $updateUsers[$record['userId']] = ['orgIds' => $orgIds, 'orgCodes' => $orgCodes];
            }
        }
        if (!empty($updateUsers)) {
            $this->getUserDao()->batchUpdate(array_keys($updateUsers), $updateUsers, 'id');
        }

        if (!empty($createUserOrgs)) {
            $this->getResourceDao()->batchCreate($createUserOrgs);
        }
    }

    protected function deleteUnexistPermission($fromOrg)
    {
        if (empty($fromOrg)) {
            return true;
        }
        $deletedOrgIds = [$fromOrg['id']];
        if (!empty($fromOrg['children'])) {
            $childIds = ArrayToolkit::column($fromOrg['children'], 'id');
            $deletedOrgIds = array_merge($childIds, $deletedOrgIds);
        }
        $conditions = [
            'orgIds' => $deletedOrgIds,
        ];

        $this->getManagePermissionOrgDao()->batchDelete($conditions);
        $this->getLogService()->info('org', 'delete_org_migrate_resource', '删除组织机构迁移用户管理范围', ['conditions' => $conditions]);
    }

    /**
     * @return UserOrgService
     */
    protected function getResourceService()
    {
        return $this->biz->service('User:UserOrgService');
    }

    /**
     * @return \Biz\User\Service\UserService
     */
    protected function getUserService()
    {
        return $this->biz->service('User:UserService');
    }

    /**
     * @return UserDao
     */
    protected function getUserDao()
    {
        return $this->biz->dao('CorporateTrainingBundle:User:UserDao');
    }

    /**
     * @return OrgService
     */
    protected function getOrgService()
    {
        return $this->biz->service('Org:OrgService');
    }

    /**
     * @return UserOrgDao
     */
    protected function getResourceDao()
    {
        return $this->biz->dao('User:UserOrgDao');
    }

    /**
     * @return ManagePermissionOrgDao
     */
    protected function getManagePermissionOrgDao()
    {
        return $this->biz->dao('ManagePermission:ManagePermissionOrgDao');
    }
}
