<?php

namespace CorporateTrainingBundle\Biz\OrgSync\Service\Impl;

use Biz\BaseService;
use Biz\System\Service\LogService;
use CorporateTrainingBundle\Biz\Classroom\Service\ClassroomService;
use CorporateTrainingBundle\Biz\Course\Service\CourseSetService;
use CorporateTrainingBundle\Biz\OfflineActivity\Service\OfflineActivityService;
use CorporateTrainingBundle\Biz\Org\Service\OrgService;
use CorporateTrainingBundle\Biz\OrgSync\Service\OrgSyncService;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\ProjectPlanService;
use CorporateTrainingBundle\Biz\ResourceScope\Dao\Impl\ResourceAccessScopeOrgDaoImpl;
use CorporateTrainingBundle\Biz\ResourceScope\Dao\Impl\ResourceVisibleScopeOrgDaoImpl;
use CorporateTrainingBundle\Component\EIMClient\DepartmentFactory;
use CorporateTrainingBundle\Common\Constant\CTConst;
use AppBundle\Common\ArrayToolkit;

class OrgSyncServiceImpl extends BaseService implements OrgSyncService
{
    public function syncFrom()
    {
        $result = ['success' => false, 'code' => 1];
        $setting = $this->getSettingService()->get('sync_department_setting', []);

        if (empty($setting) || !$setting['enable']) {
            return ['success' => false, 'code' => 2];
        }

        $client = DepartmentFactory::create($setting);
        $departments = $client->lists();

        if (!empty($departments['errcode'])) {
            $this->getLogService()->error('dingtalk', 'department_sync_from', $departments['errmsg']);

            return ['success' => false, 'code' => $departments['errcode'], 'message' => $departments['errmsg']];
        }

        try {
            $this->beginTransaction();

            if ($setting['times'] <= 0) {
                $this->firstSyncFrom($departments);
            } else {
                $this->commonSyncFrom($departments);
            }

            $setting['syncTime'] = time();
            $setting['times'] = $setting['times'] + 1;
            $this->getSettingService()->set('sync_department_setting', $setting);

            $this->commit();
            $result = ['success' => true, 'code' => 0];
        } catch (\Exception $e) {
            $this->rollback();
            throw $e;
        }

        return $result;
    }

    private function firstSyncFrom($departments)
    {
        $this->getOrgService()->resetOrg(CTConst::ROOT_ORG_CODE);
        $departments = ArrayToolkit::index($departments, 'id');
        foreach ($departments as $department) {
            $this->createOrg($departments, $department);
        }
        $this->initOrgsRelation();
        $this->initResourceScopeOrg();
    }

    private function commonSyncFrom($departments)
    {
        $departments = ArrayToolkit::index($departments, 'id');
        foreach ($departments as $department) {
            $this->createOrg($departments, $department);
        }
        $this->deleteDingTalkDeprecatedOrgs($departments);
    }

    //删除钉钉弃用的组织机构
    private function deleteDingTalkDeprecatedOrgs($departments)
    {
        $orgs = $this->getOrgService()->searchOrgs(['syncId_GT' => 0], [], 0, PHP_INT_MAX, ['syncId']);
        $syncIds = ArrayToolkit::column($orgs, 'syncId');
        $departmentIds = ArrayToolkit::column($departments, 'id');
        $syncIds = array_diff($syncIds, $departmentIds);
        foreach ($syncIds as $syncId) {
            $org = $this->getOrgService()->getOrgBySyncId($syncId);
            if ($org) {
                $this->getOrgService()->deleteOrg($org['id']);
            }
        }
    }

    private function createOrg($departments, $department)
    {
        if (!isset($department['parentid'])) {
            $this->updateRootOrg($department);

            return;
        }
        $parentOrg = $this->getOrgService()->getOrgBySyncId($department['parentid']);
        if ($parentOrg) {
            $this->syncChildOrg($parentOrg, $department);
        } else {
            $department = $departments[$department['parentid']];
            $this->createOrg($departments, $department);
        }
    }

    //创建或更新子部门
    private function syncChildOrg($parentOrg, $department)
    {
        $org = $this->getOrgService()->getOrgBySyncId($department['id']);
        if ($org) {
            $fields = [
                    'name' => $department['name'],
                    'code' => $org['code'],
                    'parentId' => $parentOrg['id'],
                    'depth' => $parentOrg['depth'] + 1,
                    ];
            $this->getOrgService()->updateOrg($org['id'], $fields);
        } else {
            $department['code'] = 'dingtalk'.$department['id'];
            $department['parentId'] = $parentOrg['id'];
            $department['syncId'] = $department['id'];
            $this->getOrgService()->createOrg($department);
        }
    }

    //更新顶级默认部门
    private function updateRootOrg($department)
    {
        $rootOrg = $this->getOrgService()->getOrgByOrgCode(CTConst::ROOT_ORG_CODE);
        $fields = [
            'name' => $department['name'],
            'code' => $rootOrg['code'],
            'syncId' => $department['id'],
        ];
        $this->getOrgService()->updateOrg($rootOrg['id'], $fields);
    }

    //资源挂载到全站
    private function initOrgsRelation()
    {
        $orgModules = $this->getOrgModules();
        $biz = $this->biz;
        foreach ($orgModules as $key => $module) {
            $module = 'org.module.'.$key;
            $biz[$module]->initOrgsRelation();
        }
    }

    //清除发布范围，加入范围设置
    private function initResourceScopeOrg()
    {
        $this->getResourceVisibleScopeOrgDao()->batchDelete(['existId' => -1]);
        $this->getResourceAccessScopeOrgDao()->batchDelete(['existId' => -1]);
    }

    public function syncTo()
    {
        $result = ['success' => false, 'code' => 1];
        $setting = $this->getSettingService()->get('sync_department_setting', []);

        if (empty($setting) || !$setting['enable']) {
            $this->getLogService()->error('org', 'sync_department_to', '同步失败，请先开启同步开关', []);

            return ['successs' => false, 'code' => 2];
        }

        $client = DepartmentFactory::create($setting);
        $departments = $client->lists();

        if (count($departments) > 1) {
            $this->getLogService()->error('org', 'sync_department_to', '同步失败，请先清空钉钉下组织机构', ['count' => count($departments)]);

            return ['success' => false, 'code' => 3];
        }

        $orgs = $this->getOrgService()->searchOrgs([], ['parentId' => 'ASC'], 0, PHP_INT_MAX);

        try {
            $this->beginTransaction();

            foreach ($orgs as $org) {
                if (1 == $org['id']) {
                    $this->updateOrgSync($org);
                    $department = $client->get($org['id']);
                    $org['syncId'] = $department['id'];
                    $this->getOrgService()->updateOrg($org['id'], $org);
                    continue;
                }
                $result = $this->createOrgSync($org);
                if (0 == $result['errcode']) {
                    $org['syncId'] = $result['id'];
                    $this->getOrgService()->updateOrg($org['id'], $org);
                }
            }

            $setting['syncTime'] = time();
            $setting['times'] = $setting['times'] + 1;
            $this->getSettingService()->set('sync_department_setting', $setting);

            $this->commit();
            $result = ['success' => true, 'code' => 0];
        } catch (\Exception $e) {
            $this->rollback();
            throw $e;
        }

        return $result;
    }

    public function createOrgSync($org)
    {
        $parentOrg = $this->getOrgService()->getOrg($org['parentId']);
        $org['parentid'] = $parentOrg['syncId'];
        $orgInfo = ArrayToolkit::parts(
            $org, [
                'name',
                'parentid',
                'deptHiding',
                'deptPerimits',
                'userPerimits',
                'outerDept',
                'outerPermitDepts',
                'outerPermitUsers',
            ]
        );

        if (!ArrayToolkit::requireds($orgInfo, ['name'])) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }

        $setting = $this->getSettingService()->get('sync_department_setting', []);

        if (empty($setting) || !$setting['enable']) {
            return;
        }

        $client = DepartmentFactory::create($setting);
        $result = $client->create($orgInfo);

        if (0 === $result['errcode']) {
            $fields = [
                'name' => $org['name'],
                'code' => $org['code'],
                'syncId' => $result['id'],
            ];
            $this->getOrgService()->updateOrg($org['id'], $fields);
            $this->getLogService()->info('org-sync', 'create-org', "创建同步部门(#{$result['id']})", $org);
        } else {
            $this->getLogService()->error('org-sync', 'create-org', '同步借口调用失败', $result);
        }

        return $result;
    }

    public function updateOrgSync($org)
    {
        $parentOrg = $this->getOrgService()->getOrg($org['parentId']);
        if ($parentOrg) {
            $org['parentid'] = $parentOrg['syncId'];
        }
        $org['id'] = $org['syncId'];

        $org = ArrayToolkit::parts(
            $org, [
                'id',
                'name',
                'parentid',
                'autoAddUser',
                'deptHiding',
                'deptPerimits',
                'userPerimits',
                'outerDept',
                'outerPermitDepts',
                'outerPermitUsers',
            ]
        );
        $org['autoAddUser'] = true;

        if (!ArrayToolkit::requireds($org, ['name'])) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }

        $setting = $this->getSettingService()->get('sync_department_setting', []);

        if (empty($setting) || !$setting['enable']) {
            return;
        }

        $client = DepartmentFactory::create($setting);

        $result = $client->update($org);

        if (0 === $result['errcode']) {
            $this->getLogService()->info('org-sync', 'update-org', "更新同步部门(#{$org['id']})", $org);
        } else {
            $this->getLogService()->error('org-sync', 'update-org', '同步借口调用失败', $result);
        }

        return $result;
    }

    public function deleteOrgSync($id)
    {
        $setting = $this->getSettingService()->get('sync_department_setting', []);

        if (empty($setting) || !$setting['enable']) {
            return;
        }

        $client = DepartmentFactory::create($setting);

        $result = $client->delete($id);

        if (0 === $result['errcode']) {
            $this->getLogService()->info('org-sync', 'delete-org', "删除同步部门(#{$id})", ['id' => $id]);
        } else {
            $this->getLogService()->error('org-sync', 'delete-org', '删除借口调用失败', $result);
        }

        return $result;
    }

    /**
     * @return OrgService
     */
    protected function getOrgService()
    {
        return $this->createService('Org:OrgService');
    }

    protected function getSettingService()
    {
        return $this->createService('System:SettingService');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->createService('System:LogService');
    }

    protected function getCategoryService()
    {
        return $this->createService('Taxonomy:CategoryService');
    }

    protected function getTagService()
    {
        return $this->createService('Taxonomy:TagService');
    }

    protected function getNavigationService()
    {
        return $this->createService('Content:NavigationService');
    }

    protected function getAnnouncementService()
    {
        return $this->createService('Announcement:AnnouncementService');
    }

    /**
     * @return CourseSetService
     */
    protected function getCourseSetService()
    {
        return $this->createService('Course:CourseSetService');
    }

    /**
     * @return ClassroomService
     */
    protected function getClassroomService()
    {
        return $this->createService('Classroom:ClassroomService');
    }

    protected function getArticleService()
    {
        return $this->createService('Article:ArticleService');
    }

    protected function getUserService()
    {
        return $this->createService('User:UserService');
    }

    /**
     * @return ProjectPlanService
     */
    protected function getProjectPlanService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:ProjectPlanService');
    }

    /**
     * @return OfflineActivityService
     */
    protected function getOfflineActivityService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineActivity:OfflineActivityService');
    }

    /**
     * @return ResourceVisibleScopeOrgDaoImpl
     */
    protected function getResourceVisibleScopeOrgDao()
    {
        return $this->createDao('ResourceScope:ResourceVisibleScopeOrgDao');
    }

    /**
     * @return ResourceAccessScopeOrgDaoImpl
     */
    protected function getResourceAccessScopeOrgDao()
    {
        return $this->createDao('ResourceScope:ResourceAccessScopeOrgDao');
    }

    protected function getOrgModules()
    {
        global $kernel;

        return $kernel->getContainer()->get('corporatetraining.extension.manager')->getOrgModules();
    }
}
