<?php

namespace CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl;

use AppBundle\Common\ArrayToolkit;
use Biz\BaseService;
use Biz\Content\Service\FileService;
use Biz\System\Service\LogService;
use Codeages\Biz\Framework\Event\Event;
use ExamPlugin\Biz\Exam\Service\ExamService;
use CorporateTrainingBundle\Biz\ProjectPlan\Dao\Impl\ProjectPlanDaoImpl;
use CorporateTrainingBundle\Biz\ProjectPlan\Dao\Impl\ProjectPlanItemDaoImpl;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\ProjectPlanService;
use CorporateTrainingBundle\Biz\ResourceScope\Service\ResourceAccessScopeService;
use CorporateTrainingBundle\Biz\ResourceScope\Service\ResourceVisibleScopeService;

class ProjectPlanServiceImpl extends BaseService implements ProjectPlanService
{
    public function createProjectPlan($projectPlan)
    {
        if (!$this->hasManageProjectPlanPermission()) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $fields = $this->filterProjectPlanFields($projectPlan);

        if (!ArrayToolkit::requireds($fields, ['name', 'orgId', 'orgCode'])) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }

        $currentUser = $this->getCurrentUser();
        $fields['createdUserId'] = $currentUser['id'];

        return $this->getProjectPlanDao()->create($fields);
    }

    public function updateProjectPlan($id, $fields)
    {
        if (!$this->canManageProjectPlan($id)) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $fields = $this->filterProjectPlanFields($fields);
        $oldProjectPlan = $this->getProjectPlan($id);
        $projectPlan = $this->getProjectPlanDao()->update($id, $fields);

        $this->dispatchEvent('project_plan.update', new Event($projectPlan, ['oldProjectPlan' => $oldProjectPlan]));

        return $projectPlan;
    }

    public function deleteProjectPlan($id)
    {
        if (!$this->canManageProjectPlan($id)) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $projectPlan = $this->getProjectPlan($id);
        if (empty($projectPlan)) {
            throw $this->createNotFoundException('Training ProjectPlan Not Found');
        }

        if ('published' == $projectPlan['status']) {
            throw $this->createAccessDeniedException('Delete Published Training ProjectPlan Is Not Allowed');
        }

        try {
            $this->beginTransaction();

            $this->deleteItemByProjectPlanId($id);
            $this->getProjectPlanMemberService()->deleteMemberByProjectPlanId($id);
            $result = $this->getProjectPlanDao()->delete($id);

            $this->commit();

            return $result;
        } catch (\Exception $e) {
            $this->rollback();
            throw $e;
        }
    }

    public function getProjectPlan($id)
    {
        return $this->getProjectPlanDao()->get($id);
    }

    public function getMonthlyProjectPlansNumAndMembersNumByOrgAndCategory($date, $orgCode, $categoryId)
    {
        $projectPlanIds = $this->getProjectPlanDao()->findMonthlyProjectPlanIdsByOrgAndCategory($date, $orgCode, $categoryId);
        $projectPlansNum = count($projectPlanIds);
        $ids = ArrayToolkit::column($projectPlanIds, 'id');
        $projectPlanIds = empty($ids) ? [-1] : $ids;
        $projectPlanMembersNum = $this->getProjectPlanMemberService()->countProjectPlanMembers(
            [
                'projectPlanIds' => $projectPlanIds,
            ]
        );

        return [$projectPlansNum, $projectPlanMembersNum];
    }

    public function findProjectPlansByIds($ids)
    {
        return $this->getProjectPlanDao()->findByIds($ids);
    }

    public function findProjectPlansByCreatedUserId($createdUserId)
    {
        return $this->getProjectPlanDao()->findByCreatedUserId($createdUserId);
    }

    public function findProjectPlanByCategoryId($categoryId)
    {
        return $this->getProjectPlanDao()->findByCategoryId($categoryId);
    }

    public function findUnfinishedProjectPlansByCurrentUserId($start, $limit)
    {
        $projectPlanIds = $this->getUnfinishedProjectPlanIds();
        $projectPlanIds = empty($projectPlanIds) ? [-1] : $projectPlanIds;

        return $this->searchProjectPlans(['ids' => $projectPlanIds, 'excludeStatus' => ['archived']], [], $start, $limit);
    }

    public function findProjectPlansByDateAndIds($date, $projectIds, $start, $limit)
    {
        if (empty($projectIds)) {
            return [];
        }

        return $this->getProjectPlanDao()->findByDateAndIds($date, $projectIds, $start, $limit);
    }

    public function countProjectPlansByDateAndIds($date, $projectIds)
    {
        if (empty($projectIds)) {
            return 0;
        }

        return $this->getProjectPlanDao()->countByDateAndIds($date, $projectIds);
    }

    protected function getUnfinishedProjectPlanIds()
    {
        $userId = $this->getCurrentUser()->getId();
        $members = $this->getProjectPlanMemberService()->findMembersByUserId($userId);
        $projectPlanIds = !empty($members) ? ArrayToolkit::column($members, 'projectPlanId') : [];
        if (!empty($projectPlanIds)) {
            $projectPlans = $this->searchProjectPlans(['ids' => $projectPlanIds, 'excludeStatus' => ['archived']], [], 0, PHP_INT_MAX);
            $projectPlanIds = ArrayToolkit::column($projectPlans, 'id');
        }
        foreach ($projectPlanIds as $key => &$projectPlanId) {
            $personalProgress = $this->getPersonalProjectPlanProgress($projectPlanId, $userId);
            if (100 == $personalProgress) {
                unset($projectPlanIds[$key]);
            }
        }

        return $projectPlanIds;
    }

    public function countUnfinishedProjectPlansByCurrentUserId()
    {
        $projectPlanIds = $this->getUnfinishedProjectPlanIds();

        return count($projectPlanIds);
    }

    public function changeCover($id, $coverArray)
    {
        if (empty($coverArray)) {
            throw $this->createInvalidArgumentException('Invalid Param: cover');
        }
        $projectPlan = $this->getProjectPlan($id);
        $covers = [];
        foreach ($coverArray as $cover) {
            $file = $this->getFileService()->getFile($cover['id']);
            $covers[$cover['type']] = $file['uri'];
        }

        return $this->getProjectPlanDao()->update($projectPlan['id'], ['cover' => $covers]);
    }

    public function countProjectPlans(array $conditions)
    {
        return $this->getProjectPlanDao()->count($conditions);
    }

    public function searchProjectPlans(array $conditions, array $orderBys, $start, $limit)
    {
        $conditions = $this->prepareSearchConditions($conditions);

        return $this->getProjectPlanDao()->search($conditions, $orderBys, $start, $limit);
    }

    public function initOrgsRelation()
    {
        return $this->getProjectPlanDao()->initOrgsRelation();
    }

    public function publishProjectPlan($id)
    {
        if (!$this->canManageProjectPlan($id)) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $projectPlan = $this->getProjectPlan($id);
        if (empty($projectPlan)) {
            throw $this->createNotFoundException('Training ProjectPlan Not Found');
        }

        return $this->updateProjectPlan($id, ['status' => 'published']);
    }

    public function closeProjectPlan($id)
    {
        if (!$this->canManageProjectPlan($id)) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $projectPlan = $this->getProjectPlan($id);
        if (empty($projectPlan)) {
            throw $this->createNotFoundException('Training ProjectPlan Not Found');
        }

        return $this->updateProjectPlan($id, ['status' => 'closed']);
    }

    public function archiveProjectPlan($id)
    {
        if (!$this->canManageProjectPlan($id)) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $projectPlan = $this->getProjectPlan($id);
        if (empty($projectPlan)) {
            throw $this->createNotFoundException('Training ProjectPlan Not Found');
        }

        return $this->updateProjectPlan($id, ['status' => 'archived']);
    }

    public function canUserVisitResource($projectPlanId)
    {
        $member = $this->getProjectPlanMemberService()->getProjectPlanMemberByUserIdAndProjectPlanId($this->getCurrentUser()->getId(), $projectPlanId);
        if (!empty($member)) {
            return true;
        }

        $canUserAccessResource = $this->getResourceVisibleService()->canUserVisitResource('projectPlan', $projectPlanId, $this->getCurrentUser()->getId());
        if ($canUserAccessResource) {
            return true;
        }

        return false;
    }

    public function getUserApplyStatus($projectPlanId, $userId)
    {
        return $this->getEnrollStrategyContext()->createStrategy('projectPlan')->getUserEnrollStatus($projectPlanId, $userId);
    }

    public function getProjectPlanProgress($projectPlanId)
    {
        $finishedItemNum = 0;
        $projectPlanItems = $this->findProjectPlanItemsByProjectPlanId($projectPlanId);
        foreach ($projectPlanItems as $projectPlanItem) {
            $strategy = $this->createProjectPlanStrategy($projectPlanItem['targetType']);
            $finishedMemberNum = $strategy->getFinishedMembersNum($projectPlanItem);
            $finishedItemNum += $finishedMemberNum;
        }
        $member = $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $projectPlanId]);
        $itemNum = $this->countProjectPlanItems(['projectPlanId' => $projectPlanId]);

        return (0 != $itemNum && 0 != $member) ? round(($finishedItemNum / ($itemNum * $member)) * 100, 0) : 0;
    }

    public function getPersonalProjectPlanProgress($projectPlanId, $userId)
    {
        $projectPlanItems = $this->findProjectPlanItemsByProjectPlanId($projectPlanId);
        $finishedItemNum = $this->getFinishedItemNum($projectPlanItems, $userId);
        $itemNum = count($projectPlanItems);

        return (0 != $itemNum) ? round(($finishedItemNum / $itemNum) * 100, 2) : 0;
    }

    public function getFinishedItemNum($projectPlanItems, $userId)
    {
        $finishedItemNum = 0;
        $user = $this->getUserService()->getUser($userId);
        foreach ($projectPlanItems as $projectPlanItem) {
            $strategy = $this->createProjectPlanStrategy($projectPlanItem['targetType']);
            $isFinished = $strategy->isFinished($projectPlanItem, $user);
            if ($isFinished) {
                ++$finishedItemNum;
            }
        }

        return $finishedItemNum;
    }

    public function appendProgress($projectPlanId, $members)
    {
        foreach ($members as &$member) {
            $member['progress'] = $this->getPersonalProjectPlanProgress($projectPlanId, $member['userId']);
        }

        return $members;
    }

    public function getUserProjectPlanAttendRateData($projectPlan, $userId)
    {
        $projectPlanItems = $this->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlan['id'], 'offline_course');
        $offlineCourseIds = empty($projectPlanItems) ? [-1] : ArrayToolkit::column($projectPlanItems, 'targetId');
        $offlineCourseMembers = $this->getOfflineCourseMemberService()->searchMembers(
            ['offlineCourseIds' => $offlineCourseIds, 'userId' => $userId],
            [],
            0,
            PHP_INT_MAX,
            ['id', 'status']
        );
        $offlineCourseMembers = ArrayToolkit::group($offlineCourseMembers, 'status');
        $offlineCourseAttendNum = (empty($offlineCourseMembers['attended']) ? 0 : count($offlineCourseMembers['attended'])) + (empty($offlineCourseMembers['partial']) ? 0 : count($offlineCourseMembers['partial']));
        $offlineCourseAttendRate = empty($offlineCourseIds) ? 0 : round($offlineCourseAttendNum / count($offlineCourseIds) * 100, 2);

        return [
            'courseCount' => count($offlineCourseIds),
            'attendCount' => $offlineCourseAttendNum,
            'rate' => $offlineCourseAttendRate,
        ];
    }

    public function getUserProjectPlanExamRateData($projectPlan, $userId)
    {
        $onlineExamData = $this->getUserOnlineExamPassRateData($projectPlan, $userId);
        $offlineExamData = $this->getUserOfflineExamPassRateData($projectPlan, $userId);
        $examCount = $onlineExamData['examCount'] + $offlineExamData['examCount'];
        $passedCount = $onlineExamData['passedCount'] + $offlineExamData['passedCount'];

        return [
            'examCount' => $examCount,
            'passedCount' => $passedCount,
            'rate' => empty($examCount) ? 0 : round($passedCount / $examCount * 100, 2),
        ];
    }

    public function getUserOnlineExamPassRateData($projectPlan, $userId)
    {
        if (!$this->isPluginInstalled('Exam')) {
            return ['examCount' => 0, 'passedCount' => 0, 'rate' => 0];
        }
        $projectPlanItems = $this->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlan['id'], 'exam');
        $examIds = empty($projectPlanItems) ? [-1] : ArrayToolkit::column($projectPlanItems, 'targetId');
        $passedMembers = $this->getExamService()->searchMembers(['examIds' => $examIds, 'userId' => $userId, 'passStatus' => 'passed'], [], 0, count($examIds));

        $examCount = count($projectPlanItems);
        $passedCount = count($passedMembers) > $examCount ? $examCount : count($passedMembers);

        return [
            'examCount' => $examCount,
            'passedCount' => $passedCount,
            'rate' => empty($examCount) ? 0 : round($passedCount / $examCount * 100, 2),
        ];
    }

    public function getUserOfflineExamPassRateData($projectPlan, $userId)
    {
        $projectPlanItems = $this->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlan['id'], 'offline_exam');
        $offlineExamIds = empty($projectPlanItems) ? [-1] : ArrayToolkit::column($projectPlanItems, 'targetId');
        $passedMembers = $this->getOfflineExamMemberService()->searchMembers(['offlineExamIds' => $offlineExamIds, 'status' => 'passed', 'userId' => $userId], [], 0, count($offlineExamIds));
        $examCount = count($projectPlanItems);
        $passedCount = count($passedMembers) > $examCount ? $examCount : count($passedMembers);

        return [
            'examCount' => $examCount,
            'passedCount' => $passedCount,
            'rate' => empty($examCount) ? 0 : round($passedCount / $examCount * 100, 2),
        ];
    }

    protected function prepareSearchConditions($conditions)
    {
        if (!empty($conditions['currentState']) && 'ongoing' === $conditions['currentState']) {
            $conditions['enrollmentEndDate_GE'] = time();
        }

        if (!empty($conditions['currentState']) && 'end' === $conditions['currentState']) {
            $conditions['enrollmentEndDate_LE'] = time();
        }

        if (!empty($conditions['currentState']) && 'all' === $conditions['currentState']) {
            unset($conditions['currentState']);
        }

        return $conditions;
    }

    public function createProjectPlanItem($item)
    {
        $fields = $this->filterProjectPlanItemFields($item);

        if (!$this->canManageProjectPlan($fields['projectPlanId'])) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        if (!ArrayToolkit::requireds($fields, ['projectPlanId', 'targetType', 'targetId'])) {
            throw $this->createInvalidArgumentException('Lack of required fields');
        }

        if (!in_array($fields['targetType'], ['course', 'offline_course', 'exam', 'offline_exam', 'questionnaire'])) {
            throw $this->createInvalidArgumentException('Param Invalid: TargetType');
        }

        $item = $this->getProjectPlanItemDao()->create($fields);
        $count = $this->getProjectPlanItemDao()->count(['projectPlanId' => $item['projectPlanId']]);
        $this->getProjectPlanDao()->update($item['projectPlanId'], ['itemNum' => $count]);

        return $item;
    }

    public function batchCreateProjectPlanItems($items)
    {
        if (!empty($items)) {
            $projectPlanItems = [];
            foreach ($items as $item) {
                $projectPlanItems[] = ArrayToolkit::parts($item, [
                    'projectPlanId', 'targetId', 'targetType', 'seq',
                ]);
            }
            $this->getProjectPlanItemDao()->batchCreate($projectPlanItems);
        }
    }

    public function updateProjectPlanItem($id, $fields)
    {
        $item = $this->getProjectPlanItem($id);

        if (empty($item)) {
            $this->createNotFoundException("ProjectPlan#{$id} Not Found");
        }

        if (!$this->canManageProjectPlan($item['projectPlanId'])) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $fields = $this->filterProjectPlanItemFields($fields);

        return $this->getProjectPlanItemDao()->update($id, $fields);
    }

    public function updateProjectPlanItemTime($id, $fields)
    {
        $item = $this->getProjectPlanItem($id);

        if (empty($item)) {
            $this->createNotFoundException("ProjectPlan#{$id} Not Found");
        }

        $fields = ArrayToolkit::parts(
            $fields,
            [
                'startTime',
                'endTime',
            ]);

        return $this->getProjectPlanItemDao()->update($id, $fields);
    }

    public function updatePlanItem($id, $params, $type)
    {
        $this->beginTransaction();

        try {
            $strategy = $this->createProjectPlanStrategy($type);
            $strategy->updateItem($id, $params);
            $this->commit();

            return true;
        } catch (\Exception $e) {
            $this->rollback();
            $this->getLogService()->error('projectPlan', 'update_projectPlan_item', 'update projectPlan item'.$e->getMessage());

            throw $e;
        }
    }

    public function deleteProjectPlanItem($id)
    {
        $item = $this->getProjectPlanItem($id);

        if (empty($item)) {
            $this->createNotFoundException("ProjectPlan#{$id} Not Found");
        }

        if (!$this->canManageProjectPlan($item['projectPlanId'])) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $strategy = $this->createProjectPlanStrategy($item['targetType']);
        $strategy->deleteItem($item);
        $result = $this->getProjectPlanItemDao()->delete($id);
        $count = $this->getProjectPlanItemDao()->count(['projectPlanId' => $item['projectPlanId']]);
        $this->getProjectPlanDao()->update($item['projectPlanId'], ['itemNum' => $count]);

        return $result;
    }

    public function deleteItemByProjectPlanId($projectPlanId)
    {
        if (!$this->canManageProjectPlan($projectPlanId)) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $this->getProjectPlanItemDao()->deleteItemByProjectPlanId($projectPlanId);

        $items = $this->findProjectPlanItemsByProjectPlanId($projectPlanId);
        foreach ($items as &$item) {
            $this->deleteProjectPlanItem($item['id']);
        }

        return true;
    }

    public function getProjectPlanItem($id)
    {
        $item = $this->getProjectPlanItemDao()->get($id);
        $strategy = $this->createProjectPlanStrategy($item['targetType']);
        $item = $strategy->getItem($item);

        return $item;
    }

    public function findTaskDetailByTimeRangeAndProjectPlanId($startTime, $endTime, $projectPlanId)
    {
        $projectPlanItems = $this->findProjectPlanItemsByProjectPlanId($projectPlanId);
        foreach ($projectPlanItems as $key => $projectPlanItem) {
            $tasks = $this->findTasksByTimeRangeAndProjectPlanItem($startTime, $endTime, $projectPlanItem);
            if (!empty($tasks)) {
                foreach ($tasks as $task) {
                    if ($this) {
                        $reviewNum = $this->getReviewNum($task, $projectPlanItem['targetType']);
                    }
                    $detail[] = [
                        'seq' => $key,
                        'id' => $task['id'],
                        'itemId' => $projectPlanItem['targetId'],
                        'itemType' => $projectPlanItem['targetType'],
                        'tagName' => (in_array($projectPlanItem['targetType'], ['exam', 'offline_exam'])) ? '考试' : '课程',
                        'type' => isset($task['type']) ? $task['type'] : '',
                        'title' => isset($task['title']) ? $task['title'] : $task['name'],
                        'place' => isset($task['place']) ? $task['place'] : '',
                        'homeworkDeadline' => isset($task['homeworkDeadline']) ? $task['homeworkDeadline'] : '',
                        'reviewNum' => isset($reviewNum) ? $reviewNum : '',
                        'startTime' => $task['startTime'],
                        'endTime' => $task['endTime'],
                    ];
                }
            }
        }

        if (!empty($detail)) {
            $startTime = ArrayToolkit::column($detail, 'startTime');
            $seq = ArrayToolkit::column($detail, 'seq');
            array_multisort($startTime, SORT_ASC, $seq, SORT_ASC, $detail);
        }

        return !empty($detail) ? $detail : [];
    }

    protected function getReviewNum($task, $projectPlanItemType)
    {
        $strategy = $this->createProjectPlanStrategy($projectPlanItemType);
        $reviewNum = $strategy->getTaskReviewNum($task['id']);

        return $reviewNum;
    }

    protected function findTasksByTimeRangeAndProjectPlanItem($startTime, $endTime, $projectPlanItem)
    {
        $timeRange = ['startTime' => $startTime, 'endTime' => $endTime];
        $strategy = $this->createProjectPlanStrategy($projectPlanItem['targetType']);
        $tasks = $strategy->findTasksByItemIdAndTimeRange($projectPlanItem['targetId'], $timeRange);

        return $tasks;
    }

    public function getProjectPlanItemByTargetIdAndTargetType($targetId, $targetType)
    {
        return $this->getProjectPlanItemDao()->getByTargetIdAndTargetType($targetId, $targetType);
    }

    public function getProjectPlanItemByProjectPlanIdAndTargetIdAndTargetType($projectPlanId, $targetId, $targetType)
    {
        return $this->getProjectPlanItemDao()->getByProjectPlanIdAndTargetIdAndTargetType($projectPlanId, $targetId, $targetType);
    }

    public function findProjectPlanItemsByIds($ids)
    {
        $projectPlanItems = $this->getProjectPlanItemDao()->findByIds($ids);

        if (!$this->isPluginInstalled('Exam')) {
            foreach ($projectPlanItems as $key => $projectPlanItem) {
                if ('exam' == $projectPlanItem['targetType']) {
                    unset($projectPlanItems[$key]);
                }
            }
        }
        $projectPlanItems = $this->spliceItemDetail($projectPlanItems);

        return $projectPlanItems;
    }

    public function findProjectPlanItemsByProjectPlanId($projectPlanId)
    {
        $projectPlanItems = $this->getProjectPlanItemDao()->findByProjectPlanId($projectPlanId);

        if (!$this->isPluginInstalled('Exam')) {
            foreach ($projectPlanItems as $key => $projectPlanItem) {
                if ('exam' == $projectPlanItem['targetType']) {
                    unset($projectPlanItems[$key]);
                }
            }
        }
        $projectPlanItems = $this->spliceItemDetail($projectPlanItems);

        return $projectPlanItems;
    }

    public function findProjectPlanItemsByProjectPlanIds(array $projectPlanIds)
    {
        $projectPlanItems = $this->getProjectPlanItemDao()->findByProjectPlanIds($projectPlanIds);

        if (!$this->isPluginInstalled('Exam')) {
            foreach ($projectPlanItems as $key => $projectPlanItem) {
                if ('exam' == $projectPlanItem['targetType']) {
                    unset($projectPlanItems[$key]);
                }
            }
        }
        $projectPlanItems = $this->spliceItemDetail($projectPlanItems);

        return $projectPlanItems;
    }

    public function findHasFinishedSurveyResultProjectPlanItemIds($id, $userIds)
    {
        return $this->getProjectPlanItemDao()->findHasFinishedSurveyResultProjectPlanItemIds($id, $userIds);
    }

    public function findProjectPlanItemsByTargetIdAndTargetType($targetId, $targetType)
    {
        return $this->getProjectPlanItemDao()->findByTargetIdAndTargetType($targetId, $targetType);
    }

    public function countProjectPlanItems(array $conditions)
    {
        if (!$this->isPluginInstalled('Exam')) {
            $conditions['excludeTargetTypes'] = ['exam'];
        }

        return $this->getProjectPlanItemDao()->count($conditions);
    }

    public function searchProjectPlanItems(array $conditions, array $orderBys, $start, $limit)
    {
        if (!$this->isPluginInstalled('Exam')) {
            $conditions['excludeTargetTypes'] = ['exam'];
        }
        $projectPlanItems = $this->getProjectPlanItemDao()->search($conditions, $orderBys, $start, $limit);

        return $this->spliceItemDetail($projectPlanItems);
    }

    protected function spliceItemDetail($projectPlanItems)
    {
        $projectPlanItems = ArrayToolkit::index($projectPlanItems, 'id');
        $projectPlanItemsGroup = ArrayToolkit::group($projectPlanItems, 'targetType');
        foreach ($projectPlanItemsGroup as $key => $itemsGroup) {
            $strategy = $this->createProjectPlanStrategy($key);
            $itemsWithDetail = $strategy->findItemsDetail($itemsGroup);
            foreach ($itemsWithDetail as $key => $itemWithDetail) {
                if (isset($itemWithDetail['detail'])) {
                    $projectPlanItems[$key]['detail'] = $itemWithDetail['detail'];
                }
            }
        }

        return $projectPlanItems;
    }

    public function findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, $targetType)
    {
        return $this->getProjectPlanItemDao()->findByProjectPlanIdAndTargetType($projectPlanId, $targetType);
    }

    public function sortItems($ids)
    {
        foreach ($ids as $index => $id) {
            $this->getProjectPlanItemDao()->update($id, ['seq' => $index + 1]);
        }
    }

    public function setProjectPlanItems($projectPlanId, $params, $type)
    {
        $this->beginTransaction();

        try {
            $strategy = $this->createProjectPlanStrategy($type);
            $strategy->createItems($projectPlanId, $params);
            $this->dispatchEvent('projectPlan.item.set', new Event($projectPlanId, ['params' => $params, 'type' => $type]));
            $this->getLogService()->info('projectPlan', 'set_projectPlan_items', 'add projectPlan items', $params);
            $this->commit();

            return true;
        } catch (\Exception $e) {
            $this->rollback();
            $this->getLogService()->error('projectPlan', 'set_projectPlan_items', 'add projectPlan items'.$e->getMessage());

            throw $e;
        }
    }

    public function createProjectPlanAdvancedOption($fields)
    {
        if (!ArrayToolkit::requireds($fields, ['projectPlanId'])) {
            throw $this->createInvalidArgumentException('parameter is invalid!');
        }

        $fields = $this->filterAdvanceFields($fields);

        return $this->getProjectPlanAdvancedOptionDao()->create($fields);
    }

    public function updateProjectPlanAdvancedOption($id, $advancedOption)
    {
        if (!$this->canManageProjectPlan($advancedOption['projectPlanId'])) {
            throw $this->createAccessDeniedException('Access Denied');
        }

        $advancedOption = $this->filterAdvanceFields($advancedOption);

        return $this->getProjectPlanAdvancedOptionDao()->update($id, $advancedOption);
    }

    public function countProjectPlanAdvancedOptions($conditions)
    {
        return $this->getProjectPlanAdvancedOptionDao()->count($conditions);
    }

    public function getProjectPlanAdvancedOption($id)
    {
        return $this->getProjectPlanAdvancedOptionDao()->get($id);
    }

    public function searchProjectPlanAdvancedOptions(array $conditions, array $orderBys, $start, $limit)
    {
        return $this->getProjectPlanAdvancedOptionDao()->search($conditions, $orderBys, $start, $limit);
    }

    public function getProjectPlanAdvancedOptionByProjectPlanId($projectPlanId)
    {
        return $this->getProjectPlanAdvancedOptionDao()->getByProjectPlanId($projectPlanId);
    }

    public function waveProjectPlanItemNum($id, $num)
    {
        return $this->getProjectPlanDao()->wave(
            [$id],
            ['itemNum' => $num]
        );
    }

    protected function filterAdvanceFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'projectPlanId',
                'orgIds',
                'userGroupIds',
                'postIds',
                'requireRemark',
                'requireMaterial',
                'remarkRequirement',
                'materialRequirement',
                'material1',
                'material2',
                'material3',
            ]
        );
    }

    protected function filterProjectPlanFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'name',
                'startTime',
                'endTime',
                'status',
                'summary',
                'createdUserId',
                'orgId',
                'orgCode',
                'cover',
                'itemNum',
                'memberNum',
                'maxStudentNum',
                'requireAudit',
                'enrollmentEndDate',
                'requireEnrollment',
                'enrollmentStartDate',
                'categoryId',
                'conditionalAccess',
                'showable',
                'enrollLockDays',
                'templateId',
            ]
        );
    }

    protected function filterProjectPlanItemFields($fields)
    {
        return ArrayToolkit::parts(
            $fields,
            [
                'projectPlanId',
                'targetType',
                'targetId',
                'seq',
                'startTime',
                'endTime',
            ]
        );
    }

    protected function createProjectPlanStrategy($type)
    {
        return $this->biz->offsetGet('projectPlan_item_strategy_context')->createStrategy($type);
    }

    public function canManageProjectPlan($id)
    {
        $currentUser = $this->getCurrentUser();

        if (!$currentUser->isLogin()) {
            return false;
        }

        $projectPlan = $this->getProjectPlan($id);
        if (empty($projectPlan)) {
            return false;
        }

        if (!$currentUser->hasManagePermissionWithOrgCode($projectPlan['orgCode'])) {
            return false;
        }

        return $this->hasManageProjectPlanPermission();
    }

    public function hasManageProjectPlanPermission()
    {
        $currentUser = $this->getCurrentUser();
        if (!$currentUser->isLogin()) {
            return false;
        }

        return $currentUser->hasPermission('admin_project_plan_manage');
    }

    protected function isPluginInstalled($pluginName)
    {
        return $this->biz['pluginConfigurationManager']->isPluginInstalled($pluginName);
    }

    /**
     * @return ProjectPlanDaoImpl
     */
    protected function getProjectPlanDao()
    {
        return $this->createDao('CorporateTrainingBundle:ProjectPlan:ProjectPlanDao');
    }

    /**
     * @return ProjectPlanItemDaoImpl
     */
    protected function getProjectPlanItemDao()
    {
        return $this->createDao('CorporateTrainingBundle:ProjectPlan:ProjectPlanItemDao');
    }

    /**
     * @return AdvancedOptionDao
     */
    protected function getProjectPlanAdvancedOptionDao()
    {
        return $this->createDao('CorporateTrainingBundle:ProjectPlan:AdvancedOptionDao');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\MemberService
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return FileService
     */
    protected function getFileService()
    {
        return $this->createService('Content:FileService');
    }

    protected function getUserService()
    {
        return $this->createService('User:UserService');
    }

    /**
     * @return LogService
     */
    protected function getLogService()
    {
        return $this->createService('System:LogService');
    }

    /**
     * @return ResourceVisibleScopeService
     */
    protected function getResourceVisibleService()
    {
        return $this->createService('CorporateTrainingBundle:ResourceScope:ResourceVisibleScopeService');
    }

    /**
     * @return ResourceAccessScopeService
     */
    protected function getResourceAccessService()
    {
        return $this->createService('CorporateTrainingBundle:ResourceScope:ResourceAccessScopeService');
    }

    /**
     * @return ResourceVisibleScopeService
     */
    protected function getResourceVisibleScopeService()
    {
        return $this->createService('ResourceScope:ResourceVisibleScopeService');
    }

    protected function getEnrollStrategyContext()
    {
        return $this->biz['enroll_strategy_context'];
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineExam\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineExamMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }

    /**
     * @return ExamService
     */
    public function getExamService()
    {
        return $this->createService('ExamPlugin:Exam:ExamService');
    }
}
