<?php

namespace CorporateTrainingBundle\Component\CSVExport\Admin;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Component\Export\Exporter;
use CorporateTrainingBundle\Common\DateToolkit;
use Topxia\Service\Common\ServiceKernel;

class DataCenterOnlineCourseExporter extends Exporter
{
    public function canExport()
    {
        $user = $this->getUser();

        return $user->isAdmin();
    }

    public function getTitles()
    {
        $title = [
            'admin.data_center.course_detail.course_title',
            'admin.data_center.course_detail.create_time',
            'admin.data_center.course_detail.org',
            'admin.data_center.course_detail.category',
            'admin.data_center.course_detail.student_num',
            'admin.data_center.course_detail.average_learn_time',
            'admin.data_center.course_detail.finishing_rate',
        ];

        if ($this->isPluginInstalled('Survey')) {
            $title[] = 'admin.data_center.course_detail.survey';
        }

        return $title;
    }

    public function getContent($start, $limit)
    {
        $courseSets = $this->getCourseSetService()->searchCourseSets($this->conditions, ['createdTime' => 'DESC'], $start, $limit);
        $courseSets = $this->getTaskResultService()->buildOnlineCourseDetail($courseSets);
        $courseData = $this->buildExportData($courseSets);

        return $courseData;
    }

    public function getCount()
    {
        return $this->getCourseSetService()->countCourseSets($this->conditions);
    }

    public function buildCondition($fields)
    {
        if (!empty($fields['createdTime'])) {
            $dateSearchTime = explode('-', $fields['createdTime']);
            $fields['startDateTime'] = strtotime($dateSearchTime[0]);
            $fields['endDateTime'] = strtotime($dateSearchTime[1].' 23:59:59');
            unset($fields['createdTime']);
        }

        return $this->prepareSearchConditions($fields);
    }

    protected function prepareSearchConditions($fields)
    {
        if (empty($fields['startDateTime']) || empty($fields['endDateTime'])) {
            list($startDateTime, $endDateTime) = DateToolkit::generateStartDateAndEndDate('year');
            $conditions['startTime'] = strtotime($startDateTime);
            $conditions['endTime'] = strtotime($endDateTime.' 23:59:59');
        } else {
            $conditions['startTime'] = $fields['startDateTime'];
            $conditions['endTime'] = $fields['endDateTime'];
        }

        if (!empty($fields['orgCode'])) {
            $conditions['orgCode'] = $fields['orgCode'];
        }

        if (!empty($fields['title'])) {
            $conditions['title'] = $fields['title'];
        }

        if (!empty($fields['status'])) {
            $conditions['status'] = $fields['status'];
        }

        $conditions = $this->fillOrgCode($conditions);
        $conditions['excludeStatus'] = ['draft'];

        return $conditions;
    }

    protected function getPageConditions()
    {
        return [$this->parameter['start'], 10];
    }

    protected function buildExportData($courseSets)
    {
        $courseData = [];
        $categories = $this->getCategoryService()->findCategoriesByIds(ArrayToolkit::column($courseSets, 'categoryId'));

        foreach ($courseSets as $k => $courseSet) {
            $org = $this->getOrgService()->getOrgByOrgCode($courseSet['orgCode']);

            $courseData[] = [
                'title' => ($courseSet['title'] ?? '').'/('.$this->statusTrans($courseSet['status']).')',
                'createdTime' => date('Y-m-d', $courseSet['createdTime']),
                'orgTitle' => $org['name'] ?? '',
                'categoryTitle' => $categories[$courseSet['categoryId']]['name'] ?? '',
                'studentNum' => $courseSet['studentNum'],
                'averageLength' => empty($courseSet['studentNum']) ? 0 : $this->timeFormatHour($courseSet['totalLearnTime'] / $courseSet['studentNum']),
                'progress' => empty($courseSet['studentNum']) ? '0%' : $this->percent($courseSet['courseProgress'], $courseSet['studentNum']),
            ];

            if ($this->isPluginInstalled('Survey')) {
                $avgEvaluate = $this->getSurveyResultService()->getOnlineCourseSurveyScoreByCourseId($courseSet['defaultCourseId']);
                $courseData[$k]['teachingEvaluation'] = (empty($avgEvaluate) ? '--' : $avgEvaluate).'/5.00';
            }
        }

        return $courseData;
    }

    protected function percent($number, $total)
    {
        if (0 == $number || 0 == $total) {
            return '0%';
        }

        if ($number >= $total) {
            return '100%';
        }

        return (int) ($number / $total * 100).'%';
    }

    protected function timeFormatHour($time)
    {
        return substr(sprintf('%.2f', $time / 3600), 0, -1).' '.ServiceKernel::instance()->trans('site.date.hour');
    }

    protected function fillOrgCode($conditions)
    {
        if (!isset($conditions['orgCode'])) {
            $conditions['likeOrgCode'] = $this->getUser()->getSelectOrgCode();
        } else {
            $conditions['likeOrgCode'] = $conditions['orgCode'];
            unset($conditions['orgCode']);
        }

        return $conditions;
    }

    protected function statusTrans($key)
    {
        $status = [
            'unpublished' => ServiceKernel::instance()->trans('course.status.unpublished'),
            'published' => ServiceKernel::instance()->trans('course.status.published'),
            'closed' => ServiceKernel::instance()->trans('course.status.closed'),
        ];
        if (!in_array($key, array_keys($status))) {
            return $status['unpublished'];
        }

        return $status[$key];
    }

    protected function getOrgService()
    {
        return $this->getBiz()->service('CorporateTrainingBundle:Org:OrgService');
    }

    /**
     * @return \SurveyPlugin\Biz\Survey\Service\Impl\SurveyResultServiceImpl
     */
    protected function getSurveyResultService()
    {
        return $this->getBiz()->service('SurveyPlugin:Survey:SurveyResultService');
    }

    /**
     * @return CategoryService
     */
    protected function getCategoryService()
    {
        return $this->getBiz()->service('CorporateTrainingBundle:Taxonomy:CategoryService');
    }

    /**
     * @return CourseSetService
     */
    protected function getCourseSetService()
    {
        return $this->getBiz()->service('CorporateTrainingBundle:Course:CourseSetService');
    }

    /**
     * @return TaskResultService
     */
    protected function getTaskResultService()
    {
        return $this->getBiz()->service('CorporateTrainingBundle:Task:TaskResultService');
    }

    public function isPluginInstalled($code)
    {
        return $this->container->get('api.plugin.config.manager')->isPluginInstalled($code);
    }
}
