<?php

namespace CorporateTrainingBundle\Controller\Admin;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\ChangelogToolkit;
use AppBundle\Common\CurlToolkit;
use AppBundle\Controller\Admin\AnalysisController as BaseAnalysisController;
use Biz\CloudPlatform\CloudAPIFactory;
use Biz\Role\Util\PermissionBuilder;
use CorporateTrainingBundle\Biz\Focus\Service\FocusService;
use CorporateTrainingBundle\Biz\OfflineActivity\Service\OfflineActivityService;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\ProjectPlanService;
use CorporateTrainingBundle\System;
use ExamPlugin\Biz\Exam\Service\ExamService;
use OfflineCourseProPlugin\Biz\OfflineClass\Service\OfflineClassService;
use Symfony\Component\HttpFoundation\Request;
use Topxia\Service\Common\ServiceKernel;

class AnalysisController extends BaseAnalysisController
{
    private $homePageQuickEntranceCodes = [
        'admin_train_teach_manage_my_teaching_courses',
        'admin_train_teach_manage_my_teaching_offline_courses',
        'admin_train_exam_manage_list',
        'admin_project_plan_manage',
        'admin_role_manage',
        'admin_block',
        'admin_setting_theme',
        'admin_setting_mobile_settings',
    ];

    public function indexAction(Request $request)
    {
        $currentUser = $this->getCurrentUser();

        $quickEntrances = $this->getUserQuickEntrances();
        $userProfile = $this->getUserService()->getUserProfile($currentUser['id']);
        $homePageQuickEntranceCodes = $this->homePageQuickEntranceCodes;
        if ($userProfile['quick_entrance']) {
            $homePageQuickEntranceCodes = $userProfile['quick_entrance'];
        }

        $quickEntrances = $this->getHomePageQuickEntrances($quickEntrances, $homePageQuickEntranceCodes);

        return $this->render('admin/default/corporate-training-admin-index.html.twig', [
            'quickEntrances' => $quickEntrances,
        ]);
    }

    public function chooseQuickEntrancesAction(Request $request)
    {
        $currentUser = $this->getCurrentUser();

        if ('POST' == $request->getMethod()) {
            $params = $request->request->all();
            $params['data'] = json_decode($params['data'], true);

            if (empty($params['data'])) {
                $params['data'] = [-1];
            }
            $updatedUserProfile = $this->getUserService()->updateUserProfile($currentUser['id'], ['quick_entrance' => $params['data']]);

            $quickEntrances = $this->getUserQuickEntrances();
            $quickEntrances = $this->getHomePageQuickEntrances($quickEntrances, $updatedUserProfile['quick_entrance']);

            return $this->render('admin/default/quick-entrance.html.twig', [
                'quickEntrances' => $quickEntrances,
            ]);
        }

        $userProfile = $this->getUserService()->getUserProfile($currentUser['id']);
        $quickEntrances = $this->getUserQuickEntrances();
        $homePageQuickEntranceCodes = $this->homePageQuickEntranceCodes;
        if ($userProfile['quick_entrance']) {
            $homePageQuickEntranceCodes = $userProfile['quick_entrance'];
        }

        foreach ($quickEntrances as &$item) {
            if (!empty($item['data'])) {
                foreach ($item['data'] as &$entrance) {
                    if (in_array($entrance['code'], $homePageQuickEntranceCodes)) {
                        $entrance['checked'] = true;
                    } else {
                        $entrance['checked'] = false;
                    }
                }
            }
        }

        return $this->render('admin/default/quick-entrance-model.html.twig', [
            'quickEntrances' => $quickEntrances,
        ]);
    }

    protected function getHomePageQuickEntrances($quickEntrances, $homePageQuickEntranceCodes)
    {
        foreach ($quickEntrances as &$item) {
            if (!empty($item['data'])) {
                $item['data'] = array_filter(
                    $item['data'],
                    function ($entrance) use ($homePageQuickEntranceCodes) {
                        return in_array($entrance['code'], $homePageQuickEntranceCodes, true);
                    }
                );
            }
        }

        return $quickEntrances;
    }

    protected function getUserQuickEntrances()
    {
        $currentUser = $this->getCurrentUser();
        $tree = PermissionBuilder::instance()->findAllPermissionTree(true);

        $allPermissions = $tree->toArray();
        $modules = $allPermissions['children'][0]['children'];

        $quickEntrances = [];
        foreach ($modules as $module) {
            $quickEntrances[$module['code']]['data'] = $this->getQuickEntrancesArray($module, [], $currentUser);
            $quickEntrances[$module['code']]['name'] = $this->trans($module['name'], [], 'menu');
        }

        return $quickEntrances;
    }

    private function getQuickEntrancesArray($module, $moduleQuickEntrances, $currentUser)
    {
        if (isset($module['quick_entrance']) && $module['quick_entrance'] && $currentUser->hasPermission($module['code'])) {
            $module['name'] = $this->trans($module['name'], [], 'menu');
            $moduleQuickEntrances = [$module];
        } else {
            $moduleQuickEntrances = [];
        }

        if (isset($module['children'])) {
            foreach ($module['children'] as $child) {
                $moduleQuickEntrances = array_merge($moduleQuickEntrances, $this->getQuickEntrancesArray($child, $moduleQuickEntrances, $currentUser));
            }
        }

        return $moduleQuickEntrances;
    }

    public function systemStatusAction(Request $request)
    {
        list($mainAppUpgrade, $upgradeAppCount) = $this->checkCloudAppUpgradeInfo();
        list($android, $ios) = $this->getMobileAppVersion();
        $mobileSetting = $this->getSettingService()->get('mobile', []);
        $mailStatus = $this->isMailEnabled();
        $eduCloudStatus = $this->getEduCloudStatus();

        return $this->render('admin/default/corporate-training-system-status.html.twig', [
            'upgradeAppCount' => $upgradeAppCount,
            'mainAppUpgrade' => $mainAppUpgrade,
            'android' => $android,
            'ios' => $ios,
            'mobileSetting' => $mobileSetting,
            'mailServiceStatus' => $mailStatus,
            'eduCloudStatus' => $eduCloudStatus,
        ]);
    }

    public function systemDataOverviewAction()
    {
        $userData = $this->getUsersNumberData();
        $courseDate = $this->getCoursesData();
        $teacherData = $this->getTeachersData();
        $projectData = $this->getProjectData();
        $examData = $this->getExamData();
        $activityData = $this->getActivityData();

        return $this->render('admin/default/data-overview.html.twig', [
            'userData' => $userData,
            'teacherData' => $teacherData,
            'courseData' => $courseDate,
            'projectData' => $projectData,
            'examData' => $examData,
            'activityData' => $activityData,
        ]);
    }

    private function isMailEnabled()
    {
        $cloudEmail = $this->getSettingService()->get('cloud_email_crm', []);

        if (!empty($cloudEmail) && 'enable' === $cloudEmail['status']) {
            return true;
        }

        $mailer = $this->getSettingService()->get('mailer', []);

        if (!empty($mailer) && $mailer['enabled']) {
            return true;
        }

        return false;
    }

    private function getUsersNumberData()
    {
        $defaultConditions = [
            'locked' => 0,
            'noType' => 'system',
        ];
        $allUsersCount = $this->getUserService()->countUsers($defaultConditions);
        $defaultConditions['startTime'] = strtotime(date('Y').'-01-01 00:00:00');
        $newUsersCount = $this->getUserService()->countUsers($defaultConditions);

        return ['allUsersCount' => $allUsersCount, 'newUsersCount' => $newUsersCount];
    }

    private function getCoursesData()
    {
        $defaultConditions = [];
        $allOfflineCoursesCount = 0;
        $newOfflineCoursesCount = 0;
        $allOnlineCoursesCount = $this->getCourseSetService()->countCourseSets($defaultConditions);
        $defaultConditions['startTime'] = strtotime(date('Y').'-01-01 00:00:00');
        $newOnlineCoursesCount = $this->getCourseSetService()->countCourseSets($defaultConditions);
        $allOfflineCoursesCount = $this->getOfflineCourseService()->countOfflineCourses([]);
        $newOfflineCoursesCount = $this->getOfflineCourseService()->countOfflineCourses($defaultConditions);

        return ['allCoursesCount' => $allOnlineCoursesCount + $allOfflineCoursesCount, 'newCoursesCount' => $newOnlineCoursesCount + $newOfflineCoursesCount];
    }

    private function getExamData()
    {
        $allExamCount = 0;
        $newExamCount = 0;
        if ($this->isPluginInstalled('exam')) {
            $allExamCount = $this->getExamService()->countExams([]);
            $newExamCount = $this->getExamService()->countExams(['createdTime_GE' => strtotime(date('Y').'-01-01 00:00:00')]);
        }

        return ['allExamCount' => $allExamCount, 'newExamCount' => $newExamCount];
    }

    private function getActivityData()
    {
        $allActivityCount = $this->getOfflineActivityService()->countOfflineActivities([]);
        $newActivityCount = $this->getOfflineActivityService()->countOfflineActivities(['createdTime_GE' => strtotime(date('Y').'-01-01 00:00:00')]);

        return ['allActivityCount' => $allActivityCount, 'newActivityCount' => $newActivityCount];
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return OfflineActivityService
     */
    protected function getOfflineActivityService()
    {
        return $this->createService('OfflineActivity:OfflineActivityService');
    }

    /**
     * @return ExamService
     */
    protected function getExamService()
    {
        return $this->createService('ExamPlugin:Exam:ExamService');
    }

    private function getTeachersData()
    {
        $defaultConditions = [
            'locked' => 0,
            'noType' => 'system',
            'roles' => 'ROLE_TEACHER',
        ];
        $allTeachersCount = $this->getUserService()->countUsers($defaultConditions);
        $defaultConditions['startTime'] = strtotime(date('Y').'-01-01 00:00:00');
        $newTeachersCount = $this->getUserService()->countUsers($defaultConditions);

        return ['allTeachersCount' => $allTeachersCount, 'newTeachersCount' => $newTeachersCount];
    }

    private function getProjectData()
    {
        $defaultConditions = [];
        $allOfflineClassCount = 0;
        $newOfflineClassCount = 0;
        $allProjectPlansCount = $this->getProjectPlanService()->countProjectPlans($defaultConditions);
        $defaultConditions['createdTime_GE'] = strtotime(date('Y').'-01-01 00:00:00');
        $newProjectPlansCount = $this->getProjectPlanService()->countProjectPlans($defaultConditions);
        if ($this->isPluginInstalled('OfflineCoursePro')) {
            $allOfflineClassCount = $this->getOfflineClassService()->countClasses([]);
            $newOfflineClassCount = $this->getOfflineClassService()->countClasses($defaultConditions);
        }

        return ['allProjectCount' => $allProjectPlansCount + $allOfflineClassCount, 'newProjectCount' => $newProjectPlansCount + $newOfflineClassCount];
    }

    private function checkCloudAppUpgradeInfo()
    {
        $apps = $this->getAppService()->checkAppUpgrades();
        $upgradeAppCount = count($apps);
        if (!is_null($apps)) {
            $indexApps = ArrayToolkit::index($apps, 'code');
        }
        $mainAppUpgrade = empty($indexApps['TRAININGMAIN']) ? [] : $indexApps['TRAININGMAIN'];

        if ($mainAppUpgrade) {
            $upgradeAppCount = $upgradeAppCount - 1;
        }

        return [$mainAppUpgrade, $upgradeAppCount];
    }

    private function getMobileAppVersion()
    {
        return [$this->getMobileVersion('android'), $this->getMobileVersion('iphone')];
    }

    private function getEduCloudStatus()
    {
        try {
            $api = CloudAPIFactory::create('root');
            $overview = $api->get("/cloud/{$api->getAccessKey()}/overview");
            if (!$overview['enabled']) {
                return false;
            }

            if (!($overview['accessCloud'])) {
                return false;
            }

            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    public function userStatisticsAction(Request $request)
    {
        $userCount = $this->getUserService()->countUsers(['noType' => 'system', 'locked' => 0]);

        $conditions = [
            'loginStartTime' => strtotime(date('Y-m-d')),
            'loginEndTime' => strtotime(date('Y-m-d')) + 86400,
        ];

        $loginCount = $this->getUserService()->countUsers($conditions);

        $conditions = [
            'updatedTime_GE' => strtotime(date('Y-m-d')),
            'finishedTime_LT' => strtotime(date('Y-m-d')) + 86400,
        ];

        $todayLearned = $this->getTaskResultService()->searchTaskResults($conditions, ['finishedTime' => 'desc'], '0', PHP_INT_MAX);

        $users = ArrayToolkit::column($todayLearned, 'userId');
        $taskNum = count(array_unique($users));

        $userCountInfo = $this->getUserService()->countUsersByLockedStatus();

        return $this->render('default/user-statistics.html.twig', [
            'userCount' => $userCount,
            'userCountInfo' => $userCountInfo,
            'loginCount' => $loginCount,
            'learnedCount' => $taskNum,
        ]);
    }

    public function userActiveChartAction(Request $request)
    {
        $data = [];
        $count = 0;

        $condition = $request->request->all();
        $endTime = strtotime($condition['endTime']);
        $startTime = strtotime('+1 year', strtotime($condition['startTime']));
        if ($endTime > $startTime) {
            $condition['endTime'] = date('Y-m-d', $startTime);
        }
        $condition['analysisDateType'] = 'login';
        $timeRange = $this->getTimeRange($condition);

        if (!$timeRange) {
            $this->setFlashMessage('danger', 'admin.analysis.message.date_error');

            return $this->redirect($this->generateUrl('admin'));
        }

        $loginData = $this->getLogService()->analysisLoginDataByTime($timeRange['startTime'], $timeRange['endTime']);

        $data = $this->fillAnalysisData($condition, $loginData);
        foreach ($loginData as $key => $value) {
            $count += $value['count'];
        }

        $loginStartData = $this->getLogService()->searchLogs(['action' => 'login_success'], 'createdByAsc', 0, 1);

        if ($loginStartData) {
            $loginStartDate = date('Y-m-d', $loginStartData[0]['createdTime']);
        }

        $dataInfo = $this->getDataInfo($condition, $timeRange);

        return $this->render('default/user-active-chart.html.twig', [
            'data' => $data,
            'count' => $count,
            'loginStartDate' => empty($loginStartDate) ? [] : $loginStartDate,
            'dataInfo' => $dataInfo,
        ]);
    }

    public function validateSyncOrgAction(Request $request)
    {
        $inspectList = [
            $this->addInspectRole('host', $this->inspectSyncOrg($request)),
        ];
        $inspectList = array_filter($inspectList);

        return $this->render('admin/default/domain.html.twig', [
            'inspectList' => $inspectList,
        ]);
    }

    private function addInspectRole($name, $value)
    {
        if (empty($value) || 'ok' == $value['status']) {
            return [];
        }

        return ['name' => $name, 'value' => $value];
    }

    private function inspectSyncOrg($request)
    {
        $sync = $this->getSettingService()->get('sync_department_setting', []);
        $siteSetting = $this->getSettingService()->get('site');

        if ($sync && $sync['enable'] && $sync['times'] <= 0) {
            return [
                'status' => 'warning',
                'errorMessage' => ServiceKernel::instance()->trans('admin.analysis.message.inspect_sync_org'),
                'except' => $siteSetting['url'],
                'actually' => $request->server->get('HTTP_HOST'),
                'settingUrl' => $this->generateUrl('admin_org'),
            ];
        }

        return [];
    }

    protected function getDataInfo($condition, $timeRange)
    {
        $startTime = strtotime(date('Y-m', $timeRange['startTime']).'-01 00:00:01');
        $endTime = strtotime(date('Y-m', $timeRange['endTime']).'-01 00:00:01');

        return [
            'startTime' => date('Y-m-d', $timeRange['startTime']),
            'endTime' => date('Y-m-d', $timeRange['endTime'] - 24 * 3600),
            'currentMonthStart' => date('Y-m-d', strtotime(date('Y-m', time()))),
            'currentMonthEnd' => date('Y-m-d', strtotime(date('Y-m-d', time()))),
            'lastMonthStart' => date('Y-m-d', strtotime(date('Y-m', strtotime('-1 month', $startTime)))),
            'lastMonthEnd' => date('Y-m-d', strtotime(date('Y-m', $endTime)) - 24 * 3600),
            'lastThreeMonthsStart' => date('Y-m-d', strtotime(date('Y-m', strtotime('-2 month', $startTime)))),
            'lastThreeMonthsEnd' => date('Y-m-d', strtotime(date('Y-m-d', time()))),
            'analysisDateType' => $condition['analysisDateType'], ];
    }

    private function getMobileVersion($system)
    {
        $mobileVersion = CurlToolkit::request('GET', "http://www.edusoho.com/version/edusoho-training-{$system}", []);
        if (empty($mobileVersion)) {
            return;
        }

        return $mobileVersion;
    }

    /**
     * @return FocusService
     */
    protected function getFocusService()
    {
        return $this->createService('CorporateTrainingBundle:Focus:FocusService');
    }

    protected function getAppService()
    {
        return $this->createService('CloudPlatform:AppService');
    }

    protected function getSettingService()
    {
        return $this->createService('System:SettingService');
    }

    /**
     * @return ProjectPlanService
     */
    protected function getProjectPlanService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:ProjectPlanService');
    }

    /**
     * 管理后台-首页-更新日志
     *
     * @return \Symfony\Component\HttpFoundation\JsonResponse
     */
    public function changelogAction(Request $request)
    {
        $rootDir = $this->getParameter('kernel.root_dir');
        $changelogPath = dirname($rootDir).'/src/CorporateTrainingBundle/CHANGELOG';
        $changelog = explode(PHP_EOL.PHP_EOL, file_get_contents($changelogPath));
        $currentChangeLog = ChangelogToolkit::parseSingleChangelog($changelog[0]);
        $currentChangeLog['jumpUrl'] = $this->generateUrl('changelog_list');

        return $this->createJsonResponse($currentChangeLog);
    }

    protected function getCacheService()
    {
        return $this->createService('System:CacheService');
    }

    /**
     * 管理后台-首页-帮助中心
     *
     * @return \Symfony\Component\HttpFoundation\JsonResponse
     */
    public function helpCenterAction(Request $request)
    {
        $cacheName = 'admin_help_center';
        $helpCenterInfo = $this->getCacheService()->get($cacheName);

        if (null == $helpCenterInfo) {
            $ctHelpCenterUri = 'https://ct.edusoho.com/faq/api/hotlist?';
            $params = [
                'version' => System::CT_VERSION,
                'url' => $request->getUri(),
            ];
            $ctHelpCenterUrl = $ctHelpCenterUri.http_build_query($params);

            $helpCenterInfo = CurlToolkit::request('GET', $ctHelpCenterUrl, [], []);
            if (!empty($helpCenterInfo)) {
                $this->getCacheService()->clear($cacheName);
                $this->getCacheService()->set($cacheName, $helpCenterInfo, time() + 24 * 60 * 60);
            }
        }

        return $this->createJsonResponse($helpCenterInfo);
    }

    /**
     * @return OfflineClassService
     */
    protected function getOfflineClassService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineClass:OfflineClassService');
    }
}
