<?php

namespace CorporateTrainingBundle\Controller\Admin\Data;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\Paginator;
use AppBundle\Controller\BaseController;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\MemberService;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\OfflineCourseSurveyService;
use CorporateTrainingBundle\Common\DateToolkit;
use CorporateTrainingBundle\Common\OrgToolkit;
use SurveyPlugin\Biz\Survey\Service\SurveyMemberService;
use Symfony\Component\HttpFoundation\Request;

class TeacherDataController extends BaseController
{
    public function teacherOverviewAction(Request $request)
    {
        return $this->render(
            'CorporateTrainingBundle::admin/data/teacher/teacher-overview.html.twig',
            [
            ]
        );
    }

    public function teacherSurveyDataAction(Request $request)
    {
        $fields = $request->query->all();

        $conditions['roles'] = 'ROLE_TEACHER';
        $conditions['locked'] = 0;
        $users = $this->getUserService()->searchUsers(
            $conditions,
            ['createdTime' => 'DESC'],
            0,
            PHP_INT_MAX
        );
        if (!empty($fields['year']) && 'lastYear' == $fields['year']) {
            $conditions['courseCreateTime'] = ['startTime' => strtotime((date('Y') - 1).'/01/01 00:00'), 'endTime' => strtotime(date('Y').'/01/01 00:00')];
        } else {
            $conditions['courseCreateTime'] = ['startTime' => strtotime(date('Y').'/01/01 00:00'), 'endTime' => time()];
        }
        $userIds = ArrayToolkit::column($users, 'id');
        $data = $this->buildTeacherSurveyData($userIds, $conditions, $fields['type']);
        $names = [$this->trans('admin.data_center.chart.x_data'), '1~2', '2~3', '3~4', '4~5'];
        foreach ($names as $key => $name) {
            $pieData[] = ['name' => $name, 'value' => $data[$key]];
        }
        $chartData = [
                'names' => $names,
                'data' => $data,
                'count' => count($users),
                'pieData' => empty($pieData) ? [] : $pieData,
        ];

        return $this->createJsonResponse($chartData);
    }

    public function teacherProfessionFieldDataAction(Request $request)
    {
        $data = [];
        $users = $this->getUserService()->searchUsers(
            ['roles' => 'ROLE_TEACHER', 'locked' => 0],
            ['createdTime' => 'DESC'],
            0,
            PHP_INT_MAX
        );
        $userIds = empty($users) ? [-1] : ArrayToolkit::column($users, 'id');
        $professionFields = $this->getTeacherProfessionFieldService()->findAllTeacherProfessionFields();
        foreach ($professionFields as $professionField) {
            $userProfiles = $this->getProfileService()->searchProfiles(['userIds' => $userIds, 'likeTeacherProfessionFieldIds' => '|'.$professionField['id'].'|'], [], 0, PHP_INT_MAX);
            $data['name'][] = $professionField['name'];
            $data['count'][] = count($userProfiles);
            $pieData[] = ['name' => $professionField['name'], 'value' => count($userProfiles)];
        }

        $chartData = [
            'professionFieldNames' => empty($data['name']) ? [] : $data['name'],
            'data' => empty($data['count']) ? [] : $data['count'],
            'pieData' => empty($pieData) ? [] : $pieData,
        ];

        return $this->createJsonResponse($chartData);
    }

    public function teacherLevelDataAction(Request $request)
    {
        $data = [];
        $levels = $this->getLevelService()->findAllLevels();
        $users = $this->getUserService()->searchUsers(
            ['roles' => 'ROLE_TEACHER', 'locked' => 0],
            ['createdTime' => 'DESC'],
            0,
            PHP_INT_MAX
        );
        $userIds = empty($users) ? [-1] : ArrayToolkit::column($users, 'id');
        foreach ($levels as $level) {
            $userProfiles = $this->getProfileService()->searchProfiles(['userIds' => $userIds, 'levelId' => $level['id']], [], 0, PHP_INT_MAX);
            $data['name'][] = $level['name'];
            $data['count'][] = count($userProfiles);
            $seriesData[] = ['name' => $level['name'], 'value' => count($userProfiles)];
        }

        $chartData = [
            'levelNames' => empty($data['name']) ? [] : $data['name'],
            'data' => empty($data['count']) ? [] : $data['count'],
            'pieData' => empty($seriesData) ? [] : $seriesData,
        ];

        return $this->createJsonResponse($chartData);
    }

    public function teacherDetailAction(Request $request)
    {
        $conditions = $request->query->all();

        $conditions = $this->buildConditions($conditions);
        $paginator = new Paginator(
            $this->get('request'),
            $this->getUserService()->countUsers($conditions),
            10
        );
        $paginator->setBaseUrl($this->generateUrl('admin_data_center_teacher_detail_ajax'));

        $users = $this->getUserService()->searchUsers(
            $conditions,
            ['createdTime' => 'DESC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        $users = $this->buildTeacherProfile($users, $conditions);

        return $this->render('admin/data/teacher/teacher-detail.html.twig', [
            'courseCreateTime' => $conditions['courseCreateTime'],
            'paginator' => $paginator,
            'data' => json_encode($users),
        ]);
    }

    public function ajaxTeacherDetailAction(Request $request)
    {
        $conditions = $request->request->all();

        $conditions = $this->buildConditions($conditions);

        $paginator = new Paginator(
            $this->get('request'),
            $this->getUserService()->countUsers($conditions),
            10
        );

        $users = $this->getUserService()->searchUsers(
            $conditions,
            ['createdTime' => 'DESC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );
        $users = $this->buildTeacherProfile($users, $conditions);

        return $this->render('admin/data/teacher/detail-tr.html.twig', [
            'paginator' => $paginator,
            'data' => json_encode($users),
        ]);
    }

    protected function buildConditions($conditions)
    {
        $conditions = $this->fillOrgCode($conditions);
        $conditions['roles'] = 'ROLE_TEACHER';
        $conditions['locked'] = 0;
        list($startDateTime, $endDateTime) = DateToolkit::generateStartDateAndEndDate('year');

        $courseTime = ['startTime' => strtotime($startDateTime), 'endTime' => strtotime($endDateTime)];

        if (!empty($conditions['courseCreateTime'])) {
            $courseCreateTime = explode('-', $conditions['courseCreateTime']);
            $courseTime['startTime'] = strtotime($courseCreateTime[0]);
            $courseTime['endTime'] = strtotime($courseCreateTime[1].' 23:59:59');
        }

        $conditions['courseCreateTime'] = $courseTime;

        if (isset($conditions['keyword'])) {
            $conditions['keyword'] = trim($conditions['keyword']);
        }
        if (isset($conditions['keywordType']) && !empty($conditions['keyword'])) {
            $conditions[$conditions['keywordType']] = $conditions['keyword'];
            unset($conditions['keywordType']);
            unset($conditions['keyword']);
        }

        return $conditions;
    }

    protected function buildTeacherProfile($users, $conditions)
    {
        $userIds = ArrayToolkit::column($users, 'id');
        $teacherProfiles = $this->getProfileService()->searchProfiles(['userIds' => $userIds], [], 0, PHP_INT_MAX);
        $teacherProfiles = ArrayToolkit::index($teacherProfiles, 'userId');
        $teacherLevels = $this->getLevelService()->findAllLevels();
        $teacherLevels = ArrayToolkit::index($teacherLevels, 'id');
        $posts = $this->getPostService()->findPostsByIds(ArrayToolkit::column($users, 'postId'));
        $posts = ArrayToolkit::index($posts, 'id');

        $data = [];
        foreach ($users as &$user) {
            $userData = [];
            $courseSets = $this->getCourseSetService()->searchUserTeachingCourseSets(
                $user['id'],
                array_merge(['excludeStatus' => ['draft']], $conditions['courseCreateTime']),
                0,
                PHP_INT_MAX
            );
            if (!empty($courseSets)) {
                $userData['courseNum'] = count($courseSets);
                $courseSetIds = ArrayToolkit::column($courseSets, 'id');
                $userData['courseStudentNum'] = $this->getCourseSetService()->sumStudentNumByCourseSetIds($courseSetIds);
            }

            $offlineCourses = $this->getOfflineCourseService()->findPublishedOfflineCoursesByTeacherIdAndTimeRange($user['id'], $conditions['courseCreateTime']);
            if (!empty($offlineCourses)) {
                $courseIds = ArrayToolkit::column($offlineCourses, 'id');
                $time = $this->getOfflineCourseService()->statisticsOfflineCourseTimeByTimeRangeAndCourseIds($conditions['courseCreateTime'], $courseIds);
                $userData['offlineCourseTime'] = round($time, 2);
                $userData['offlineCourseStudentNum'] = $this->getOfflineCourseMemberService()->countMembers(['offlineCourseIds' => ArrayToolkit::column($offlineCourses, 'id')]);
            }

            if ($this->isPluginInstalled('Survey')) {
                $courseIds = ArrayToolkit::column($courseSets, 'defaultCourseId');
                $userData['courseSurveyScore'] = $this->getSurveyResultService()->getOnlineCoursesAverageSurveyScore($courseIds);
                $userData['offlineCourseSurveyScore'] = $this->getSurveyResultService()->getOfflineCoursesAverageSurveyScore(ArrayToolkit::column($offlineCourses, 'id'));
            }

            $data[] = $this->buildTeacherDetailUserData($user, $posts, $teacherProfiles, $teacherLevels, $userData);
        }

        return $data;
    }

    protected function buildTeacherDetailUserData($user, $posts, $teacherProfiles, $teacherLevels, $userData)
    {
        $teacherProfile = empty($teacherProfiles[$user['id']]) ? [] : $teacherProfiles[$user['id']];
        $orgs = $this->getOrgService()->findOrgsByIds($user['orgIds']);
        $orgs = OrgToolkit::buildOrgsNamesAndCodes($user['orgIds'], $orgs);
        $org['name'] = $orgs[0]['name'];
        $org['code'] = $orgs[0]['code'];
        $org['orgs'] = $orgs;
        $teacherProfessionFieldIds = empty($teacherProfiles[$user['id']]) ? [-1] : $teacherProfiles[$user['id']]['teacherProfessionFieldIds'];
        $teacherProfessionFields = $this->getTeacherProfessionFieldService()->findTeacherProfessionFieldsByIds($teacherProfessionFieldIds);
        $userProfile = $this->getUserService()->getUserProfile($user['id']);
        $userData['truename'] = empty($userProfile['truename']) ? $user['nickname'] : $userProfile['truename'];
        $userData['post'] = empty($posts[$user['postId']]) ? '-' : $posts[$user['postId']]['name'];
        $userData['level'] = empty($teacherProfile['levelId']) ? '--' : $teacherLevels[$teacherProfile['levelId']]['name'];
        $userData['courseNum'] = isset($userData['courseNum']) ? $userData['courseNum'] : 0;
        $userData['courseStudentNum'] = isset($userData['courseStudentNum']) ? $userData['courseStudentNum'] : 0;
        $userData['offlineCourseTime'] = isset($userData['offlineCourseTime']) ? $userData['offlineCourseTime'] : 0;
        $userData['offlineCourseStudentNum'] = isset($userData['offlineCourseStudentNum']) ? $userData['offlineCourseStudentNum'] : 0;
        $userData['courseSurveyScore'] = (empty($userData['courseSurveyScore']) ? '--' : $userData['courseSurveyScore']).'/5.00';
        $userData['offlineCourseSurveyScore'] = (empty($userData['offlineCourseSurveyScore']) ? '--' : $userData['offlineCourseSurveyScore']).'/5.00';
        $userData['teacherTotalScore'] = (empty($userData['teacherTotalScore']) ? '--' : $userData['teacherTotalScore']).'/5.00';
        $userData['archivesPatch'] = $this->generateUrl('admin_teacher_course_archives', ['userId' => $user['id'], 'from' => 'data-center']);
        $userData['userDetailPatch'] = $this->generateUrl('admin_user_show', ['uuid' => $user['uuid']]);
        $userData['professionField'] = $this->buildProfessionFields($teacherProfessionFieldIds, $teacherProfessionFields);
        $userData['org'] = $org;
        $userData['coverImgUri'] = $this->getCoverImgUri($user, 'middle');

        return $userData;
    }

    protected function getCoverImgUri($user, $type)
    {
        $coverPath = $this->get('web.twig.app_extension')->userAvatar($user, 'medium');

        return $this->getWebExtension()->getFpath($coverPath, 'avatar.png');
    }

    protected function buildProfessionFields(array $professionFieldIds, array $professionFields, $delimiter = '/')
    {
        $teacherProfessionField = reset($professionFields);

        $teacherProfessionFields = ['fieldNames' => '--', 'fieldName' => '--'];

        if (empty($professionFieldIds) || empty($professionFields)) {
            return $teacherProfessionFields;
        }

        $teacherProfessionFieldNames = '';
        $professionFields = ArrayToolkit::index($professionFields, 'id');
        foreach ($professionFieldIds as $professionFieldId) {
            if (!empty($professionFields[$professionFieldId])) {
                $teacherProfessionFieldNames = $teacherProfessionFieldNames.$delimiter.$professionFields[$professionFieldId]['name'];
            }
        }

        $teacherProfessionFields['fieldNames'] = trim($teacherProfessionFieldNames, $delimiter);
        $teacherProfessionFields['fieldName'] = empty($teacherProfessionField) ? '' : $teacherProfessionField['name'];

        return  $teacherProfessionFields;
    }

    protected function buildTeacherSurveyData($userIds, $conditions, $type)
    {
        $data = [0, 0, 0, 0, 0];
        foreach ($userIds as &$id) {
            $surveyScore = 0;
            $surveyResultCount = 0;
            if ('offline' != $type) {
                list($onlineSurveyScore, $onlineSurveyResultCount) = $this->buildTeacherSurveyOnlineCourseData($conditions, $id);
                $surveyScore += $onlineSurveyScore;
                $surveyResultCount += $onlineSurveyResultCount;
            }

            if ('online' != $type) {
                list($offlineSurveyScore, $offlineSurveyResultCount) = $this->buildTeacherSurveyOfflineCourseData($conditions, 2);
                $surveyScore += $offlineSurveyScore;
                $surveyResultCount += $offlineSurveyResultCount;
            }

            $surveyScore = $surveyResultCount > 0 ? sprintf('%.2f', $surveyScore / $surveyResultCount) : 0;

            if ($surveyScore < 1) {
                ++$data[0];
            } elseif ($surveyScore >= 1 && $surveyScore <= 2) {
                ++$data[1];
            } elseif ($surveyScore > 2 && $surveyScore <= 3) {
                ++$data[2];
            } elseif ($surveyScore > 3 && $surveyScore <= 4) {
                ++$data[3];
            } else {
                ++$data[4];
            }
        }

        return $data;
    }

    protected function buildTeacherSurveyOnlineCourseData($conditions, $userId)
    {
        $surveyScore = 0;
        $surveyResultCount = 0;
        $courseSets = $this->getCourseSetService()->searchUserTeachingCourseSets(
            $userId,
            array_merge(['excludeStatus' => ['draft']], $conditions['courseCreateTime']),
            0,
            PHP_INT_MAX
        );
        if (!empty($courseSets) && $this->isPluginInstalled('Survey')) {
            $courseIds = ArrayToolkit::column($courseSets, 'defaultCourseId');
            $activities = $this->getActivityService()->findActivitiesByCourseIdsAndType($courseIds, 'questionnaire');

            $surveyIds = empty($activities) ? [-1] : ArrayToolkit::column($activities, 'mediaId');
            $members = $this->getSurveyMemberService()->searchMembers(['surveyIds' => $surveyIds], [], 0, PHP_INT_MAX);
            $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');
            $surveyScore = $this->getSurveyResultService()->sumScoreBySurveyIdsAndUserIds($surveyIds, $userIds);

            $surveyResultCount = $this->getSurveyResultService()->countSurveyResults([
                'surveyIds' => $surveyIds,
                'status' => 'finished',
                'userIds' => $userIds,
            ]);
        }

        return [$surveyScore, $surveyResultCount];
    }

    protected function buildTeacherSurveyOfflineCourseData($conditions, $userId)
    {
        $surveyScore = 0;
        $surveyResultCount = 0;
        $offlineCourses = $this->getOfflineCourseService()->searchOfflineCourses(array_merge($conditions['courseCreateTime'], ['status' => 'published', 'teacherId' => $userId]), [], 0, PHP_INT_MAX);
        if (!empty($offlineCourses) && $this->isPluginInstalled('Survey')) {
            $offlineCourseIds = empty($offlineCourses) ? [-1] : ArrayToolkit::column($offlineCourses, 'id');
            $offlineCourseSurveys = $this->getOfflineCourseSurveyService()->findOfflineCourseSurveysByOfflineCourseIds($offlineCourseIds);
            $offlineCourseSurveyIds = ArrayToolkit::column($offlineCourseSurveys, 'surveyId');

            $surveyIds = (!empty($offlineCourseSurveyIds)) ? $offlineCourseSurveyIds : [-1];
            $offlineMembers = $this->getSurveyMemberService()->searchMembers(['surveyIds' => $surveyIds], [], 0, PHP_INT_MAX);
            $userIds = empty($offlineMembers) ? [-1] : ArrayToolkit::column($offlineMembers, 'userId');
            $surveyScore = $this->getSurveyResultService()->sumScoreBySurveyIdsAndUserIds($surveyIds, $userIds);

            $surveyResultCount = $this->getSurveyResultService()->countSurveyResults([
                'surveyIds' => $surveyIds,
                'status' => 'finished',
                'userIds' => $userIds,
            ]);
        }

        return [$surveyScore, $surveyResultCount];
    }

    /**
     * @return SurveyMemberService
     */
    protected function getSurveyMemberService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyMemberService');
    }

    /**
     * @return MemberService
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }

    /**
     * @return OfflineCourseSurveyService
     */
    protected function getOfflineCourseSurveyService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseSurveyService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\Course\Service\Impl\CourseSetServiceImpl
     */
    protected function getCourseSetService()
    {
        return $this->createService('CorporateTrainingBundle:Course:CourseSetService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return \SurveyPlugin\Biz\Survey\Service\Impl\SurveyResultServiceImpl
     */
    protected function getSurveyResultService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyResultService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\Teacher\Service\Impl\ProfileServiceImpl
     */
    protected function getProfileService()
    {
        return $this->createService('CorporateTrainingBundle:Teacher:ProfileService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\TeacherProfessionField\Service\Impl\TeacherProfessionFieldServiceImpl
     */
    protected function getTeacherProfessionFieldService()
    {
        return $this->createService('CorporateTrainingBundle:TeacherProfessionField:TeacherProfessionFieldService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\Teacher\Service\Impl\LevelServiceImpl
     */
    protected function getLevelService()
    {
        return $this->createService('CorporateTrainingBundle:Teacher:LevelService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\ProjectPlanServiceImpl
     */
    protected function getProjectPlanService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:ProjectPlanService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    protected function getPostService()
    {
        return $this->createService('CorporateTrainingBundle:Post:PostService');
    }

    /**
     * @return \Biz\Activity\Service\Impl\ActivityServiceImpl
     */
    protected function getActivityService()
    {
        return $this->createService('Activity:ActivityService');
    }

    /**
     * @return \Biz\Course\Service\Impl\CourseServiceImpl
     */
    protected function getCourseService()
    {
        return $this->createService('Course:CourseService');
    }
}
