<?php

namespace CorporateTrainingBundle\Controller\Admin\Train;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\Paginator;
use AppBundle\Controller\Admin\BaseController;
use Biz\File\Service\UploadFileService;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\OfflineCourseSurveyService;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\MemberService;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\ProjectPlanService;
use CorporateTrainingBundle\Biz\User\Service\UserOrgService;
use OfflineCourseProPlugin\Biz\OfflineClass\Service\OfflineClassService;
use OfflineCourseProPlugin\Biz\OfflineCourseTemplate\Service\OfflineCourseTemplateService;
use SurveyPlugin\Biz\Survey\Service\SurveyResultService;
use Symfony\Component\HttpFoundation\Request;

class OfflineCourseController extends BaseController
{
    public function teachingAction(Request $request)
    {
        if (!$this->getCurrentUser()->isTeacher()) {
            return $this->createMessageResponse('error', $this->trans('my.classroom.teaching.no_permission'));
        }

        $conditions = $request->request->all();
        $conditions = $this->buildConditions($conditions);

        $paginator = new Paginator(
            $request,
            $this->getOfflineCourseService()->countPublishedOfflineClassOrProjectPlanOfflineCourses($conditions),
            20
        );

        $offlineCourses = $this->getOfflineCourseService()->searchPublishedOfflineClassOrProjectPlanOfflineCourses(
            $conditions,
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );
        $group = ArrayToolkit::group($offlineCourses, 'targetType');
        $group['offlineClass'] = empty($group['offlineClass']) ? [] : $group['offlineClass'];
        $group['projectPlan'] = empty($group['projectPlan']) ? [] : $group['projectPlan'];
        if ($this->isPluginInstalled('OfflineCoursePro')) {
            $offlineClasses = $this->getOfflineClassService()->findClassesByIds(ArrayToolkit::column($group['offlineClass'], 'targetId'));
            $offlineCourseTemplates = $this->getOfflineCourseTemplateService()->findTemplatesByIds(ArrayToolkit::column($offlineCourses, 'templateId'));
        }
        $projectPlans = $this->getProjectPlanService()->findProjectPlansByIds(ArrayToolkit::column($group['projectPlan'], 'targetId'));

        foreach ($offlineCourses as &$offlineCourse) {
            $offlineCourse['hasSign'] = $this->getOfflineCourseTaskService()->countTasks(['offlineCourseId' => $offlineCourse['id'], 'type' => 'sign']);
            $offlineCourse['hasFiles'] = $this->hasFiles($offlineCourse['id']);
            if (!$this->isPluginInstalled('Survey')) {
                continue;
            }
            $memberUserIds = ArrayToolkit::column($this->findMembers($offlineCourse['targetType'], $offlineCourse['targetId']), 'userId');
            $offlineCourseSurveys = $this->getOfflineCourseSurveyService()->findOfflineCourseSurveysByOfflineCourseId($offlineCourse['id']);
            $surveyIds = ArrayToolkit::column($offlineCourseSurveys, 'surveyId');
            $surveyResultNum = $this->getSurveyResultService()->countSurveyResults(['surveyIds' => $surveyIds ?: [-1], 'status' => 'finished', 'userIds' => $memberUserIds]);
            $offlineCourse['surveyCount'] = count($surveyIds) * count($memberUserIds);
            $offlineCourse['averageScore'] = (0 == $surveyResultNum) ? 0 : sprintf('%.2f', $this->getSurveyResultService()->sumScoreBySurveyIdsAndUserIds($surveyIds, $memberUserIds) / $surveyResultNum);
        }

        return $this->render('CorporateTrainingBundle::admin/train/teach-manage/offline-courses/index.html.twig', [
            'offlineCourses' => $offlineCourses,
            'offlineCourseTemplates' => empty($offlineCourseTemplates) ? [] : ArrayToolkit::index($offlineCourseTemplates, 'id'),
            'offlineClasses' => empty($offlineClasses) ? [] : ArrayToolkit::index($offlineClasses, 'id'),
            'projectPlans' => empty($projectPlans) ? [] : ArrayToolkit::index($projectPlans, 'id'),
            'paginator' => $paginator,
            'orgIds' => implode(',', $conditions['orgIds']),
        ]);
    }

    public function homeworkListAction($id)
    {
        $homeworks = $this->getTaskService()->searchTasks(
            ['offlineCourseId' => $id, 'type' => 'homework'],
            ['offlineCourseId' => 'ASC', 'seq' => 'ASC'],
            0,
            PHP_INT_MAX
        );

        $offlineCourse = $this->getOfflineCourseService()->getOfflineCourse($id);

        $memberUserIds = ArrayToolkit::column($this->findMembers($offlineCourse['targetType'], $offlineCourse['targetId']), 'userId');

        foreach ($homeworks as &$homework) {
            $homework['homeworkCount'] = $this->getTaskService()->countTaskResults(['taskId' => $homework['id'], 'userIds' => $memberUserIds]);
            $homework['finishedCount'] = $this->getTaskService()->countTaskResults(['taskId' => $homework['id'], 'status' => 'finished', 'userIds' => $memberUserIds]);
        }

        return $this->render('CorporateTrainingBundle::admin/train/teach-manage/offline-courses/homework-list.html.twig', [
            'offlineCourse' => $offlineCourse,
            'homeworks' => $homeworks,
            'memberCount' => $this->countMembers($offlineCourse['targetType'], $offlineCourse['targetId']),
        ]);
    }

    public function homeworkManageAction(Request $request, $id)
    {
        $homework = $this->getOfflineCourseTaskService()->getTask($id);
        $offlineCourse = $this->getOfflineCourseService()->tryManageOfflineCourse($homework['offlineCourseId']);
        $renderData = ('offlineClass' == $offlineCourse['targetType']) ? $this->buildOfflineClassHomeworkManageData($request, $id)
            : $this->buildProjectPlanHomeworkManageData($request, $id);

        return $this->render('CorporateTrainingBundle::admin/train/teach-manage/offline-courses/homework-manage.html.twig', $renderData);
    }

    public function filesAction($id)
    {
        $offlineCourse = $this->getOfflineCourseService()->getOfflineCourse($id);

        return $this->render('CorporateTrainingBundle::admin/train/teach-manage/offline-courses/files.html.twig', [
            'targetType' => 'offlineCourseTemplate.files',
            'attachments' => $this->getUploadFileService()->findUseFilesByTargetTypeAndTargetIdAndType('offlineCourseTemplate.files', $offlineCourse['templateId'], 'attachment'),
        ]);
    }

    protected function buildConditions($conditions)
    {
        $conditions['orgIds'] = $this->prepareOrgIds($conditions);
        $conditions['teacherId'] = $this->getUser()->getId();

        if (empty($conditions['keyword'])) {
            unset($conditions['keywordType']);
            unset($conditions['keyword']);

            return $conditions;
        }
        if ('offlineCourseTitle' == $conditions['keywordType']) {
            $conditions['likeTitle'] = $conditions['keyword'];
        }
        if ('offlineCourseTemplateId' == $conditions['keywordType']) {
            $conditions['templateId'] = $conditions['keyword'];
        }

        unset($conditions['keywordType']);
        unset($conditions['keyword']);

        return $conditions;
    }

    protected function buildProjectPlanHomeworkManageData($request, $taskId, $type = '')
    {
        $offlineCourseTask = $this->getOfflineCourseTaskService()->getTask($taskId);
        $offlineCourse = $this->getOfflineCourseService()->tryManageOfflineCourse($offlineCourseTask['offlineCourseId']);

        $projectPlan = $this->getProjectPlanService()->getProjectPlan($offlineCourse['targetId']);
        $conditions = $request->request->all();
        $conditions['projectPlanId'] = $projectPlan['id'];
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $conditions['projectPlanId']], [], 0, PHP_INT_MAX, ['userId']);
        $conditions = $this->prepareConditionByUserIds(ArrayToolkit::column($members, 'userId'), $conditions);
        $conditions = $this->prepareConditionsByHomeworkStatus($conditions, $taskId);

        $paginator = new Paginator(
            $request,
            $this->getProjectPlanMemberService()->countProjectPlanMembers($conditions),
            20
        );

        if ('ajax' != $type) {
            $paginator->setBaseUrl($this->generateUrl('project_plan_offline_course_homework_manage_ajax', ['taskId' => $taskId]));
        }

        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(
            $conditions,
            ['id' => 'DESC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');
        $users = $this->getUserService()->findUsersByIds($userIds);
        $taskResults = $this->getOfflineCourseTaskResultService()->searchTaskResults(['taskId' => $taskId, 'userIds' => $userIds], [], 0, count($userIds));

        return [
            'offlineCourse' => $offlineCourse,
            'members' => $members,
            'paginator' => $paginator,
            'users' => $users,
            'taskId' => $taskId,
            'taskResults' => ArrayToolkit::index($taskResults, 'userId'),
            'orgIds' => implode(',', $this->prepareOrgIds($conditions)),
            'resourceId' => $projectPlan['id'],
            'type' => 'project_plan',
            'projectPlan' => $this->getProjectPlanService()->getProjectPlan($projectPlan['id']),
        ];
    }

    protected function buildOfflineClassHomeworkManageData($request, $taskId, $type = '')
    {
        $offlineCourseTask = $this->getOfflineCourseTaskService()->getTask($taskId);
        $offlineCourse = $this->getOfflineCourseService()->tryManageOfflineCourse($offlineCourseTask['offlineCourseId']);
        $offlineClass = $this->getOfflineClassService()->getClass($offlineCourse['targetId']);

        $conditions = $request->request->all();
        $conditions['offlineClassId'] = $offlineClass['id'];
        $members = $this->getOfflineClassMemberService()->searchMembers(['offlineClassId' => $offlineClass['id']], [], 0, PHP_INT_MAX, ['userId']);
        $conditions = $this->prepareConditionByUserIds(ArrayToolkit::column($members, 'userId'), $conditions);
        $conditions = $this->prepareConditionsByHomeworkStatus($conditions, $taskId);

        $paginator = new Paginator(
            $request,
            $this->getOfflineClassMemberService()->countMembers($conditions),
            20
        );

        if ('ajax' != $type) {
            $paginator->setBaseUrl($this->generateUrl('offline_class_offline_course_homework_manage_ajax', ['taskId' => $taskId]));
        }

        $members = $this->getOfflineClassMemberService()->searchMembers(
            $conditions,
            ['id' => 'DESC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');
        $users = $this->getUserService()->findUsersByIds($userIds);
        $taskResults = $this->getOfflineCourseTaskResultService()->searchTaskResults(['taskId' => $taskId, 'userIds' => $userIds], [], 0, count($userIds));

        return [
            'offlineCourse' => $offlineCourse,
            'members' => $members,
            'paginator' => $paginator,
            'users' => $users,
            'taskId' => $taskId,
            'taskResults' => ArrayToolkit::index($taskResults, 'userId'),
            'orgIds' => implode(',', $this->prepareOrgIds($conditions)),
            'offlineClass' => $this->getOfflineClassService()->getClass($offlineClass['id']),
            'resourceId' => $offlineClass['id'],
            'type' => 'offline_class',
        ];
    }

    protected function prepareConditionByUserIds($userIds, $conditions)
    {
        if (isset($conditions['orgIds'])) {
            $conditions['orgIds'] = explode(',', $conditions['orgIds']);
            $users = $this->getUserOrgService()->searchUserOrgs(
                ['orgIds' => $conditions['orgIds'], 'userIds' => $userIds],
                [],
                0,
                PHP_INT_MAX,
                ['userId']
            );
            $conditions['userIds'] = ArrayToolkit::column($users, 'userId');
        }

        if (!empty($conditions['username'])) {
            $userIds = $this->getUserService()->findUserIdsByNickNameOrTrueName($conditions['username']);
            $conditions['userIds'] = (empty($conditions['userIds']) || empty($userIds)) ? [] : array_intersect($conditions['userIds'], $userIds);
            unset($conditions['username']);
        }

        if (isset($conditions['userIds'])) {
            $conditions['userIds'] = empty($conditions['userIds']) ? [-1] : $conditions['userIds'];
        }

        return $conditions;
    }

    protected function prepareConditionsByHomeworkStatus($conditions, $taskId)
    {
        if (!empty($conditions['homeworkStatus']) && !in_array($conditions['homeworkStatus'], ['none', 'all'])) {
            $results = $this->getOfflineCourseTaskResultService()->searchTaskResults(['status' => $conditions['homeworkStatus'], 'userIds' => $conditions['userIds'], 'taskId' => $taskId], [], 0, count($conditions['userIds']), ['userId']);
            $userIds = ArrayToolkit::column($results, 'userId');
            $conditions['userIds'] = (empty($conditions['userIds']) || empty($userIds)) ? [] : array_intersect($conditions['userIds'], $userIds);
        }

        if (!empty($conditions['homeworkStatus']) && 'none' == $conditions['homeworkStatus']) {
            $results = $this->getOfflineCourseTaskResultService()->searchTaskResults(['taskId' => $taskId], [], 0, PHP_INT_MAX, ['userId']);
            $userIds = ArrayToolkit::column($results, 'userId');
            $conditions['userIds'] = empty($conditions['userIds']) ? [] : array_diff($conditions['userIds'], $userIds);
        }

        unset($conditions['homeworkStatus']);
        if (isset($conditions['userIds'])) {
            $conditions['userIds'] = empty($conditions['userIds']) ? [-1] : $conditions['userIds'];
        }

        return $conditions;
    }

    protected function hasFiles($id)
    {
        $files = $this->findFiles($id);

        return !empty($files);
    }

    protected function findFiles($id)
    {
        $offlineCourse = $this->getOfflineCourseService()->getOfflineCourse($id);

        return $this->getUploadFileService()->findUseFilesByTargetTypeAndTargetIdAndType('offlineCourseTemplate.files', $offlineCourse['templateId'], 'attachment');
    }

    protected function countMembers($type, $id)
    {
        if ('projectPlan' == $type) {
            return $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $id]);
        }

        if ('offlineClass' == $type) {
            return $this->getOfflineClassMemberService()->countMembers(['offlineClassId' => $id]);
        }

        return 0;
    }

    protected function findMembers($type, $id)
    {
        if ('projectPlan' == $type) {
            return $this->getProjectPlanMemberService()->findMembersByProjectPlanId($id);
        }

        if ('offlineClass' == $type) {
            return $this->getOfflineClassMemberService()->findMembersByOfflineClassId($id);
        }

        return [];
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return OfflineCourseTemplateService
     */
    protected function getOfflineCourseTemplateService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineCourseTemplate:OfflineCourseTemplateService');
    }

    /**
     * @return ProjectPlanService
     */
    protected function getProjectPlanService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:ProjectPlanService');
    }

    /**
     * @return MemberService
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return OfflineClassService
     */
    protected function getOfflineClassService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineClass:OfflineClassService');
    }

    /**
     * @return \OfflineCourseProPlugin\Biz\OfflineClass\Service\MemberService
     */
    protected function getOfflineClassMemberService()
    {
        return $this->createService('OfflineCourseProPlugin:OfflineClass:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return UploadFileService
     */
    protected function getUploadFileService()
    {
        return $this->createService('File:UploadFileService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskResultService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return OfflineCourseSurveyService
     */
    protected function getOfflineCourseSurveyService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseSurveyService');
    }

    /**
     * @return SurveyResultService
     */
    protected function getSurveyResultService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyResultService');
    }

    /**
     * @return UserOrgService
     */
    protected function getUserOrgService()
    {
        return $this->createService('CorporateTrainingBundle:User:UserOrgService');
    }
}
