<?php

namespace CorporateTrainingBundle\Controller\Admin;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\Exception\InvalidArgumentException;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkService;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkSyncUserService;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkUserService;
use CorporateTrainingBundle\Component\EIMClient\DepartmentFactory;
use Symfony\Component\HttpFoundation\Request;
use AppBundle\Component\OAuthClient\OAuthClientFactory;
use Codeages\Biz\Framework\Scheduler\Service\SchedulerService;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;
use AppBundle\Controller\Admin\UserSettingController as BaseController;
use Topxia\Service\Common\ServiceKernel;

class UserSettingController extends BaseController
{
    public function loginConnectAction(Request $request)
    {
        $clients = OAuthClientFactory::clients();
        $default = $this->getDefaultLoginConnect($clients);
        $loginConnect = $this->getSettingService()->get('login_bind', array());
        $loginConnect = array_merge($default, $loginConnect);
        $syncSetting = $this->getSettingService()->get('sync_department_setting', array());

        if ($request->isMethod('POST')) {
            $loginConnect = $request->request->all();
            $loginConnect = ArrayToolkit::trim($loginConnect);
            $this->validateTemporaryLockSetting($loginConnect);
            $loginConnect = $this->fillTemporaryLockSetting($loginConnect);
            $syncSetting['enable'] = isset($syncSetting['enable']) ? $syncSetting['enable'] : 0;

            if ('close' == $loginConnect['dingtalkMode'] && !$syncSetting['enable']) {
                $loginConnect['dingtalkweb_enabled'] = 0;
                $loginConnect['dingtalkmob_enabled'] = 0;
            } else {
                $loginConnect['dingtalkweb_enabled'] = 1;
                $loginConnect['dingtalkmob_enabled'] = 1;
            }

            $loginConnect = $this->decideEnabledLoginConnect($loginConnect);
            $this->getSettingService()->set('login_bind', $loginConnect);
            $this->updateWeixinMpFile($loginConnect['weixinmob_mp_secret']);
            $this->getLogService()->info('system', 'update_settings', '更新登录设置', $loginConnect);
        }

        return $this->render('admin/system/login-connect.html.twig', array(
            'loginConnect' => $loginConnect,
            'clients' => $clients,
        ));
    }

    public function syncAccountDockingAction(Request $request)
    {
        $syncSetting = $this->getSettingService()->get('sync_department_setting', array());
        $loginConnect = $this->getSettingService()->get('login_bind', array());
        if ($request->isMethod('POST')) {
            $data = $request->request->all();

            if ('dingtalk' == $data['sync_mode']) {
                $this->setDingTalkSyncMode($data, $loginConnect, $syncSetting);
            } elseif ('closed' == $data['sync_mode']) {
                $this->closeSyncMode($loginConnect, $syncSetting);
            }
            $this->setFlashMessage('success', 'admin.user_setting.message.sync_account_docking');

            $syncSetting = $this->getSettingService()->get('sync_department_setting', array());
            $loginConnect = $this->getSettingService()->get('login_bind', array());
        }

        $syncMode = $this->getSyncMode($syncSetting);
        $host = $this->generateUrl('homepage', array(), UrlGeneratorInterface::ABSOLUTE_URL);

        return $this->render('@CorporateTraining/admin/system/sync-account-docking.html.twig', array(
            'loginConnect' => $loginConnect,
            'syncSetting' => $syncSetting,
            'sync_mode' => $syncMode,
            'host' => $host,
            'ip' => gethostbyname($request->getHost()),
        ));
    }

    protected function setDingTalkSyncMode($data, $loginConnect, $syncSetting)
    {
        $biz = $this->getBiz();

        $biz['db']->beginTransaction();
        try {
            if (isset($data['dingtalksync_key']) && isset($data['dingtalksync_secret'])) {
                $setting = array(
                    'enable' => 1,
                    'type' => 'dingtalk',
                    'agentId' => $data['dingtalksync_agentId'],
                    'key' => $data['dingtalksync_key'],
                    'secret' => $data['dingtalksync_secret'],
                    'syncTime' => (empty($syncSetting['syncTime'])) ? 0 : $syncSetting['syncTime'],
                    'times' => (empty($syncSetting['times'])) ? 0 : $syncSetting['times'],
                    'dingtalk_jobnumber_to_nickname' => isset($data['dingtalk_jobnumber_to_nickname']) ? 1 : 0,
                );

                $this->getSettingService()->set('sync_department_setting', $setting);
            }

            $loginConnect['dingtalkweb_key'] = $data['dingtalkweb_key'];
            $loginConnect['dingtalkweb_secret'] = $data['dingtalkweb_secret'];
            $loginConnect['dingtalkmob_key'] = $loginConnect['dingtalkweb_key'];
            $loginConnect['dingtalkmob_secret'] = $loginConnect['dingtalkweb_secret'];
            $loginConnect['dingtalkMode'] = 'login';
            $loginConnect['enabled'] = 1;
            $loginConnect['dingtalkweb_enabled'] = 1;
            $loginConnect['dingtalkmob_enabled'] = 1;
            $this->getSettingService()->set('login_bind', $loginConnect);

            $this->getDingTalkSyncUserService()->createSyncDingTalkJobs();
            $biz['db']->commit();
        } catch (\Exception $e) {
            $biz['db']->rollback();
            throw $e;
        }
    }

    public function checkDingTalkSyncSettingAction(Request $request)
    {
        $data = $request->request->all();
        if (empty($data['dingtalksync_key']) || empty($data['dingtalksync_secret']) || empty($data['dingtalksync_agentId'])) {
            return $this->createJsonResponse(array('status' => 'error', 'message' => '请先完善配置！'));
        }
        $syncSetting = $this->getSettingService()->get('sync_department_setting', array());
        $syncSetting['key'] = $data['dingtalksync_key'];
        $syncSetting['secret'] = $data['dingtalksync_secret'];
        $syncSetting['agentId'] = $data['dingtalksync_agentId'];
        $this->getSettingService()->set('sync_department_setting', $syncSetting);
        $setting = $this->getSettingService()->get('sync_department_setting', []);

        $client = DepartmentFactory::create($setting);
        $result = $client->lists();
        if (!empty($result['errcode'])) {
            $this->getLogService()->error('dingtalk', 'department_sync_from', $result['errmsg']);
            $result = array('status' => 'error', 'message' => '配置存在错误，详情请在系统日志中查看');
        } else {
            $result = array('status' => 'success', 'message' => $this->trans('admin.user_setting.account_synchronization.setting_success'));
        }

        return $this->createJsonResponse($result);
    }

    public function syncUserAction(Request $request)
    {
        if ('POST' == $request->getMethod()) {
            $status = $request->request->get('status', '');
            if (empty($status)) {
                return $this->createJsonResponse(['status' => 'end']);
            }

            if ('syncOrg' == $status) {
                return  $this->syncDingTalkOrg();
            }

            if ('lockUser' == $status) {
                return  $this->lockDingTalkUser();
            }

            return  $this->syncDingTalkUser($request);
        }

        return $this->render('@CorporateTraining/admin/system/sync-user-modal.html.twig');
    }

    protected function syncOrgBeforeFirstSyncUser()
    {
        return $this->getOrgSyncService()->syncFrom();
    }

    protected function getNextOrgSyncId($syncId)
    {
        $org = $this->getOrgService()->searchOrgs(['syncId_GT' => $syncId], ['syncId' => 'ASC'], 0, 1);

        return empty($org) ? 0 : $org[0]['syncId'];
    }

    protected function getLastOrgSyncId()
    {
        $org = $this->getOrgService()->searchOrgs([], ['syncId' => 'DESC'], 0, 1);

        return empty($org) ? 0 : $org[0]['syncId'];
    }

    protected function closeSyncMode($loginConnect, $syncSetting)
    {
        $biz = $this->getBiz();

        $biz['db']->beginTransaction();
        try {
            $syncSetting['enable'] = 0;
            $this->getSettingService()->set('sync_department_setting', $syncSetting);

            $loginConnect['dingtalkMode'] = 'close';
            $loginConnect['dingtalkweb_enabled'] = 0;
            $loginConnect['dingtalkmob_enabled'] = 0;
            if (empty($loginConnect['weixinweb_enabled']) && empty($loginConnect['weixinmob_enabled'])) {
                $loginConnect['enabled'] = 0;
            }
            $this->getSettingService()->set('login_bind', $loginConnect);
            $this->getSettingService()->delete('dingtalk_notification');
            $this->getDingTalkSyncUserService()->deleteSyncDingTalkJobs();
            $biz['db']->commit();
        } catch (\Exception $e) {
            $biz['db']->rollback();
            throw $e;
        }
    }

    private function getSyncMode($syncSetting)
    {
        if (!empty($syncSetting) && $syncSetting['enable']) {
            return $syncMode = 'dingtalk';
        }

        return $syncMode = 'closed';
    }

    private function getDefaultLoginConnect($clients)
    {
        $default = array(
            'login_limit' => 0,
            'enabled' => 0,
            'verify_code' => '',
            'captcha_enabled' => 0,
            'temporary_lock_enabled' => 0,
            'temporary_lock_allowed_times' => 5,
            'ip_temporary_lock_allowed_times' => 20,
            'temporary_lock_minutes' => 20,
        );

        foreach ($clients as $type => $client) {
            $default["{$type}_enabled"] = 0;
            $default["{$type}_key"] = '';
            $default["{$type}_secret"] = '';
            $default["{$type}_set_fill_account"] = 0;
            if ('weixinmob' == $type) {
                $default['weixinmob_mp_secret'] = '';
            }
        }

        return $default;
    }

    private function decideEnabledLoginConnect($loginConnect)
    {
        if (0 == $loginConnect['enabled']) {
            $loginConnect['only_third_party_login'] = 0;
            $loginConnect['weibo_enabled'] = 0;
            $loginConnect['qq_enabled'] = 0;
            $loginConnect['renren_enabled'] = 0;
            $loginConnect['weixinweb_enabled'] = 0;
            $loginConnect['weixinmob_enabled'] = 0;
            $loginConnect['dingtalkweb_enabled'] = 0;
        }
        //新增第三方登陆方式，加入下列列表计算，以便判断是否关闭第三方登陆功能
        $loginConnects = ArrayToolkit::parts($loginConnect, array('weibo_enabled', 'qq_enabled', 'renren_enabled', 'weixinweb_enabled', 'weixinmob_enabled', 'dingtalkweb_enabled'));
        $sum = 0;
        foreach ($loginConnects as $value) {
            $sum += $value;
        }

        if ($sum < 1) {
            if (1 == $loginConnect['enabled']) {
                $this->setFlashMessage('danger', 'admin.user_setting.message.login_connect.error');
            }
            if (0 == $loginConnect['enabled']) {
                $this->setFlashMessage('success', 'admin.user_setting.message.login_connect.success');
            }
            $loginConnect['enabled'] = 0;
        } else {
            $loginConnect['enabled'] = 1;
            $this->setFlashMessage('success', 'admin.user_setting.message.login_connect.success');
        }

        if (isset($loginConnect['dingtalkweb_enabled'])) {
            $loginConnect['dingtalkmob_enabled'] = $loginConnect['dingtalkweb_enabled'];
        }

        if (isset($loginConnect['dingtalkweb_key'])) {
            $loginConnect['dingtalkmob_key'] = $loginConnect['dingtalkweb_key'];
        }

        if (isset($loginConnect['dingtalkweb_secret'])) {
            $loginConnect['dingtalkmob_secret'] = $loginConnect['dingtalkweb_secret'];
        }

        return $loginConnect;
    }

    private function validateTemporaryLockSetting($loginConnect)
    {
        if (empty($loginConnect['temporary_lock_enabled'])) {
            return;
        }
        $checkFields = array(
            'temporary_lock_allowed_times',
            'ip_temporary_lock_allowed_times',
            'temporary_lock_minutes',
        );
        foreach ($checkFields as $checkField) {
            if (empty($loginConnect[$checkField]) || !preg_match('/^[1-9][0-9]{0,2}$/', $loginConnect[$checkField])) {
                throw new InvalidArgumentException('Temporary lock args is invalid');
            }
        }
    }

    private function fillTemporaryLockSetting($loginConnect)
    {
        $originLoginConnect = $this->getSettingService()->get('login_bind', array());
        if (!empty($loginConnect['temporary_lock_enabled']) && empty($originLoginConnect['temporary_lock_enabled'])) {
            $loginConnect['temporary_lock_allowed_times'] = $originLoginConnect['temporary_lock_allowed_times'];
            $loginConnect['ip_temporary_lock_allowed_times'] = $originLoginConnect['ip_temporary_lock_allowed_times'];
            $loginConnect['temporary_lock_minutes'] = $originLoginConnect['temporary_lock_minutes'];
        }

        return $loginConnect;
    }

    protected function syncDingTalkOrg()
    {
        $setting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
        $this->getSettingService()->set('dingTalk_sync_user_setting', array_merge($setting, ['syncStartTime' => time()]));

        $result = $this->syncOrgBeforeFirstSyncUser();
        if (empty($result['success'])) {
            $this->getLogService()->error('dingtalk', 'department_sync_from', $result['message']);

            return $this->createJsonResponse(['status' => 'error', 'message' => '配置存在错误，详情请在系统日志中查看']);
        }
        $org = $this->getOrgService()->searchOrgs(['syncId_GT' => 0], ['syncId' => 'ASC'], 0, 1);
        $departmentId = empty($org) ? 0 : $org[0]['syncId'];
        $totalSyncCount = $this->getOrgService()->countOrgs(['syncId_GT' => 0]) + 2;

        return $this->createJsonResponse(['hasMore' => false, 'progress' => round(100 / $totalSyncCount, 2), 'status' => 'syncUser', 'offset' => 0, 'departmentId' => $departmentId]);
    }

    protected function lockDingTalkUser()
    {
        $setting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
        $count = $this->getDingTalkUserService()->countDingTalkUsers(['syncTime_LT' => empty($setting['syncStartTime']) ? strtotime(date('Y-m-d', time()).' 02:00:00') : $setting['syncStartTime']]);
        if ($count <= 0) {
            return $this->createJsonResponse(['status' => 'end']);
        }
        $this->getDingTalkSyncUserService()->lockDingTalkUsers();

        return $this->createJsonResponse(['progress' => 99.9, 'status' => 'lockUser', 'offset' => 0, 'departmentId' => 0]);
    }

    protected function syncDingTalkUser(Request $request)
    {
        $departmentId = $request->request->get('departmentId', 0);
        $needSyncCount = $this->getOrgService()->countOrgs(['syncId_GT' => $departmentId]);
        $totalSyncCount = $this->getOrgService()->countOrgs(['syncId_GT' => 0]) + 2;
        if (empty($departmentId) || 0 == $totalSyncCount) {
            return $this->createJsonResponse(['status' => 'end']);
        }

        $offset = $request->request->get('offset', 0);
        $progress = round(($totalSyncCount - $needSyncCount) * 100 / $totalSyncCount, 2);
        $result = $this->getDingTalkService()->syncDingTalkUser($departmentId, $offset, 50);

        if (isset($result['hasMore']) && $result['hasMore']) {
            return  $this->createJsonResponse(['hasMore' => true, 'progress' => $progress, 'status' => 'syncUser', 'offset' => $offset + 100, 'departmentId' => $departmentId]);
        }

        $lastOrgId = $this->getLastOrgSyncId();
        if ($departmentId == $lastOrgId) {
            $setting = $this->getSettingService()->get('dingTalk_sync_user_setting', []);
            $this->getSettingService()->set('dingTalk_sync_user_setting', array_merge($setting, ['finishTime' => time()]));

            return $this->createJsonResponse(['progress' => round(($totalSyncCount - 1) * 100 / $totalSyncCount, 2), 'status' => 'lockUser', 'offset' => 0, 'departmentId' => $departmentId]);
        }

        return $this->createJsonResponse(['hasMore' => false, 'progress' => $progress, 'status' => 'syncUser', 'offset' => 0, 'departmentId' => $this->getNextOrgSyncId($departmentId)]);
    }

    /**
     * @return DingTalkService
     */
    protected function getDingTalkService()
    {
        return $this->createService('CorporateTrainingBundle:DingTalk:DingTalkService');
    }

    /**
     * @return \LDAPPlugin\Biz\LDAP\Service\Impl\LDAPServiceImpl
     */
    protected function getLdapService()
    {
        return $this->createService('LDAPPlugin:LDAP:LdapService');
    }

    /**
     * @return SchedulerService
     */
    protected function getSchedulerService()
    {
        return $this->createService('Scheduler:SchedulerService');
    }

    /**
     * @return DingTalkSyncUserService
     */
    protected function getDingTalkSyncUserService()
    {
        return ServiceKernel::instance()->createService('CorporateTrainingBundle:DingTalk:DingTalkSyncUserService');
    }

    /**
     * @return DingTalkUserService
     */
    protected function getDingTalkUserService()
    {
        return $this->createService('CorporateTrainingBundle:DingTalk:DingTalkUserService');
    }

    protected function getOrgSyncService()
    {
        return $this->createService('OrgSync:OrgSyncService');
    }
}
