<?php

namespace CorporateTrainingBundle\Controller;

use AppBundle\Common\Exception\InvalidArgumentException;
use AppBundle\Common\SimpleValidator;
use AppBundle\Controller\OAuth2\OAuthUser;
use Biz\System\Service\SessionService;
use Biz\User\CurrentUser;
use Biz\User\Dao\UserDao;
use Biz\User\Dao\UserProfileDao;
use Codeages\Biz\Framework\Service\Exception\NotFoundException;
use CorporateTrainingBundle\Biz\DingTalk\Client\DingTalkClientFactory;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkSyncUserService;
use CorporateTrainingBundle\Biz\DingTalk\Service\DingTalkUserService;
use CorporateTrainingBundle\Biz\OrgSync\Service\OrgSyncService;
use Symfony\Component\HttpFoundation\Request;
use AppBundle\Controller\LoginBindController as BaseLoginBindController;
use Topxia\Service\Common\ServiceKernel;

class LoginBindController extends BaseLoginBindController
{
    public function callbackAction(Request $request, $type)
    {
        $code = $request->query->get('code');
        $inviteCode = $request->query->get('inviteCode');
        $token = $request->query->get('token', '');

        $this->validateToken($request, $type);

        $callbackUrl = $this->generateUrl('login_bind_callback', ['type' => $type, 'token' => $token], true);
        $oauthClient = $this->createOAuthClient($type);
        $token = $oauthClient->getAccessToken($code, $callbackUrl);
        $bind = $this->getUserService()->getUserBindByTypeAndFromId($type, $token['userId']);
        $this->createDingTalkUserRecord($token);

        $request->getSession()->set('oauth_token', $token);
        if ($bind) {
            return $this->forward(
                'CorporateTrainingBundle:LoginBind:loginWithBindUser',
                ['request' => $request, 'bind' => $bind, 'type' => $type, 'token' => $token]
            );
        }

        if (in_array($type, ['dingtalkweb', 'dingtalkmob']) && $this->isEnableSyncDepartment()) {
            //dingding第三方登录需要同步数据
            return $this->forward(
                'CorporateTrainingBundle:LoginBind:dingTalkCallbackSyncDepartment',
                ['request' => $request, 'type' => $type, 'token' => $token]
            );
        }

        if (in_array($type, ['dingtalkweb', 'dingtalkmob'])) {
            $oUser = [
                'id' => $token['userId'],
                'name' => $token['name'],
                'avatar' => $token['avatar'],
            ];
        } else {
            $oUser = $oauthClient->getUserInfo($token);
        }

        $this->storeOauthUserToSession($request, $oUser, $type);

        return $this->redirect($this->generateUrl('oauth2_login_index', ['inviteCode' => $inviteCode]));
    }

    public function loginWithBindUserAction(Request $request, $bind, $type, $token)
    {
        $user = $this->getUserService()->getUser($bind['toId']);
        if (empty($user)) {
            $this->setFlashMessage('danger', 'login_bind.call_back.message.user_empty_error');

            return $this->redirect($this->generateUrl('login'));
        }

        if (1 == $user['locked']) {
            $this->setFlashMessage('danger', 'login_bind.call_back.message.user_locked_error');

            return $this->redirect($this->generateUrl('login'));
        }

        if ($this->isPluginInstalled('LDAP')) {
            return $this->forward('LDAPPlugin:LoginBind:LDAPCallback',
                ['request' => $request, 'type' => $type, 'token' => $token, 'user' => $user]
            );
        }

        $this->authenticateUser($user);

        if ($this->getAuthService()->hasPartnerAuth()) {
            return $this->redirect($this->generateUrl('partner_login', ['goto' => $this->getTargetPath($request)]));
        }

        $goto = $this->getTargetPath($request);

        if (!$user['pwdInit'] && in_array($type, ['dingtalkweb', 'dingtalkmob'])) {
            $url = $this->generateUrl('pwd_init');
            $queries = ['goto' => $goto];
            $url = $url.'?'.http_build_query($queries);

            return $this->redirect($url);
        }

        return $this->redirect($goto);
    }

    //dingding第三方登录需要同步数据
    public function dingTalkCallbackSyncDepartmentAction(Request $request, $type, $token)
    {
        if (empty($token['dingtalkUserid'])) {
            $this->setFlashMessage('danger', 'login_bind.call_back.dingtalkweb.message.user_empty');

            return $this->redirect($this->generateUrl('login'));
        }

        $dingTalkClient = DingTalkClientFactory::create();
        $result = $dingTalkClient->getUserInfoByUserid($token['dingtalkUserid']);
        $this->getDingTalkSyncUserService()->syncDingTalkUsers([$result]);
        $bind = $this->getUserService()->getUserBindByTypeAndFromId($type, $token['userId']);

        if (empty($bind)) {
            $this->setFlashMessage('danger', 'login_bind.call_back.message.register_error');

            return $this->redirect($this->generateUrl('login'));
        }

        return $this->forward('CorporateTrainingBundle:LoginBind:loginWithBindUser',
            ['request' => $request, 'bind' => $bind, 'type' => $type, 'token' => $token]
        );
    }

    public function initDingTalkUserAction(Request $request)
    {
        $currentUser = $this->getCurrentUser();

        if ($request->isMethod('POST')) {
            $oauthUser = $request->getSession()->get(OAuthUser::SESSION_KEY);
            $oauthUser->accountType = $request->request->get('accountType');
            $oauthUser->account = $request->request->get('account');
            $user = $this->getUserByTypeAndAccount($oauthUser->accountType, $oauthUser->account);
            $bind = $this->getUserService()->getUserBindByTypeAndUserId('dingtalk', $user['id']);
            if (!empty($bind) && $bind['toId'] != $currentUser['id']) {
                throw new InvalidArgumentException('选择的用户已绑定钉钉！');
            }

            return $this->render('@CorporateTraining/login/init-user/bind-login.html.twig',
                [
                    'oauthUser' => $oauthUser,
                    'esUser' => $user,
                ]
            );
        }

        return $this->render('@CorporateTraining/login/init-user/bind-user.html.twig',
            ['oauthUser' => $this->getOauthUser($request)]
        );
    }

    public function bindDingTalkUserAction(Request $request)
    {
        $accountType = $request->request->get('accountType');
        $account = $request->request->get('account');
        $password = $request->request->get('password');
        $newUser = $this->getUserByTypeAndAccount($accountType, $account);
        if (empty($newUser)) {
            throw new InvalidArgumentException('绑定用户失败！');
        }
        $bind = $this->getUserService()->getUserBindByTypeAndUserId('dingtalk', $newUser['id']);
        if (!empty($bind)) {
            throw new InvalidArgumentException('用户已绑定');
        }
        $isCorrectPassword = $this->getUserService()->verifyPassword($newUser['id'], $password);
        if (!$isCorrectPassword) {
            throw new InvalidArgumentException('密码匹配失败！');
        }
        $currentUser = $this->getCurrentUser();
        $bind = $this->getUserService()->getUserBindByTypeAndUserId('dingtalk', $currentUser->getId());
        $this->processUserBind($bind, $newUser);

        return $this->createJsonResponse(['url' => $this->generateUrl('homepage')]);
    }

    protected function getOauthUser($request)
    {
        $currentUser = $this->getCurrentUser();
        $bind = $this->getUserService()->getUserBindByTypeAndUserId('dingtalk', $currentUser['id']);
        if (empty($bind)) {
            throw new NotFoundException('你还未绑定用户！');
        }
        $dingTalkClient = DingTalkClientFactory::create();
        $oUser = $dingTalkClient->getUserInfoByUnionid($bind['fromId']);
        $oUser = [
            'id' => $currentUser->getId(),
            'name' => $oUser['name'],
            'avatar' => $oUser['avatar'],
        ];

        return $this->storeOauthUserToSession($request, $oUser, 'dingtalk');
    }

    protected function processUserBind($bind, $newUser)
    {
        $currentUser = $this->getCurrentUser();
        if ($currentUser['pwdInit'] || empty($bind)) {
            return true;
        }
        $this->getUserBindDao()->update($bind['id'], ['toId' => $newUser['id']]);
        $this->getUserDao()->update($currentUser['id'], [
            'email' => $this->getUserService()->generateEmail([]),
            'nickname' => $this->getUserService()->generateNickname([]),
            'locked' => 1,
            'type' => 'deleted',
            'verifiedMobile' => '',
            ]);
        $this->getSessionService()->clearByUserId($currentUser['id']);
        $this->authenticateUser($newUser);
    }

    protected function getUserByTypeAndAccount($type, $account)
    {
        $user = null;
        switch ($type) {
            case OAuthUser::EMAIL_TYPE:
                $user = $this->getUserService()->getUserByEmail($account);
                break;
            case OAuthUser::MOBILE_TYPE:
                $user = $this->getUserService()->getUserByVerifiedMobile($account);
                break;
            case OAuthUser::NICKNAME_TYPE:
                $user = $this->getUserService()->getUserByNickname($account);
                break;
            default:
                throw new NotFoundHttpException();
        }

        return $user;
    }

    protected function createDingTalkUserRecord($token)
    {
        $syncSetting = $this->getSettingService()->get('sync_department_setting', []);
        if (empty($token['dingtalkUserid']) || empty($syncSetting['key']) || empty($syncSetting['secret'])) {
            return true;
        }
        $this->getDingTalkUserService()->createUser(
            ['unionid' => $token['dingtalkUnionid'], 'userid' => $token['dingtalkUserid']]
        );
    }

    protected function isEnableSyncDepartment()
    {
        $enable = false;
        $setting = $this->getSettingService()->get('sync_department_setting', []);
        if (!empty($setting['enable'])) {
            $enable = true;
        }

        return $enable;
    }

    protected function generateUser($type, $token, $oauthUser, $setData)
    {
        $registration = [];
        $randString = base_convert(sha1(uniqid(mt_rand(), true)), 16, 36);
        $oauthUser['name'] = preg_replace('/[^\x{4e00}-\x{9fa5}a-zA-z0-9_.]+/u', '', $oauthUser['name']);
        $oauthUser['name'] = str_replace(['-'], ['_'], $oauthUser['name']);

        if (!SimpleValidator::nickname($oauthUser['name'])) {
            $oauthUser['name'] = '';
        }

        $tempType = $type;

        if (empty($oauthUser['name'])) {
            if ('weixinmob' == $type || 'weixinweb' == $type) {
                $tempType = 'weixin';
            }

            $oauthUser['name'] = $tempType.substr($randString, 9, 3);
        }

        $nameLength = mb_strlen($oauthUser['name'], 'utf-8');

        if ($nameLength > 10) {
            $oauthUser['name'] = mb_substr($oauthUser['name'], 0, 11, 'utf-8');
        }

        if (!empty($setData['nickname']) && !empty($setData['email'])) {
            $registration['nickname'] = $setData['nickname'];
            $registration['email'] = $setData['email'];
            $registration['emailOrMobile'] = $setData['email'];
        } else {
            $nicknames = [];
            $nicknames[] = isset($setData['nickname']) ? $setData['nickname'] : $oauthUser['name'];
            $nicknames[] = mb_substr($oauthUser['name'], 0, 8, 'utf-8').substr($randString, 0, 3);
            $nicknames[] = mb_substr($oauthUser['name'], 0, 8, 'utf-8').substr($randString, 3, 3);
            $nicknames[] = mb_substr($oauthUser['name'], 0, 8, 'utf-8').substr($randString, 6, 3);

            foreach ($nicknames as $name) {
                if ($this->getUserService()->isNicknameAvaliable($name)) {
                    $registration['nickname'] = $name;
                    break;
                }
            }

            if (empty($registration['nickname'])) {
                return [];
            }

            $syncSetting = $this->getSettingService()->get('sync_department_setting', []);
            if (!empty($syncSetting) && $syncSetting['enable'] && 'dingtalk' == $syncSetting['type']) {
                if (!empty($setData['emailDingtalk'])) {
                    $registration['email'] = $setData['emailDingTalk'];
                } else {
                    $registration['email'] = 'u_'.substr($randString, 0, 12).'@edusoho.net';
                }
            } else {
                $registration['email'] = 'u_'.substr($randString, 0, 12).'@edusoho.net';
            }
        }

        if ($this->getSensitiveService()->scanText($registration['nickname'])) {
            return $this->createMessageResponse('error', 'login_bind.generate_user.message.nickname_sensitive');
        }

        $registration['password'] = substr(base_convert(sha1(uniqid(mt_rand(), true)), 16, 36), 0, 8);
        $registration['token'] = $token;
        $registration['createdIp'] = $oauthUser['createdIp'];

        if (isset($setData['mobile']) && !empty($setData['mobile'])) {
            $registration['mobile'] = $setData['mobile'];
            $registration['emailOrMobile'] = $setData['mobile'];
        }

        if (isset($setData['invite_code']) && !empty($setData['invite_code'])) {
            $registration['invite_code'] = $setData['invite_code'];
        }

        $user = $this->getAuthService()->register($registration, $type);

        return $user;
    }

    private function isDepartmentMember($department)
    {
        $result = false;
        $orgs = $this->getOrgService()->findOrgsBySyncIds($department);

        if ($orgs) {
            $result = true;
        }

        return $result;
    }

    private function isExistedUserByMobile($mobile)
    {
        $result = false;
        $user = $this->getUserService()->getUserByVerifiedMobile($mobile);

        if ($user) {
            $result = true;
        }

        return $result;
    }

    public function existAction(Request $request, $type)
    {
        $token = $request->getSession()->get('oauth_token');
        $client = $this->createOAuthClient($type);
        $oauthUser = $client->getUserInfo($token);
        $data = $request->request->all();

        $message = ServiceKernel::instance()->trans('login_bind.exist.message.email_mobile_error');

        if (SimpleValidator::email($data['emailOrMobileOrNickname'])) {
            $user = $this->getUserService()->getUserByEmail($data['emailOrMobileOrNickname']);
            $message = ServiceKernel::instance()->trans('login_bind.exist.message.email_error');
        } elseif (SimpleValidator::mobile($data['emailOrMobileOrNickname'])) {
            $user = $this->getUserService()->getUserByVerifiedMobile($data['emailOrMobileOrNickname']);
            $message = ServiceKernel::instance()->trans('login_bind.exist.message.mobile_error');
        } elseif (SimpleValidator::nickname($data['emailOrMobileOrNickname'])) {
            $user = $this->getUserService()->getUserByNickname($data['emailOrMobileOrNickname']);
            $message = '该用户名尚未注册';
        }

        if (empty($user)) {
            $response = ['success' => false, 'message' => $message];
        } elseif (!$this->getUserService()->verifyPassword($user['id'], $data['password'])) {
            $response = [
                'success' => false,
                'message' => ServiceKernel::instance()->trans('login_bind.exist.message.password_error'),
            ];
        } elseif ($this->getUserService()->getUserBindByTypeAndUserId($type, $user['id'])) {
            $response = [
                'success' => false,
                'message' => sprintf(
                    ServiceKernel::instance()->trans(
                        'login_bind.exist.message.user_exit',
                        ['%s%' => $this->setting('site.name')]
                    )
                ),
            ];
        } else {
            $response = ['success' => true, '_target_path' => $this->getTargetPath($request)];
            $this->getUserService()->bindUser($type, $oauthUser['id'], $user['id'], $token);
            $currentUser = new CurrentUser();
            $currentUser->fromArray($user);
            $this->switchUser($request, $currentUser);
        }

        return $this->createJsonResponse($response);
    }

    protected function getOrgService()
    {
        return $this->createService('Org:OrgService');
    }

    /**
     * @return OrgSyncService
     */
    protected function getOrgSyncService()
    {
        return $this->createService('OrgSync:OrgSyncService');
    }

    protected function getSettingService()
    {
        return $this->createService('System:SettingService');
    }

    /**
     * @return DingTalkUserService
     */
    protected function getDingTalkUserService()
    {
        return $this->createService('CorporateTrainingBundle:DingTalk:DingTalkUserService');
    }

    protected function getLdapUserService()
    {
        return $this->createService('LDAPPlugin:User:UserService');
    }

    /**
     * @return DingTalkSyncUserService
     */
    protected function getDingTalkSyncUserService()
    {
        return $this->createService('CorporateTrainingBundle:DingTalk:DingTalkSyncUserService');
    }

    protected function getUserBindDao()
    {
        return $this->getBiz()->dao('User:UserBindDao');
    }

    /**
     * @return UserDao
     */
    protected function getUserDao()
    {
        return $this->getBiz()->dao('User:UserDao');
    }

    /**
     * @return UserProfileDao
     */
    protected function getUserProfileDao()
    {
        return $this->getBiz()->dao('User:UserProfileDao');
    }

    /**
     * @return SessionService
     */
    protected function getSessionService()
    {
        return $this->createService('System:SessionService');
    }
}
