<?php

namespace CorporateTrainingBundle\Controller\ProjectPlan\Item;

use AppBundle\Common\Paginator;
use AppBundle\Common\ArrayToolkit;
use Symfony\Component\HttpFoundation\Request;

class OfflineCourseItemController extends BaseItemController
{
    public function createAction(Request $request, $projectPlanId)
    {
        $this->hasManageRole();
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($projectPlanId);
        $user = $this->getCurrentUser();
        $userProfile = $this->getUserService()->getUserProfile($user['id']);
        $user['truename'] = !empty($userProfile['truename']) ? $userProfile['truename'] : '';

        return $this->render(
            'project-plan/item/offline-course.html.twig',
            array(
                'projectPlanId' => $projectPlanId,
                'projectPlan' => $projectPlan,
                'user' => $user,
                'teacher' => ['truename' => $user['truename'], 'nickname' => $user['nickname'], 'id' => $user['id']],
            )
        );
    }

    public function updateAction(Request $request, $projectPlanId, $offlineCourseId)
    {
        $this->hasManageRole();

        $offlineCourse = $this->getOfflineCourseService()->getOfflineCourse($offlineCourseId);
        $item = $this->getProjectPlanService()->getProjectPlanItemByProjectPlanIdAndTargetIdAndTargetType($projectPlanId, $offlineCourseId, 'offline_course');
        $tasks = $this->getTaskService()->findTasksByOfflineCourseId($offlineCourseId);
        $signTaskNum = $this->getTaskService()->countTasks(array('offlineCourseId' => $offlineCourseId, 'type' => 'sign'));
        $homeworkTaskNum = $this->getTaskService()->countTasks(array('offlineCourseId' => $offlineCourseId, 'type' => 'homework'));
        $users = $this->getUserService()->searchUsers(['userIds' => [$offlineCourse['teacherId']]], ['id' => 'ASC'], 0, 20, ['truename', 'nickname', 'id']);
        $tasks = ArrayToolkit::group($tasks, 'type');

        if (!isset($users[0])) {
            $user = $this->getCurrentUser();
            $userProfile = $this->getUserService()->getUserProfile($user['id']);
            $user['truename'] = !empty($userProfile['truename']) ? $userProfile['truename'] : '';
        }

        return $this->render(
            'project-plan/item/offline-course.html.twig',
            array(
                'offlineCourse' => $offlineCourse,
                'user' => $this->getCurrentUser(),
                'projectPlanId' => $projectPlanId,
                'teacher' => isset($users[0]) ? $users[0] : ['truename' => $user['truename'], 'nickname' => $user['nickname'], 'id' => $user['id']],
                'item' => $item,
                'signTaskNum' => $signTaskNum,
                'homeworkTaskNum' => $homeworkTaskNum,
                'tasks' => $tasks,
            )
        );
    }

    public function showAction(Request $request, $projectPlanId, $offlineCourseId)
    {
        $this->hasManageRole();

        $offlineCourse = $this->getOfflineCourseService()->getOfflineCourse($offlineCourseId);
        $item = $this->getProjectPlanService()->getProjectPlanItemByProjectPlanIdAndTargetIdAndTargetType($projectPlanId, $offlineCourseId, 'offline_course');
        $tasks = $this->getTaskService()->findTasksByOfflineCourseId($offlineCourseId);
        $signTaskNum = $this->getTaskService()->countTasks(array('offlineCourseId' => $offlineCourseId, 'type' => 'sign'));
        $homeworkTaskNum = $this->getTaskService()->countTasks(array('offlineCourseId' => $offlineCourseId, 'type' => 'homework'));
        $users = $this->getUserService()->searchUsers(['userIds' => [$offlineCourse['teacherId']]], ['id' => 'ASC'], 0, 20, ['truename', 'nickname', 'id']);
        $tasks = ArrayToolkit::group($tasks, 'type');

        return $this->render(
            'project-plan/item/offline-course-show.html.twig',
            array(
                'offlineCourse' => $offlineCourse,
                'user' => $this->getCurrentUser(),
                'projectPlanId' => $projectPlanId,
                'teacher' => empty($users[0]) ? [] : $users[0],
                'item' => $item,
                'signTaskNum' => $signTaskNum,
                'homeworkTaskNum' => $homeworkTaskNum,
                'tasks' => $tasks,
            )
        );
    }

    public function listAction(Request $request, $id)
    {
        $this->hasManageRole();
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($id);
        $projectPlanMemberCount = $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $projectPlan['id']]);
        $conditions = array(
            'projectPlanId' => $id,
            'targetType' => 'offline_course',
        );
        $itemNum = $this->getProjectPlanService()->countProjectPlanItems($conditions);

        $paginator = new Paginator(
            $request,
            $itemNum,
            20
        );

        $items = $this->getProjectPlanService()->searchProjectPlanItems(
            $conditions,
            array('createdTime' => 'DESC'),
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        return $this->render(
            '@CorporateTraining/project-plan/offline-course/offline-course-list.html.twig',
            array(
                'items' => $items,
                'paginator' => $paginator,
                'projectPlan' => $projectPlan,
                'memberCount' => $projectPlanMemberCount,
                'courses' => $this->buildOfflineCourses($items, $id),
            )
        );
    }

    public function homeworkListAction(Request $request, $id)
    {
        $this->hasManageRole();
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($id);
        $items = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($id, 'offline_course');
        $offlineCourseIds = empty($items) ? array(-1) : ArrayToolkit::column($items, 'targetId');
        $conditions = ['offlineCourseIds' => $offlineCourseIds, 'type' => 'homework'];
        $paginator = new Paginator(
            $request,
            $this->getOfflineCourseTaskService()->countTasks($conditions),
            20
        );

        $offlineCourseTasks = $this->getOfflineCourseTaskService()->searchTasks(
            $conditions,
            ['offlineCourseId' => 'ASC', 'seq' => 'ASC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        $offlineCourseIds = ArrayToolkit::column($offlineCourseTasks, 'offlineCourseId');
        $offlineCourses = $this->getOfflineCourseService()->findOfflineCoursesByIds($offlineCourseIds);
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $projectPlan['id']], [], 0, PHP_INT_MAX, ['userId']);
        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');
        foreach ($offlineCourseTasks as &$offlineCourseTask) {
            $offlineCourseTask['homeworkCount'] = $this->getTaskService()->countTaskResults(['taskId' => $offlineCourseTask['id'], 'userIds' => $userIds]);
            $offlineCourseTask['finishedCount'] = $this->getTaskService()->countTaskResults(['taskId' => $offlineCourseTask['id'], 'userIds' => $userIds, 'status' => 'finished']);
        }

        return $this->render('@CorporateTraining/project-plan/offline-course/homework-list.html.twig', [
            'projectPlan' => $projectPlan,
            'offlineCourseTasks' => $offlineCourseTasks,
            'paginator' => $paginator,
            'offlineCourses' => ArrayToolkit::index($offlineCourses, 'id'),
            'memberCount' => $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $id]),
            ]);
    }

    protected function buildOfflineCourses($items, $projectPlanId)
    {
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $projectPlanId], [], 0, PHP_INT_MAX, ['userId']);
        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');
        $offlineCourses = $this->getOfflineCourseService()->findOfflineCoursesByIds(ArrayToolkit::column($items, 'targetId'));

        foreach ($offlineCourses as &$offlineCourse) {
            $offlineCourse['tasks'] = $this->getOfflineCourseTaskService()->searchTasks(['offlineCourseId' => $offlineCourse['id'], 'type' => 'sign'], [], 0, PHP_INT_MAX);
            $offlineCourse['taskCount'] = count($offlineCourse['tasks']);
            $taskIds = empty($offlineCourse['tasks']) ? array(-1) : ArrayToolkit::column($offlineCourse['tasks'], 'id');
            $offlineCourse['attendResultCount'] = $this->getOfflineCourseTaskResultService()->countTaskResults(['offlineCourseId' => $offlineCourse['id'], 'userIds' => $userIds, 'taskIds' => $taskIds]);
        }

        return ArrayToolkit::index($offlineCourses, 'id');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskResultService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }
}
