<?php

namespace CorporateTrainingBundle\Controller\ProjectPlan\Item;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\Exception\AccessDeniedException;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\OfflineCourseService;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\OfflineCourseSurveyService;
use CorporateTrainingBundle\Biz\ProjectPlan\Service\MemberService;
use SurveyPlugin\Biz\Questionnaire\Service\QuestionnaireService;
use SurveyPlugin\Biz\Survey\Service\SurveyMemberService;
use SurveyPlugin\Biz\Survey\Service\SurveyResultService;
use SurveyPlugin\Biz\Survey\Service\SurveyService;
use Symfony\Component\HttpFoundation\Request;

class QuestionnaireItemController extends BaseItemController
{
    public function createAction(Request $request, $projectPlanId)
    {
        $this->hasManageRole();
        $offlineItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'offline_course');
        $courseIds = ArrayToolkit::column($offlineItems, 'targetId');
        $offlineCourses = $this->getOfflineCourseService()->findOfflineCoursesByIds($courseIds);

        return $this->render(
            'project-plan/item/questionnaire.html.twig',
            [
                'offlineCourses' => $offlineCourses,
                'projectPlanId' => $projectPlanId,
            ]
        );
    }

    public function updateAction(Request $request, $projectPlanId, $itemId)
    {
        $item = $this->getProjectPlanService()->getProjectPlanItem($itemId);
        $records = $this->getOfflineCourseSurveyService()->findOfflineCourseSurveysByTargetIdAndTargetType($item['id'], 'projectPlanItem');
        $offlineCourseIds = ArrayToolkit::column($records, 'offlineCourseId');

        return $this->render(
            'project-plan/item/questionnaire.html.twig',
            [
                'item' => $item,
                'questionnaire' => $this->getQuestionnaireService()->getQuestionnaire($item['targetId']),
                'offlineCourses' => $this->getOfflineCourseService()->findOfflineCoursesByIds($offlineCourseIds),
                'projectPlanId' => $projectPlanId,
            ]
        );
    }

    public function offlineItemMatchAction(Request $request, $projectPlanId)
    {
        $courseIds = $request->query->get('courseIds', []);
        $name = $request->query->get('q', '');
        $offlineItems = $this->getProjectPlanService()->searchProjectPlanItems(['projectPlanId' => $projectPlanId, 'targetType' => 'offline_course', 'excludeTargetIds' => $courseIds], [], 0, PHP_INT_MAX);
        $courseIds = empty($offlineItems) ? [-1] : ArrayToolkit::column($offlineItems, 'targetId');
        $offlineCourses = $this->getOfflineCourseService()->searchOfflineCourses(['ids' => $courseIds, 'likeTitle' => $name], [], 0, PHP_INT_MAX);

        return $this->createJsonResponse($offlineCourses);
    }

    public function viewSurveyQrcodeAction($itemId)
    {
        $projectPlanItem = $this->getProjectPlanService()->getProjectPlanItem($itemId);

        return $this->render('@CorporateTraining/project-plan/evaluation/qr-code.html.twig',
            [
                'projectPlanItem' => $projectPlanItem,
                'qrcodeImgUrl' => $this->qrcode('project_plan_evaluation_do_survey', ['projectPlanItemId' => $projectPlanItem['id']]),
            ]
        );
    }

    public function canDoSurveyAction(Request $request, $projectPlanItemId)
    {
        $item = $this->getProjectPlanService()->getProjectPlanItem($projectPlanItemId);
        if ($item['startTime'] > time()) {
            throw new AccessDeniedException('问卷暂未开放！');
        }
        if ('POST' == $request->getMethod()) {
            $user = $this->getCurrentUser();
            $member = $this->getProjectPlanMemberService()->getProjectPlanMemberByUserIdAndProjectPlanId($user['id'], $item['projectPlanId']);
            if (empty($member)) {
                throw new AccessDeniedException('你不是该项目成员！');
            }

            $surveyAnswers = [];
            $answers = $request->request->all();

            foreach ($answers as $key => $answer) {
                $key = explode('_', $key);
                $surveyId = $key[0];
                $questionnaireItemId = $key[1];
                $surveyAnswers[$surveyId][$questionnaireItemId] = $answer;
            }
            $this->finishSurvey($surveyAnswers);

            return $this->createJsonResponse(true);
        }

        return $this->buildSurvey($projectPlanItemId);
    }

    public function resultSurveyAction(Request $request, $projectPlanItemId)
    {
        return $this->buildSurvey($projectPlanItemId);
    }

    protected function buildSurvey($projectPlanItemId)
    {
        $item = $this->getProjectPlanService()->getProjectPlanItem($projectPlanItemId);
        $user = $this->getCurrentUser();
        $member = $this->getProjectPlanMemberService()->getProjectPlanMemberByUserIdAndProjectPlanId($user['id'], $item['projectPlanId']);
        if (empty($member)) {
            throw new AccessDeniedException('你不是该项目成员！');
        }
        $questionnaireRecords = $this->getOfflineCourseSurveyService()->findOfflineCourseSurveysByTargetIdAndTargetType($item['id'], 'projectPlanItem');
        $offlineCourseIds = ArrayToolkit::column($questionnaireRecords, 'offlineCourseId');

        $data = [
            'projectPlanItem' => $item,
            'offlineCourses' => $this->getOfflineCourseService()->findOfflineCoursesByIds($offlineCourseIds),
            'questionnaireRecords' => ArrayToolkit::index($questionnaireRecords, 'offlineCourseId'),
            'questionnaire' => $this->getQuestionnaireService()->getQuestionnaire($item['targetId']),
            'questionnaireItems' => $this->getQuestionnaireService()->showQuestionnaireItems($item['targetId']),
            'user' => $user,
        ];

        if ($this->checkSubmitted($item)) {
            $data['action'] = 'result';
        }

        return $this->render(
            '@CorporateTraining/project-plan/evaluation/do.html.twig',
            $data
        );
    }

    protected function finishSurvey($surveyAnswers)
    {
        $user = $this->getCurrentUser();

        foreach ($surveyAnswers as $key => $surveyAnswer) {
            $member = $this->getSurveyMemberService()->getMemberBySurveyIdAndUserId($key, $user['id']);
            if (empty($member)) {
                $fields = [
                    'surveyId' => $key,
                    'userId' => $user['id'],
                ];
                $this->getSurveyMemberService()->createMember($fields);
            }
            $this->getSurveyResultService()->deleteSurveyResultByUserIdAndSurveyId($user['id'], $key);
            $surveyResult = $this->getSurveyService()->startSurvey($key, $user['id']);
            $this->getSurveyService()->finishSurvey($surveyResult['id'], ['answers' => $surveyAnswer]);
        }
    }

    protected function checkSubmitted($item)
    {
        $questionnaireRecords = $this->getOfflineCourseSurveyService()->findOfflineCourseSurveysByTargetIdAndTargetType($item['id'], 'projectPlanItem');
        $surveyIds = ArrayToolkit::column($questionnaireRecords, 'surveyId');
        $surveyResults = $this->getSurveyResultService()->searchSurveyResults(['userId' => $this->getCurrentUser()->getId(), 'surveyIds' => empty($surveyIds) ? [-1] : $surveyIds, 'status' => 'finished'], [], 0, PHP_INT_MAX);

        if (empty($surveyResults)) {
            return false;
        }

        return true;
    }

    protected function qrcode($url, $array)
    {
        $token = $this->getTokenService()->makeToken('qrcode', [
            'data' => [
                'url' => $this->generateUrl($url, $array, true),
            ],
            'duration' => 3600 * 24 * 365,
        ]);

        $url = $this->generateUrl('common_parse_qrcode', ['token' => $token['token']], true);

        return $this->generateUrl('common_qrcode', ['text' => $url], true);
    }

    /**
     * @return OfflineCourseService
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return OfflineCourseSurveyService
     */
    protected function getOfflineCourseSurveyService()
    {
        return $this->getBiz()->service('CorporateTrainingBundle:OfflineCourse:OfflineCourseSurveyService');
    }

    /**
     * @return QuestionnaireService
     */
    protected function getQuestionnaireService()
    {
        return $this->createService('SurveyPlugin:Questionnaire:QuestionnaireService');
    }

    /**
     * @return MemberService
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return SurveyService
     */
    protected function getSurveyService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyService');
    }

    /**
     * @return SurveyResultService
     */
    protected function getSurveyResultService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyResultService');
    }

    /**
     * @return SurveyMemberService
     */
    protected function getSurveyMemberService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyMemberService');
    }

    /**
     * @return \Biz\User\Service\Impl\TokenServiceImpl
     */
    protected function getTokenService()
    {
        return $this->createService('User:TokenService');
    }
}
