<?php

namespace CorporateTrainingBundle\Controller\ProjectPlan;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\Paginator;
use AppBundle\Controller\BaseController;
use ExamPlugin\Biz\Exam\Service\ExamService;
use Symfony\Component\HttpFoundation\Request;

class ProjectPlanStudyDataController extends BaseController
{
    public function studyDataOverviewAction($id)
    {
        $user = $this->getCurrentUser();
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($id);
        $hasAdminDataCenterPermission = $user->hasManagePermissionWithOrgCode($projectPlan['orgCode']) && $user->hasPermission('admin_data_center_project_plan');
        $hasProjectPlanManagePermission = $this->getProjectPlanService()->canManageProjectPlan($id);
        if (!$hasProjectPlanManagePermission && !$hasAdminDataCenterPermission) {
            return $this->createMessageResponse('error', 'project_plan.message.can_not_manage_message');
        }

        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $id], [], 0, PHP_INT_MAX, ['id', 'userId']);
        $memberNum = count($members);

        $completionRate = $this->getCompletionRate($id, $members);
        $usersTotalLearnTime = $this->createProjectPlanStrategy()->calculateMembersTotalLearnTime($projectPlan, ArrayToolkit::column($members, 'userId'));

        $totalLearnTime = 0;
        foreach ($usersTotalLearnTime as $userTotalLearnTime) {
            $totalLearnTime += $userTotalLearnTime;
        }

        $completionRate['usersTotalLearnTime'] = round($totalLearnTime / 3600, 2);

        return $this->render(
            'project-plan/study-data/statistic-chart.html.twig',
            [
                'isEmpty' => $this->isStudyDataEmpty($id),
                'projectPlan' => $projectPlan,
                'completionRate' => $completionRate,
                'averageLearnTime' => (0 != $memberNum) ? round($totalLearnTime / (3600 * $memberNum), 2) : 0,
                'hasProjectPlanManagePermission' => $hasProjectPlanManagePermission,
            ]
        );
    }

    public function studyDataListAction(Request $request, $projectPlanId)
    {
        $user = $this->getCurrentUser();
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($projectPlanId);
        $hasAdminDataCenterPermission = $user->hasManagePermissionWithOrgCode($projectPlan['orgCode']) && $user->hasPermission('admin_data_center_project_plan');
        $hasProjectPlanManagePermission = $this->getProjectPlanService()->canManageProjectPlan($projectPlanId);
        if (!$hasProjectPlanManagePermission && !$hasAdminDataCenterPermission) {
            return $this->createMessageResponse('error', 'project_plan.message.can_not_manage_message');
        }
        $renderData = $this->buildStudyListData($request, $projectPlan);

        $renderData['hasProjectPlanManagePermission'] = $hasProjectPlanManagePermission;

        return $this->render('project-plan/study-data/member-statistic-data.html.twig', $renderData);
    }

    public function studyDataAjaxListAction(Request $request, $projectPlanId)
    {
        $user = $this->getCurrentUser();
        if (!$this->getProjectPlanService()->canManageProjectPlan($projectPlanId) && !$user->hasPermission('admin_data_center_project_plan')) {
            return $this->createMessageResponse('error', 'project_plan.message.can_not_manage_message');
        }
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($projectPlanId);

        $renderData = $this->buildStudyListData($request, $projectPlan, 'ajax');

        return $this->render('project-plan/study-data/member-statistic-data-list.html.twig', $renderData);
    }

    public function studyDataUserDetailAction(Request $request, $projectPlanId, $userId)
    {
        $user = $this->getCurrentUser();
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($projectPlanId);
        $hasAdminDataCenterPermission = $user->hasManagePermissionWithOrgCode($projectPlan['orgCode']) && $user->hasPermission('admin_data_center_project_plan');
        $hasProjectPlanManagePermission = $this->getProjectPlanService()->canManageProjectPlan($projectPlanId);
        if (!$hasProjectPlanManagePermission && !$hasAdminDataCenterPermission) {
            return $this->createMessageResponse('error', 'project_plan.message.can_not_manage_message');
        }

        return $this->render(
            'project-plan/study-data/index.html.twig',
            [
                'projectPlan' => $projectPlan,
                'hasProjectPlanManagePermission' => $hasProjectPlanManagePermission,
                'user' => $this->getUserService()->getUser($userId),
            ]
        );
    }

    public function studyDataUserDetailListAction(Request $request, $projectPlanId, $userId)
    {
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($projectPlanId);
        $onlineItems = $this->memberOnlineCourseStatistic($projectPlanId, $userId);
        $onlineCourseIds = ArrayToolkit::column($onlineItems, 'targetId');

        $offlineExamItems = $this->memberOfflineExamStatistic($projectPlanId, $userId);
        $offlineCourseItems = $this->memberOfflineCourseStatistic($projectPlanId, $userId);

        $onlineCourses = $this->getCourseService()->findCoursesByIds($onlineCourseIds);

        $user = $this->getUserService()->getUser($userId);
        $post = $this->getPostService()->getPost($user['postId']);
        $itemCount = $this->getProjectPlanService()->countProjectPlanItems(['projectPlanId' => $projectPlanId]);

        return $this->render(
            'project-plan/study-data/user-detail.html.twig',
            [
                'onlineItems' => $onlineItems,
                'offlineCourseItems' => $offlineCourseItems,
                'offlineExamItems' => $offlineExamItems,
                'user' => $user,
                'courses' => $onlineCourses,
                'post' => $post,
                'projectPlan' => $projectPlan,
                'itemCount' => $itemCount,
            ]
        );
    }

    protected function buildStudyListData($request, $projectPlan, $type = '')
    {
        $conditions = $request->request->all();
        $projectPlanId = $projectPlan['id'];
        $conditions = $this->prepareStudyDataListSearchConditions($conditions, $projectPlanId);

        $paginator = new Paginator(
            $request,
            $this->getProjectPlanMemberService()->countProjectPlanMembers($conditions),
            20
        );
        if ('ajax' != $type) {
            $paginator->setBaseUrl($this->generateUrl('project_plan_study_data_ajax_list', ['projectPlanId' => $projectPlanId]));
        }

        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(
            $conditions,
            [],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        $userIds = ArrayToolkit::column($members, 'userId');
        $users = $this->getUserService()->findUsersByIds($userIds);
        $postIds = ArrayToolkit::column($users, 'postId');
        $posts = $this->getPostService()->findPostsByIds($postIds);
        $posts = ArrayToolkit::index($posts, 'id');

        return [
            'onlineCourseLearnTime' => $this->getMemberOnlineCourseLearnTime($projectPlanId, $userIds),
            'paginator' => $paginator,
            'orgIds' => implode(',', $conditions['orgIds']),
            'projectPlan' => $projectPlan,
            'members' => $this->processStudyListMembers($projectPlanId, $members),
            'users' => $users,
            'posts' => $posts,
            'membersCount' => $this->getProjectPlanMemberService()->countProjectPlanMembers(
                ['projectPlanId' => $projectPlanId]
            ),
        ];
    }

    protected function getMemberOnlineCourseLearnTime($projectPlanId, $userIds)
    {
        $projectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'course');
        $onlineCourseIds = isset($projectPlanItems) ? ArrayToolkit::column($projectPlanItems, 'targetId') : [-1];
        $onlineCourseLearnTime = $this->getTaskResultService()->sumLearnTimeByCourseIdsAndUserIdsGroupByUserId($onlineCourseIds, $userIds);

        return ArrayToolkit::index($onlineCourseLearnTime, 'userId');
    }

    protected function processStudyListMembers($projectPlanId, $members)
    {
        $offlineProjectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'offline_course');
        $onlineProjectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'course');
        $onlineCourseIds = isset($onlineProjectPlanItems) ? ArrayToolkit::column($onlineProjectPlanItems, 'targetId') : [-1];

        $offlineCourseIds = empty($offlineProjectPlanItems) ? ArrayToolkit::column($offlineProjectPlanItems, 'targetId') : [-1];
        foreach ($members as &$member) {
            $member['finishedOnlineCourseNum'] = $this->getCourseMemberService()->countLearnedMember(['m.userId' => $member['userId'], 'courseId' => $onlineCourseIds]);
            $member['onlineCourseFinishedRate'] = (isset($onlineProjectPlanItems) && 0 != count($onlineProjectPlanItems)) ? round(($member['finishedOnlineCourseNum'] / count($onlineProjectPlanItems)) * 100, 2) : 0;
            $member = array_merge($member, $this->getOfflineCourseService()->getUserOfflineCoursesStatisticData([$member['userId']], $offlineCourseIds));
        }

        return ArrayToolkit::index($members, 'id');
    }

    protected function isStudyDataEmpty($projectPlanId)
    {
        $isEmpty = ['offlineCourse' => true, 'offlineExam' => true, 'exam' => true, 'onlineCourse' => true];
        $count = $this->getProjectPlanMemberService()->countProjectPlanMembers(['projectPlanId' => $projectPlanId]);
        if (empty($count)) {
            return  $isEmpty;
        }
        $items = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanId($projectPlanId);
        $items = ArrayToolkit::group($items, 'targetType');
        $isEmpty['offlineCourse'] = empty($items['offline_course']);

        $isEmpty['offlineExam'] = empty($items['offline_exam']);
        if ($this->isPluginInstalled('Exam')) {
            $isEmpty['exam'] = empty($items['exam']);
        }

        $isEmpty['onlineCourse'] = empty($items['course']);

        return $isEmpty;
    }

    protected function memberOnlineCourseStatistic($projectPlanId, $userId)
    {
        $onlineItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType(
            $projectPlanId,
            'course'
        );

        foreach ($onlineItems as &$item) {
            $courseMember = $this->getCourseMemberService()->getCourseMember($item['targetId'], $userId);
            $item['courseTaskNum'] = $this->getTaskService()->countTasks(['courseId' => $item['targetId'], 'isOptional' => 0]);
            $item['courseFinishTaskNum'] = $courseMember['learnedCompulsoryTaskNum'];
            $item['learnTime'] = $this->getTaskResultService()->sumLearnTimeByCourseIdAndUserId(
                $item['targetId'],
                $userId
            );
        }

        return $onlineItems;
    }

    protected function prepareStudyDataListSearchConditions($conditions, $projectPlanId)
    {
        if (!isset($conditions['orgIds'])) {
            $orgs = $this->getOrgService()->findOrgsByPrefixOrgCodes(['1.']);
            $orgIds = ArrayToolkit::column($orgs, 'id');
        }

        $conditions['orgIds'] = empty($orgIds) ? explode(',', $conditions['orgIds']) : $orgIds;
        $conditions['projectPlanId'] = $projectPlanId;
        $orgUserIds = $this->getProjectPlanOrgUserIds($projectPlanId, $conditions['orgIds']);
        $conditions['userIds'] = $orgUserIds;

        if (!empty($conditions['postId'])) {
            $users = $this->getUserService()->searchUsers(
                ['postId' => $conditions['postId']],
                ['id' => 'DESC'],
                0,
                PHP_INT_MAX
            );

            $userIds = ArrayToolkit::column($users, 'id');
            if (empty($conditions['userIds']) || empty($userIds)) {
                $conditions['userIds'] = [];
            } else {
                $conditions['userIds'] = array_intersect($conditions['userIds'], $userIds);
            }
        }

        if (!empty($conditions['username'])) {
            $userIds = $this->getUserService()->findUserIdsByNickNameOrTrueName($conditions['username']);
            if (empty($conditions['userIds']) || empty($userIds)) {
                $conditions['userIds'] = [];
            } else {
                $conditions['userIds'] = array_intersect($conditions['userIds'], $userIds);
            }

            unset($conditions['username']);
        }

        $conditions['userIds'] = empty($conditions['userIds']) ? [-1] : $conditions['userIds'];

        return $conditions;
    }

    protected function getProjectPlanOrgUserIds($projectPlanId, $orgIds)
    {
        $members = $this->getProjectPlanMemberService()->searchProjectPlanMembers(['projectPlanId' => $projectPlanId], [], 0, PHP_INT_MAX, ['id', 'userId']);

        $userIds = ArrayToolkit::column($members, 'userId');

        $users = $this->getUserOrgService()->searchUserOrgs(
            ['orgIds' => $orgIds, 'userIds' => $userIds],
            [],
            0,
            PHP_INT_MAX
        );

        return ArrayToolkit::column($users, 'userId');
    }

    protected function memberOfflineExamStatistic($projectPlanId, $userId)
    {
        $offlineExamItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'offline_exam');
        $examIds = ArrayToolkit::column($offlineExamItems, 'targetId');
        $offlineExams = $this->getOfflineExamService()->findOfflineExamByIds($examIds);

        foreach ($offlineExams as &$offlineExam) {
            $offlineExam['result'] = $this->getOfflineExamMemberService()->getMemberByOfflineExamIdAndUserId($offlineExam['id'], $userId);
        }

        return $offlineExams;
    }

    protected function memberOfflineCourseStatistic($projectPlanId, $userId)
    {
        $offlineItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanIdAndTargetType($projectPlanId, 'offline_course');
        $offlineCourseIds = ArrayToolkit::column($offlineItems, 'targetId');
        $offlineCourses = $this->getOfflineCourseService()->findOfflineCoursesByIds($offlineCourseIds);
        $offlineCourseMembers = $this->getOfflineCourseMemberService()->searchMembers(
            ['offlineCourseIds' => $offlineCourseIds, 'userId' => $userId],
            [],
            0,
            PHP_INT_MAX,
            ['id', 'offlineCourseId', 'status']
        );
        $offlineCourseMembers = ArrayToolkit::index($offlineCourseMembers, 'offlineCourseId');

        foreach ($offlineCourses as &$offlineCourse) {
            $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(
                ['offlineCourseId' => $offlineCourse['id'], 'type' => 'homework'],
                [],
                0,
                PHP_INT_MAX
            );
            $homeworkTaskIds = empty($homeworkTasks) ? [-1] : ArrayToolkit::column($homeworkTasks, 'id');

            $member['offlineCoursePassedHomeworkNum'] = $this->getOfflineCourseTaskService()->countTaskResults(['taskIds' => $homeworkTaskIds, 'status' => 'finished', 'userId' => $userId]);
            $offlineCourse['attendStatus'] = empty($offlineCourseMembers[$offlineCourse['id']]) ? 'none' : $offlineCourseMembers[$offlineCourse['id']]['status'];

            $offlineCourse['offlineCoursePassedHomeworkRate'] = empty($member['offlineCoursePassedHomeworkNum']) ? 0 : round(($member['offlineCoursePassedHomeworkNum'] / count($homeworkTasks)) * 100, 2).'%';
        }

        return $offlineCourses;
    }

    protected function getCompletionRate($id, $members)
    {
        $userIds = empty($members) ? [-1] : ArrayToolkit::column($members, 'userId');
        $projectPlanItems = $this->getProjectPlanService()->findProjectPlanItemsByProjectPlanId($id);
        $projectPlanItems = ArrayToolkit::group($projectPlanItems, 'targetType');

        return [
            'offline_exam' => $this->getOfflineExamCompletionRate($projectPlanItems, $userIds),
            'exam' => $this->getOnlineExamCompletionRate($projectPlanItems, $userIds),
            'course' => $this->getOnlineCourseCompletionRate($projectPlanItems, $userIds),
            'offline_course' => $this->getOfflineCompletionRate($projectPlanItems, $userIds),
        ];
    }

    protected function getOfflineExamCompletionRate($projectPlanItems, $userIds)
    {
        if (empty($projectPlanItems['offline_exam']) || empty($userIds)) {
            return 0;
        }
        $offlineExamIds = ArrayToolkit::column($projectPlanItems['offline_exam'], 'targetId');
        $passedNum = $this->getOfflineExamMemberService()->countMembers(['offlineExamIds' => $offlineExamIds, 'userIds' => $userIds, 'status' => 'passed']);

        return empty($projectPlanItems['offline_exam']) ? 0 : $passedNum / (count($projectPlanItems['offline_exam']) * count($userIds)) * 100;
    }

    protected function getOnlineExamCompletionRate($projectPlanItems, $userIds)
    {
        if (empty($projectPlanItems['exam']) || !$this->isPluginInstalled('Exam') || empty($userIds)) {
            return 0;
        }
        $onlineExamIds = ArrayToolkit::column($projectPlanItems['exam'], 'targetId');
        $passedNum = $this->getExamService()->countMembers(['examIds' => $onlineExamIds, 'userIds' => $userIds, 'passStatus' => 'passed']);

        return empty($projectPlanItems['exam']) ? 0 : $passedNum / (count($projectPlanItems['exam']) * count($userIds)) * 100;
    }

    protected function getOnlineCourseCompletionRate($projectPlanItems, $userIds)
    {
        if (empty($projectPlanItems['course']) || empty($userIds)) {
            return 0;
        }
        $onlineCourseIds = ArrayToolkit::column($projectPlanItems['course'], 'targetId');
        $allFinishedOnlineCourseNum = $this->getCourseMemberService()->countLearnedMember(['userId' => $userIds, 'courseId' => $onlineCourseIds]);

        return empty($projectPlanItems['course']) ? 0 : $allFinishedOnlineCourseNum / (count($projectPlanItems['course']) * count($userIds)) * 100;
    }

    protected function getOfflineCompletionRate($projectPlanItems, $userIds)
    {
        if (empty($projectPlanItems['offline_course']) || empty($userIds)) {
            return ['offlineCourse' => []];
        }
        $offlineCoursesIds = ArrayToolkit::column($projectPlanItems['offline_course'], 'targetId');
        $homeworkTasks = $this->getOfflineCourseTaskService()->searchTasks(
            ['offlineCourseIds' => $offlineCoursesIds, 'type' => 'homework'],
            [],
            0,
            PHP_INT_MAX
        );
        $homeworkTaskIds = empty($homeworkTasks) ? [-1] : ArrayToolkit::column($homeworkTasks, 'id');

        $homeworkResults = $this->getOfflineCourseTaskService()->selectResultCountGroupByStatus(['offlineCourseIds' => $offlineCoursesIds, 'taskIds' => $homeworkTaskIds, 'userIds' => $userIds]);
        $homeworkResults = ArrayToolkit::index($homeworkResults, 'status');
        $offlineCourseMembers = $this->getOfflineCourseMemberService()->selectMemberCountGroupByStatus(['offlineCourseIds' => $offlineCoursesIds, 'userIds' => $userIds]);
        $projectMemberCount = count($userIds);
        $offlineCourseMembers = ArrayToolkit::index($offlineCourseMembers, 'status');
        $courseCount = count($projectPlanItems['offline_course']);

        if (!empty($projectMemberCount)) {
            $attendData = [
                'attended' => isset($offlineCourseMembers['attended']) && isset($projectPlanItems['offline_course']) ? round($offlineCourseMembers['attended']['count'] / ($courseCount * $projectMemberCount) * 100, 2) : 0,
                'partial' => isset($offlineCourseMembers['partial']) && isset($projectPlanItems['offline_course']) ? round($offlineCourseMembers['partial']['count'] / ($courseCount * $projectMemberCount) * 100, 2) : 0,
                'absenteeism' => isset($offlineCourseMembers['absenteeism']) && isset($projectPlanItems['offline_course']) ? round($offlineCourseMembers['absenteeism']['count'] / ($courseCount * $projectMemberCount) * 100, 2) : 0,
                'leave' => isset($offlineCourseMembers['leave']) && isset($projectPlanItems['offline_course']) ? round($offlineCourseMembers['leave']['count'] / ($courseCount * $projectMemberCount) * 100, 2) : 0,
            ];
            $attendData['none'] = 100 - array_sum($attendData);
            $passedHomeworkRate = [
                'finished' => empty($homeworkResults['finished']) || empty($homeworkTasks) ? 0 : round($homeworkResults['finished']['count'] / (count($homeworkTasks) * $projectMemberCount) * 100, 2),
                'submitted' => empty($homeworkResults['submitted']) || empty($homeworkTasks) ? 0 : round($homeworkResults['submitted']['count'] / (count($homeworkTasks) * $projectMemberCount) * 100, 2),
                'unpassed' => empty($homeworkResults['unpassed']) || empty($homeworkTasks) ? 0 : round($homeworkResults['unpassed']['count'] / (count($homeworkTasks) * $projectMemberCount) * 100, 2),
            ];
            $passedHomeworkRate['none'] = 100 - array_sum($passedHomeworkRate);
        }

        return [
            'homeworkNum' => count($homeworkTasks),
            'passedHomeworkRate' => empty($passedHomeworkRate) ? ['finished' => 0, 'submitted' => 0, 'unpassed' => 0, 'none' => 0] : $passedHomeworkRate,
            'attendData' => empty($attendData) ? ['attended' => 0, 'partial' => 0, 'absenteeism' => 0, 'leave' => 0, 'none' => 0] : $attendData,
        ];
    }

    protected function createProjectPlanStrategy()
    {
        return $this->getBiz()->offsetGet('projectPlan_item_strategy_context')->createStrategy('course');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\ProjectPlanServiceImpl
     */
    protected function getProjectPlanService()
    {
        return $this->createService('ProjectPlan:ProjectPlanService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\Activity\Service\Impl\ActivityServiceImpl
     */
    protected function getActivityService()
    {
        return $this->createService('Activity:ActivityService');
    }

    protected function getTaskService()
    {
        return $this->createService('Task:TaskService');
    }

    protected function getTaskResultService()
    {
        return $this->createService('CorporateTrainingBundle:Task:TaskResultService');
    }

    protected function getCourseService()
    {
        return $this->createService('Course:CourseService');
    }

    protected function getCourseMemberService()
    {
        return $this->createService('Course:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineExam\Service\Impl\OfflineExamServiceImpl
     */
    protected function getOfflineExamService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:OfflineExamService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineExam\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineExamMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:MemberService');
    }

    protected function getPostService()
    {
        return $this->createService('CorporateTrainingBundle:Post:PostService');
    }

    protected function getUserOrgService()
    {
        return $this->createService('CorporateTrainingBundle:User:UserOrgService');
    }

    /**
     * @return ExamService
     */
    protected function getExamService()
    {
        return $this->createService('ExamPlugin:Exam:ExamService');
    }
}
