<?php

namespace CorporateTrainingBundle\Controller;

use AppBundle\Common\ArrayToolkit;
use AppBundle\Common\Paginator;
use AppBundle\Controller\BaseController;
use Biz\Course\Service\MemberService;
use Codeages\Biz\Framework\Service\Exception\NotFoundException;
use CorporateTrainingBundle\Biz\Course\Service\CourseService;
use CorporateTrainingBundle\Biz\Enroll\Service\EnrollRecordService;
use CorporateTrainingBundle\Biz\OfflineCourse\Service\OfflineCourseSurveyService;
use CorporateTrainingBundle\Biz\PostCourse\Service\PostCourseService;
use CorporateTrainingBundle\Biz\Task\Service\TaskResultService;
use CorporateTrainingBundle\Biz\Task\Service\TaskService;
use CorporateTrainingBundle\Common\DateToolkit;
use SurveyPlugin\Biz\Survey\Service\SurveyResultService;
use Symfony\Component\HttpFoundation\Request;

class StudyRecordController extends BaseController
{
    public function projectPlanRecordAction(Request $request, $userId)
    {
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }
        $conditions = $request->query->all();
        $conditions['userId'] = $userId;
        if (empty($conditions['dataSearchTime'])) {
            list($startDateTime, $endDateTime) = DateToolkit::generateStartDateAndEndDate('year');
            $dataSearchTime['startDateTime'] = strtotime($startDateTime);
            $dataSearchTime['endDateTime'] = strtotime($endDateTime.' 23:59:59');
        } else {
            $date = explode('-', $conditions['dataSearchTime']);
            $dataSearchTime['startDateTime'] = strtotime($date[0]);
            $dataSearchTime['endDateTime'] = strtotime($date[1].' 23:59:59');
        }

        $conditions['startTime_GE'] = $dataSearchTime['startDateTime'];
        $conditions['startTime_LE'] = $dataSearchTime['endDateTime'];

        $members = $this->getProjectPlanMemberService()->findMembersByUserId($userId);
        $conditions['ids'] = empty($members) ? [-1] : ArrayToolkit::column($members, 'projectPlanId');
        $paginator = new Paginator(
            $request,
            $this->getProjectPlanService()->countProjectPlans($conditions),
            10
        );

        $projectPlans = $this->getProjectPlanService()->searchProjectPlans(
            $conditions,
            ['startTime' => 'DESC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        foreach ($projectPlans as &$projectPlan) {
            $attendRateData = $this->getProjectPlanService()->getUserProjectPlanAttendRateData($projectPlan, $userId);
            $projectPlan['attendanceRate'] = $attendRateData['rate'];
            $examRateData = $this->getProjectPlanService()->getUserProjectPlanExamRateData($projectPlan, $userId);
            $projectPlan['examRateData'] = $examRateData['rate'];
        }

        return $this->render('study-record/project-plan-record.html.twig',
            [
                'projectPlans' => !empty($projectPlans) ? $projectPlans : [],
                'userId' => $userId,
                'paginator' => $paginator,
                'dataSearchTime' => $dataSearchTime,
            ]
        );
    }

    public function studyDataUserDetailAction(Request $request, $projectPlanId, $userId)
    {
        if (empty($userId)) {
            $userId = $this->getUser()->getId();
        }
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }
        $projectPlan = $this->getProjectPlanService()->getProjectPlan($projectPlanId);

        return $this->render('study-record/record-user-detail.html.twig',
            [
                'user' => $this->getUserService()->getUser($userId),
                'projectPlan' => $projectPlan,
            ]
        );
    }

    public function offlineActivityRecordAction(Request $request, $userId)
    {
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }
        $members = $this->getOfflineActivityMemberService()->searchMembers(
            ['userId' => $userId],
            [],
            0,
            PHP_INT_MAX
        );

        $paginator = new Paginator(
            $request,
            count($members),
            10
        );

        if (!empty($members)) {
            $offlineActivityIds = ArrayToolkit::column($members, 'offlineActivityId');
            $members = ArrayToolkit::index($members, 'offlineActivityId');

            $offlineActivities = $this->getOfflineActivityService()->searchOfflineActivities(
                ['ids' => $offlineActivityIds],
                ['startTime' => 'DESC'],
                $paginator->getOffsetCount(),
                $paginator->getPerPageCount()
            );

            foreach ($offlineActivities as &$offlineActivity) {
                $offlineActivity['member'] = $members[$offlineActivity['id']];
                $offlineActivity['month'] = date('m', $offlineActivity['startTime']);
                $offlineActivity['day'] = date('d', $offlineActivity['startTime']);
                $offlineActivity['year'] = date('Y', $offlineActivity['startTime']);
            }
            $offlineActivities = ArrayToolkit::group($offlineActivities, 'year');
        } else {
            $offlineActivities = [];
        }

        return $this->render(
            'study-record/offline-activity/offline-activity.html.twig',
            [
                'offlineActivities' => $offlineActivities,
                'paginator' => $paginator,
                'type' => 'offline_activity',
                'userId' => $userId,
            ]
        );
    }

    public function offlineActivityEnrollmentRecordAction(Request $request, $userId)
    {
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }
        $conditions['userId'] = $userId;
        $conditions['targetType'] = 'offlineActivity';

        $count = $this->getEnrollRecordService()->countEnrollRecords($conditions);
        $paginator = new Paginator(
            $request,
            $count,
            10
        );

        $enrollmentRecords = $this->getEnrollRecordService()->searchEnrollRecords(
            $conditions,
            ['submittedTime' => 'DESC'],
            $paginator->getOffsetCount(),
            $paginator->getPerPageCount()
        );

        $offlineActivityIds = ArrayToolkit::column($enrollmentRecords, 'targetId');
        $offlineActivities = $this->getOfflineActivityService()->findOfflineActivitiesByIds($offlineActivityIds);
        $offlineActivities = ArrayToolkit::index($offlineActivities, 'id');

        foreach ($enrollmentRecords as &$enrollmentRecord) {
            $enrollmentRecord['offlineActivity'] = $offlineActivities[$enrollmentRecord['targetId']];
        }

        return $this->render(
            'study-record/offline-activity/offline-activity.html.twig',
            [
                'enrollmentRecords' => $enrollmentRecords,
                'type' => 'enrollment_record',
                'paginator' => $paginator,
                'userId' => $userId,
            ]
        );
    }

    public function postCourseRecordAction(Request $request, $userId)
    {
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }

        $user = $this->getUserService()->getUser($userId);
        $postCourses = $this->getPostCourseService()->findPostCoursesByPostId($user['postId']);

        if (!empty($postCourses)) {
            $courseIds = ArrayToolkit::column($postCourses, 'courseId');
            $courses = $this->getCourseService()->findCoursesByIds($courseIds);
            list($courses, $totalLearnTime) = $this->calculateLearnTime($user['id'], $courses);
            $learnedCoursesNum = $this->getPostCourseService()->countUserLearnedPostCourses($user['id'], $courseIds);
            $recentLearnTask = $this->findRecentLearnTask($user['id'], $courseIds);
        }

        return $this->render(
            'study-record/post-course-record.html.twig',
            [
                'postCourseCount' => empty($postCourses) ? 0 : count($postCourses),
                'totalLearnTime' => empty($totalLearnTime) ? 0 : $totalLearnTime,
                'learnedCoursesNum' => empty($learnedCoursesNum) ? 0 : $learnedCoursesNum,
                'recentLearnTask' => empty($recentLearnTask) ? [] : $recentLearnTask,
                'user' => $user,
                'userId' => $userId,
            ]
        );
    }

    public function postCourseListAction(Request $request, $userId)
    {
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }

        $user = $this->getUserService()->getUser($userId);
        $postCourses = $this->getPostCourseService()->findPostCoursesByPostId($user['postId']);

        if (!empty($postCourses)) {
            $courseIds = ArrayToolkit::column($postCourses, 'courseId');
            $courses = $this->getCourseService()->findCoursesByIds($courseIds);
            foreach ($courses as $key => &$course) {
                $course['learnTime'] = $this->getTaskResultService()->sumLearnTimeByCourseIdAndUserId($key, $userId);
                $course['isCourseMember'] = $this->getMemberService()->isCourseMember($course['id'], $userId);
            }
        }

        return $this->render(
            'study-record/post-course-list.html.twig',
            [
                'courses' => empty($courses) ? [] : $courses,
                'tab_type' => 'record',
                'userId' => $user['id'],
            ]
        );
    }

    public function statisticsOverviewAction(Request $request, $userId)
    {
        if (!$this->hasManageRole($userId)) {
            return $this->createMessageResponse('warning', '您没有权限查看');
        }

        $learnCourses = $this->getCourseService()->findUserLearnCourses($userId);

        $learnCourseIds = ArrayToolkit::column($learnCourses, 'id');
        $learnCourseCount = count($learnCourses);
        $coursesLearnTime = $this->getTaskResultService()->sumLearnTimeByCourseIdsAndUserIdsGroupByUserId($learnCourseIds, [$userId]);

        list($startDateTime, $endDateTime) = DateToolkit::generateStartDateAndEndDate('year', 0, 'date');

        $yearTotalDataConditions = ['startDateTime' => strtotime($startDateTime), 'endDateTime' => strtotime($endDateTime), 'userId' => $userId];
        $yearFinishedCourseNum = $this->getUserDailyLearnRecordService()->countFinishedCourseNumByConditions($yearTotalDataConditions);
        $yearLearnTime = $this->getUserDailyLearnRecordService()->sumLearnTimeByConditions($yearTotalDataConditions);

        $totalDataConditions = ['userId' => $userId];
        $totalFinishedCourseNum = $this->getUserDailyLearnRecordService()->countFinishedCourseNumByConditions($totalDataConditions);
        $totalLearnTime = $this->getUserDailyLearnRecordService()->sumLearnTimeByConditions($totalDataConditions);

        return $this->render(
            'study-record/online-course-record/statistics-overview.html.twig',
            [
                'learnCourseCount' => $learnCourseCount,
                'coursesLearnTime' => empty($coursesLearnTime) ? '' : $coursesLearnTime[0]['totalLearnTime'],
                'totalData' => ['finishedCourseNum' => $totalFinishedCourseNum, 'totalLearnTime' => $totalLearnTime],
                'yearTotalData' => ['finishedCourseNum' => $yearFinishedCourseNum, 'totalLearnTime' => $yearLearnTime],
                'userId' => $userId,
            ]
        );
    }

    protected function hasManageRole($userId)
    {
        $user = $this->getUserService()->getUser($userId);
        if (empty($user)) {
            throw new NotFoundException('User'.$userId.'Not Found');
        }
        $currentUser = $this->getCurrentUser();

        if ($this->getCurrentUser()->getId() == $userId) {
            return true;
        }

        $userManageIds = $this->getCurrentUser()->getManageOrgIdsRecursively();
        $orgIds = array_intersect($user['orgIds'], $userManageIds);

        if (empty($orgIds)) {
            return false;
        }

        if ($this->getCurrentUser()->hasPermission('admin_user_learn_data')) {
            return true;
        }

        $canManage = false;
        $roles = [
            'ROLE_TRAINING_ADMIN',
            'ROLE_DEPARTMENT_ADMIN',
            'ROLE_SUPER_ADMIN',
        ];

        foreach ($roles as $role) {
            if (in_array($role, $currentUser['roles'])) {
                $canManage = true;
                break;
            }
        }

        return $canManage;
    }

    private function findRecentLearnTask($userId, $courseIds)
    {
        $taskResults = $this->getTaskResultService()->searchTaskResults(
            ['userId' => $userId, 'courseIds' => $courseIds],
            ['updatedTime' => 'DESC'],
            0, 1
        );

        $taskResult = array_shift($taskResults);

        if (empty($taskResult)) {
            return [];
        }

        $task = $this->getTaskService()->getTask($taskResult['courseTaskId']);

        return $task;
    }

    private function calculateLearnTime($userId, $courses)
    {
        $totalLearnTime = 0;
        foreach ($courses as $key => &$course) {
            $course['finishedTaskNum'] = $this->getTaskResultService()->countTaskResults([
                'courseId' => $key,
                'userId' => $userId,
                'status' => 'finish',
            ]);

            $course['learnTime'] = $this->getTaskResultService()->sumLearnTimeByCourseIdAndUserId($key, $userId);
            $totalLearnTime += $course['learnTime'];
        }

        return [$courses, $totalLearnTime];
    }

    /**
     * @return PostCourseService
     */
    protected function getPostCourseService()
    {
        return $this->createService('CorporateTrainingBundle:PostCourse:PostCourseService');
    }

    /**
     * @return CourseService
     */
    protected function getCourseService()
    {
        return $this->createService('Course:CourseService');
    }

    /**
     * @return MemberService
     */
    protected function getMemberService()
    {
        return $this->createService('Course:MemberService');
    }

    /**
     * @return TaskService
     */
    protected function getTaskService()
    {
        return $this->createService('Task:TaskService');
    }

    /**
     * @return TaskResultService
     */
    protected function getTaskResultService()
    {
        return $this->createService('Task:TaskResultService');
    }

    protected function getOfflineActivityMemberService()
    {
        return $this->createService('OfflineActivity:MemberService');
    }

    protected function getCategoryService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineActivity:CategoryService');
    }

    protected function getOfflineActivityService()
    {
        return $this->createService('OfflineActivity:OfflineActivityService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineExam\Service\Impl\OfflineExamServiceImpl
     */
    protected function getOfflineExamService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineExam:OfflineExamService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\MemberServiceImpl
     */
    protected function getProjectPlanMemberService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\MemberServiceImpl
     */
    protected function getOfflineCourseMemberService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:MemberService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\ProjectPlan\Service\Impl\ProjectPlanServiceImpl
     */
    protected function getProjectPlanService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:ProjectPlanService');
    }

    protected function getOfflineCourseAttendanceService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:OfflineCourseAttendanceService');
    }

    protected function getOfflineCourseActivityService()
    {
        return $this->createService('CorporateTrainingBundle:Activity:OfflineCourseActivityService');
    }

    protected function getOfflineExamResultService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:OfflineExamResultService');
    }

    protected function getActivityService()
    {
        return $this->createService('Activity:ActivityService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\OfflineCourseServiceImpl
     */
    protected function getOfflineCourseService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:OfflineCourseService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\OfflineCourse\Service\Impl\TaskServiceImpl
     */
    protected function getOfflineCourseTaskService()
    {
        return $this->createService('CorporateTrainingBundle:OfflineCourse:TaskService');
    }

    protected function getOfflineCourseHomeworkService()
    {
        return $this->createService('CorporateTrainingBundle:ProjectPlan:OfflineCourseHomeworkService');
    }

    protected function getExamService()
    {
        return $this->createService('ExamPlugin:Exam:ExamService');
    }

    protected function getTestpaperService()
    {
        return $this->createService('ExamPlugin:TestPaper:TestPaperService');
    }

    protected function getCourseSetService()
    {
        return $this->createService('Course:CourseSetService');
    }

    /**
     * @return EnrollRecordService
     */
    protected function getEnrollRecordService()
    {
        return $this->createService('CorporateTrainingBundle:Enroll:EnrollRecordService');
    }

    /**
     * @return \CorporateTrainingBundle\Biz\DataStatistics\Service\Impl\UserDailyLearnRecordServiceImpl
     */
    protected function getUserDailyLearnRecordService()
    {
        return $this->createService('CorporateTrainingBundle:UserDailyLearnRecord:UserDailyLearnRecordService');
    }

    /**
     * @return OfflineCourseSurveyService
     */
    protected function getOfflineCourseSurveyService()
    {
        return $this->getBiz()->service('CorporateTrainingBundle:OfflineCourse:OfflineCourseSurveyService');
    }

    /**
     * @return SurveyResultService
     */
    protected function getSurveyResultService()
    {
        return $this->createService('SurveyPlugin:Survey:SurveyResultService');
    }
}
