// Copyright (c) Microsoft Corporation.
// Licensed under the MIT license.
import { ChainedTokenCredential } from "./chainedTokenCredential";
import { AzureCliCredential } from "./azureCliCredential";
import { AzurePowerShellCredential } from "./azurePowerShellCredential";
import { EnvironmentCredential } from "./environmentCredential";
import { ManagedIdentityCredential } from "./managedIdentityCredential";
import { VisualStudioCodeCredential } from "./visualStudioCodeCredential";
/**
 * A shim around ManagedIdentityCredential that adapts it to accept
 * `DefaultAzureCredentialOptions`.
 *
 * @internal
 */
export class DefaultManagedIdentityCredential extends ManagedIdentityCredential {
    constructor(options) {
        var _a;
        const managedIdentityClientId = (_a = options === null || options === void 0 ? void 0 : options.managedIdentityClientId) !== null && _a !== void 0 ? _a : process.env.AZURE_CLIENT_ID;
        if (managedIdentityClientId !== undefined) {
            super(managedIdentityClientId, options);
        }
        else {
            super(options);
        }
    }
}
export const defaultCredentials = [
    EnvironmentCredential,
    DefaultManagedIdentityCredential,
    VisualStudioCodeCredential,
    AzureCliCredential,
    AzurePowerShellCredential,
];
/**
 * Provides a default {@link ChainedTokenCredential} configuration that should
 * work for most applications that use the Azure SDK.
 */
export class DefaultAzureCredential extends ChainedTokenCredential {
    /**
     * Creates an instance of the DefaultAzureCredential class.
     *
     * This credential provides a default {@link ChainedTokenCredential} configuration that should
     * work for most applications that use the Azure SDK.
     *
     * The following credential types will be tried, in order:
     *
     * - {@link EnvironmentCredential}
     * - {@link ManagedIdentityCredential}
     * - {@link VisualStudioCodeCredential}
     * - {@link AzureCliCredential}
     * - {@link AzurePowerShellCredential}
     *
     * Consult the documentation of these credential types for more information
     * on how they attempt authentication.
     *
     * **Note**: `VisualStudioCodeCredential` is provided by a plugin package:
     * `@azure/identity-vscode`. If this package is not installed and registered
     * using the plugin API (`useIdentityPlugin`), then authentication using
     * `VisualStudioCodeCredential` will not be available.
     *
     * @param options - Optional parameters. See {@link DefaultAzureCredentialOptions}.
     */
    constructor(options) {
        super(...defaultCredentials.map((ctor) => new ctor(options)));
        this.UnavailableMessage =
            "DefaultAzureCredential => failed to retrieve a token from the included credentials. To troubleshoot, visit https://aka.ms/azsdk/js/identity/defaultazurecredential/troubleshoot.";
    }
}
//# sourceMappingURL=defaultAzureCredential.js.map