import type { OpenAPIV3_1 } from 'openapi-types';
declare type Document = OpenAPIV3_1.Document;
/**
 * OAS Operation Object containing the path and method so it can be placed in a flat array of operations
 *
 * @export
 * @interface Operation
 * @extends {OpenAPIV3_1.OperationObject}
 */
export interface Operation extends OpenAPIV3_1.OperationObject {
    path: string;
    method: string;
}
export interface Request {
    method: string;
    path: string;
    headers: {
        [key: string]: string | string[];
    };
    query?: {
        [key: string]: string | string[];
    } | string;
    body?: any;
}
export interface ParsedRequest extends Request {
    params: {
        [key: string]: string | string[];
    };
    cookies: {
        [key: string]: string | string[];
    };
    query: {
        [key: string]: string | string[];
    };
    requestBody: any;
}
/**
 * Class that handles routing
 *
 * @export
 * @class OpenAPIRouter
 */
export declare class OpenAPIRouter {
    definition: Document;
    apiRoot: string;
    private ignoreTrailingSlashes;
    /**
     * Creates an instance of OpenAPIRouter
     *
     * @param opts - constructor options
     * @param {Document} opts.definition - the OpenAPI definition, file path or Document object
     * @param {string} opts.apiRoot - the root URI of the api. all paths are matched relative to apiRoot
     * @memberof OpenAPIRouter
     */
    constructor(opts: {
        definition: Document;
        apiRoot?: string;
        ignoreTrailingSlashes?: boolean;
    });
    /**
     * Matches a request to an API operation (router)
     *
     * @param {Request} req
     * @param {boolean} [strict] strict mode, throw error if operation is not found
     * @returns {Operation }
     * @memberof OpenAPIRouter
     */
    matchOperation(req: Request): Operation | undefined;
    matchOperation(req: Request, strict: boolean): Operation;
    /**
     * Flattens operations into a simple array of Operation objects easy to work with
     *
     * @returns {Operation[]}
     * @memberof OpenAPIRouter
     */
    getOperations(): Operation[];
    /**
     * Gets a single operation based on operationId
     *
     * @param {string} operationId
     * @returns {Operation}
     * @memberof OpenAPIRouter
     */
    getOperation(operationId: string): Operation | undefined;
    /**
     * Normalises request:
     * - http method to lowercase
     * - remove path leading slash
     * - remove path query string
     *
     * @export
     * @param {Request} req
     * @returns {Request}
     */
    normalizeRequest(req: Request): Request;
    /**
     * Normalises path for matching: strips apiRoot prefix from the path
     *
     * Also depending on configuration, will remove trailing slashes
     *
     * @export
     * @param {string} path
     * @returns {string}
     */
    normalizePath(pathInput: string): string;
    /**
     * Parses and normalizes a request
     * - parse json body
     * - parse query string
     * - parse cookies from headers
     * - parse path params based on uri template
     *
     * @export
     * @param {Request} req
     * @param {string} [patbh]
     * @returns {ParsedRequest}
     */
    parseRequest(req: Request, operation?: Operation): ParsedRequest;
}
export {};
