"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
const MAX = (1 << 16) - 1;
const UNKNOWN_PLP_LEN = Buffer.from([0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]);
const PLP_TERMINATOR = Buffer.from([0x00, 0x00, 0x00, 0x00]);
const NULL_LENGTH = Buffer.from([0xFF, 0xFF]);
const MAX_NULL_LENGTH = Buffer.from([0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const NVarChar = {
  id: 0xE7,
  type: 'NVARCHAR',
  name: 'NVarChar',
  maximumLength: 4000,
  declaration: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.

    let length;

    if (parameter.length) {
      length = parameter.length;
    } else if (value != null) {
      length = value.toString().length || 1;
    } else if (value === null && !parameter.output) {
      length = 1;
    } else {
      length = this.maximumLength;
    }

    if (length <= this.maximumLength) {
      return 'nvarchar(' + length + ')';
    } else {
      return 'nvarchar(max)';
    }
  },
  resolveLength: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.

    if (parameter.length != null) {
      return parameter.length;
    } else if (value != null) {
      if (Buffer.isBuffer(value)) {
        return value.length / 2 || 1;
      } else {
        return value.toString().length || 1;
      }
    } else {
      return this.maximumLength;
    }
  },

  generateTypeInfo(parameter) {
    const buffer = Buffer.alloc(8);
    buffer.writeUInt8(this.id, 0);

    if (parameter.length <= this.maximumLength) {
      buffer.writeUInt16LE(parameter.length * 2, 1);
    } else {
      buffer.writeUInt16LE(MAX, 1);
    }

    if (parameter.collation) {
      parameter.collation.toBuffer().copy(buffer, 3, 0, 5);
    }

    return buffer;
  },

  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      if (parameter.length <= this.maximumLength) {
        return NULL_LENGTH;
      } else {
        return MAX_NULL_LENGTH;
      }
    }

    let value = parameter.value;

    if (parameter.length <= this.maximumLength) {
      let length;

      if (value instanceof Buffer) {
        length = value.length;
      } else {
        value = value.toString();
        length = Buffer.byteLength(value, 'ucs2');
      }

      const buffer = Buffer.alloc(2);
      buffer.writeUInt16LE(length, 0);
      return buffer;
    } else {
      return UNKNOWN_PLP_LEN;
    }
  },

  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }

    let value = parameter.value;

    if (parameter.length <= this.maximumLength) {
      if (value instanceof Buffer) {
        yield value;
      } else {
        value = value.toString();
        yield Buffer.from(value, 'ucs2');
      }
    } else {
      if (value instanceof Buffer) {
        const length = value.length;

        if (length > 0) {
          const buffer = Buffer.alloc(4);
          buffer.writeUInt32LE(length, 0);
          yield buffer;
          yield value;
        }
      } else {
        value = value.toString();
        const length = Buffer.byteLength(value, 'ucs2');

        if (length > 0) {
          const buffer = Buffer.alloc(4);
          buffer.writeUInt32LE(length, 0);
          yield buffer;
          yield Buffer.from(value, 'ucs2');
        }
      }

      yield PLP_TERMINATOR;
    }
  },

  validate: function (value) {
    if (value == null) {
      return null;
    }

    if (typeof value !== 'string') {
      throw new TypeError('Invalid string.');
    }

    return value;
  }
};
var _default = NVarChar;
exports.default = _default;
module.exports = NVarChar;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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