"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Connector = exports.SequentialConnectionStrategy = exports.ParallelConnectionStrategy = void 0;

var _net = _interopRequireDefault(require("net"));

var _dns = _interopRequireDefault(require("dns"));

var punycode = _interopRequireWildcard(require("punycode"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

class AbortError extends Error {
  constructor() {
    super('The operation was aborted');
    this.code = void 0;
    this.code = 'ABORT_ERR';
    this.name = 'AbortError';
  }

}

class ParallelConnectionStrategy {
  constructor(addresses, signal, options) {
    this.addresses = void 0;
    this.options = void 0;
    this.signal = void 0;
    this.addresses = addresses;
    this.options = options;
    this.signal = signal;
  }

  connect(callback) {
    const signal = this.signal;

    if (signal.aborted) {
      return process.nextTick(callback, new AbortError());
    }

    const addresses = this.addresses;
    const sockets = new Array(addresses.length);
    let errorCount = 0;

    function onError(_err) {
      errorCount += 1;
      this.removeListener('error', onError);
      this.removeListener('connect', onConnect);
      this.destroy();

      if (errorCount === addresses.length) {
        signal.removeEventListener('abort', onAbort);
        callback(new Error('Could not connect (parallel)'));
      }
    }

    function onConnect() {
      signal.removeEventListener('abort', onAbort);

      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];

        if (this === socket) {
          continue;
        }

        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }

      callback(null, this);
    }

    const onAbort = () => {
      for (let j = 0; j < sockets.length; j++) {
        const socket = sockets[j];
        socket.removeListener('error', onError);
        socket.removeListener('connect', onConnect);
        socket.destroy();
      }

      callback(new AbortError());
    };

    for (let i = 0, len = addresses.length; i < len; i++) {
      const socket = sockets[i] = _net.default.connect({ ...this.options,
        host: addresses[i].address,
        family: addresses[i].family
      });

      socket.on('error', onError);
      socket.on('connect', onConnect);
    }

    signal.addEventListener('abort', onAbort, {
      once: true
    });
  }

}

exports.ParallelConnectionStrategy = ParallelConnectionStrategy;

class SequentialConnectionStrategy {
  constructor(addresses, signal, options) {
    this.addresses = void 0;
    this.options = void 0;
    this.signal = void 0;
    this.addresses = addresses;
    this.options = options;
    this.signal = signal;
  }

  connect(callback) {
    if (this.signal.aborted) {
      return process.nextTick(callback, new AbortError());
    }

    const next = this.addresses.shift();

    if (!next) {
      return callback(new Error('Could not connect (sequence)'));
    }

    const socket = _net.default.connect({ ...this.options,
      host: next.address,
      family: next.family
    });

    const onAbort = () => {
      socket.removeListener('error', onError);
      socket.removeListener('connect', onConnect);
      socket.destroy();
      callback(new AbortError());
    };

    const onError = _err => {
      this.signal.removeEventListener('abort', onAbort);
      socket.removeListener('error', onError);
      socket.removeListener('connect', onConnect);
      socket.destroy();
      this.connect(callback);
    };

    const onConnect = () => {
      this.signal.removeEventListener('abort', onAbort);
      socket.removeListener('error', onError);
      socket.removeListener('connect', onConnect);
      callback(null, socket);
    };

    this.signal.addEventListener('abort', onAbort, {
      once: true
    });
    socket.on('error', onError);
    socket.on('connect', onConnect);
  }

}

exports.SequentialConnectionStrategy = SequentialConnectionStrategy;

class Connector {
  constructor(options, signal, multiSubnetFailover) {
    var _options$lookup;

    this.options = void 0;
    this.multiSubnetFailover = void 0;
    this.lookup = void 0;
    this.signal = void 0;
    this.options = options;
    this.lookup = (_options$lookup = options.lookup) !== null && _options$lookup !== void 0 ? _options$lookup : _dns.default.lookup;
    this.signal = signal;
    this.multiSubnetFailover = multiSubnetFailover;
  }

  execute(cb) {
    if (this.signal.aborted) {
      return process.nextTick(cb, new AbortError());
    }

    this.lookupAllAddresses(this.options.host, (err, addresses) => {
      if (this.signal.aborted) {
        return cb(new AbortError());
      }

      if (err) {
        return cb(err);
      }

      if (this.multiSubnetFailover) {
        new ParallelConnectionStrategy(addresses, this.signal, this.options).connect(cb);
      } else {
        new SequentialConnectionStrategy(addresses, this.signal, this.options).connect(cb);
      }
    });
  }

  lookupAllAddresses(host, callback) {
    if (_net.default.isIPv6(host)) {
      process.nextTick(callback, null, [{
        address: host,
        family: 6
      }]);
    } else if (_net.default.isIPv4(host)) {
      process.nextTick(callback, null, [{
        address: host,
        family: 4
      }]);
    } else {
      this.lookup.call(null, punycode.toASCII(host), {
        all: true
      }, callback);
    }
  }

}

exports.Connector = Connector;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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