"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _jsbi = _interopRequireDefault(require("jsbi"));

var _token = require("./token");

var _colmetadataTokenParser = _interopRequireDefault(require("./colmetadata-token-parser"));

var _doneTokenParser = require("./done-token-parser");

var _envChangeTokenParser = _interopRequireDefault(require("./env-change-token-parser"));

var _infoerrorTokenParser = require("./infoerror-token-parser");

var _fedauthInfoParser = _interopRequireDefault(require("./fedauth-info-parser"));

var _featureExtAckParser = _interopRequireDefault(require("./feature-ext-ack-parser"));

var _loginackTokenParser = _interopRequireDefault(require("./loginack-token-parser"));

var _orderTokenParser = _interopRequireDefault(require("./order-token-parser"));

var _returnstatusTokenParser = _interopRequireDefault(require("./returnstatus-token-parser"));

var _returnvalueTokenParser = _interopRequireDefault(require("./returnvalue-token-parser"));

var _rowTokenParser = _interopRequireDefault(require("./row-token-parser"));

var _nbcrowTokenParser = _interopRequireDefault(require("./nbcrow-token-parser"));

var _sspiTokenParser = _interopRequireDefault(require("./sspi-token-parser"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const tokenParsers = {
  [_token.TYPE.DONE]: _doneTokenParser.doneParser,
  [_token.TYPE.DONEINPROC]: _doneTokenParser.doneInProcParser,
  [_token.TYPE.DONEPROC]: _doneTokenParser.doneProcParser,
  [_token.TYPE.ENVCHANGE]: _envChangeTokenParser.default,
  [_token.TYPE.ERROR]: _infoerrorTokenParser.errorParser,
  [_token.TYPE.FEDAUTHINFO]: _fedauthInfoParser.default,
  [_token.TYPE.FEATUREEXTACK]: _featureExtAckParser.default,
  [_token.TYPE.INFO]: _infoerrorTokenParser.infoParser,
  [_token.TYPE.LOGINACK]: _loginackTokenParser.default,
  [_token.TYPE.ORDER]: _orderTokenParser.default,
  [_token.TYPE.RETURNSTATUS]: _returnstatusTokenParser.default,
  [_token.TYPE.RETURNVALUE]: _returnvalueTokenParser.default,
  [_token.TYPE.SSPI]: _sspiTokenParser.default
};

class StreamBuffer {
  constructor(iterable) {
    this.iterator = void 0;
    this.buffer = void 0;
    this.position = void 0;
    this.iterator = (iterable[Symbol.asyncIterator] || iterable[Symbol.iterator]).call(iterable);
    this.buffer = Buffer.alloc(0);
    this.position = 0;
  }

  async waitForChunk() {
    const result = await this.iterator.next();

    if (result.done) {
      throw new Error('unexpected end of data');
    }

    if (this.position === this.buffer.length) {
      this.buffer = result.value;
    } else {
      this.buffer = Buffer.concat([this.buffer.slice(this.position), result.value]);
    }

    this.position = 0;
  }

}

class Parser {
  static async *parseTokens(iterable, debug, options, colMetadata = []) {
    let token;

    const onDoneParsing = t => {
      token = t;
    };

    const streamBuffer = new StreamBuffer(iterable);
    const parser = new Parser(streamBuffer, debug, options);
    parser.colMetadata = colMetadata;

    while (true) {
      try {
        await streamBuffer.waitForChunk();
      } catch (err) {
        if (streamBuffer.position === streamBuffer.buffer.length) {
          return;
        }

        throw err;
      }

      if (parser.suspended) {
        // Unsuspend and continue from where ever we left off.
        parser.suspended = false;
        const next = parser.next;
        next(); // Check if a new token was parsed after unsuspension.

        if (!parser.suspended && token) {
          if (token instanceof _token.ColMetadataToken) {
            parser.colMetadata = token.columns;
          }

          yield token;
        }
      }

      while (!parser.suspended && parser.position + 1 <= parser.buffer.length) {
        const type = parser.buffer.readUInt8(parser.position);
        parser.position += 1;

        if (type === _token.TYPE.COLMETADATA) {
          const token = await (0, _colmetadataTokenParser.default)(parser);
          parser.colMetadata = token.columns;
          yield token;
        } else if (type === _token.TYPE.ROW) {
          yield (0, _rowTokenParser.default)(parser);
        } else if (type === _token.TYPE.NBCROW) {
          yield (0, _nbcrowTokenParser.default)(parser);
        } else if (tokenParsers[type]) {
          tokenParsers[type](parser, parser.options, onDoneParsing); // Check if a new token was parsed after unsuspension.

          if (!parser.suspended && token) {
            if (token instanceof _token.ColMetadataToken) {
              parser.colMetadata = token.columns;
            }

            yield token;
          }
        } else {
          throw new Error('Unknown type: ' + type);
        }
      }
    }
  }

  constructor(streamBuffer, debug, options) {
    this.debug = void 0;
    this.colMetadata = void 0;
    this.options = void 0;
    this.suspended = void 0;
    this.next = void 0;
    this.streamBuffer = void 0;
    this.debug = debug;
    this.colMetadata = [];
    this.options = options;
    this.streamBuffer = streamBuffer;
    this.suspended = false;
    this.next = undefined;
  }

  get buffer() {
    return this.streamBuffer.buffer;
  }

  get position() {
    return this.streamBuffer.position;
  }

  set position(value) {
    this.streamBuffer.position = value;
  }

  suspend(next) {
    this.suspended = true;
    this.next = next;
  }

  awaitData(length, callback) {
    if (this.position + length <= this.buffer.length) {
      callback();
    } else {
      this.suspend(() => {
        this.awaitData(length, callback);
      });
    }
  }

  readInt8(callback) {
    this.awaitData(1, () => {
      const data = this.buffer.readInt8(this.position);
      this.position += 1;
      callback(data);
    });
  }

  readUInt8(callback) {
    this.awaitData(1, () => {
      const data = this.buffer.readUInt8(this.position);
      this.position += 1;
      callback(data);
    });
  }

  readInt16LE(callback) {
    this.awaitData(2, () => {
      const data = this.buffer.readInt16LE(this.position);
      this.position += 2;
      callback(data);
    });
  }

  readInt16BE(callback) {
    this.awaitData(2, () => {
      const data = this.buffer.readInt16BE(this.position);
      this.position += 2;
      callback(data);
    });
  }

  readUInt16LE(callback) {
    this.awaitData(2, () => {
      const data = this.buffer.readUInt16LE(this.position);
      this.position += 2;
      callback(data);
    });
  }

  readUInt16BE(callback) {
    this.awaitData(2, () => {
      const data = this.buffer.readUInt16BE(this.position);
      this.position += 2;
      callback(data);
    });
  }

  readInt32LE(callback) {
    this.awaitData(4, () => {
      const data = this.buffer.readInt32LE(this.position);
      this.position += 4;
      callback(data);
    });
  }

  readInt32BE(callback) {
    this.awaitData(4, () => {
      const data = this.buffer.readInt32BE(this.position);
      this.position += 4;
      callback(data);
    });
  }

  readUInt32LE(callback) {
    this.awaitData(4, () => {
      const data = this.buffer.readUInt32LE(this.position);
      this.position += 4;
      callback(data);
    });
  }

  readUInt32BE(callback) {
    this.awaitData(4, () => {
      const data = this.buffer.readUInt32BE(this.position);
      this.position += 4;
      callback(data);
    });
  }

  readBigInt64LE(callback) {
    this.awaitData(8, () => {
      const result = _jsbi.default.add(_jsbi.default.leftShift(_jsbi.default.BigInt(this.buffer[this.position + 4] + this.buffer[this.position + 5] * 2 ** 8 + this.buffer[this.position + 6] * 2 ** 16 + (this.buffer[this.position + 7] << 24) // Overflow
      ), _jsbi.default.BigInt(32)), _jsbi.default.BigInt(this.buffer[this.position] + this.buffer[this.position + 1] * 2 ** 8 + this.buffer[this.position + 2] * 2 ** 16 + this.buffer[this.position + 3] * 2 ** 24));

      this.position += 8;
      callback(result);
    });
  }

  readInt64LE(callback) {
    this.awaitData(8, () => {
      const data = Math.pow(2, 32) * this.buffer.readInt32LE(this.position + 4) + ((this.buffer[this.position + 4] & 0x80) === 0x80 ? 1 : -1) * this.buffer.readUInt32LE(this.position);
      this.position += 8;
      callback(data);
    });
  }

  readInt64BE(callback) {
    this.awaitData(8, () => {
      const data = Math.pow(2, 32) * this.buffer.readInt32BE(this.position) + ((this.buffer[this.position] & 0x80) === 0x80 ? 1 : -1) * this.buffer.readUInt32BE(this.position + 4);
      this.position += 8;
      callback(data);
    });
  }

  readBigUInt64LE(callback) {
    this.awaitData(8, () => {
      const low = _jsbi.default.BigInt(this.buffer.readUInt32LE(this.position));

      const high = _jsbi.default.BigInt(this.buffer.readUInt32LE(this.position + 4));

      this.position += 8;
      callback(_jsbi.default.add(low, _jsbi.default.leftShift(high, _jsbi.default.BigInt(32))));
    });
  }

  readUInt64LE(callback) {
    this.awaitData(8, () => {
      const data = Math.pow(2, 32) * this.buffer.readUInt32LE(this.position + 4) + this.buffer.readUInt32LE(this.position);
      this.position += 8;
      callback(data);
    });
  }

  readUInt64BE(callback) {
    this.awaitData(8, () => {
      const data = Math.pow(2, 32) * this.buffer.readUInt32BE(this.position) + this.buffer.readUInt32BE(this.position + 4);
      this.position += 8;
      callback(data);
    });
  }

  readFloatLE(callback) {
    this.awaitData(4, () => {
      const data = this.buffer.readFloatLE(this.position);
      this.position += 4;
      callback(data);
    });
  }

  readFloatBE(callback) {
    this.awaitData(4, () => {
      const data = this.buffer.readFloatBE(this.position);
      this.position += 4;
      callback(data);
    });
  }

  readDoubleLE(callback) {
    this.awaitData(8, () => {
      const data = this.buffer.readDoubleLE(this.position);
      this.position += 8;
      callback(data);
    });
  }

  readDoubleBE(callback) {
    this.awaitData(8, () => {
      const data = this.buffer.readDoubleBE(this.position);
      this.position += 8;
      callback(data);
    });
  }

  readUInt24LE(callback) {
    this.awaitData(3, () => {
      const low = this.buffer.readUInt16LE(this.position);
      const high = this.buffer.readUInt8(this.position + 2);
      this.position += 3;
      callback(low | high << 16);
    });
  }

  readUInt40LE(callback) {
    this.awaitData(5, () => {
      const low = this.buffer.readUInt32LE(this.position);
      const high = this.buffer.readUInt8(this.position + 4);
      this.position += 5;
      callback(0x100000000 * high + low);
    });
  }

  readUNumeric64LE(callback) {
    this.awaitData(8, () => {
      const low = this.buffer.readUInt32LE(this.position);
      const high = this.buffer.readUInt32LE(this.position + 4);
      this.position += 8;
      callback(0x100000000 * high + low);
    });
  }

  readUNumeric96LE(callback) {
    this.awaitData(12, () => {
      const dword1 = this.buffer.readUInt32LE(this.position);
      const dword2 = this.buffer.readUInt32LE(this.position + 4);
      const dword3 = this.buffer.readUInt32LE(this.position + 8);
      this.position += 12;
      callback(dword1 + 0x100000000 * dword2 + 0x100000000 * 0x100000000 * dword3);
    });
  }

  readUNumeric128LE(callback) {
    this.awaitData(16, () => {
      const dword1 = this.buffer.readUInt32LE(this.position);
      const dword2 = this.buffer.readUInt32LE(this.position + 4);
      const dword3 = this.buffer.readUInt32LE(this.position + 8);
      const dword4 = this.buffer.readUInt32LE(this.position + 12);
      this.position += 16;
      callback(dword1 + 0x100000000 * dword2 + 0x100000000 * 0x100000000 * dword3 + 0x100000000 * 0x100000000 * 0x100000000 * dword4);
    });
  } // Variable length data


  readBuffer(length, callback) {
    this.awaitData(length, () => {
      const data = this.buffer.slice(this.position, this.position + length);
      this.position += length;
      callback(data);
    });
  } // Read a Unicode String (BVARCHAR)


  readBVarChar(callback) {
    this.readUInt8(length => {
      this.readBuffer(length * 2, data => {
        callback(data.toString('ucs2'));
      });
    });
  } // Read a Unicode String (USVARCHAR)


  readUsVarChar(callback) {
    this.readUInt16LE(length => {
      this.readBuffer(length * 2, data => {
        callback(data.toString('ucs2'));
      });
    });
  } // Read binary data (BVARBYTE)


  readBVarByte(callback) {
    this.readUInt8(length => {
      this.readBuffer(length, callback);
    });
  } // Read binary data (USVARBYTE)


  readUsVarByte(callback) {
    this.readUInt16LE(length => {
      this.readBuffer(length, callback);
    });
  }

}

var _default = Parser;
exports.default = _default;
module.exports = Parser;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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