#include <iostream>
#include <cstdint>
#include <string>
#include <fstream>
#include <vector>
#include <cstring>

using namespace std;

const unsigned char ident[] = {0x44, 0x53, 0x41, 0x52, 0x43, 0x49, 0x44, 0x58};

struct toc_entry {
    string name;
    uint32_t start;
    uint32_t file_size;
    void *data;
};

vector<toc_entry> get_toc(ifstream& file);
void read_file(toc_entry& entry);
int next_pos(int cur);
void fix_toc(vector<toc_entry>& toc);

int main(int argc, char *argv[])
{
    if(argc < 3) {
        cout << "cgio tpk archive tool\n\n"
             << "usage:\n"
             << "cgio_tpk_arctool unpack archive.tpk\n"
             << "cgio_tpk_arctool repack archive.tpk\n";
        return(0);
    }

    string command(argv[1]);
    if(command != "unpack" && command != "repack") {
        cout << "invalid command\n";
        return(0);
    }

    if(command == "unpack") {
        ifstream in(argv[2], ifstream::in | ifstream::binary);
        if(!in.is_open()) {
            cout << "unable to open archive\n";
            return 0;
        }

        vector<toc_entry> toc = get_toc(in);

        for(vector<toc_entry>::iterator it = toc.begin(); it != toc.end(); it++) {
            ofstream out(it->name.c_str(), ofstream::out | ofstream::binary);
            if(!out.is_open()) {
                cout << "Unable to create " << it->name << "\n";
                return 0;
            }

            uint8_t *buffer = new uint8_t[it->file_size];

            in.seekg(it->start);
            in.read((char*)buffer, it->file_size);
            out.write((char*)buffer, it->file_size);

            delete [] buffer;
            out.close();

            cout << "Extracted: " << it->name << "\n";
        }
        in.close();
    }
    else {
        ifstream in(argv[2], ifstream::in | ifstream::binary);
        if(!in.is_open()) {
            cout << "unable to open archive\n";
            return 0;
        }

        vector<toc_entry> toc = get_toc(in);
        in.close();

        for(vector<toc_entry>::iterator it = toc.begin(); it != toc.end(); it++) {
            read_file(*it);
            if(it->data == 0) {
                cout << "Failed to read file: " << it->name << "\n";
                return 0;
            }
        }

        fix_toc(toc);

        ofstream out(argv[2], ofstream::out | ofstream::binary | ofstream::trunc);
        if(!out.is_open()) {
            cout << "unable to open archive for writing\n";
            return 0;
        }

        out.write((char*)ident, 8);

        uint32_t num_files = toc.size();
        out.write((char*)&num_files, sizeof(num_files));

        out.put(0);
        out.put(0);
        out.put(0);
        out.put(0);
        out.put(0);
        out.put(0);
        out.put(1);
        out.put(0);

        for(vector<toc_entry>::iterator it = toc.begin(); it != toc.end(); it++) {
            uint8_t *name_buffer = new uint8_t[0x28];
            memset(name_buffer, 0, 0x28);
            memcpy(name_buffer, it->name.c_str(), it->name.size());
            out.write((char*)name_buffer, 0x28);
            delete [] name_buffer;

            out.write((char*)&(it->file_size), 4);
            out.write((char*)&(it->start), 4);
        }

        int toc_padding_size = 0x0204 - (0x30 * toc.size()) - 0x14;
        uint8_t *toc_padding = new uint8_t[toc_padding_size];
        memset(toc_padding, 0, toc_padding_size);
        out.write((char*)toc_padding, toc_padding_size);
        delete [] toc_padding;

        uint32_t pos = 0x204;

        for(vector<toc_entry>::iterator it = toc.begin(); it != toc.end(); it++) {
            out.write((char*)it->data, it->file_size);
            delete [] it->data;
            it->data = NULL;
            pos += it->file_size;

            int padding_size = next_pos(pos) - pos;
            if(padding_size != 0) {
                uint8_t *padding = new uint8_t[padding_size];
                memset(padding, 0, padding_size);
                out.write((char*)padding, padding_size);
                delete [] padding;
                pos += padding_size;
            }
        }

        int padding_size = next_pos(pos) - pos;
        if(padding_size != 0) {
            uint8_t *padding = new uint8_t[padding_size];
            memset(padding, 0, padding_size);
            out.write((char*)padding, padding_size);
            delete [] padding;
            pos += padding_size;
        }

        cout << "Done\n";

        out.close();
    }



    return 0;
}

vector<toc_entry> get_toc(ifstream& file) {
    vector<toc_entry> toc;

    file.seekg(0x08);
    uint32_t num_files;
    file.read((char*)&num_files, sizeof(num_files));

    for(int i = 0; i < num_files; i++) {
        file.seekg((0x30 * i) + 0x14);

        uint8_t *name_buffer = new uint8_t[0x28 + 1];
        name_buffer[0x28] = 0;
        file.read((char*)name_buffer, 0x28);

        uint32_t file_size;
        file.read((char*)&file_size, sizeof(file_size));

        uint32_t start;
        file.read((char*)&start, sizeof(start));

        toc_entry entry;
        entry.name = string((char*)name_buffer);
        entry.start = start;
        entry.file_size = file_size;
        entry.data = 0;

        toc.push_back(entry);
    }

    return toc;
}

void read_file(toc_entry& entry) {
    ifstream in(entry.name, ifstream::in | ifstream::binary);
    if(!in.is_open()) {
        return;
    }

    in.seekg(0, ios_base::end);
    int file_size = in.tellg();

    in.seekg(0);
    uint8_t *buffer = new uint8_t[file_size];
    in.read((char*)buffer, file_size);

    entry.file_size = file_size;
    entry.data = buffer;

    in.close();
}

int next_pos(int cur) {
    int m4 = cur - 4;
    int new_base = (m4 / 0x200) * 0x200;
    int new_pos = new_base + 4;
    if(new_pos < cur) {
        new_pos += 0x200;
    }
    return new_pos;
}

void fix_toc(vector<toc_entry>& toc) {
    uint32_t pos = 0x204;
    for(vector<toc_entry>::iterator it = toc.begin(); it != toc.end(); it++) {
        it->start = pos;
        pos += it->file_size;
        pos = next_pos(pos);
    }
}
