/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw;

import net.sf.jaxodraw.gui.launch.JaxoLauncher;
import net.sf.jaxodraw.gui.launch.JaxoShutDown;
import net.sf.jaxodraw.io.JaxoCLIConvert;
import net.sf.jaxodraw.util.JaxoPrefs;
import net.sf.jaxodraw.util.JaxoInfo;
import net.sf.jaxodraw.util.JaxoDictionary;
import net.sf.jaxodraw.util.JaxoLog;


/**
 * Starts JaxoDraw from the command line.
 *
 * @since 1.0
 */
public final class JaxoDraw {
    private static final JaxoDictionary LANGUAGE;

    /** Empty private constructor: do not instantiate. */
    private JaxoDraw() {
        // empty on purpose
    }

    static {
        // Preferences are initialized first so that:
        //  * command line arguments may override them
        //  * command line logs are already localized
        JaxoPrefs.initSessionPrefs();
        LANGUAGE = new JaxoDictionary(JaxoDraw.class);
    }

    /**
     * The method that executes JaxoDraw from the command line:
     * reads preferences and command line arguments,
     * before bringing up the graphical user interface.
     *
     * @param args The array of command line arguments.
     */
    public static void main(final String[] args) {

        final String[] fileNames = getCommandLineArgs(args);

        final JaxoLauncher launcher = new JaxoLauncher();
        launcher.setFilesToOpen(fileNames);
        launcher.setShutDownHook(new JaxoShutDown() {
                    public void shutdown(final int status) {
                        JaxoDraw.shutdown(status);
                    }
                });

        launcher.launch(JaxoPrefs.getBooleanPref(JaxoPrefs.PREF_SHOWSPLASHWINDOW));
    }

    /**
     * Processes the command line arguments and returns an
     * array of file names to be opened.
     *
     * @param args The array of command line parameters.
     * as passed from the main method.
     * @return An array of files to be opened.
     */
    private static String[] getCommandLineArgs(final String[] args) {
        boolean shutdown = false;
        boolean convert = false;

        int files = 0;
        String[] filesToOpen = new String[args.length];

        for (int i = 0; i < args.length; i++) {
            if (args[i].startsWith("--")) {
                if (args[i].equals("--version")) {
                    JaxoLog.info(JaxoInfo.VERSION);
                } else if (args[i].equals("--info")) {
                    JaxoLog.info(JaxoInfo.info());
                } else if (args[i].equals("--help")) {
                    JaxoLog.info(JaxoInfo.help());
                } else if (args[i].equals("--convert")) {
                    convert = true;
                } else {
                    JaxoLog.warn(LANGUAGE.message(
                        "unknownOption%0", args[i]));
                }
                shutdown = true;
            } else if (args[i].endsWith(JaxoInfo.DOT_EXTENSION)
                    || args[i].endsWith(".tex")) {
                filesToOpen[files] = args[i];
                files++;
            } else if (args[i].equals("-verbose") || args[i].equals("-debug")) {
                JaxoLog.setLevel(JaxoLog.DEBUG);
            } else if (args[i].equals("-quiet")) {
                JaxoLog.setLevel(JaxoLog.WARN);
            } else if (args[i].equals("-nosplash")) {
                JaxoPrefs.setBooleanPref(JaxoPrefs.PREF_SHOWSPLASHWINDOW, false);
            } else {
                JaxoLog.warn(LANGUAGE.message(
                    "unknownOption%0", args[i]));
            }
        }

        if (shutdown && !convert) {
            shutdown(0);
        }

        final String[] returnFiles = new String[files];
        System.arraycopy(filesToOpen, 0, returnFiles, 0, files);

        if (convert) {
            JaxoCLIConvert.convert(returnFiles);
            shutdown(0);
        }

        return returnFiles;
    }

    /**
     * Calls System.exit(status).
     *
     * @param status The exit status code.
     */
    private static void shutdown(final int status) {
        if (status != 0) {
            JaxoLog.warn("JaxoDraw terminated with exit status: " + status);
        }

        System.exit(status);
    }

}
