/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */

package net.sf.jaxodraw.gui.grid;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Transparency;

import javax.swing.event.ChangeListener;


/**
 * A grid that can be painted.
 *
 * @since 2.1
 */
public interface JaxoPaintableGrid extends JaxoGrid, Transparency {

    // Type

    /** Rectangular type. */
    int TYPE_RECTANGULAR = 0;

    /** Hexagonal type. */
    int TYPE_HEXAGONAL = 1;

    // Styles

    /** Dot style. */
    int STYLE_DOT = 0;

    /** Cross style. */
    int STYLE_CROSS = 1;

    /** Line style. */
    int STYLE_LINE = 2;

    /** Honeycomb style. */
    int STYLE_LINE_HONEYCOMB = 3;

    /**
     * ChangeEvents will be fired everytime the actual image of
     * the grid changes (not on canvasSize changes).
     *
     * @param l The listener to add.
     */
    void addChangeListener(ChangeListener l);

    /**
     * Removes a change listener.
     *
     * @param l The listener to remove.
     */
    void removeChangeListener(ChangeListener l);

    /**
     * Paints the grid.
     *
     * @param g The graphics context to paint the grid.
     */
    void paint(Graphics2D g);

    /**
     * Size of the grid canvas.
     * This is the size of the area on which the grid will be painted.
     *
     * @return The size of the grid canvas.
     */
    Dimension getCanvasSize();

    /**
     * Sets the size of the grid canvas.
     * This is the size of the area on which the grid will be painted.
     *
     * @param value The size to set.
     */
    void setCanvasSize(Dimension value);

    /**
     * Sets the size of the grid to the given value.
     * This is the distance between two grid points.
     * For a hexagonal grid, the grid size is always forced to be even.
     *
     * @param value The grid size to be set.
     */
    void setGridSize(int value);

    /**
     * Returns the current size of the grid.
     * This is the distance between two grid points.
     *
     * @return The grid size.
     */
    int getGridSize();

    /**
     * Sets the type of the grid (TYPE_XYZ).
     *
     * @param value The gridtype to be set.
     */
    void setGridType(int value);

    /**
     * Returns the type of the grid.
     *
     * @return The gridtype (TYPE_XYZ).
     */
    int getGridType();


    /**
     * Grid style (STYLE_XYZ constant). The default is STYLE_DOT.
     *
     * @return The grid style.
     */
    int getGridStyle();


    /**
     * Set style, and repaint.
     *
     * @param value The style to set.
     */
    void setGridStyle(int value);

    /**
     * Color of the grid.
     *
     * @return The color of the grid.
     */
    Color getGridColor();

    /**
     * Sets the color of the grid points.
     *
     * @param value The color to set.
     */
    void setGridColor(Color value);

    /**
     * Determine if this grid gets painted. By default, a grid is painted.
     *
     * @return true if the grid is switched on, false otherwise.
     */
    boolean isPainted();

    /**
     * Determine if this grid gets painted.
     *
     * @param painted true if this grid should get painted, false otherwise.
     */
    void setPainted(boolean painted);

    /**
     * Check if points are currently snapped to the grid. By default snapping is off.
     *
     * @return true if this grid snaps points, false otherwise.
     */
    boolean isSnapping();

    /**
     * Switch on/off the snapping of this grid.
     *
     * @param snapping true if this grid should snap points, false otherwise.
     */
    void setSnapping(boolean snapping);
}
