/**
 *  Licensed under GPL. For more information, see
 *    http://jaxodraw.sourceforge.net/license.html
 *  or the LICENSE file in the jaxodraw distribution.
 */
package net.sf.jaxodraw.gui.menu;

import java.beans.PropertyChangeEvent;

import javax.swing.ButtonGroup;
import javax.swing.JMenuItem;
import javax.swing.JRadioButtonMenuItem;

import net.sf.jaxodraw.util.JaxoPrefs;
import net.sf.jaxodraw.util.JaxoConstants;
import net.sf.jaxodraw.util.JaxoLanguage;

/**
 * A JaxoMenu for default mode MenuItems.
 */
public class DefaultMenu extends JaxoMenu {
    private static final long serialVersionUID = 7526471155622776147L;

    private static final int NOF_DEFAULTS =
        JaxoConstants.defaultModes().length;
    private final JMenuItem[] defMIs =
            new JMenuItem[NOF_DEFAULTS];

    /**
     * Creates a new instance of DefaultMenu.
     *
     * @param initValue the initial value.
     */
    public DefaultMenu(final int initValue) {
        super();
        final ButtonGroup g = new ButtonGroup();

        for (int i = 0; i < NOF_DEFAULTS; i++) {
            defMIs[i] = new JRadioButtonMenuItem(JaxoLanguage.translate(
                    JaxoConstants.defaultActionKeys()[i]));
            g.add(defMIs[i]);
            JaxoMenuUtils.setActionCommand(defMIs[i], JaxoConstants.defaultModes()[i]);
            this.add(defMIs[i]);

            if ((i == getDefIndex(JaxoConstants.DEF_NONE))
                    || (i == getDefIndex(JaxoConstants.DEF_LATEX))) {
                this.addSeparator();
            }
        }
        updateDefaultAction(initValue);
        updateLanguage();
    }

    /**
     * Applies a property change event.
     *
     * @param e the change event.
     */
    public void propertyChange(final PropertyChangeEvent e) {
        final String name = e.getPropertyName();
        if ("Jaxo.defaultAction".equals(name)) {
            updateDefaultAction(((Integer) e.getNewValue()).intValue());
        }
    }

     /** {@inheritDoc} */
    @Override
    public final void updateLanguage() {
        final int j = getDefIndex(JaxoPrefs.getIntPref(JaxoPrefs.PREF_DEFAULTACTION));
        this.setText(JaxoLanguage.message(
                "JaxoOptionsMenu.defaultAction%0",
                JaxoLanguage.translate(JaxoConstants.defaultActionKeys()[j])));

        for (int i = 0; i < NOF_DEFAULTS; i++) {
            defMIs[i].setText(JaxoLanguage.translate(
                    JaxoConstants.defaultActionKeys()[i]));
        }
    }

      //
     // private methods
    //

       /** Notification on change of default mode.
        Sets the label of the default entry to the selected one.
     * @param value Specifies the current default action mode.
     */
    private void updateDefaultAction(final int value) {
        int newValue = value;
        if (!JaxoConstants.isDefaultMode(value)) {
            newValue = JaxoConstants.DEF_NONE;
        }

        final int i = getDefIndex(newValue);

        this.setText(JaxoLanguage.message(
                "JaxoOptionsMenu.defaultAction%0",
                JaxoLanguage.translate(JaxoConstants.defaultActionKeys()[i])));
        defMIs[i].setSelected(true);
    }

    private int getDefIndex(final int value) {
        int j = NOF_DEFAULTS + 1;

        for (int jj = 0; jj < NOF_DEFAULTS; jj++) {
            if (value == JaxoConstants.defaultModes()[jj]) {
                j = jj;
                break;
            }
        }

        return j;
    }
}
